/*
 *  Copyright 2001-2005 Internet2
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* internal.h - internally visible classes

   Scott Cantor
   9/6/02

   $History:$
*/

#ifndef __saml_internal_h__
#define __saml_internal_h__

#ifdef WIN32
# define SAML_EXPORTS __declspec(dllexport)
#endif

#include <xsec/framework/XSECProvider.hpp>
#include <log4cpp/Category.hh>

// eventually we might be able to support autoconf via cygwin...
#if defined (_MSC_VER) || defined(__BORLANDC__)
# include "config_win32.h"
#else
# include "config.h"
#endif

#include "saml.h"

#define SAML_LOGCAT "SAML"
#define SAML_log (*reinterpret_cast<log4cpp::Category*>(m_log))

#define DEFAULT_REPLAYCACHE_PROVIDER "org.opensaml.provider.MemoryReplayCacheProvider"
#define DEFAULT_BROWSERPROFILE_PROVIDER "org.opensaml.provider.BrowserProfileProvider"
#define DEFAULT_SOAPBINDING_PROVIDER "org.opensaml.provider.SOAPHTTPBindingProvider"

#define DECLARE_DEF_NAMESPACE(element,ns) \
    if (!element->hasAttributeNS(XML::XMLNS_NS,L(xmlns))) \
        element->setAttributeNS(XML::XMLNS_NS,L(xmlns),ns)
#define DECLARE_NAMESPACE(element,prefix,ns) \
    if (!element->hasAttributeNS(XML::XMLNS_NS,L(prefix))) \
        element->setAttributeNS(XML::XMLNS_NS,L_QNAME(xmlns,prefix),ns)

namespace saml
{
    class SAMLInternalConfig : public SAMLConfig
    {
    public:
        SAMLInternalConfig() : wide_schema_dir(NULL), wide_inclusive_namespace_prefixes(NULL),
            m_lock(NULL), m_pool(NULL), m_compat_pool(NULL), m_log(NULL), m_xsec(NULL) {}

        // global per-process setup and shutdown of runtime
        bool init();
        void term();

        // global mutex available to library applications
        void saml_lock() const;
        void saml_unlock() const;

        // cross-platform extension library loader
        // extensions must provide:
        //      extern "C" int saml_extension_init(void* context);
        //      extern "C" void saml_extension_term();
        void saml_register_extension(const char* path, void* context=NULL) const;

        PlugManager& getPlugMgr() {return m_plugMgr;}
        const char* getDefaultBindingProvider(const XMLCh* binding) const;
        void setDefaultBindingProvider(const XMLCh* binding, const char* type);
        
        // supplement xmlsec so callers are insulated from alg constants
        signatureMethod sigAlgFromURI(const char* algURI) const {
            std::map<std::string,signatureMethod>::const_iterator i=m_sigAlgFromURI.find(algURI);
            return (i!=m_sigAlgFromURI.end()) ? i->second : SIGNATURE_NONE;
        }
        hashMethod digestAlgFromURI(const char* algURI) const {
            std::map<std::string,hashMethod>::const_iterator i=m_digestAlgFromURI.find(algURI);
            return (i!=m_digestAlgFromURI.end()) ? i->second : HASH_NONE;
        }
        
    private:
        friend class XML;
        friend class XML::ParserPool;
        friend class XML::Parser;
        friend class SAMLSignedObject;
        friend class SAMLAssertion;
        friend class SAMLRequest;
        friend class SAMLResponse;

        XMLCh* wide_schema_dir;
        XMLCh* wide_inclusive_namespace_prefixes;
        mutable std::vector<void*> m_libhandles;
        void* m_lock;
        XML::ParserPool* m_pool;
        XML::ParserPool* m_compat_pool;
        log4cpp::Category* m_log;
        XSECProvider* m_xsec;
        PlugManager m_plugMgr;
#ifdef HAVE_GOOD_STL
        std::map<xstring,std::string> m_bindingMap;
#else
        std::map<std::string,std::string> m_bindingMap;
#endif
        // These will get removed once xmlsec has a URI-based API
        std::map<std::string,signatureMethod> m_sigAlgFromURI;
        std::map<std::string,hashMethod> m_digestAlgFromURI;
    };

    void soap_pool_init();
    void soap_pool_term();
}

#endif
