#include <sstream>
#include <iostream>

#include <qcheckbox.h>
#include <qlayout.h>
#include <qlabel.h>
#include <QMainWindow>
#include <qmessagebox.h>
#include <qpoint.h>
#include <QAbstractItemView>
#include <QVBoxLayout>

//#include <ept/configuration/apt.h>
//#include <ept/cache/debtags/tagmap.h>


#include <helpers.h>
#include <exception.h>

// NPlugin
#include <packagenotfoundexception.h>
#include <iprovider.h>

#include "debtagsplugin.h"

#include <ept/debtags/debtags.h>

// NUtil
#include "debtagshelper.h"



#include "selectioninputanddisplay.h"
#include "choosentagsdisplay.h"

// NTagModel
#include "vocabularymodel.h"

typedef std::string Package;
typedef ept::debtags::Tag Tag;

using namespace std;

namespace NPlugin
{

const QString DebtagsPlugin::PLUGIN_NAME = "DebtagsPlugin";


/////////////////////////////////////////////////////
// Constructors/ Destructors
/////////////////////////////////////////////////////

DebtagsPlugin::DebtagsPlugin(const DebtagsPluginContainer& container) :
	_container(container)
{ 
	
	
	_pProvider = 0;
	_pTagSelection = 0;
	_pChooserWidget = 0;
	_isInactive = true;
}

DebtagsPlugin::~DebtagsPlugin() 
{ 
	delete _pChooserWidget;
	delete _pTagSelection;
}

/////////////////////////////////////////////////////
// Plugin Interface
/////////////////////////////////////////////////////

void DebtagsPlugin::init(IProvider* pProvider)
{
	_pProvider = pProvider;

	// this will be garbage collected if the parent is deleted
	_pTagSelection = new NWidgets::SelectionInputAndDisplay(&_container, vocabularyModel(), this);

	// currently disabled as the exclude tag option is not too useful and wastes some space
	// assume that the vocabulary is not accessible if one of the pointers is not set
// 	bool debtagsEnabled = (_container.collection() != 0);
	connect(vocabularyModel(), SIGNAL(dataChanged(const QModelIndex&, const QModelIndex&)), SLOT(onTagSelectionChanged()));
	connect(vocabularyModel(), SIGNAL(modelReset()), SLOT(onTagSelectionChanged()));
	
	if ( _container.collection()==0 )
		setWidgetsEnabled(false);
}


QString DebtagsPlugin::title() const 
{ 
	return QString("Debtags Plugin"); 
}

QString DebtagsPlugin::briefDescription() const 
{ 
	return QString("Offers information and search using the debtags system"); 
}

QString DebtagsPlugin::description() const 
{ 
	return QString("This plugin shows the tags for a program in the detailed view.\n"
		"It also offers searching by tags that can be selected from a list."); 
}

/////////////////////////////////////////////////////
// Search Plugin Interface 
/////////////////////////////////////////////////////

QWidget* DebtagsPlugin::shortInputAndFeedbackWidget() const
{
	return _pTagSelection->tagDisplayWidget();
	
}

QWidget* DebtagsPlugin::inputWidget() const
{
	return _pTagSelection->tagSelectorWidget();
}	

const set<string>& DebtagsPlugin::searchResult() const
{ 
	return _searchResult; 
};

void DebtagsPlugin::clearSearch()
{
	vocabularyModel()->setAllUnselected();
}

/////////////////////////////////////////////////////
// Information Plugin Interface 
/////////////////////////////////////////////////////

QString DebtagsPlugin::informationText(const string& package)
{
	{	// add the tags of the package to the description
		Package packageEntity= NUtil::getPackageByName(package, _pProvider->apt());
		if (packageEntity.empty())
			throw NPlugin::PackageNotFoundException(package);
		set<string> tagset = NUtil::tagsToStrings(_container.collection()->getTagsOfItem(packageEntity));
		if (tagset.empty()) return _emptyString;
		QString detailsString = "<b>Tags:</b> ";
		for (set<string>::iterator it = tagset.begin(); ; )
		{
			detailsString += toQString( *it );
			if ( ++it == tagset.end() )
			{
				detailsString.append("\n");
				break;
			}
			detailsString += ", ";
		}
		return detailsString+"<br>";
	}
}

/////////////////////////////////////////////////////
// Helper Methods
/////////////////////////////////////////////////////

void DebtagsPlugin::showExcludeWidgets(bool )
{
	// TODO reimplement
/*	_pExcludeSelection->setShown(display);
	_pChooserWidget->_pExcludeInputLabel->setShown(display);*/
}


/** This creates a search expression which can be used as input for an ExpressionFiler.
  * @returns the filter string
  */
std::string DebtagsPlugin::createSearchExpression()
{
	ostringstream oexpr;
	// the list view that displays the selected tags
	set<Tag> includeTags = vocabularyModel()->selectedTags();
	bool first = true;
	for (set<Tag>::const_iterator it = includeTags.begin(); it != includeTags.end(); ++it)
	{
		if (!first)
			oexpr << " && ";
		else 
			first = false;

		oexpr << it->fullname();
	}
	// the list view that displays the selected tags
// 	set<Tag> excludeTags = fill in here;
// 	for (set<Tag>::const_iterator it = excludeTags.begin(); it != excludeTags.end(); ++it)
// 	{
// 		if (!first)
// 		{
// 			first = false;
// 			oexpr << " && ";
// 		}
// 		oexpr << "!" << it->fullname();
// 	}
	return oexpr.str();	
}


void DebtagsPlugin::evaluateSearch()
{
	_pProvider->reportBusy(this, tr("Performing full text search on Package Database"));
//	statusBar()->message(tr("Searching Package Database for tags"));
	_searchResult.clear();
	string expression = createSearchExpression();
	qStrDebug("searching for " + toQString(expression));
	if (expression.empty())	// if nothing is selected, do not show anything
	{
		_isInactive = true;
	}
	else
	{
		_isInactive = false;
 		_searchResult = NUtil::packagesToStrings( _container.collection()->getItemsHavingTags(
 			vocabularyModel()->selectedTags()) );
	}
	_pProvider->reportReady(this);
	emit searchChanged(this);
}


void DebtagsPlugin::debtagsDataChanged()
{
	clearSearch();
	if (_container.collection()==0)
		setWidgetsEnabled(false);
	else
	{
// 		_pIncludeSelection->loadVocabulary(_container.facets());
		setWidgetsEnabled(true);
	}
//	clearSearch();
}

void DebtagsPlugin::setWidgetsEnabled(bool enabled)
{
	_pTagSelection->setEnabled(enabled);
}

void DebtagsPlugin::onTagSelectionChanged()
{
	evaluateSearch();
	if ( vocabularyModel()->selectedTags().size() == 0 )
		_pTagSelection->setSelectedTagDisplayShown(false);
	else
		_pTagSelection->setSelectedTagDisplayShown(true);
}


NTagModel::VocabularyModel* DebtagsPlugin::vocabularyModel()
{
	return _container.vocabularyModel();
}

}	// namespace NPlugin

#undef emit
/*
#include <ept/cache/apt/packages.tcc>
#include <ept/cache/debtags/vocabulary.tcc>
#include <ept/cache/tag.tcc>
#include <tagcoll/coll/base.tcc>
*/
