#include <sys/types.h>
#include <stdio.h>
#include <string.h>

#ifdef WIN32
extern "C" {
#endif
#include <regex.h>
#ifdef WIN32
}
#endif

#include "Fraction.h"
#include "Win32PluginAPI.cpp"

#define PREFS_HL_COMPLETE   "FractionPlugin_hl_complete"
#define PREFS_FADE_MIN      "FractionPlugin_fade_min"
#define PREFS_FADE_MID      "FractionPlugin_fade_mid"
#define PREFS_FADE_MAX      "FractionPlugin_fade_max"
#define PREFS_FADE_TRIPLET  "FractionPlugin_fade_triplet" 

static Fraction * fraction;

#define MAJOR "1"
#define MINOR "3"

extern "C" G_MODULE_EXPORT void plugin_init(plugin_address_table_t * pat) {
  plugin_address_table_init(pat);
  fraction = new Fraction();
}

extern "C" G_MODULE_EXPORT void plugin_cleanup(void) {
  delete fraction;
}

extern "C" G_MODULE_EXPORT char * plugin_query_name() {
  return "Fraction";
}

extern "C" G_MODULE_EXPORT char * plugin_query_description() {
  return _("Colours all fractional values depending on their closeness to 0 or 1");
}

extern "C" G_MODULE_EXPORT char * plugin_query_major() {
  return MAJOR;
}

extern "C" G_MODULE_EXPORT char * plugin_query_minor() {
  return MINOR;
}

Fraction::Fraction() {
  pattern = strdup("([0-9]+)/([ ]*)([0-9]+)");
  regcomp(&regexp, pattern, REG_ICASE|REG_EXTENDED);
  version = 1.2;
  name = strdup("Fraction highlighter");
  memset(&prefs_scratch, 0, sizeof(prefs_scratch));

  register_plugin(this, VERSION);
  plugin_handler_add_output_filter(get_plugin_handler(), this);
}

Fraction::~Fraction() {
  // Delete the list in the plugin destructor.
  regfree(&regexp);

  free(pattern);
  pattern = NULL;

  free(name);
  name = NULL;

  unregister_plugin(this);
}

void Fraction::output(Connection * c, char * buf) {

  char * ptr = buf;

  struct fraction_data * data = find_data(c);
  if (!data)
    data = setupConnection(c);

  // Look for pattern in buf.  If we find it, diddle buf so that colour codes occur before the numbers and reset at the end.  Make sure that we reset the pointer to the end of the diddled section so that we don't loop.

  while (1) {
    int nmatch = 4;
    regmatch_t match[4];
    char first[128];
    char second[128];
    char code[128];
    char endcode[128];

    int numerator = 0;
    int denominator = 0;

    int res = regexec(&regexp, ptr, nmatch, match, 0);
    if (res == REG_NOMATCH)
      return; // out should be in decent condition...

    // This is where the first number is found
    strncpy(first, ptr + match[1].rm_so, match[1].rm_eo - match[1].rm_so);
    first[match[1].rm_eo - match[1].rm_so] = '\0';

    // This is where the second number is found
    strncpy(second, ptr + match[3].rm_so, match[3].rm_eo - match[3].rm_so);
    second[match[3].rm_eo - match[3].rm_so] = '\0';

    numerator = atoi(first);
    denominator = atoi(second);

    if (denominator == 0) {
      ptr += match[0].rm_eo; // Set ptr to the end of the matched text.
      ptr++; // Skip over final character of matched text.
      continue;
    }

    if (numerator < 0 || denominator < 0) {
      ptr += match[0].rm_eo; // Set ptr to the end of the matched text.
      ptr++; // Skip over final character of matched text.
      continue;
    }

    if (!data->hl_complete && numerator >= denominator) {
      ptr += match[0].rm_eo; // Set ptr to the end of the matched text.
      ptr++; // Skip over final character of matched text.
      continue;
    }

    // Now need to markup the text for processing by the VT.  This will
    // only work with the Papaya VT.  Ugh.

    //   [<rrr>;<ggg>;<bbb>p  - 0-255 in decimal. RGB values.
    
    GdkColor * shade = fade_get_shade(data->fade, numerator, denominator);

    // Scale the values from Fade::getShade() to 0-255 from 0-65535
    int red   = ((shade->red    * 100 / 65535) * 255) / 100;
    int green = ((shade->green  * 100 / 65535) * 255) / 100;
    int blue  = ((shade->blue   * 100 / 65535) * 255) / 100;

    sprintf(code, "\033[%3.3d;%3.3d;%3.3dp", red, green, blue);
    sprintf(endcode, "\033[q");

    memmove(ptr + match[3].rm_eo + strlen(endcode), ptr + match[3].rm_eo, strlen(ptr + match[3].rm_eo + 1) + 2);
    memcpy(ptr + match[3].rm_eo, endcode, strlen(endcode));

    // We now need to insert this code into buf at ptr + match[1].rm_so

    // First though, we have to move the data at ptr + match[1].rm_so forward
    // by strlen(code) and make sure the terminating nul is still there.

    memmove(ptr + match[1].rm_so + strlen(code), ptr + match[1].rm_so, strlen(ptr + match[1].rm_so) + 1);
    memcpy(ptr + match[1].rm_so, code, strlen(code));

    ptr += strlen(code);
    ptr += match[3].rm_eo; // Set ptr to the end of the matched text.
    ptr++;
  }
}

struct fraction_data * Fraction::setupConnection(Connection * c) {

  struct fraction_data * fdata =
    (struct fraction_data *) g_malloc(sizeof(struct fraction_data));

  resetConnection(connection_get_mud(c), fdata);
  fdata->connection = c;

  add_data(fdata);

  return fdata;
}

void Fraction::resetConnection(MUD * m, struct fraction_data * fdata) {

  char * triplet = NULL;
  char * minc = NULL;
  char * midc = NULL;
  char * maxc = NULL;
  char * hl_complete = NULL;

  if (m) {
    Prefs * currPref = mud_get_preferences(m);
    if (currPref) {
      triplet = preferences_get_preference(currPref, PREFS_FADE_TRIPLET);
      minc = preferences_get_preference(currPref, PREFS_FADE_MIN);
	  midc = preferences_get_preference(currPref, PREFS_FADE_MID);
	  maxc = preferences_get_preference(currPref, PREFS_FADE_MAX);
	  hl_complete = preferences_get_preference(currPref, PREFS_HL_COMPLETE);
    }
  }


  Prefs * globalPrefs = get_global_preferences();

  // If preferences for the MUD are missing fall back to globals
  if (!m || !triplet)
	  triplet = preferences_get_preference(globalPrefs, PREFS_FADE_TRIPLET);

  if (!m || !minc)
	minc = preferences_get_preference(globalPrefs, PREFS_FADE_MIN);

  if (!m || !midc)
	midc = preferences_get_preference(globalPrefs, PREFS_FADE_MID);

  if (!m || !maxc)  
	maxc = preferences_get_preference(globalPrefs, PREFS_FADE_MAX);

  if (!m || !hl_complete)
	hl_complete = preferences_get_preference(globalPrefs, PREFS_HL_COMPLETE);

  if (triplet)
    fdata->fade = fade_new((gboolean) atoi(triplet), minc, midc, maxc);
  else
    fdata->fade = fade_new(FALSE, minc, midc, maxc);

  if (hl_complete)
    fdata->hl_complete = (gboolean) atoi(hl_complete);
  else
    fdata->hl_complete = TRUE;
}

GtkWidget * Fraction::getPrefsWidget(MUD * m) {

  GtkWidget * panel;
  GtkWidget * table;
  GtkWidget * widget;

  resetConnection(m, &prefs_scratch);

  panel = gtk_frame_new (NULL);
  gtk_widget_ref(panel);

#ifdef USE_GTK_2
  g_object_set_data(G_OBJECT(panel), "panel", panel);
#else
  gtk_object_set_data(GTK_OBJECT(panel), "panel", panel);
#endif

  table = gtk_table_new (5, 3, TRUE);
  gtk_widget_show (table);
  gtk_container_add (GTK_CONTAINER (panel), table);

#ifdef USE_GTK_2
  g_object_set_data_full(G_OBJECT(panel), "table", table,
			 (GtkDestroyNotify) gtk_widget_unref);
#else
  gtk_object_set_data_full(GTK_OBJECT(panel), "table", table,
			   (GtkDestroyNotify) gtk_widget_unref);
#endif

  prefs_scratch.hl_c_check = gtk_check_button_new_with_label
                              (_("Highlight complete fractions"));
  gtk_widget_show(prefs_scratch.hl_c_check);
  gtk_table_attach (GTK_TABLE (table), prefs_scratch.hl_c_check, 0, 1, 0, 1,
                    (GtkAttachOptions) 0, (GtkAttachOptions) 0, 0, 0);

  if (prefs_scratch.hl_complete)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(prefs_scratch.hl_c_check), TRUE);

#ifdef USE_GTK_2
  g_object_set_data_full(G_OBJECT(panel), "prefs_scratch.hl_c_check", prefs_scratch.hl_c_check,
			 (GtkDestroyNotify) gtk_widget_unref);
#else
  gtk_object_set_data_full(GTK_OBJECT(panel), "prefs_scratch.hl_c_check", prefs_scratch.hl_c_check,
			   (GtkDestroyNotify) gtk_widget_unref);
#endif


  widget = fade_get_editor(prefs_scratch.fade);
  gtk_widget_show(widget);
  gtk_table_attach (GTK_TABLE (table), widget, 0, 3, 1, 5,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);

#ifdef USE_GTK_2
  g_object_set_data_full(G_OBJECT(panel), "widget", widget,
			 (GtkDestroyNotify) gtk_widget_unref);
#else
  gtk_object_set_data_full(GTK_OBJECT(panel), "widget", widget,
			   (GtkDestroyNotify) gtk_widget_unref);
#endif


  return panel;
}

void Fraction::onPrefsCancel(MUD * m) {
  fade_on_prefs_cancel(prefs_scratch.fade);
}

void Fraction::onPrefsOk(MUD * m) {
  // We only clean up the widgets here - onPrefsApply is always called
  // before an onPrefsOk()
  fade_on_prefs_cancel(prefs_scratch.fade);
}

void Fraction::onPrefsApply(MUD * m) {

  char * maxc;
  char * midc;
  char * minc;
  char * triplet;
  char checkb[10];

  fade_on_prefs_apply(prefs_scratch.fade);

  maxc     = fade_string_max_colour(prefs_scratch.fade);
  midc     = fade_string_mid_colour(prefs_scratch.fade);
  minc     = fade_string_min_colour(prefs_scratch.fade);
  triplet  = fade_string_use_three(prefs_scratch.fade);

  prefs_scratch.hl_complete =
    (GTK_TOGGLE_BUTTON(prefs_scratch.hl_c_check)->active);

  sprintf(checkb, "%d", prefs_scratch.hl_complete);

  Prefs * currPref = NULL;
  if (m)
	currPref = mud_get_preferences(m);

  if (!currPref)
	  currPref = get_global_preferences();

  preferences_set_preference(currPref, PREFS_HL_COMPLETE, checkb);
  preferences_set_preference(currPref, PREFS_FADE_MIN, minc);
  preferences_set_preference(currPref, PREFS_FADE_MID, midc);
  preferences_set_preference(currPref, PREFS_FADE_MAX, maxc);
  preferences_set_preference(currPref, PREFS_FADE_TRIPLET, triplet);

  free(minc);
  free(midc);
  free(maxc);
  free(triplet);

  for (FractionDataList::iterator i = fractionList.begin(); i != fractionList.end(); i++)
    resetConnection(connection_get_mud((*i)->connection), (*i));

}

void Fraction::onEvent(Event * e, Connection * c) {

  struct fraction_data * fdata = find_data(c);

  if (event_get_type(e) == EvConnect) {
    if (!fdata)
      setupConnection(c);
    else
      resetConnection(connection_get_mud(c), fdata);
    return;
  }

  if (event_get_type(e) == EvDisconnect) {
    if (!fdata)
      return;

    // I strongly suggest running new code with dmalloc enabled
    // - that picked this up.
    fade_delete(fdata->fade);
    remove_data(fdata);

    return;
  }
}

static int FractionCmp(struct fraction_data * l1, struct fraction_data *l2) {
  return (l1 < l2);
}

void Fraction::remove_data(struct fraction_data * data) {
  FractionDataList::iterator i = std::lower_bound(fractionList.begin(),
                                               fractionList.end(),
                                               data,
                                               FractionCmp);
  if (i == fractionList.end() || (*i) != data)
    return;

  fractionList.erase(i);
}

struct fraction_data * Fraction::find_data(Connection * connection) {
  for (FractionDataList::iterator i = fractionList.begin(); i != fractionList.end(); 
i++)
    if ((*i)->connection == connection)
      return (*i);

  return NULL;
}

void Fraction::add_data(struct fraction_data * data) {
    FractionDataList::iterator i = std::lower_bound(fractionList.begin(),
                                                 fractionList.end(),
                                                 data,
                                                 FractionCmp);
    
    fractionList.insert(i, data);
}
