/*

Copyright (C) 2007 Robert S. Edmonds 

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

*/

#include <errno.h>
#include <fcntl.h>
#include <getopt.h>
#include <grp.h>
#include <pwd.h>
#include <stdbool.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#include "util.h"

/* externally visible globals */

bool util_daemonized = false;

/* functions */

/* scan_* and fmt_* derived from public domain libdjbdns code */

inline unsigned scan_ulong(const char *s, unsigned long *u){
	unsigned pos = 0;
	unsigned long result = 0;
	unsigned long c;

	while((c = (unsigned long) (unsigned char) (s[pos] - '0')) < 10){
		result = result * 10 + c;
		++pos;
	}
	*u = result;
	return pos;
}

inline unsigned fmt_ulong(char *s, unsigned long u){
	unsigned len = 1;
	unsigned long q = u;

	while(q > 9){
		++len; q /= 10;
	}
	if(s){
		s += len;
		do{
			*--s = '0' + (u % 10); u /= 10;
		} while(u);
	}
	return len;
}

unsigned scan_ip4(const char *s, ipaddr_t *n){
	unsigned i;
	unsigned len;
	unsigned long u;
	char *ip = (char *) n;

	len = 0;
	i = scan_ulong(s,&u); if (!i) return 0; ip[0] = u; s += i; len += i;
	if (*s != '.') return 0; ++s; ++len;
	i = scan_ulong(s,&u); if (!i) return 0; ip[1] = u; s += i; len += i;
	if (*s != '.') return 0; ++s; ++len;
	i = scan_ulong(s,&u); if (!i) return 0; ip[2] = u; s += i; len += i;
	if (*s != '.') return 0; ++s; ++len;
	i = scan_ulong(s,&u); if (!i) return 0; ip[3] = u; s += i; len += i;
	return len;
}

unsigned fmt_ip4(char *s, ipaddr_t n){
	unsigned i;
	unsigned len = 0;
	char *ip = (char *) &n;

	i = fmt_ulong(s,(unsigned long) (unsigned char) ip[0]); len += i; if (s) s += i;
	if (s) *s++ = '.'; ++len;
	i = fmt_ulong(s,(unsigned long) (unsigned char) ip[1]); len += i; if (s) s += i;
	if (s) *s++ = '.'; ++len;
	i = fmt_ulong(s,(unsigned long) (unsigned char) ip[2]); len += i; if (s) s += i;
	if (s) *s++ = '.'; ++len;
	i = fmt_ulong(s,(unsigned long) (unsigned char) ip[3]); len += i; if (s) s += i;
	*s = '\0';
	return len;
}

uint32_t bswap32(uint32_t x){
	return  ((x << 24) & 0xff000000 ) |
		((x <<  8) & 0x00ff0000 ) |
		((x >>  8) & 0x0000ff00 ) |
		((x >> 24) & 0x000000ff );
}

uint16_t bswap16(uint16_t x){
	return (uint16_t) ((x & 0xff) << 8 | (x & 0xff00) >> 8);
}

void reset_getopt(void){
#ifdef __GLIBC__
	optind = 0;
#else
	optind = 1;
#endif
#ifdef HAVE_OPTRESET
	optreset = 1;
#endif
}

ipaddr_range_t sips_to_range(char *sip0, char *sip1){
	ipaddr_range_t ipr;
	scan_ip4(sip0, &ipr.ip0);
	scan_ip4(sip1, &ipr.ip1);
	ipr.ip0 = my_ntohl(ipr.ip0);
	ipr.ip1 = my_ntohl(ipr.ip1);
	return ipr;
}

void make_pidfile(char *pidfile){
	if(pidfile == NULL)
		return;
	pid_t pid = getpid();
	FILE *fpidfile = fopen(pidfile, "w");
	if(!fpidfile)
		ERROR("unable to open pidfile for writing: %s", strerror(errno));
	fprintf(fpidfile, "%d\n", pid);
	fclose(fpidfile);
}

void util_daemonize(char *program_name, char *pidfile){
	if(daemon(0, 0) != 0)
		ERROR("%s", strerror(errno));
	make_pidfile(pidfile);
	openlog(program_name, LOG_PID, LOG_DAEMON);
	util_daemonized = true;
}

char *human_readable_rate(int64_t packets, int64_t bytes, unsigned interval){
	char prefix_packets[] = "\0\0";
	char prefix_bytes[] = "\0\0";
	char *str;
	CALLOC(str, 64);

	double div_packets = ((double) packets) / interval;
	double div_bytes = ((double) bytes) / interval;

	if(div_packets >= 1E9){
		prefix_packets[0] = 'G';
		div_packets /= 1E9;
	}else if(div_packets >= 1E6){
		prefix_packets[0] = 'M';
		div_packets /= 1E6;
	}else if(div_packets >= 1E3){
		prefix_packets[0] = 'K';
		div_packets /= 1E3;
	}

	if(div_bytes >= 1E9){
		prefix_bytes[0] = 'G';
		div_bytes /= 1E9;
	}else if(div_bytes >= 1E6){
		prefix_bytes[0] = 'M';
		div_bytes /= 1E6;
	}else if(div_bytes >= 1E3){
		prefix_bytes[0] = 'K';
		div_bytes /= 1E3;
	}

	snprintf(str, 64, "%.2f %spps, %.2f %sBps",
		div_packets, prefix_packets, div_bytes, prefix_bytes);
	return str;
}

int creat_mog(const char *pathname, mode_t mode, char *owner, char *group){
	int fd;
	uid_t uid;
	gid_t gid;
	struct passwd *pw = getpwnam(owner);
	struct group *gr = getgrnam(group);

	if(pw)
		uid = pw->pw_uid;
	else
		uid = geteuid();
	if(gr)
		gid = gr->gr_gid;
	else
		gid = getegid();
	
	if((fd = creat(pathname, mode)) == -1)
		ERROR("creat: %s", strerror(errno));
	if(geteuid() == 0)
		if(fchown(fd, uid, gid) != 0)
			ERROR("fchown: %s", strerror(errno));
	return fd;
}

void seed_rng(void){
	unsigned seed;
	FILE *random = fopen("/dev/random", "r");
	if(!random)
		ERROR("unable to open /dev/random for reading: %s", strerror(errno));
	if(fread(&seed, sizeof(seed), 1, random) != 1)
		ERROR("unable to read %zd bytes from /dev/random: %s", sizeof(seed), strerror(errno));
	srandom(seed);
}
