# Phatch - Photo Batch Processor
# Copyright (C) 2007-2008 www.stani.be
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see http://www.gnu.org/licenses/
#
# Phatch recommends SPE (http://pythonide.stani.be) for editing python files.

#---import modules

#standard library
import codecs, cPickle, os, pprint, time, traceback
from cStringIO import StringIO

#gui-independent
from core import ct, pil
from models import Action
from message import send
from settings import create_settings
from unicoding import ensure_unicode, exception_to_unicode, ENCODING

#---classes
    
class PathError(Exception):
    def __init__(self, filename):
        """PathError for invalid path."""
        self.filename   = filename
        
    def __str__(self):
        return u'"%s" '%self.filename+_('is not a valid path')+'.'
 
class ReadOnlyDict:
    def __init__(self,data):
        self.data   = data
        
    def __call__(self,key):
        return self.data[key]
    
    def update_filename(self,result,parent_index,filename):
        dirname, basename   = os.path.split(filename)
        message             = "%s: %s\n%s: %s\n"%(_('In'),dirname,_('File'),
                                basename)
        self.update(result,parent_index*self.child_max,newmsg=message)
        self.sleep()
        
    def update_index(self,result,parent_index,child_index):
        self.update(result,parent_index*self.child_max + child_index + 1)
        
    #---overwrite
    def close(self):
        pass
        
    def update(self,result,value,newmsg=''):
        pass
        
    def sleep(self):
        pass
    
#---init/exit
def init():
    verify_app_user_path()
    import_actions()
    
def exit():
    pass
        

def verify_app_user_path():
    for path in [ct.APP_USER_PATH,ct.APP_USER_PATH,ct.USER_ACTIONLISTS_PATH,
            ct.USER_ACTIONS_PATH,ct.USER_MASKS_PATH,ct.USER_WATERMARKS_PATH]:
        if not os.path.exists(path):
            os.mkdir(path)

#---various
def title(f):
    return os.path.splitext(os.path.basename(f))[0].replace('_',' ')\
        .replace('-',' ').title()
#---error logs
def init_error_log_file():
    global ERROR_LOG_FILE, ERROR_LOG_COUNTER
    ERROR_LOG_COUNTER   = 0
    ERROR_LOG_FILE      = codecs.open(ct.ERROR_LOG_PATH, 'wb',
                            encoding=ENCODING,errors='replace')

def log_error(message,filename,action=None):
    global ERROR_LOG_COUNTER
    details = ''
    if action:
        details += os.linesep + 'Action: ' + \
                    pprint.pformat(action.dump())
    ERROR_LOG_FILE.write(os.linesep.join([
        u'Error %d: %s'%(ERROR_LOG_COUNTER,message),
        details,
        os.linesep,
    ]))
    try:
        traceback.print_exc(file=ERROR_LOG_FILE)
    except UnicodeDecodeError:
        stringio    = StringIO()
        traceback.print_exc(file=stringio)
        traceb      = stringio.read()
        ERROR_LOG_FILE.write(unicode(traceb,ENCODING,'replace'))
    ERROR_LOG_FILE.write('*'+os.linesep)
    ERROR_LOG_FILE.flush()
    ERROR_LOG_COUNTER += 1
    return details

#---collect image files
def filter_image_files(folder,files, extensions):
    files   = [os.path.join(folder,file) for file in files]
    return [file for file in files if os.path.isfile(file) and \
        os.path.splitext(file)[1].lower() in extensions]
       
def get_image_files_folder(folder,extensions,recursive):
    if recursive:
        image_files  = []
        for folder, dirs, files in os.walk(folder):
            image_files.extend(filter_image_files(folder, files, extensions))
        return image_files
    else:
        return filter_image_files(folder, os.listdir(folder), extensions)
    

#---check
def check_actionlist(actions,settings):
    #Check if there is something to do
    if actions == []:
        send.frame_show_error('%s %s'%(_('Nothing to do.'),
            _('The action list is empty.')))
        return None
    
    #Skip disabled actions
    actions = [action for action in actions if action.is_enabled()]
    if actions == []:
        send.frame_show_error('%s %s'%(_('Nothing to do.'),
            _('There is no action enabled.')))
        return None
    
    #Check if there is a save statement
    last_action = actions[-1]
    if not (last_action.is_valid_last_action() or file_only(actions)):
        send.frame_append_save_action()
        return None
    
    #Check if overwrite is forced
    settings['overwrite_existing_images_forced'] = \
        actions[-1].is_overwrite_existing_images_forced()

    return actions

def file_only(actions):
    for action in actions:
        if not ('file' in action.tags):
            return False
    return True

def check_images(image_files):
    #show dialog
    send.frame_show_progress(   title       = _("Checking images"),
                                parent_max  = len(image_files))
                                
    #check files
    valid = []; invalid = []
    for index, (folder,image_file) in enumerate(image_files):
        pil.report_invalid_image(image_file,valid,invalid,folder)
        result              = {}
        send.progress_update_filename(result,index,image_file)
        if not result['keepgoing']:
            return
    send.progress_close()
    
    #show invalid messages
    if invalid:
        result      = {}
        send.frame_show_files_message(result,
            message = _('Phatch can not handle %d image(s):')%len(invalid), 
            title   = ct.FRAME_TITLE%('',_('Invalid images')), 
            files   = invalid)
        if result['cancel']:
            return
    image_files              = valid

    #Check if there are files
    if image_files:
        return image_files
    else:
        send.frame_show_error('%s, %s.'%(_("Sorry"),_("no valid files found")))
        return

#---get
def get_image_files(paths,extensions,recursive):
    try:
        result  = []
        for p in paths:
            p   = os.path.abspath(p.strip())
            if os.path.isfile(p):
                result.append((None,p))
            elif os.path.isdir(p):
                result.extend([(p,f) 
                    for f in get_image_files_folder(p,extensions,recursive)])
            else:
                raise PathError(p)
        result.sort()
        return result
    except PathError, error:
        send.frame_show_error('%s, "%s" %s.'%(_('Sorry'),
                ensure_unicode(error.filename),
                _('is not a valid path')))
        return []

def get_paths(paths, settings,drop=False):
    if drop or (paths is None):
        result  = {}
        send.frame_show_execute_dialog(result,settings,paths)
        if result['cancel']:return
        paths    = settings['paths']
        if not paths:
            send.frame_show_error(_('No files or folder selected.'))
            return None
    return paths

def get_photo(image_file,image_index,result,save_metadata,folder):
    try:
        photo   = pil.Photo(image_file,image_index,save_metadata,folder)
        result['skip'] = False
        result['abort'] = False
        return photo, result
    except Exception, details:
        photo   = None
        reason  = exception_to_unicode(details)
        #log error details
        message = u'%s: %s:\n%s'%(_('Unable to open file'),image_file,reason)
        ignore  = False
        action  = None
        return process_error(photo,message,image_file,action,result,ignore)
    
#---apply
def process_error(photo,message,image_file,action,result,ignore):
    """Logs error to file and show dialog box allowing the user to 
    skip,abort or ignore."""
    log_error(message,image_file,action)
    #show error dialog
    if result['stop_for_errors']:
        send.frame_show_progress_error(result,message,ignore = ignore)
        #if result:
        answer = result['answer']
        if answer == _('abort'):
            send.progress_close()
            result['skip']  = False
            result['abort'] = True
            return photo, result
        result['last_answer'] = answer
        if answer == _('skip'):
            result['skip']  = True
            result['abort'] = False
            return photo, result
    elif result['last_answer'] == _('skip'):
        result['skip']  = True
        result['abort'] = False
        return photo, result
    result['skip']  = False
    result['abort'] = False
    return photo, result

def apply_action(action,photo,setting,cache,image_file,result):
    try:
        photo   = action.apply(photo,setting,cache)
        result['skip']  = False
        result['abort'] = False
        return photo, result
    except Exception, details:
        folder, image  = os.path.split(ensure_unicode(image_file))
        reason  = exception_to_unicode(details)
        message = u'%s\n%s\n\n%s'%(
            _("Can not apply action %(a)s on image '%(i)s' in folder:")%\
                {'a':_(action.label),'i':image},
            folder,
            reason,
        )
        ignore  = True
        return process_error(photo,message,image_file,action,result,ignore)
    
def apply_actions(actions,settings,paths=None,drop=False):
    """Do all the actions."""
    
    #Start log file
    init_error_log_file()
    
    #Check action list
    actions = check_actionlist(actions,settings)
    if not actions: return
    
    #Get paths (and update settings)
    paths    = get_paths(paths,settings,drop=drop)
    if not paths: return
            
    #Check if all files exist
    extensions      = ['.'+x for x in settings['extensions']]
    image_files     = get_image_files(paths,extensions,settings['recursive'])
    if not image_files: return
    
    #Check if all the images are valid
    if settings['check_images_first']:
        image_files = check_images(image_files)
    if not image_files: return

    #Initialize actions
    for action in actions:
        try:
            action.init()
        except Exception, details:
            reason  = exception_to_unicode(details)
            message = u'%s\n\n%s'%(
                _("Can not apply action %(a)s:")%\
                    {'a':_(action.label)},
                reason,
            )
            send.frame_show_error(message)
            return
        
    #Retrieve settings
    skip_existing_images    = not (settings['overwrite_existing_images'] or\
                                settings['overwrite_existing_images_forced'])
    save_metadata           = settings['save_metadata']
    result                  = {
        'stop_for_errors'   : settings['stop_for_errors'],
        'last_answer'       : None,
    }
    
    #Execute action list
    actions_amount          = len(actions) + 1 #open image is extra action
    cache                   = {} 
    is_done                 = actions[-1].is_done #checking method for resuming
    read_only_settings      = ReadOnlyDict(settings)
    
    #Start progress dialog
    send.frame_show_progress(   title       = _("Executing action list"),
                                parent_max  = len(image_files),
                                child_max   = actions_amount,
                                message     = _('Starting...'),
                            )
    
    for image_index, (folder,image_file) in enumerate(image_files):
        #update image file & progress dialog box
        progress_result = {}
        send.progress_update_filename(progress_result,image_index,image_file)
        if progress_result and not progress_result['keepgoing']:
            send.progress_close
            return
        
        #open image and check for errors
        photo, result = get_photo(image_file,image_index, result,save_metadata,
                            folder)
        if      result['abort']:  return
        elif    result['skip']:   break
        
        #check if already not done
        if skip_existing_images and is_done(photo):
            continue
        
        #do the actions
        for action_index, action in enumerate(actions):
            #update progress
            progress_result = {}
            send.progress_update_index(progress_result,image_index,action_index)
            if progress_result and not progress_result['keepgoing']:
                send.progress_close()
                return
            #apply action
            photo, result  = apply_action(action,photo,
                                    read_only_settings, cache,image_file,
                                    result)
            if      result['abort']: return
            elif    result['skip']:
                break
        del photo, progress_result, action_index, action
    send.progress_close()
    
#---common
import glob
#---classes
def import_module(module,folder=None):
    if folder is None:
        return __import__(module)
    return getattr(__import__('%s.%s'%(folder,module)),module)

def import_actions():
    global ACTIONS, ACTION_LABELS, ACTION_FIELDS
    #if bitmaps
    from core.lib.events import send
    #actions=result
    modules = \
        [import_module(os.path.basename(os.path.splitext(filename)[0]),
            'actions') 
            for filename in glob.glob(os.path.join(ct.ACTIONS_PATH,'*.py'))]+\
        [import_module(os.path.basename(os.path.splitext(filename)[0])) for 
            filename in glob.glob(os.path.join(ct.USER_ACTIONS_PATH,'*.py'))]
##    for filename in glob.glob(os.path.join(ct.ACTIONS_PATH,'*.py')):
##        basename= os.path.basename(os.path.splitext(filename)[0])
##        module  = import_module(basename,'actions')
    ACTIONS = {}
    for module in modules:
        try:
            cl  = getattr(module,ct.ACTION)
        except AttributeError:
            continue
        #register action
        ACTIONS[cl.label]   = cl
    #ACTION_LABELS
    ACTION_LABELS                          = ACTIONS.keys()
    ACTION_LABELS.sort()
    #ACTION_FIELDS
    ACTION_FIELDS = {}
    for label in ACTIONS:
        ACTION_FIELDS[label]  = ACTIONS[label]()._fields

def save_actionlist(filename,data):
    """data = {'actions':...}"""
    #check filename
    if os.path.splitext(filename)[1].lower() != ct.EXTENSION:
        filename    += ct.EXTENSION
    #prepare data
    data['actions'] = [action.dump() for action in data['actions']]
    #backup previous
    if os.path.isfile(filename):
        os.rename(filename,filename+'~')
    #write it
    f       = open(filename,'wb')
    f.write(pprint.pformat(data))
    f.close()

def open_actionlist(filename):
    #read source
    f       = open(filename,'rb')
    source  = f.read()
    f.close()
    #load data
    data                = eval(source)
    result              = []
    invalid_labels      = []
    for action in data['actions']:
        actionLabel     = action['label']
        actionFields    = action['fields']
        newAction       = ACTIONS[actionLabel]()
        invalid_labels.extend(['- %s (%s)'%(label,actionLabel) 
                                for label in newAction.load(actionFields)])
        result.append(newAction)
    data['actions']         = result
    data['invalid labels']  = invalid_labels
    return data
