/*
	 pingpong - Really free server for Amateur Radio convers
	 Copyright (C) 2005 Joop Stakenborg <pg4i@amsat.org>

	 This program is free software; you can redistribute it and/or modify
	 it under the terms of the GNU General Public License as published by
	 the Free Software Foundation; either version 2, or (at your option)
	 any later version.

	 This program is distributed in the hope that it will be useful,
	 but WITHOUT ANY WARRANTY; without even the implied warranty of
	 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the
	 GNU General Public License for more details.

	 You should have received a copy of the GNU General Public License
	 along with this program; if not, write to the Free Software Foundation,
	 Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.	
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <syslog.h>
#include <glib.h>
#include "pingpong.h"
#include "types.h"
#include "net.h"
#include "client.h"
#include "messages.h"
#include "init.h"
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

gchar *buffer, *confdir;
extern GList *client_list;

int main (int argc, char **argv) {
	gint server_fd, i, port;
	gchar *conf;
	pid_t pid, sid;

	/* parse the command line for options */
	for(i = 1; i < argc; i++)
	{
		if(argv[i][0] == '-' && strlen(argv[i]) > 1)
		{
			if(strcmp(argv[i], "-h") == 0 || strcmp(argv[i], "--help") == 0)
			{
				g_print("pingpong, a free server for amateur radio convers\n"
						"\n"
						"usage: pingpong [options]\n"
						"\n"
						"where options are:\n"
						" -h, --help                show this help message\n"
						" -V, --version             show the version of pingpong\n"
						" -c <dir>, --confdir <dir> read configuration files from <dir>\n"
						"\n"
						"the configuration directory defaults to %s\n"
						"\n"
						"pingpong is distributed under the GNU General Public License\n", PINGPONG_SYSCONFDIR);
				return 0;
			}
			else if(strcmp(argv[i], "-V") == 0 || strcmp(argv[i], "--version") == 0)
			{
				g_print("pingpong version %s\n", VERSION);
				return 0;
			}
			else if(strcmp(argv[i], "-c") == 0 || strcmp(argv[i], "--confdir") == 0)
			{
				if(i > argc - 2)
				{
					g_print("pingpong: error, no value for the %s option, type pingpong -h for help\n", argv[i]);
					return 1;
				}
				else
				{
					i++;
					confdir = g_strdup(argv[i]);
				}
			} 
			else
			{
				g_print("pingpong: error, unknown option: %s, type pingpong -h for help\n",argv[i]);
				return 1;
			}
		}
		else
		{
			g_print("pingpong: error, unknown argument: %s, type pingpong -h for help\n",argv[i]);
			return 1;
		}
	}

	pid = fork ();
	if (pid < 0)
	{
		g_print ("%s: Fork failed: %s\n", PACKAGE, strerror (errno));
		return 1;
	}
	if (pid > 0) exit (0);

	openlog ("pingpong", LOG_PID, LOG_DAEMON);
	if ((sid = setsid ()) < 0)
	{
		syslog (LOG_ERR, "%s\n", "setsid");
		return 1;
	}
	if ((chdir ("/")) < 0)
	{
		syslog (LOG_ERR, "%s\n", "chdir");
		return 1;
	}
	umask (0);

	/* read configuration and see if we can create server */
	if (confdir) conf = g_strdup_printf("%s/pingpong.conf", confdir);
		else conf = g_strdup_printf("%s/pingpong.conf", PINGPONG_SYSCONFDIR);
	port = readconf(conf);

	server_fd = create_server(port, 8);
	if (server_fd == -1)
	{
		syslog (LOG_ERR, "%s\n", "create server");
		return 1;
	}

	/* loop for select() */
	for(;;)
	{
		gint descr;
		guint i;
		Client *cli;
		fd_set s;
		gchar dest[256];
		struct timeval tv;

		FD_ZERO(&s);
		FD_SET(server_fd, &s);

		for (i = 0; i < g_list_length(client_list); i++)
		{
			cli = g_list_nth_data(client_list, i);
			FD_SET(cli->fd, &s);
		}

		tv.tv_sec = 1;
		tv.tv_usec = 0;
		descr = select(FD_SETSIZE, &s, NULL, NULL, &tv);

		if (descr < 0)
			syslog (LOG_ERR, "%s: %s\n", "select", strerror (errno));

		/* check if a new client wants to connect */
		if (FD_ISSET(server_fd, &s))
		{
			struct sockaddr addr;
			socklen_t len;
			if ((descr = accept(server_fd, &addr, &len)) != -1)
			{
				struct hostent *host;
				if (addr.sa_family == AF_INET)
				{
					struct sockaddr_in *addr_in = (struct sockaddr_in *)&addr;
					len = (socklen_t)sizeof(addr_in->sin_addr);
					host = gethostbyaddr ((gchar *)&addr_in->sin_addr, len, AF_INET);
					len = (socklen_t)sizeof(dest);
					syslog (LOG_INFO, "connection from %s(%s)\n", host->h_name,
						inet_ntop(AF_INET, host->h_addr, dest, len));
				}
				else
					syslog (LOG_INFO, "connection from (address family not supported)\n");
				if (!create_client(descr)) close(descr);
			}
		} /* if (FD_ISSET (server_fd)) */

		/* go through the list of clients and see if their fd is changed */
		for (i = 0; i < g_list_length(client_list); i++)
		{
			cli = g_list_nth_data(client_list, i);
			if (FD_ISSET(cli->fd, &s))
			{

				gchar *msg = recv_msg(cli->fd);

				/* disconnect */
				if (!msg)
				{ 
					syslog (LOG_INFO, "disconnect from '%s'\n", cli->name);
					remove_client(cli, NULL);
					continue;
				}

				/* commands that are allowed prior to login */
				else if (g_strncasecmp(msg, PINGPONG_CMD_SET_USER_NAME, 2) == 0)
				{ /* "/n pa4tu" */
					if (!set_name(cli)) 
						continue;
				}
				else if (g_strncasecmp(msg, PINGPONG_CMD_QUIT, 2) == 0)
				{ /* "/q" */
					remove_client(cli, msg);
					continue;
				}

				/* following commands only allowed if a name is set */
				else if (cli->name)
				{
					if (g_strncasecmp(msg, PINGPONG_CMD_QUESTION, 2) == 0)
					{ /* "/?" */
						gchar *helpmsg = g_strdup_printf(
"*** %s release %s Command Summary:\n\
 ?      channel      help      leave      quit\n\
***\n", PACKAGE, VERSION);
						send_msg(cli->fd, PINGPONG_MSG, helpmsg);
						g_free(helpmsg);
						continue;
					}
					else if (g_strncasecmp(msg, PINGPONG_CMD_HELP, 2) == 0)
					{ /* "/h" */
						help_msg(cli->fd);
						continue;
					}
					else if (g_strncasecmp(msg, PINGPONG_CMD_CHANNEL, 2) == 0)
					{ /* "/c" */
						goto_channel(cli);
						continue;
					}
					else if (g_strncasecmp(msg, PINGPONG_CMD_LEAVE, 2) == 0)
					{ /* "/l" */
						leave_channel(cli);
						continue;
					}
					else if (g_strncasecmp(msg, PINGPONG_CMD_WHO, 2) == 0)
					{ /* "/w" */
						who_msg(cli->fd);
						continue;
					}
					else if (g_strncasecmp(msg, "/", 1) == 0)
					{ /* other command */
						send_msg(cli->fd, PINGPONG_MSG, "*** Command not implemented (yet)\n");
						continue;
					}
					else
					{
						user_msg(cli);
						continue;
					}
				}
			} /* if (FD_ISSET) */
		} /* for (i)*/
	} /* for (;;) */
	return 0;
}
