#include <stdlib.h>
#include <errno.h>
#include <sys/stat.h>
#include <string.h>

#include "sys/wcebase.h"
#include "sys/wceerror.h"
#include "sys/wcefile.h"
#include "sys/wcetrace.h"
#include "sys/fixpath.h"
#include "sys/dirent.h"

#define MAXPATHLEN 255

static struct dirent dir_static;

#define IS_DIRECTORY_SEP(x) (x == '\\' || x == '/')

DIR *
opendir(char *path)
{
  DIR *dirp;
  struct stat statbuf;
  char fpath[MAXPATHLEN];
  char *p, *d;

  WCETRACE(WCE_IO, "opendir(%s)", path);

#if 0
  /* Current directory implementation - TBD */
  if (strcmp(path, ".") == 0)
    XCEGetCurrentDirectoryA(sizeof(fpath), path);
  else
    XCEFixPathA(path, fpath);
#endif
  fixpath(path, fpath);
  WCETRACE(WCE_IO, "opendir2(%s)", fpath);

  if (stat(fpath, &statbuf) < 0)
    return(NULL);

  if (!(statbuf.st_mode & S_IFDIR))
    return(NULL);

  if (!(dirp = (DIR *) malloc(sizeof(DIR)))) {
    errno = ENOMEM;
    return NULL;
  }

  dirp->dd_handle = INVALID_HANDLE_VALUE;
  dirp->dd_isfat = 1;
  dirp->dd_fd = 0;
  dirp->dd_loc = 0;
  dirp->dd_size = 0;

  strcpy(dirp->dd_path, fpath);

  return(dirp);
}

int
closedir(DIR *dirp)
{
  BOOL retval = FALSE;

  if (dirp == NULL) {
    errno = EBADF;
    return(-1);
  }

  if (dirp->dd_handle != INVALID_HANDLE_VALUE) {
    retval = FindClose((HANDLE)dirp->dd_handle);
  }

  if (retval) {
    WCETRACE(WCE_IO, "closedir: FindClose succeeded");
    free((char *) dirp);
    return(0);
  } else {
    WCETRACE(WCE_IO, "closedir: FindClose FAILED");
    errno = EBADF;
    free((char *) dirp);
    return(-1);
  }
}

struct dirent *
readdir(DIR *dirp)
{
  WIN32_FIND_DATAW fdw;
  char buf[MAX_PATH];

  if (dirp->dd_loc == 0) {
    mbstowcs(fdw.cFileName, ".", MAX_PATH);
  } else if(dirp->dd_loc == 1) {
    mbstowcs(fdw.cFileName, "..", MAX_PATH);
  } else if (dirp->dd_handle == INVALID_HANDLE_VALUE) {
    wchar_t fpathw[MAXNAMLEN];
    char fpath[MAXNAMLEN];
    int ln;

    fixpath(dirp->dd_path, fpath);

    ln = strlen(fpath) - 1;
    if (!IS_DIRECTORY_SEP(fpath[ln]))
      strcat(fpath, "\\");
    strcat(fpath, "*");

    WCETRACE(WCE_IO, "FindFirstFile: %s\n", fpath);

    mbstowcs(fpathw, fpath, MAXNAMLEN);
    memset(&fdw, 0, sizeof(WIN32_FIND_DATAW));
    dirp->dd_handle = (HANDLE)FindFirstFileW(fpathw, &fdw);

    if (dirp->dd_handle == INVALID_HANDLE_VALUE) {
      WCETRACE(WCE_IO, "readdir: FindFirstFileW failed for \"%s\"", fpath);
      return(NULL);
    }
  } else {
    if (!FindNextFileW((HANDLE)dirp->dd_handle, &fdw))
      return NULL;
  }
  
  dir_static.d_ino = 1;
  dirp->dd_loc++;

  dir_static.d_reclen = sizeof(struct dirent) - MAXNAMLEN + 3 +
    dir_static.d_namlen - dir_static.d_namlen % 4;
  
  wcstombs(buf, fdw.cFileName, MAX_PATH);
  dir_static.d_namlen = strlen(buf);
  strcpy(dir_static.d_name, buf);

  return &dir_static;
}
