//
// UTF8Encoding.cpp
//
// $Id: //poco/1.2/Foundation/src/UTF8Encoding.cpp#2 $
//
// Library: Foundation
// Package: Text
// Module:  UTF8Encoding
//
// Copyright (c) 2004-2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#include "Poco/UTF8Encoding.h"


namespace Poco {


const TextEncoding::CharacterMap UTF8Encoding::_charMap = 
{
	/* 00 */	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 
	/* 10 */	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 
	/* 20 */	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 
	/* 30 */	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 
	/* 40 */	0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 
	/* 50 */	0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 
	/* 60 */	0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 
	/* 70 */	0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 
	/* 80 */	  -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1, 
	/* 90 */	  -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1, 
	/* a0 */	  -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1, 
	/* b0 */	  -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,   -1, 
	/* c0 */	  -2,   -2,   -2,   -2,   -2,   -2,   -2,   -2,   -2,   -2,   -2,   -2,   -2,   -2,   -2,   -2, 
	/* d0 */	  -2,   -2,   -2,   -2,   -2,   -2,   -2,   -2,   -2,   -2,   -2,   -2,   -2,   -2,   -2,   -2, 
	/* e0 */	  -3,   -3,   -3,   -3,   -3,   -3,   -3,   -3,   -3,   -3,   -3,   -3,   -3,   -3,   -3,   -3, 
	/* f0 */	  -4,   -4,   -4,   -4,   -4,   -4,   -4,   -4,   -5,   -5,   -5,   -5,   -6,   -6,   -1,   -1, 
};


UTF8Encoding::UTF8Encoding()
{
}


UTF8Encoding::~UTF8Encoding()
{
}


const TextEncoding::CharacterMap& UTF8Encoding::characterMap() const
{
	return _charMap;
}


int UTF8Encoding::convert(const unsigned char* bytes) const
{
	int n = _charMap[*bytes];
	int uc;
	switch (n)
	{
	case -6:
		uc = *bytes & 0x01; break;
	case -5:
		uc = *bytes & 0x03; break;
	case -4:
		uc = *bytes & 0x07; break;
	case -3:
		uc = *bytes & 0x0F; break;
	case -2:
		uc = *bytes & 0x1F; break;
	default:
		uc = n;
	}
	while (n++ < -1) 
	{	
		// TODO: check for malformed or overlong sequences
		uc <<= 6;
		uc |= (*++bytes & 0x3F);
	}
	return uc;
}


int UTF8Encoding::convert(int ch, unsigned char* bytes, int length) const
{
	if (ch <= 0x7F)
	{
		if (bytes && length >= 1)
			*bytes = (unsigned char) ch;
		return 1;
	}
	else if (ch <= 0x7FF)
	{
		if (bytes && length >= 2)
		{
			*bytes++ = (unsigned char) ((ch >> 6) & 0x1F | 0xC0);
			*bytes   = (unsigned char) ((ch & 0x3F) | 0x80);
		}
		return 2;
	}
	else if (ch <= 0xFFFF)
	{
		if (bytes && length >= 3)
		{
			*bytes++ = (unsigned char) ((ch >> 12) & 0x0F | 0xE0);
			*bytes++ = (unsigned char) ((ch >> 6) & 0x3F | 0x80);
			*bytes   = (unsigned char) ((ch & 0x3F) | 0x80);
		}
		return 3;
	}
	else if (ch <= 0x1FFFFF)
	{
		if (bytes && length >= 4)
		{
			*bytes++ = (unsigned char) ((ch >> 18) & 0x07 | 0xF0);
			*bytes++ = (unsigned char) ((ch >> 12) & 0x3F | 0x80);
			*bytes++ = (unsigned char) ((ch >> 6) & 0x3F | 0x80);
			*bytes   = (unsigned char) ((ch & 0x3F) | 0x80);
		}
		return 4;
	}
	else if (ch <= 0x3FFFFFF)
	{
		if (bytes && length >= 5)
		{
			*bytes++ = (unsigned char) ((ch >> 24) & 0x03 | 0xF8);
			*bytes++ = (unsigned char) ((ch >> 18) & 0x3F | 0x80);
			*bytes++ = (unsigned char) ((ch >> 12) & 0x3F | 0x80);
			*bytes++ = (unsigned char) ((ch >> 6) & 0x3F | 0x80);
			*bytes   = (unsigned char) ((ch & 0x3F) | 0x80);
		}
		return 5;
	}
	else if (ch <= 0x7FFFFFFF)
	{
		if (bytes && length >= 6)
		{
			*bytes++ = (unsigned char) ((ch >> 30) & 0x01 | 0xFC);
			*bytes++ = (unsigned char) ((ch >> 24) & 0x3F | 0x80);
			*bytes++ = (unsigned char) ((ch >> 18) & 0x3F | 0x80);
			*bytes++ = (unsigned char) ((ch >> 12) & 0x3F | 0x80);
			*bytes++ = (unsigned char) ((ch >> 6) & 0x3F | 0x80);
			*bytes   = (unsigned char) ((ch & 0x3F) | 0x80);
		}
		return 6;
	}
	else return 0;
}


} // namespace Poco
