/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
#include "polyxedit-ui-findmass-results.h"
#include "polyxedit-ui-cleave-results.h"
#include "polyxedit-ui-fragment-results.h"
#include "polyxedit-ui-searchmass-results.h"
#include "polyxmass-ui-seqed-widget.h"



/* columns */

enum
{
  /* The name of the oligo ! */
  COLUMN_OLIG_NAME = 0,

  /* The mass that was found. */
  COLUMN_OLIG_TO_FIND_MASS,

  /* the difference between found and found masses */
  COLUMN_OLIG_MASS_ERROR,

  /* The index + 1 in the array of oligos ! */
  COLUMN_OLIG_NUMBER,

  /* MONO or AVG: the mass type of the find */
  COLUMN_OLIG_MASS_TYPE,

  /* Mono mass */
  COLUMN_OLIG_MONO,

  /* Average mass */
  COLUMN_OLIG_AVG,

  /* Pointer to the oligomer */
  COLUMN_OLIG_POINTER,

  COLUMN_OLIG_COL_COUNT
};



GtkWidget *
polyxedit_findmass_res_wnd_setup (GtkWidget *parent,
				  GPtrArray *alloligsGPA)
{
  GtkWidget *window = NULL;
  GtkWidget *widget = NULL;
  GtkWidget *grand_parent = NULL;
  

  GladeXML *xml = NULL;

  gchar *gui_file = NULL;
  gchar *help = NULL;


  g_assert (parent != NULL);
  g_assert (alloligsGPA != NULL);

  gui_file = 
    g_strdup_printf ("%s/polyxedit-findmass-results.glade", 
		     userspec->gladedir);
  
  g_assert (gui_file != NULL);
  
  xml = glade_xml_new (gui_file, "findmass_results_wnd", 
		       PACKAGE);

  g_free (gui_file);

  if (xml == NULL)
    {
      g_error (_("%s@%d: failed to load the interface\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  window = glade_xml_get_widget (xml, "findmass_results_wnd");
  
  if (window == NULL)
    {
      g_error (_("%s@%d: failed to create the findmass results window\n"),
	     __FILE__, __LINE__);

      g_object_unref (G_OBJECT (xml));

      return NULL;
    }

  /* There is a GtkEntry in the window that is commonly used to
     display messages. We need to set this pointer to window
     immediately because we'll need this entry when displaying
     messages, in particular when oligomers are to be displayed in the
     treeview.
   */
  widget = glade_xml_get_widget (xml, "messages_entry");
  g_object_set_data (G_OBJECT (window),
		     "messages_entry", widget);


  /* Set the parent window pointer to the relative entry.
   */
  /* The window from which _this_ window was asked to setup has a
     datum that corresponds to the one-generation-up-level parent,
     that is the window in which the cleave/fragment results mass data
     were in the first place and from which mass(es) where asked to be
     found.  See the polyxedit-ui-findmass-options.c file at the place
     where the options window is setup.

     We will need to access that datum later, so we put it in an
     identical manner to _this window. We will use that 'grand_parent'
     pointer to connect _this window really_close () function to the
     window-closing function belonging to the grand_parent
     window. Indeed, it makes no sense to still have _this found
     masses-window open if the initial masses window does not exist
     anymore.
  */
  grand_parent = g_object_get_data (G_OBJECT (parent), "grand_parent");
  g_assert (grand_parent != NULL);
  g_object_set_data (G_OBJECT (window), "grand_parent", grand_parent);
  

  widget = glade_xml_get_widget (xml, "results_set_identity_number_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), "results_set_identity_number_entry", 
		     widget);
  
  help = g_strdup_printf ("%p", grand_parent);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);

  
  /* At this point we ought to set an explanation string to the
     created window, so that we later can determine for what purpose
     this window was created. This is particularly important to know
     how to sort the names of the oligomers that will be displayed in
     the findmass results window.
     
     We actually did set that explanatory string in the oligomer
     results window, either the cleavage oligomer results window,
     or the fragmentation oligomer results window. We have to get
     it out of _this_ window and duplicate it to the findmass
     results window.
  */
  help = g_object_get_data (G_OBJECT (parent), 
			    "oligomer_type");
  g_assert (help != NULL);
  
  help = g_strdup (help);
  
  g_object_set_data_full (G_OBJECT (window), 
			  "oligomer_type", help, g_free);
  
  /* We also want to provide an identification number of this window
     to the user, so that when a reporting of the results is asked it
     will be possible to identify *this* specific window of mass find
     results.
   */
  widget =
    glade_xml_get_widget (xml, 
			  "mass_find_results_set_identity_number_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "mass_find_results_set_identity_number_entry", 
		     widget);
  
  help = g_strdup_printf ("%p", window);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);

  

  /* The oligomers' frame must be correctly set up:
   */
  widget = glade_xml_get_widget (xml, "findmass_results_oligomers_frame");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "findmass_results_oligomers_frame", widget);

  /* In the first frame we'll later setup a treeview with all the
     oligomers arranged depending on the findmass specification
     from which they originated.
  */
  widget = glade_xml_get_widget (xml, "findmass_results_oligomers_vbox");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "findmass_results_oligomers_vbox", widget);


  /* We have to fill the treeview with the oligomers passed to this 
     function in the alloligsGPA parameter. There is a function to 
     do just that:
  */
  if (alloligsGPA)
    {
      g_object_set_data (G_OBJECT (window), "alloligsGPA", alloligsGPA);
      
      if (FALSE == polyxedit_findmass_res_wnd_show_oligs (window,
							    alloligsGPA))
	return NULL;
    }
  

  /* We don't need the GladeXML object any more, so unref it
     to save some memory 
   */
  g_object_unref (G_OBJECT (xml));

  g_signal_connect (G_OBJECT (window),
		    "delete_event",
		    G_CALLBACK (polyxedit_findmass_res_wnd_delete_event),
		    window);
  
  g_signal_connect (G_OBJECT (window),
		    "destroy_event",
		    G_CALLBACK (polyxedit_findmass_res_wnd_destroy_event),
		    window);

  /* Set this window pointer as a full datum to the window where the
     results were displayed, and in which the find mass operation was
     performed to produce _this findmass results window, so that when
     it is closed this window is closed also.

     There might be more than one findmass results' window opened for
     a given oligomer results window, and we do not want that the
     second window destroys the datum name of the first window, so we
     create an uambiguous datum name each time.
  */
  help = g_strdup_printf ("findmass_results_wnd-%p", window);
  
  g_object_set_data_full 
    (G_OBJECT (grand_parent),
     help, GTK_WIDGET (window), 
     (GDestroyNotify) polyxedit_findmass_res_wnd_really_close);
  
  g_free (help);
  


  return window;
}



gboolean
polyxedit_findmass_res_wnd_show_oligs (GtkWidget *window,
					 GPtrArray *alloligsGPA)
{
  GtkWidget *vbox = NULL;
  GtkWidget *messages_entry = NULL;
  
  GtkTreeModel *model = NULL;

  gint *count = NULL;

  gchar *help;
  
  gboolean res = FALSE;
  
  
  g_assert (window != NULL);
  g_assert (alloligsGPA != NULL);
  
  vbox = g_object_get_data (G_OBJECT (window),
			    "findmass_results_oligomers_vbox");
  g_assert (vbox != NULL);
  
  res = polyxedit_findmass_res_setup_oligs_treeview (window, vbox);

  /* By now, model should have the number of oligomers in the 
     treeview's store.
  */
  model = (GtkTreeModel *) g_object_get_data (G_OBJECT (window), 
					      "oligomers_treeview_model");
  g_assert (model != NULL);
  
  count = g_object_get_data (G_OBJECT (model), "total_oligos");
  g_assert (count != NULL);
  
  messages_entry = g_object_get_data (G_OBJECT (window),  
				      "messages_entry");
  g_assert (messages_entry != NULL);

  help = g_strdup_printf (_("Total number of oligomers: %d"),
			  *count);
  
  gtk_entry_set_text (GTK_ENTRY (messages_entry),
		      help);
  g_free (help);
  
  return res;
}


GtkTreeModel *
polyxedit_findmass_res_create_oligs_treeview_model (GPtrArray *alloligsGPA)
{
  GtkTreeStore *model;
  GtkTreeIter treeiter;
  
  gint iter = 0;
  gint jter = 0;
  gint *count = NULL;
    
  gdouble error = 0;

  gchar *mass_type = NULL;
  gchar *coordinates = NULL;
    
  GPtrArray *oligGPA = NULL;
  
  PxmProp *prop = NULL;
  PxmFindmassOpt *fmopt = NULL;
  PxmOligomer *oligomer = NULL;

  

  g_assert (alloligsGPA != NULL);
  

  model = gtk_tree_store_new 
    (COLUMN_OLIG_COL_COUNT,
     
     /* The name of the oligo ! COLUMN_OLIG_NAME*/
     G_TYPE_STRING,
     
     /* The mass that was found. COLUMN_OLIG_TO_FIND_MASS*/
     G_TYPE_DOUBLE,
     
     /* Difference (found - found) COLUMN_OLIG_MASS_ERROR*/
     G_TYPE_DOUBLE,
     
     /* The index + 1 in the array of oligos ! COLUMN_OLIG_NUMBER*/
     G_TYPE_INT,
     
     /* MONO or AVG: the mass type of the find COLUMN_OLIG_MASS_TYPE*/
     G_TYPE_STRING,
     
     /* Mono mass COLUMN_OLIG_MONO*/
     G_TYPE_DOUBLE,
     
     /* Average mass COLUMN_OLIG_AVG*/
     G_TYPE_DOUBLE,
     
     /* Pointer to the oligomer COLUMN_OLIG_POINTER*/
     G_TYPE_POINTER);
  
  count = g_malloc0 (sizeof (gint));

  /* Remember that the array that we get is an array of arrays.  This
     array contains an array for each group of oligomers that were
     found by finding a given mass.
  */
  for (iter = 0; iter < alloligsGPA->len; iter++)
    {
      oligGPA = g_ptr_array_index (alloligsGPA, iter);
      g_assert (oligGPA != NULL);
            
      /* And now iterate in the array and analyze each oligomer in it.
       */
      for (jter = 0 ; jter < oligGPA->len ; jter++)
	{
	  oligomer = g_ptr_array_index (oligGPA, jter);

	  g_assert (oligomer != NULL);
	  g_assert (oligomer->masspair != NULL);
	  
	  /* Get the PxmFindmassOpt instance as a prop object.
	   */
	  prop = libpolyxmass_prop_find_prop (oligomer->propGPA,
					  NULL,
					  NULL,
					  "FINDMASS_OPT",
					  NULL,
					  PXM_CMP_NO_DEEP);
	  g_assert (prop != NULL);
	  
	  fmopt = prop->data;
	  g_assert (fmopt != NULL);

	  /* Now we can calculate the error:
	   */
	  if (fmopt->mass_type == PXM_MASS_MONO)
	    {
	      error = oligomer->masspair->mono - fmopt->mass;

	      /* Take advantage to set the mass type string:
	       */
	      mass_type = g_strdup_printf (_("MONO"));
	    }
	  else /*  (fmopt->mass_type == PXM_MASS_AVG) */
	    {
	      error = oligomer->masspair->avg - fmopt->mass;

	      /* Take advantage to set the mass type string:
	       */
	      mass_type = g_strdup_printf (_("AVG"));
	    }
	  
	  gtk_tree_store_append (model, &treeiter, NULL);
	  gtk_tree_store_set 
	    (model, &treeiter,

	     /* The name of the oligo ! */
	     COLUMN_OLIG_NAME, oligomer->name,

	     /* The mass that was found. */
	     COLUMN_OLIG_TO_FIND_MASS, fmopt->mass,

	     /* Difference (found - found) */
	     COLUMN_OLIG_MASS_ERROR, error,

	     /* The absolute index + 1 in the arrays of oligos ! */
	     COLUMN_OLIG_NUMBER, (*count) + 1,

	     /* MONO or AVG: the mass type of the find */
	     COLUMN_OLIG_MASS_TYPE, mass_type,

	     /* Mono mass */
	     COLUMN_OLIG_MONO, oligomer->masspair->mono,

	     /* Average mass */
	     COLUMN_OLIG_AVG, oligomer->masspair->avg,

	     /* Pointer to the oligomer */
	     COLUMN_OLIG_POINTER, oligomer,

	     -1);

	  g_free (mass_type);
	  g_free (coordinates);
	  
	  /* Store the number of oligomer treated, all partial cleavages
	     included.
	  */
	  (*count)++;
	}
      /* end of 
	 for (jter = 0 ; jter < oligGPA->len ; jter++)
      */
    }
  /* end of 
     for (iter = 0; iter < GPA->len; iter++)
  */
  
  /* Before returning the model set to it the number of items we have
     set to it. The integer that was allocated will be automatically
     freed upon destruction of the model.
  */
  g_object_set_data_full (G_OBJECT (model), "total_oligos", count,
			  (GDestroyNotify) free);

  return GTK_TREE_MODEL (model);
}


gboolean
polyxedit_findmass_res_setup_oligs_treeview (GtkWidget *window, 
					     GtkWidget *vbox)
{
  GPtrArray *alloligsGPA = NULL;

  GtkWidget *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkCellRenderer *renderer = NULL;
  GtkTreeViewColumn *column;

  GtkWidget *sw = NULL;

  gchar *oligomer_type = NULL;
  

  g_assert (window != NULL);
  g_assert (vbox != NULL);
  
  alloligsGPA = g_object_get_data (G_OBJECT (window), "alloligsGPA");
  g_assert (alloligsGPA != NULL);


  /* Create the scrolledview that we'll pack into widget.
   */
  sw = gtk_scrolled_window_new (NULL, NULL);

  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),
				       GTK_SHADOW_ETCHED_IN);

  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);

  gtk_box_pack_start (GTK_BOX (vbox), sw, TRUE, TRUE, 0);  
  

  /* Create the treeview model.
   */
  model = polyxedit_findmass_res_create_oligs_treeview_model (alloligsGPA);


  /* Set to the model a datum with a pointer to alloligsGPA (the array
     of oligomer-containing arrays) and to fragoptGPA (the array of
     fragopt instances), so that the array of oligomers' array that
     has been used to fill the model is accessible later along with
     the corresponding fragopt instances stored in the array
     'fragoptGPA'. Also a pointer to the window!
   */
  g_object_set_data (G_OBJECT (model), "alloligsGPA", alloligsGPA);
  g_object_set_data (G_OBJECT (model), "window", window);

  /* And now set the window a datum with a pointer to the mode,
   * so that later the model is accessible (add/remove button
   * handlers).
   */
  g_object_set_data (G_OBJECT (window), 
		     "oligomers_treeview_model", model);
    
  /* Create the treeview proper.
   */
  treeview = gtk_tree_view_new_with_model (model);
  
  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);

  /* Set to the window a datum with a pointer to the treeview, so that
   * is accessible later (remove item handler).
   */
  g_object_set_data (G_OBJECT (window), "oligomers_treeview", treeview);


  gtk_tree_selection_set_mode (gtk_tree_view_get_selection 
			       (GTK_TREE_VIEW (treeview)),
			       GTK_SELECTION_MULTIPLE);

  /* Oligomer name column.
   */
  renderer = gtk_cell_renderer_text_new ();

  column = 
    gtk_tree_view_column_new_with_attributes (_("Name"),
					      
					      renderer, 
					      
					      "text",
					      COLUMN_OLIG_NAME,
					      
					      NULL);
  
  gtk_tree_view_column_set_sort_column_id (column, 
					   COLUMN_OLIG_NAME);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Depending on what kind of oligomer (cleavage or fragmentation oligomer),
     we have to choose the proper name-sorting function ! 

     This is because the names of the oligomers do not share the same
     syntax and layout if they are from a cleavage or a fragmentation.
  */
  oligomer_type = g_object_get_data (G_OBJECT (window), 
				     "oligomer_type");
  g_assert (oligomer_type != NULL);
  
  if (0 == strcmp (oligomer_type, "CLEAVAGE"))
    gtk_tree_sortable_set_sort_func 
      (GTK_TREE_SORTABLE (model),
       COLUMN_OLIG_NAME,
       polyxedit_cleave_res_wnd_sort_name_column,
       NULL,
       NULL);
  else if (0 == strcmp (oligomer_type, "FRAGMENT"))
    gtk_tree_sortable_set_sort_func 
      (GTK_TREE_SORTABLE (model),
       COLUMN_OLIG_NAME,
       polyxedit_fragment_res_wnd_sort_name_column,
       NULL,
       NULL);
  else if (0 == strcmp (oligomer_type, "SEARCHMASS"))
    gtk_tree_sortable_set_sort_func 
      (GTK_TREE_SORTABLE (model),
       COLUMN_OLIG_NAME,
       polyxedit_searchmass_res_wnd_sort_name_column,
       NULL,
       NULL);
  else
    g_assert_not_reached ();


  /* The found mass column.
   */
  renderer = gtk_cell_renderer_text_new ();

  column = 
    gtk_tree_view_column_new_with_attributes (_("To Find"),
					      
					      renderer, 
					      
					      "text",
					      COLUMN_OLIG_TO_FIND_MASS,
					      
					      NULL);
  
  gtk_tree_view_column_set_sort_column_id (column, 
					   COLUMN_OLIG_TO_FIND_MASS);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);


  /* The mass error column.
   */
  renderer = gtk_cell_renderer_text_new ();

  column = 
    gtk_tree_view_column_new_with_attributes (_("Error"),
					      
					      renderer, 
					      
					      "text",
					      COLUMN_OLIG_MASS_ERROR,
					      
					      NULL);
  
  gtk_tree_view_column_set_sort_column_id (column, 
					   COLUMN_OLIG_MASS_ERROR);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  
  /* Oligomer number column.
   */
  renderer = gtk_cell_renderer_text_new ();

  column = 
    gtk_tree_view_column_new_with_attributes (_("Number"),
					      
					      renderer, 
					      
					      "text",
					      COLUMN_OLIG_NUMBER,
					      
					      NULL);
  
  gtk_tree_view_column_set_sort_column_id (column, 
					   COLUMN_OLIG_NUMBER);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Oligomer mass type column.
   */
  renderer = gtk_cell_renderer_text_new ();

  column = 
    gtk_tree_view_column_new_with_attributes (_("Mass Type"),
					      
					      renderer, 
					      
					      "text",
					      COLUMN_OLIG_MASS_TYPE,
					      
					      NULL);
  
  gtk_tree_view_column_set_sort_column_id (column, 
					   COLUMN_OLIG_MASS_TYPE);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Oligomer mono mass column.
   */
  renderer = gtk_cell_renderer_text_new ();

  column = 
    gtk_tree_view_column_new_with_attributes (_("Mono Mass"),
					      
					      renderer, 
					      
					      "text",
					      COLUMN_OLIG_MONO,

					      NULL);
  
  gtk_tree_view_column_set_sort_column_id (column, COLUMN_OLIG_MONO);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Oligomer avg mass column.
   */
  renderer = gtk_cell_renderer_text_new ();

  column = 
    gtk_tree_view_column_new_with_attributes (_("Avg Mass"),
					      
					      renderer, 
					      
					      "text",
					      COLUMN_OLIG_AVG,

					      NULL);
  
  gtk_tree_view_column_set_sort_column_id (column, COLUMN_OLIG_AVG);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Oligomer pointer column.
   */
  renderer = gtk_cell_renderer_text_new ();
  
  column = 
    gtk_tree_view_column_new_with_attributes (_("Oligo. Pointer"),
					      
					      renderer,
					      
					      "text", 
					      COLUMN_OLIG_POINTER,
					      
					      NULL);
  
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_visible (column, FALSE),
  g_object_set (G_OBJECT (renderer), "visible", FALSE, NULL);

  /* Finally add the treeview to the scrooll window.
   */
  gtk_container_add (GTK_CONTAINER (sw), treeview);
  
  gtk_widget_show_all (vbox);

  /* At this point we can finally unref the model.
   */
  g_object_unref (G_OBJECT (model));
  
  return TRUE;
}


/* REPORTING FUNCTIONS.
 */
gchar *
polyxedit_findmass_results_wnd_make_report (PxmReportOpt* reportopt,
					    PxmWinMngmt* winmngmt)
{




  g_assert (winmngmt != NULL);
  g_assert (reportopt != NULL);
  
  g_assert (winmngmt->seqed_widget != NULL);
  g_assert (winmngmt->wnd != NULL);
  

  /* Great, we are able to start doing the report. We need to know what
     kind of report we want : text or LaTeX.
  */
  if (reportopt->export_format == EXPORT_FORMAT_TEXT)
    return polyxedit_findmass_results_wnd_make_report_text_format (reportopt,
								 winmngmt);
  else if (reportopt->export_format == EXPORT_FORMAT_LATEX)
    return polyxedit_findmass_results_wnd_make_report_latex_format (reportopt,
								  winmngmt);
  else
    g_assert_not_reached ();
    
  return NULL;
}



gchar *
polyxedit_findmass_results_wnd_make_report_text_format (PxmReportOpt* reportopt,
						      PxmWinMngmt* winmngmt)
{
  GtkWidget *window = NULL;
  GtkWidget *seqed_widget = NULL;
  
  gchar *report = NULL;
  gchar *type = NULL;

  gdouble find = 0;
  gdouble error = 0;
    
  GString *report_gs = NULL;
  
  PxmPolymer *polymer = NULL;

  GtkTreeView *treeview = NULL;
  GtkTreeStore *model;
  GtkTreeIter treeiter;

  PxmOligomer *oligomer = NULL;



  g_assert (winmngmt != NULL);
  g_assert (reportopt != NULL);

  seqed_widget = winmngmt->seqed_widget;
  g_assert (seqed_widget != NULL);

  /* The window where the fragment results are displayed is in
     'winmngmt->wnd'.
   */
  window = winmngmt->wnd;
  g_assert (window != NULL);
  
  polymer = PXM_SEQED_WIDGET (seqed_widget)->polymer;
  g_assert (polymer != NULL);

 

 report_gs = g_string_new ("");



  /* Since we want to make a user-friendly report, we manage to have
     the different oligomers reported in the order that the user has
     set in the treeview (some of that treeview column are sortable).

     Thus we iterate in the treeview instead of iterating in the
     arrray of oligomers... Some tricky work is required to still
     inform the user of the partial cleavage of each oligomer.
  */
  model = g_object_get_data (G_OBJECT (window), "oligomers_treeview_model");
  g_assert (model != NULL);
  
  treeview = g_object_get_data (G_OBJECT (window), "oligomers_treeview");
  g_assert (treeview != NULL);
  
  if (FALSE == gtk_tree_model_get_iter_first ((GtkTreeModel *) model,
					      &treeiter))
    {
      g_warning (_("%s@%d: treeview is empty, nothing to report\n"),
		 __FILE__, __LINE__);

      report_gs = 
	g_string_append (report_gs, _("Treeview is empty, nothing to report\n"));
      
      report = report_gs->str;
      g_string_free (report_gs, FALSE);
      
      return report;
    }
  else
    {
      /* We have the first item of the list here. Use it !
       */
      gtk_tree_model_get ((GtkTreeModel *) model, &treeiter,

			  COLUMN_OLIG_TO_FIND_MASS,
			  &find,

			  COLUMN_OLIG_MASS_ERROR,
			  &error,

			  COLUMN_OLIG_MASS_TYPE,
			  &type,

			  COLUMN_OLIG_POINTER,
			  &oligomer,

			  -1);
      
      g_assert (oligomer != NULL);
      g_assert (oligomer->masspair != NULL);
      
      g_string_append_printf (report_gs, 
			      "%-12s %-20s %-20s %-5s %-20s %-20s\n\n",
			      _("name"), _("mass to find"), _("mass error"), _("type"), 
			      _("mono"), _("avg"));
      
      g_string_append_printf (report_gs, 
			      "%-12s %-20.5lf %-20.5lf %-5s "
			      "%-20.5lf %-20.5lf\n",
			      oligomer->name, find, error, type,
			      oligomer->masspair->mono, 
			      oligomer->masspair->avg);
    }
  /*
    End of successfully  getting the first item in the treeview.
  */


  /* And now continue to the remaining one in the while loop.
   */
  while (TRUE == gtk_tree_model_iter_next ((GtkTreeModel *) model,
					      &treeiter))
    {
      gtk_tree_model_get ((GtkTreeModel *) model, &treeiter,

			  COLUMN_OLIG_TO_FIND_MASS,
			  &find,

			  COLUMN_OLIG_MASS_ERROR,
			  &error,

			  COLUMN_OLIG_MASS_TYPE,
			  &type,

			  COLUMN_OLIG_POINTER,
			  &oligomer,

			  -1);
      
      g_assert (oligomer != NULL);
      g_assert (oligomer->masspair != NULL);
      
      g_string_append_printf (report_gs, 
			      "%-12s %-20.5lf %-20.5lf %-5s "
			      "%-20.5lf %-20.5lf\n",
			      oligomer->name, find, error, type,
			      oligomer->masspair->mono, 
			      oligomer->masspair->avg);
    }
  /*
    End of getting the remaining items in the treeview.
  */
  
  report = report_gs->str;
  g_string_free (report_gs, FALSE);
  
  return report;
}

gchar *
polyxedit_findmass_results_wnd_make_report_latex_format (PxmReportOpt* reportopt,
						      PxmWinMngmt* winmngmt)
{

  return NULL;
}









/* WINDOW LIFE-CYCLE FUNCTIONS.
 */
void
polyxedit_findmass_res_wnd_really_close (GtkWidget *window)
{
  GPtrArray *alloligsGPA = NULL;

  gint count = 0;

  g_assert (window != NULL);  


  /* This function is called as a GDestroyNotify callback function when
     the sequence editor window gets closed.
  */
  
  /* As a first step we want to un_register this window.
   */
  polyxmass_winmngmt_un_register_window (window, "POLYXEDIT");
  
  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
  

  /* We have a full array of arrays of oligomers to free before
     leaving this window! Also we have to free the array of fragopt
     objects.
  */
  alloligsGPA = g_object_get_data (G_OBJECT (window), "alloligsGPA");
  g_assert (alloligsGPA != NULL);
  count = pxmchem_oligomer_GPA_of_GPA_free (alloligsGPA);

  /*
    debug_printf (("freed %d oligomers\n", count));
  */

  gtk_widget_destroy (window);
}


gboolean
polyxedit_findmass_res_wnd_delete_event (GtkWidget *window,
					   GdkEvent *event,
					   gpointer data)
{
  GtkWidget *grand_parent = NULL;
  
  PxmEditCtxt *editctxt = data;

  GPtrArray *alloligsGPA = NULL;

  gchar *help = NULL;
  gint count = 0;
  

  g_assert (window != NULL);
  g_assert (editctxt != NULL);


  /* As a first step we want to un_register this window.
   */
  polyxmass_winmngmt_un_register_window (window, "POLYXEDIT");
  
  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
  
  /* We have a full array of arrays of oligomers to free before
     leaving this window! Also we have to free the array of fragopt
     objects.
  */
  alloligsGPA = g_object_get_data (G_OBJECT (window), "alloligsGPA");
  g_assert (alloligsGPA != NULL);
  count = pxmchem_oligomer_GPA_of_GPA_free (alloligsGPA);

  /*
    debug_printf (("freed %d oligomers\n", count));
  */

  /* This window pointer was set as a full datum to the sequence editor
     window, which means that we have to remove that pointer, without
     triggering the callback function call.
  */
  help = g_strdup_printf ("findmass_results_wnd-%p", window);
  
  grand_parent = g_object_get_data (G_OBJECT (window), "grand_parent");
  g_assert (grand_parent != NULL);
  
  window = g_object_steal_data (G_OBJECT (grand_parent), help);
  
  g_free (help);


  /* Let Gtk+ do the rest of the work.
   */
  return FALSE;
}




gboolean
polyxedit_findmass_res_wnd_destroy_event (GtkWidget *window,
					   GdkEvent *event,
					    gpointer data)
{
  return FALSE;
}


