/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005 SUSE Linux Products GmbH               *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#ifndef DEVICE_H
#define DEVICE_H

#include <list>
#include <string>

/** @brief this is the power state which gets written into the power state
 * file */
enum DPM_POWER_STATE { DPM_STATE_UNKNOWN = -1, DPM_STATE_D0, DPM_STATE_D1,
		       DPM_STATE_D2, DPM_STATE_D3 };

/** @brief a device class used to determine available devices */
enum DPM_DEVICE_CLASS { DPM_CLASS_USB, DPM_CLASS_LAN, DPM_CLASS_WLAN,
			DPM_CLASS_SOUND, DPM_CLASS_MAX };

/** @brief list of class names */
extern char *DPM_CLASS_NAMES[DPM_CLASS_MAX];

/** @brief device abstraction of a powermanagable device */
class Device {
public:
	/** @brief constructor
	 *
	 * @param udi the udi of the device given by hal
	 */
	Device(const std::string &udi);

	/** @brief destructor */
	virtual ~Device();

	/** @brief Set device into powersave state
	 *
	 * @param state the power state to set (default is DPM_STATE_D3)
	 * @param force suspend device even if active (default is false)
	 *
	 * @return true if successfull, false on error
	 */
	virtual bool suspend(DPM_POWER_STATE state = DPM_STATE_D3, bool force = false);

	/** @brief resumes a device from a powersave state
	 *
	 * @return true if successfull, false on error
	 */
	virtual bool resume();

	/** @brief check whether device is currently busy
	 *
	 * pure virtual
	 *
	 * @return true if device is busy
	 */
	virtual bool isActive() = 0;

	/** @brief get the @ref DPM_DEVICE_CLASS of the device
	 *
	 * @retval @ref DPM_DEVICE_CLASS
	 */
	virtual DPM_DEVICE_CLASS deviceClass();

	/** @brief get the udi assigned by hal of the device
	 *
	 * @retval udi of device
	 */
	virtual std::string udi();

	/** @brief get the current @ref DPM_POWER_STATE of the device
	 *
	 * reads out current power state from state file and updates local
	 * @ref _current_state variable
	 * 
	 * @return @ref DPM_POWER_STATE
	 */
	virtual DPM_POWER_STATE updateState();

	/** @brief read out current power state from file
	 *
	 * @param state_file power state file to read from
	 * 
	 * @return @ref DPM_POWER_STATE
	 */
	DPM_POWER_STATE readState(const std::string &state_file);

protected:
	/** @brief holding the udi assigned by hal*/
	std::string _udi;

	/** @brief the device class (@ref DPM_DEVICE_CLASS) */
	DPM_DEVICE_CLASS _device_class;

	/** @brief the device path in system
	 *
	 *  returned by linux.sysfs_path capability
	 */
	std::string _device_path;

	/** @brief the full path of the power state file 
	 *
	 * e.g. @ref _device_path + /device/power/state
	 */
	std::string _state_file;

private:
	/** @brief the current power state the device is in */
	DPM_POWER_STATE _current_state;
};


/** @brief class representing an usb device */
class UsbDevice : public Device {
public:
	/** @brief constructor
	 *
	 * @param udi the udi of the device given by hal
	 */
	UsbDevice(const std::string &udi);

	/** @brief destructor */
	~UsbDevice();

	/** @brief check whether device is currently busy
	 *
	 * @return true if device is busy
	 */
	virtual bool isActive();

	/** @brief get list of usb devices in system
	 *
	 * @param device_udis stl list which get filled with devices
	 */
	static void getDevices(std::list<std::string> &device_udis);

	/** @brief chef if a device is a device of this
	 *         @ref DPM_DEVICE_CLASS
	 * @param udi the device udi to check for
	 *
	 * @return true if the device is out of this class
	 */
	static bool isDevice(const char *udi);
};


/** @brief class representing an wlan device */
class WlanDevice : public Device {
public:
	/** @brief constructor
	 *
	 * @param udi the udi of the device given by hal
	 */
	WlanDevice(const std::string &udi);

	/** @brief destructor */
	~WlanDevice();

	/** @brief check whether device is currently busy
	 *
	 * @return true if device is busy
	 */
	virtual bool isActive();

	/** @brief get list of wlan devices in system
	 *
	 * @param device_udis stl list which get filled with devices
	 */
	static void getDevices(std::list<std::string> &device_udis);

	/** @brief chef if a device is a device of this
	 *         @ref DPM_DEVICE_CLASS
	 * @param udi the device udi to check for
	 *
	 * @return true if the device is out of this class
	 */
	static bool isDevice(char *udi);
};


/** @brief class representing an sound device */
class SoundDevice : public Device {
public:
	/** @brief constructor
	 *
	 * @param udi the udi of the device given by hal
	 */
	SoundDevice(const std::string &udi);

	/** @brief destructor */
	~SoundDevice();

	/** @brief check whether device is currently busy
	 *
	 * @return true if device is busy
	 */
	virtual bool isActive();

	/** @brief get list of sound devices in system
	 *
	 * @param device_udis stl list which get filled with devices
	 */
	static void getDevices(std::list<std::string> &device_udis);

	/** @brief chef if a device is a device of this
	 *         @ref DPM_DEVICE_CLASS
	 * @param udi the device udi to check for
	 *
	 * @return true if the device is out of this class
	 */
	static bool isDevice(const char *udi);
};


/** @brief class representing an network device */
class LanDevice : public Device {
public:
	/** @brief constructor
	 *
	 * @param udi the udi of the device given by hal
	 */
	LanDevice(const std::string &udi);

	/** @brief destructor */
	~LanDevice();

	/** @brief check whether device is currently busy
	 *
	 * @return true if device is busy
	 */
	virtual bool isActive();

	/** @brief get list of usb devices in system
	 *
	 * @param device_udis stl list which get filled with devices
	 */
	static void getDevices(std::list<std::string> &device_udis);

	/** @brief chef if a device is a device of this
	 *         @ref DPM_DEVICE_CLASS
	 * @param udi the device udi to check for
	 *
	 * @return true if the device is out of this class
	 */
	static bool isDevice(char *udi);
};

#endif // DEVICE_H
