/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005 SUSE Linux Products GmbH               *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#ifndef POWERSAVE_POWERLIB_H
#define POWERSAVE_POWERLIB_H

/** @defgroup libpower Libpower
 *
*/
#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#include <stdio.h>
#include <syslog.h>
#include <stdint.h>

/** @defgroup libpower_information Functions to gather acpi/apm information
 * @ingroup libpower
 * this is conform to the ACPI specification 2.0b (2002)
 *
 * <b>IMPORTANT !!!</b>
 * Test all data for UNKNOWN<br>
 * You never know if the kernel or the hw supports specific features!
 * @{ 
 */

/** @brief maximum length of a filepath
 *
 * Hope this is only under /proc/acpi where this is correct.
 */
#define MAX_FILE_PATH 512

/** @brief maximum length of a line */
#define MAX_LINE_SIZE 1024

/** @brief maximum number of batteries we support */
#define MAX_BATTERIES 4

/** @brief Max string length in ACPI Source Language: 200 bytes */
#define MAX_BAT_CHAR_LEN 200

#define MAX_THERMAL_ACTIVE 10

/** @brief error if acpi is not supported by the system */
#define NO_ACPI_ERROR -20

/** @brief error for unavailable module */
#define NO_MODULE_ERROR -10

/** @brief error for unavailable device */
#define NO_DEVICE_ERROR -5

/** @brief maximum number of supported cpus */
#define MAX_SUPPORTED_CPUS 32

/** @brief defining an unknown state
 *
 * be careful to test every variable on unknown
 */
#define UNKNOWN -1

/** @brief charging state is unknown */
#define CHARG_STATE_UNKNOWN	0
/* @brief battery is currently charging */
#define CHARG_STATE_CHARGING	1
/** @brief battery is currently discharging */
#define CHARG_STATE_DISCHARGING	2
/** @brief defines whether charging or discharging */
#define CHARG_STATE_CHARG_DISCHARG (CHARG_STATE_CHARGING | CHARG_STATE_DISCHARGING)

/** @brief acpi sleep state S1 */
#define ACPI_S1 1
/** @brief acpi sleep state S2 */
#define ACPI_S2 2
/** @brief acpi sleep state S3 */
#define ACPI_S3 4
/** @brief acpi sleep state S3 bios */
#define ACPI_S3_BIOS 8
/** @brief acpi sleep state S4 */
#define ACPI_S4 16
/** @brief acpi sleep state S4 bios */
#define ACPI_S4_BIOS 32
/** @brief apm standby */
#define APM_STANDBY 64
/** @brief apm suspend */
#define APM_SUSPEND 128

/* debugging macros and defines */
#ifdef POWERSAVE_DEBUG
#define DBG_ERR 1
#define DBG_WARN 2
#define DBG_DIAG 4
#define DBG_INFO 8
#define DBG_DEBUG 16

	extern int DEBUG_LEVEL;
#ifdef USE_SYSLOG
	static int syslog_open = 0;
// FIXME: (seife, 2006-08-04)
// pDebug does not seem to work from the library yet. Not sure if this is
// a build issue or a general problem.
// Because of this, it is no problem to hardcode the "powersaved" tag for now.
#define pDebug(level, string, args...) \
do{\
        if (!syslog_open){ \
                openlog("powersaved", LOG_PID, LOG_DAEMON); \
                syslog_open = 1; \
        } \
        if (DEBUG_LEVEL & DBG_ERR & level){ \
              syslog(LOG_ERR,     "ERROR (%s:%d) "string, __FUNCTION__, __LINE__, ## args); \
        }else if (DEBUG_LEVEL & DBG_WARN & level) { \
              syslog(LOG_WARNING, "WARNING (%s:%d) "string, __FUNCTION__, __LINE__, ## args); \
        }else if (DEBUG_LEVEL & DBG_DIAG & level) { \
              syslog(LOG_NOTICE,  "DIAG (%s:%d) "string, __FUNCTION__, __LINE__, ## args); \
        }else if (DEBUG_LEVEL & DBG_INFO & level) {\
              syslog(LOG_INFO,    "Info (%s:%d) "string, __FUNCTION__, __LINE__, ## args); \
        }else if (DEBUG_LEVEL & DBG_DEBUG & level) {\
              syslog(LOG_INFO,    "Debug (%s:%d) "string, __FUNCTION__, __LINE__, ## args); \
        } \
}while(0);
#else /* USE_SYSLOG */
#define pDebug(level, string, args...) \
do{\
        if (DEBUG_LEVEL & DBG_ERR & level){ \
              fprintf(stderr, "ERROR (%s:%d) "string"\n", __FUNCTION__, __LINE__, ## args); \
        }else if (DEBUG_LEVEL & DBG_WARN & level) { \
              fprintf(stderr, "WARNING (%s:%d) "string"\n", __FUNCTION__, __LINE__, ## args); \
        }else if (DEBUG_LEVEL & DBG_DIAG & level) { \
              printf("DIAG (%s:%d) "string"\n", __FUNCTION__, __LINE__, ## args); \
        }else if (DEBUG_LEVEL & DBG_INFO & level) {\
              printf("Info "string"\n", ## args); \
        }else {\
              printf("Debug "string"\n", ## args); \
        } \
}while(0);
/*              printf("[POWERSAVE] INFO in Function %s, line %d: "string, __FUNCTION__, __LINE__, ## args); \*/
#endif /* USE_SYSLOG */
#else /* POWERSAVE_DEBUG */
#define pDebug(level, string, args...) do {} while (0);
#endif /* POWERSAVE_DEBUG */
	
	/** @brief whether the battery is present */
	enum BATTERY_PRESENT { PRESENT_YES = 0, PRESENT_NO };
	
	/** @brief the unit of the battery capacity */
	enum CAPACITY_UNIT { WATT_H = 0, AMP };
	
	/** @brief technologie of the battery */
	enum BATTERY_TECH { TECH_NON_RECHARGEABLE = 0, TECH_RECHARGEABLE };
	
	/** because of kernel implementations separated in ACPI spec this
	 * is combined to battery_state 
	 * be careful this is kernel definition of batteries states !
	 */
	enum CAPACITY_STATE { STATE_OK, STATE_CRIT, STATE_LOW, STATE_WARN };

	/** @brief ac adapter status */
	enum AC_ADAPTER_STATE { AC_UNKNOWN, AC_ONLINE, AC_OFFLINE };
	
	/** @brief states a disk can be in */
	enum DISK_MODE { DISK_FULL_ON, DISK_STANDBY, DISK_SLEEP };

	/** @brief current state of the lid */
	enum LID_STATE { LID_OPEN, LID_CLOSED };

	/** @brief use for checkACPI function return value! */
	enum ISACPI { NOT_SUPPORTED = -1, APM, ACPI };

	/** @brief the current thermal mode
	 *
	 * modes used for active/passive cooling mode and for the devices'
	 * temperature state
	 */
	enum THERM_MODE { OK, ACTIVE, PASSIVE, HOT, CRITICAL };

	/** @brief cooling policy */
	enum _COOLING_POLICY { COOLING_MODE_ACTIVE = 0, COOLING_MODE_PASSIVE };
	
	/** @brief General battery information */
	typedef struct BatteryGeneral {
		/** @brief remaining percent of battery charge */
		int remaining_percent;
		
		/** @brief remaining time until discharged or fully charged
		 * depending on the context used */
		int remaining_minutes;
		
		/** @brief the current charging state
		 *
		 * @ref CHARG_STATE_UNKNOWN
		 * @ref CHARG_STATE_CHARGING
		 * @ref CHARG_STATE_DISCHARGING
		 * @ref CHARG_STATE_CHARG_DISCHARG
		 */
		int charging_state;
	} BatteryGeneral;
	
	/** @brief Detailed battery information
	 *
	 * only available for ACPI through the getBatteryDetailedInfo function
	 */
	typedef struct BatteryDetailed {
		/** @brief whether battery is currently present */
		int present;
		
		/* from /proc/acpi/battery/ /info */
		int design_capacity;
		int last_full_capacity;
		int battery_technology;
		int design_voltage;
		int design_capacity_warning;
		int design_capacity_low;
		int capacity_granularity_1;
		int capacity_granularity_2;
		char model_number[MAX_BAT_CHAR_LEN + 1];
		char serial_number[MAX_BAT_CHAR_LEN + 1];
		char battery_type[MAX_BAT_CHAR_LEN + 1];
		char OEM_info[MAX_BAT_CHAR_LEN + 1];
		
		/* from /proc/acpi/battery/ /status */
		int power_unit;
		int capacity_state;
		int charging_state;
		int present_rate;
		int remaining_capacity;
		int present_voltage;
		
		/* from /proc/acpi/battery/ /alarm */
		int alarm_limit;
	} BatteryDetailed;

	/** @brief object representing a thermal device */
	typedef struct ThermalDev {
		int present;
		int temperature;
		
		int cooling_mode;
		int polling_frequency;
		int critical;
		int hot;
		int passive;
		int active[MAX_THERMAL_ACTIVE];
		
		int state;
		int tc1;
		int tc2;
		int tsp;
	} ThermalDev;
	
	/** @brief fills the given Battery struct with the current battery
	 *  information for all primary batteries
	 *
	 * supports acpi and apm
	 *
	 * @param bat pointer to a BatteryGeneral struct
	 *
	 * @return integer with result of call
	 * @retval 0 on success
	 * @retval < 0 on error
	 */
	int getBatteriesInfo(BatteryGeneral * bat);

	/** @brief get Information for only one primary battery
	 *
	 * this function only returns values for primary batteries
	 *
	 * @param no number of primary battery to query starting from 0
	 * @param battery_info pointer to an @ref BatteryGeneral struct
	 * @retval 1 on success and battery is present
	 * @retval 0 on success but battery is not present
	 * @retval < 0 on error
	 */
	int getBatteryInfo(const int no, BatteryGeneral *battery_info);

	/** @brief get number of present primary batteries
	 *
	 * @return number of primary batteries (also not present ones) or
	 *         -1 on error
	 */
	int numBatteries(void);

	/** @brief use this method to get detailed information about a
	 *         primary battery
	 *
	 * <b>ACPI only!</b>
	 *
	 * @param no battery number
	 * @param bd pointer to detailed battery info structure
	 *
	 * @return: integer with result of call
	 * @retval 0 SUCCESS
	 * @retval < 0 ERROR
	 * @retval -1 general error
	 * @retval -2 if device no does not exist
	 * @retval -3 if struct has not initialized with memory
	 * @retval NO_ACPI_ERROR if it is not an ACPI system 
	 * @retval NO_MODULE_ERROR if the battery module is not loaded 
	 * @retval NO_DEVICE_ERROR if there is a fan directory, but no device 
	 * @retval > 0 END OF BATTERIES REACHED
	 */
	int getBatteryDetailedInfo(const int no, BatteryDetailed * bd);

	/** @brief sets the ACPI battery alarm for all batteries
	 *
	 * <b>ACPI only!</b><br>
	 */
	int setBatteryAlarm(const int percent);

	/** @brief returns the current state of the lid
	 *
	 * @return @ref LID_STATE
	 */
	int getLidState(void);

	/** @brief get throttling information (ACPI only)
	 *
 	 * <b>ACPI only!</b>
	 *
	 * @param num_states will be filled with number of states
	 * @param current_state will be filled with current state
	 *
	 * @return < 0 on error
	 * @retval NO_ACPI_ERROR
	 * @retval NO_MODULE_ERROR
	 * @retval -1 on general error
	 */
	int getThrottlingInfo(int *num_states, int *current_state);

	/** @brief set throttling percent
	 *
	 * <b>ACPI only!</b>
	 *
	 * @param percent percent value to which should be throttled
	 *
	 * @return < 0 on error
	 */
	int setThrottlingPercent(const int percent);

	/** @brief get throttling info for one single cpu
	 *
	 * <b>ACPI only!</b>
	 *
	 * @param cpu number of cpu starting with 0
	 * @param num_states integer where number of states will be stored
	 * @param current_state integer where current state will be stored
	 *
	 * @return < 0 on error
	 * @retval NO_ACPI_ERROR
	 * @retval NO_MODULE_ERROR
	 * @retval -2 if cpu does not exist
	 * @retval -3 on general error
	 */
	int getThrottlingInfoCPU(const int cpu, int *num_states, int *current_state);

	/** @brief set throttling state on cpu
	 *
	 * <b>ACPI only!</b>
	 *
	 * @param cpu number of cpu starting with 0
	 * @param state state to set
	 *
	 * @return < 0 on error
	 * @retval NO_ACPI_ERROR
	 * @retval NO_MODULE_ERROR
	 * @retval -3 on general error
	 */
	int setThrottlingStateCPU(const int cpu, const int state);

	/** @brief set throttling percent for single cpu
	 *
	 * <b>ACPI only!</b>
	 *
	 * @param cpu number of cpu starting with 0
	 * @param percent percent value to which should be throttled
	 *
	 * @return < 0 on error
	 */
	int setThrottlingPercentCPU(const int cpu, const int percent);

	/** @brief get the number of thermal zones 
	 *
	 * <b>ACPI only!</b>
	 *
	 * @return < 0 on error
	 */
	int getThermalZonesNum(void);

	/** @brief get thermal zone temp
	 *
	 * <b>ACPI only!</b>
	 *
	 * @param zone value from 0-n zones
	 *
	 * @return < 0 on error
	 * @retval NO_ACPI_ERROR
	 * @retval NO_MODULE_ERROR
	 * @retval -3 general error (too long file name or similar)
	 * @retval -1 if thermal_zone zone does not exist
	 */
	int getThermalZoneTemp(const int zone);

	/** @brief get thermal zone state
	 *
	 * <b>ACPI only!</b>
	 *
	 * @param zone value from 0-n zones
	 *
	 * @return < 0 on error
	 * @retval NO_ACPI_ERROR
	 * @retval NO_MODULE_ERROR
	 * @retval -3 general error (too long file name or similar)
	 * @retval -1 if thermal_zone zone does not exist
	 */
	int getThermalZoneState(int zone);

	/** @brief get a thermal zone 
	 *
	 * <b>ACPI only!</b>
	 *
	 * @param zone value from 0-n zones
	 * @param td @ref ThermalDev
	 * 
	 * @return < 0 on error
	 * @retval NO_ACPI_ERROR
	 * @retval NO_MODULE_ERROR
	 * @retval -3 if thermal device no. num does not exist
	 * @retval -4 if td is not allocated
	 */
	int getThermalZone(const int zone, ThermalDev * td);

	/** @brief set thermal trippoints
	 *
	 * <b>ACPI only!</b>
	 *
	 * @param zone value from 0-n zones
	 * @param td @ref ThermalDev
	 *
	 * @return < 0 on error
	 * @retval NO_ACPI_ERROR
	 * @retval NO_MODULE_ERROR 
	 * @retval -3 if thermal device no. num does not exist 
	 */
	int setThermalTrippoints(const int zone, const ThermalDev td);

	/** @brief set cooling mode 
	*
	* @param device_num the device number
	* @param mode the cooling mode
	*
	* @return integer value repesenting result of call func
	* @retval NO_ACPI_ERROR 
	* @retval NO_MODULE_ERROR
	* @retval -1 general error -> could not be set
	* @retval -2 if cooling_mode is not supported
	* @retval -3 if thermal device no. num does not exist
	*/
	int setCoolingMode(const int device_num, const int mode);

	/** @brief get number of fans in the system
	 *
	 * <b>ACPI only!</b>
	 *
	 * @return number of devices, < 0 on error
	 */
	int getFanNum(void);

	/** @brief get fan status
	 *
	 * @param device_num
	 *
	 * @return integer with result of get fan status 
	 * @retval NO_ACPI_ERROR 
	 * @retval NO_MODULE_ERROR
	 * @retval NO_DEVICE_ERROR
	 * @retval 0 if fan is on 
	 * @retval 1 if fan is off
	 * @retval -3 if fan device no. num does not exist
	 * @retval -4 if state file does not exist 
	 * @retval -5 if state could not be figured out
	 *           (probably changes in kernel syntax) 
	 */
	int getFanStatus(const int device_num);

	/** @brief returns if the machine is on ac power or not
	 *
	 * apm + acpi
	 *
	 * @return < 0 on error
	 * @retval NO_MODULE_ERROR
	 * @retval NO_DEVICE_ERROR
	 * @retval AC_ADAPTER_STATE
	 */
	int getACAdapterStatus(void);

	/** @brief get processor speed
	 * 
	 * be careful this method waits a second ...
	 *
	 * Function only available on i686 and x86_64 machines!
	 *
	 * @return current processor speed
	 * @retval -1.0 if sleep is interrupted, result will be wrong
	 * @retval -2.0 if compiled for a not supported architecture
	 * @retval -3.0 if /proc/cpuinfo is not accessable and supported
	 *         for architecture could not be evaluated
	 */
	float getRealProcessorSpeed(void);

	/** @brief calculate cpu load
	 *
	 * @param consider_nice give a nice value
	 *
	 * @return "Idleness" of the processor in percent 
	 *         since the last time this function has been called 
	 * @retval UNKNOWN if not supported
	 * 
	 * if you poll idleness of processor, ignore the result of the
	 * first time you invoke this function, because it will be the
	 * total idleness of cpu since the machine has been powered on
	 */
	int calcCPULoad(const int consider_nice);

	/** @brief get current cpu load
	 *
	 * @param cpu the cpu to use starting at 0
	 *
	 * @return the current cpu load
	 */
	int getCPULoad(const int cpu);

	/** @brief free data for calculating cpu time
	 *
	 * needs to be invoked, once the getCPULoad function is called to
	 * free memory. The getCPULoad may be called multiple times, when
	 * not using the func anymore call this one
	 */
	void freeCPULoadData(void);

	/** @brief get number of cpus in system
	 *
	 * @return number of cpus, < 0 on error
	 */
	int getCPUCount(void);

	/** @brief get all supported sleeping states
	 *
	 * mask out return value to evaluate supported states
	 *
	 * @retval 0 if no support
	 */
	int64_t getSupportedSleepStates(void);

	/** @brief determine whether there is apm, acpi or nothing on your Computer
	 *
	 * @return integer values representing acpi status
	 * @retval ISACPI
	 */
	int checkACPI(void);

	/** @brief set the debug level. Values: 0-31
	 *
	 * if library is compiled without -DPOWERSAVE_DEBUG variable this
	 * method has no effect. Define -D USE_SYSLOG to use the syslog
	 * daemon for debugging
	 *
	 * sum up the levels you want to see: e.g.: 7 -> error + warning + diag
	 * 
	 * levels:
	 *     - 0 = nothing
	 *     - 1 = error
	 *     - 2 = warning
	 *     - 4 = diag
	 *     - 8 = info
	 *     - 16= debug
	 *
	 * @param dbg the debug level
	 */
	void setDebugLevel(const int dbg);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* POWERSAVE_POWERLIB_H */
/** @} */// end of libpower_information group
