package PSP::FieldSpace;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: FieldSpace.pm,v 1.3 2000/12/03 15:18:42 muaddib Exp $

use strict;

use Field;
use PSP::share;
use PSP::Utils qw(&dump_object);
use PSP::Error::Field;
use PSP::Error::Verify;

@PSP::FieldSpace::ISA = qw(PSP::share);

use vars qw(@hashes @arrays);
@hashes = (
	   'field_defs',  # field definitions (read-only)
	   'group_defs',  # group definitions (read-only)
	   'verify_defs', # verify definitions (read-only)
	   'fields',      # instantiated fields
	   'groups',      # instantiated groups
	   'verifies',    # things to verify and not verify
	   'errors'       # encountered errors
	  );
@arrays = (
	  );

=head2 new

 class
 (PSP::FieldSpace $fs)new(CGI $cgi,string share_class,string propagation_class)

DESCRIPTION:

This function takes in a CGI object and returns a fieldspace object of
the appropriate type.  Since most of the action takes place in
C<initialize>, the only interesting action of this function is to call
the constructor on the parent.

=cut

sub new {
  my ($proto,$prop_class) = @_;
  $prop_class ||= "PSP::FieldSpace::Propagation";

  # create the object.
  my $this = {};
  bless $this, ref($proto) || $proto;

  # create the propagation class.
  $this->{propagation} = $prop_class->new($this);

  # initialize the members.
  ($this->{name} = ref($this)) =~ s/^(Pile|FieldSpace):://;
  map { $this->{$_} ||= {} } @hashes;
  map { $this->{$_} ||= [] } @arrays;

  return $this;
}

=head2 initialize

 instance/private
 (PSP::FieldSpace $this) initialize (CGI $cgi, 
				     string $share_class, 
				     HASH no_prop_in)

DESCRIPTION:

This is where all the action takes place. It will look at what groups
have been requested and will then make sure that all these and all
required fields are instantiated properly. The initialized fields will
have the proper values and all error messages properly associated with
them.

=cut

sub initialize {
  my ($this, $cgi, $share_class, $no_prop_in) = @_;

  # assign the CGI member and the variable sharing source class.
  $this->cgi($cgi);
  $this->source_share_class($share_class);

  # Retrieve any stored state.
  $no_prop_in or $this->retrieve_state($cgi);

  # Acquire active field values from CGI object.
  $this->import_cgi($cgi);

  # Fully instantiate each field.
  my $empty_sub = sub { };
  $this->scan_fields($empty_sub);
}

sub scan_fields {
  my ($this,$sub) = @_;

  my $ret_val;
  my $out = "";
  # Iterate through all static fields.
  my $field;
  for my $field_name ($this->fields()) {
    $field = $this->get_field($field_name);
    no strict;
    $ret_val = &{$sub}($field,$field_name);
    defined $ret_val and $out .= $ret_val;
    use strict;
  }

  # Iterate through each group.
  my $group;
  for my $gname ($this->groups()) {
    $group = $this->group($gname) or next;
    $ret_val = $group->scan_fields($sub);
    defined $ret_val and $out .= $ret_val;
  }

  return $out;
}

sub cgi {
  my ($this, $val) = @_;
  $this->{cgi} = $val if defined $val;
  return $this->{cgi};
}

sub import_cgi {
  my ($this, $cgi, @fields) = @_;

  my $nfound = 0;

  # iterate through all possible fields.  populate hashes.
  my (%fields,%dfields,$field_def);
  for my $field_name ($this->field_names()) {
    # get the characteristics of this field.
    $field_def = $this->field_def($field_name);
    $fields{$field_name} = 1;
    if (my $gname = $field_def->{group}) {
      $dfields{$field_name} = $gname;
    }
  }

  # prepare an optimization for the indexed param search.
  my $sub = sub { return ($_[0] =~ /^(\w+)(?:(?::)(\d+))?$/) };

  # iterate through all parameters.
  my ($field_name,$index,$gname);
  for my $param ($cgi->param()) {

    # see if this parameter matches
    no strict;
    ($field_name,$index) = &{$sub}($param);
    use strict;

    # continue if not
    next unless $field_name and $fields{$field_name};

    $nfound++;

    # if this field is dynamic..
    if ($gname = $dfields{$field_name}) {
      if (! $index) {
	warn "index expected for $field_name parameter\n";
	next;	
      } else {
	# set the cursor for put_field().
	$this->set_cursor($gname,$index);
      }
    } elsif ($index) {
      warn "index=$index unexpected for $field_name parameter\n";
      next;
    }

    # get the values for this parameter
    my @values = $cgi->param($param);
    # possibly create a field, and set that field to these values.
    $this->put_field($field_name,@values);

    # clear the cursor after put_field().
    $gname and $this->set_cursor($gname,0);
  }

  return @fields;
}

sub set_cursor {
  my ($this,$gname,$index) = @_;
  my $group = $this->group($gname) or return;
  return $group->set_cursor($index);
}

sub setup {
  my($this,$cgi) = @_;
  warn "This function should have been overridden by subclass.";
}

sub was_setup {
  my ($this,$val) = @_;
  defined $val and $this->{was_setup} = $val;
  $this->{was_setup};
}

sub field_was_setup {
  my ($this,$field_name,$val) = @_;

  # get the field definition for this field.
  my $field_def = $this->field_def($field_name);
  if (my $gname = $field_def->{group}) {
    my $group = $this->group($gname) or return;
    return $group->field_was_setup($field_name,$val);
  } else {
    # if a value is given, assign it.
    defined $val and $this->{field_was_setup}->{$field_name} = $val;
    # return the value of this field.
    return $this->{field_was_setup}->{$field_name};
  }
}

sub verify_def {
  my ($this,$name) = @_;
  return $this->{verify_defs}->{$name};
}

sub group_def {
  my ($this,$name,$group_def) = @_;
  if ($group_def) {
    $this->{group_defs}->{$name} = $group_def;
  } else {
    $group_def = $this->{group_defs}->{$name};
  }
  return $this->{group_defs}->{$name};
}

sub field_def {
  my ($this,$name,$field_def) = @_;

  if ($field_def) {
    $this->{field_defs}->{$name} = $field_def;
  } else {
    $field_def = $this->{field_defs}->{$name};
  }

  if (! $field_def) {
    my $warning = ("Reference to undefined field: ".
		   "$this->{name} has no definition for $name");
    #throw Error::Simple($warning);
    warn $warning;
    $field_def = { type => 'HTMLIO::Text',
		   data => 'AtomicData::AnyThing',
		   auto_generated => 1
		 };
    $this->{field_defs}->{$name} = $field_def;
  }

  return $field_def;
}

sub field_setup {
  my ($this, $field_name, $field) = @_;
  my $field_def = $this->field_def($field_name);
  my $sub = $field_def->{setup_sub} or return;
  $field ||= $this->get_field($field_name,1);
  no strict;
  return &{$sub}($this,$field);
  use strict;
}

sub field_names {
  my ($this) = @_;
  return sort keys %{$this->{field_defs}};
}

sub cgi_param_name {
  my ($this,$name,$gname,$index) = @_;
  my $param_name = $name;
  $index and $param_name .= ":$index";
  return $param_name;
}

sub from_cgi_param_name {
  my ($this,$param_name) = @_;
  my ($field_name,$gname,$index,$field_def);
  if ($param_name =~ /^(.*):(\d+)$/) {
    $field_name = $1;
    $index = $2;
    $field_def = $this->field_def($field_name);
    $gname = $field_def->{group};
  } else {
    $field_name = $param_name;
  }
  return ($field_name,$gname,$index);
}

=head2 fields

 instance
 (string[] @fields) fields ()

 DESCRIPTION:

Returns a list of all the fields present in the fieldspace.

=cut

sub fields {
  my ($this,$gname) = @_;
  if ($gname) {
    my $group = $this->group($gname) or return;
    return $group->fields();
  } else {
    return sort keys %{$this->{fields}};
  }
}

sub field {
  my ($this,$field_name,$field) = @_;
  defined $field and $this->{fields}->{$field_name} = $field;
  return $this->{fields}->{$field_name};
}

sub group {
  my ($this,$gname) = @_;
  if (!$this->{groups}->{$gname}) {
    my $group_def = $this->group_def($gname) or return;
    my $pkg = $group_def->{package};
    $this->{groups}->{$gname} = $pkg->new($this,$group_def);
  }
  return $this->{groups}->{$gname};
}

sub groups {
  my ($this) = @_;
  return sort keys %{$this->{groups}};
}

sub group_names {
  my ($this) = @_;
  return sort keys %{$this->{group_defs}};
}

=head2 get_field

 private/instance
 (field $field) get_field (string $name)

DESCRIPTION:

Will return the field contained in the field space referenced by
C<$name>.

=cut

sub get_field {
  my ($this, $field_name, $skip_setup) = @_;
  #print STDERR "get_field $field_name: entered\n";

  # get the field definition for this field.
  my $field_def = $this->field_def($field_name);

  # get the possible group and index for this field.
  my ($gname,$group,$index);
  if ($gname = $field_def->{group}) {
    $group = $this->group($gname);
    $index = $group->cursor();
  }

  # attempt to get the field from a cache.
  my $field;
  if ($group) {
    $field = $group->field($field_name,$index);
  } else {
    $field = $this->field($field_name);
  }

  # if there is no field yet, create it.
  if (! $field) {
    #print STDERR "get_field $field_name: CREATION\n";

    # get field creation information.
    my $data = $field_def->{data};
    my $type = $field_def->{type};
    my $container = $field_def->{container};

    # create this field.
    $field = Field->new($field_name,$data,$type,$container);

    if (my $set_sub = $field_def->{set_sub}) {
      no strict;
      &{$set_sub}($this,$field);
      use strict;
    }

    # transfer any field_def parameters.
    defined $field_def->{blank_ok} and
      $field->set_parameter( blank_ok => $field_def->{blank_ok} );

    # cache this value.
    if ($group) {
      $field->name($field_name.":".$index);
      $group->field($field_name,$index,$field);
    } else {
      $this->field($field_name,$field);
    }
  }

  # if this field hasn't been setup ..
  if (! $skip_setup and ! $this->field_was_setup($field_name)) {
    #print STDERR "get_field $field_name: SETUP\n";

    # call fieldspace setup if we haven't already.
    $this->was_setup() or ($this->setup() and $this->was_setup(1));

    # call field setup.
    $this->field_setup($field_name,$field);
    $this->field_was_setup($field_name,1);

    # make sure value is at least blank.
    defined $field->value() or $field->set_value("");

    # call format after field setup.
    $field->format(1);

    # force the current value to be the original value.
    $field->set_orig_value();
  }

  return $field;
}

sub put_field {
  my ($this,$field_name,@value) = @_;
  #print STDERR "put_field $field_name: @value\n";
  return $this->set_value($field_name,@value);
}

=head2 get_value

 private/instance
 () get_value (string $field_name, values[])

DESCRIPTION:

Will return the value of the field contained in the field space referenced
by C<$field_name>.

=cut

sub get_value {
  my ($this,$field_name) = @_;
  my $field = $this->get_field($field_name) or throw
    Error::Simple("get_field() called on non-existent field, $field_name");
  return $field->value();
}

=head2 set_value

 private/instance
 () set_value (string $field_name, values[])

DESCRIPTION:

Will set the value of the field contained in the field space referenced
by C<$field_name>.

=cut

sub set_value {
  my ($this,$field_name,@value) = @_;
  my $field = $this->get_field($field_name) or throw
    Error::Simple("set_field() called on non-existent field, $field_name");
  $field->set_value(@value);
  # call format after value assignment.
  $field->format(1);
  return $field;
}

=head2 get_alias

 private/instance
 (string) get_alias (string $field_name)

DESCRIPTION:

Will return the alias of the field contained in the field space referenced
by C<$field_name>.

=cut

sub get_alias {
  my ($this,$field_name) = @_;
  my $field = $this->get_field($field_name) or throw
    Error::Simple("get_field() called on non-existent field, $field_name");
  return $field->alias();
}

=head2 set_alias

 private/instance
 (string) set_alias (string $field_name)

DESCRIPTION:

Will return the alias of the field contained in the field space referenced
by C<$field_name>.

=cut

sub set_alias {
  my ($this,$field_name,$new_alias) = @_;
  my $field = $this->get_field($field_name) or throw
    Error::Simple("get_field() called on non-existent field, $field_name");
  return $field->alias($new_alias);
}

=head2 get_html_input

 private/instance
 (string) get_html_input (string $field_name, string slice[])

DESCRIPTION:

Will return the value of the field contained in the field space referenced
by C<$field_name>.

=cut

sub get_html_input {
  my ($this,$field_name,$slice) = @_;
  my $field = $this->get_field($field_name) or throw
    Error::Simple("get_field() called on non-existent field, $field_name");
  return $field->html_input($slice);
}

=head2 verify

 [private] instance
 () verify ()

 DESCRIPTION:

blah blah 

This function runs verification on all fields indicated by the member
lists vfields and dvfields (the latter being for dynamic fields). It
will also run all arbitrary tests associated with the keys of the hash
referenced by C<$this->{verifies}>, and if the test proves false, the
function keyed by the test will be associated with an error object so
that that C<$error_obj->print_report()>.

Note that fields, verifications, etc. marked for verification will
override (take precedence over) those marked for deletion from error.

=cut

sub add_error {
  my ($this, $error_type, $message, @labels) = @_;
  my $errors = $this->{errors} ||= {};
  my $error_obj = $errors->{$error_type};
  if (! $error_obj) {
    my %err_classes = 
      ('field'  => 'PSP::Error::Field',
       'verify' => 'PSP::Error::Verify');
    my $class = $err_classes{$error_type} || 'PSP::Error';
    $error_obj = $errors->{$error_type} = $class->new($error_type,$this);
  }
  return $error_obj->add_error($message,@labels);
}
sub remove_error {
  my ($this, $error_type, @label) = @_;
  my $errors = $this->{errors} or return;
  my $error_obj = $errors->{$error_type} or return;
  return $error_obj->remove_error(@label);
}

=head2 error_report

 instance
 (string) error_report ()

 DESCRIPTION:

=cut

sub error_report {
  my ($this) = @_;
  return "" unless $this->errors_p();

  my $content = join("\n",
	('<table width="100%" border="0">',
	 '<tr><td>',
	 '<font color="#ff0000"><b>Errors Were Encountered</b></font>',
	 '</td></tr>'))."\n\n";

  my %to_process;
  @to_process{ %{$this->{errors}||{}} } = 1;

  my $field_errors;
  for my $err (qw(field verify), sort keys %to_process) {
    delete $to_process{$err} or next;
    my $error_obj = $this->{errors}->{$err} or next;

    $content .= "<tr><td>";
    $error_obj->isa('field') and $field_errors++;
    $content .= $error_obj->as_bullets();
    $content .="</td></tr>\n";
  }

  if ($field_errors) {
    $content .=
      ('<tr><td>'.
       'Red asterisks (<font color="#ff0000">*</font>) '.
       "indicate fields in error.</td></tr>\n");
  }

  $content .= "</table>\n<br>\n";

  return $content;
}

=head2 verify

 [private] instance
 () verify ()

 DESCRIPTION:

This function runs verification on all fields indicated by the member
lists vfields and dvfields (the latter being for dynamic fields). It
will also run all arbitrary tests associated with the keys of the hash
referenced by C<$this->{verifies}>, and if the test proves false, the
function keyed by the test will be associated with an error object so
that that C<$error_obj->print_report()>.

Note that fields, verifications, etc. marked for verification will
override (take precedence over) those marked for deletion from error.

=cut

sub verify {
  my ($this) = @_;

  $this->verify_fields();
  $this->verify_verifies();

  return ();
}

sub verify_fields {
  my ($this,$vfields,$rvfields) = @_;
  $vfields  ||= $this->{verifies}->{field}  || {};
  $rvfields ||= $this->{verifies}->{rfield} || {};

  my $n_errors = 0;
  for my $field_name (sort keys %$vfields) {
    # do not verify field if we would just remove that field anyway.
    next if $rvfields->{$field_name};

    # get the field definition.
    my $field_def = $this->field_def($field_name);

    # verify dynamic fields in a group method.
    $field_def->{group} and next;

    # get the field.
    my $field = $this->get_field($field_name) or throw
      Error::Simple("$this->{name} contains no '$field_name'");

    # verify the field.
    my ($success,$problems) = $field->verify();
    if (! $success) {
      for my $msg (@$problems) {
	$this->add_error("field",$msg,$field_name);
      }
      $n_errors++;
    }

  }#for field_name

  # verify dynamic fields.
  for my $gname ($this->groups()) {
    my $group = $this->group($gname) or next;
    $n_errors += $group->verify_fields($vfields,$rvfields);
  }

  # remove any ignored fields.
  if (my $ferror = $this->{errors}->{field}) {
    for my $field_name (keys %$rvfields) { 
      $ferror->remove_error('field',$field_name);
    }
  }

  return $n_errors;
}

sub verify_verifies {
  my ($this,$verifies,$rverifies,$rvfields) = @_;
  $verifies  ||= $this->{verifies}->{verify}  || {};
  $rverifies ||= $this->{verifies}->{rverify} || {};
  $rvfields  ||= $this->{verifies}->{rfield}  || {};

  my $n_errors = 0;

  for my $verify_name (sort keys %$verifies) {
    # do not do verify if we would just remove that verify error anyway.
    next if $rverifies->{$verify_name};

    my $verify = $this->verify_def($verify_name) or throw
      Error::Simple("Bad verification: verify $verify_name undefined.");

    #print STDERR "executing: $verify->{test}\n";
    my $fs = $this;
    my ($bool, @args) = eval $verify->{test};

    if (! $bool) {
      my $text = "";
      (defined $@ and $@) and
	$text .= "(Verification failed: $verify_name: $@)\n";

      my $method = $verify->{disp_method};
      if (! $method) {
	$text .= "Verify $verify_name failed without display method\n";
      } elsif (! $this->can($method)) {
	$text .= "Verify $verify_name failed.  bad display method: $method\n";
      } else {
	$text .= $this->$method(@args);
      }
      $this->add_error('verify',$text,$verify_name);
      $n_errors++;
    }
  }

  # remove any ignored fields that verifies may have put in error.
  if (my $ferror = $this->{errors}->{field}) {
    for my $field_name (keys %$rvfields) {
      $ferror->remove_error('field',$field_name);
    }
  }

  # remove any ignored verifies.
  if (my $verror = $this->{errors}->{verify}) {
    for my $verify_name (keys %$rverifies) {
      $verror->remove_error('verify',$verify_name);
    }
  }

  return $n_errors;
}

=head2 changed_p

 instance
 (boon $changed) changed_p (string $field_name)

DESCRIPTION:

Returns true if C<$field_name> has changed through the action of
C<$cgi>; will return false if the changes where by the action of
C<set_value> or the default setting.

=cut

sub changed_p {
  my ($this, $field_name) = @_;
  my $field = $this->get_field($field_name);
  return $field->changed_p();
}

# aggregation for propagation.
sub retrieve_state { 
  my ($this,@args) = @_;
  return $this->{propagation}->retrieve_state($this,@args);
}
sub propagate {
  my ($this,@args) = @_;
  return $this->{propagation}->propagate($this,@args);
}

=head2 add_verify

 [private] instance
 () add_verify (string $verify_name)

 DESCRIPTION:

Adds C<$verify_name> to the internal hash C<$this->{verifies}> such that: 

 $this->{verifies}->{$verify_name}->{test} = $test

and

 $this->{verifies}->{$verify_name}->{action} = $output_func;

When C<$this->verify> is called, all tests associated with the
keys off C<$this->{verifies}> will be run, and those that return a false
value will be remembered on C<$this->{verrors}> such that when
C<$error_obj->print_report()> is called with the error object associated
with this error, the code contained within C<$output_func> will be
called.

=cut

sub add_verify {
  my ($this, @label) = @_;
  my $node = $this->{verifies} ||= {};
  my $verify_name = pop @label or return;
  map { $node = $node->{$_} ||= {} } @label;
  $node->{$verify_name}++;
}

=head2 errors_p

 instance
 (bool $errors_p) errors_p ()

DESCRIPTION:

Returns true if the any of the fields (singletons or within groups),
or verifies are in error.

=cut

sub errors_p {
  my ($this, @types) = @_;
  return unless $this->{errors};
  @types or @types = qw(field verify); 
  map { return 1 if $this->{errors}->{$_} } @types;
  return;
}

=head2 in_error

 instance
 (bool $in_error) in_error (string $field_name)

DESCRIPTION:

Returns true if the fieldspace believes that C<$field_name> is
currently in error.

=cut

sub in_error {
  my ($this, $type, $name, @label) = @_;
  return unless $this->{errors};
  my $node = $this->{errors}->{$type} or return;
  unshift @label, $name;
  for $name (@label) {
    $node = $node->find_child($name) or return;
  }
  return 1;
}

=head2 add_poss_changed_to_verify and remove_poss_changed_errors

 [private] instance
 () add_poss_changed_to_verify()

 [private] instance
 () remove_poss_changed_errors ()

DESCRIPTION:

blah blah

=cut

sub add_poss_changed_to_verify {
  my ($this) = @_;
  my $add_sub = sub {
    my ($field,$field_name,$gname,$index) = @_;
    $field->poss_changed_p() or return;
    $this->add_verify('field',$field_name);
  };
  $this->scan_fields($add_sub);
}
sub remove_verifies {
  my ($this) = @_;
  my $remove_sub = sub {
    my ($field,$field_name,$gname,$index) = @_;
    $field->poss_changed_p(0);
  };
  $this->scan_fields($remove_sub);
  $this->{verifies} = {};
}
sub remove_poss_changed_errors {
  my ($this) = @_;
  my $remove_sub = sub {
    my ($field,$field_name,$gname,$index) = @_;
    $field->poss_changed_p() and $this->add_verify('rfield',$field_name);
  };
  $this->scan_fields($remove_sub);
}

=head2 add_instantiated_to_verify and remove_instantiated_errors

 [private] instance
 () add_instantiated_to_verify

 [private] instance
 () remove_instantiated_errors ()

DESCRIPTION:

add_instantiated_to_verify() adds all fields currently within the
fieldspace to the list of fields to be verified.

remove_instantiated_errors() will remove all instantiated fields.

=cut

sub add_instantiated_to_verify {
  my ($this) = @_;
  my $add_sub = sub { 
    my ($field,$field_name,$gname,$index) = @_;
    $this->add_verify('field',$field_name);
  };
  $this->scan_fields($add_sub);
}
sub remove_instantiated_errors {
  my ($this) = @_;
  my $remove_sub = sub { 
    my ($field,$field_name,$gname,$index) = @_;
    $this->add_verify('rfield',$field_name);
  };
  $this->scan_fields($remove_sub);
}

sub dumper {
  my ($this) = @_;
  my $out = "";
  $out .= dump_object($this,'$Fieldspace',
		      [qw(propagation cgi groups)])."\n";
  $out .= dump_object($this->{propagation},'$Propagation',
		      [qw(fieldspace)])."\n";
  for my $gname ($this->groups()) {
    $out .= dump_object($this->group($gname),'$Group::'.$gname,
			[qw(fieldspace)])."\n";
  }
  return $out;
}

sub free_internals {
  my ($this) = @_;

  delete $this->{cgi};

  my $propagation = $this->{propagation};
  $propagation->free_internals();
  delete $this->{propagation};

  for my $error_name (keys %{$this->{errors}}) {
    my $error = $this->{errors}->{$error_name};
    $error->free_internals();
    delete $this->{errors}->{$error_name};
  }
  delete $this->{errors};

  for my $gname (sort keys %{$this->{groups}}) {
    my $group = $this->{groups}->{$gname};
    $group->free_internals();
    delete $this->{groups}->{$gname};
  }

  my $free_sub = sub {
    my ($field,$field_name,$gname,$index) = @_;
    $field->free_internals();
  };
  $this->scan_fields($free_sub);
  return;
}

package PSP::Pile;

sub fieldspace {
  my ($this,$fsname) = @_;
  my $fs = $this->{fieldspaces}->{$fsname};
  if (! $fs) {
    my $fspackage;
    if ($fsname =~ /::/) {
      $fspackage = $fsname;
    } else {
      $fspackage = 'FieldSpace::'.$this->name()."::$fsname";
    }
    my $cgi = $this->cgi();
    eval { $fs = $fspackage->new() };
    (!$fs or $@) and throw
      Error::Simple("Error creating $fsname fieldspace: $@");
    $this->{fieldspaces}->{$fsname} = $fs or return;
    $fs->initialize($cgi,ref($this));
  }
  return $fs;
}

sub fieldspaces {
  my ($this) = @_;
  return sort keys %{$this->{fieldspaces}};
}

sub free_fieldspaces {
  my ($this) = @_;
  for my $fsname (sort keys %{$this->{fieldspaces}}) {
    my $fs = $this->{fieldspaces}->{$fsname};
    $fs->free_internals();
    delete $this->{fieldspaces}->{$fsname};
  }
  delete $this->{loader};
  return;
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<AtomicData>, L<HTMLIO>, L<Field>.

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
