#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <qstring.h>
#include <qlabel.h>
#include <qslider.h> 
#include <qhbox.h>
#include <qvbox.h>
#include <qpushbutton.h>
#include <qcheckbox.h>
#include <qprocess.h>
#include <qsocketnotifier.h>
#include <qfiledialog.h>
#include <qwaitcondition.h>
#include <alsa/asoundlib.h>
#include "gui.h"
#include "recdata.h"
#include "meter.h"

Gui::Gui(bool enableJack, QString pcmName, long periodSize, unsigned long ringBufSize, QWidget *parent, const char *name) : QVBox(parent, name) {

  char buf[2048];
  int l1;
 
  recdata = new RecData(pcmName, ringBufSize, this); 
  recdata->periodsize = periodSize;
  recdata->enableJack = enableJack;
  getcwd(buf, 2048);
  recdata->currentDir = QString(buf);
  aboutWidget = new QMessageBox(this);
  open_seq(&seq_handle, in_ports, out_ports, 1, 0);
  initSeqNotifier();
  if (recdata->enableJack) {
    recdata->initJack();
  }
  QHBox *fileBox = new QHBox(this);
  fileBox->setMargin(10);
  QHBox *timeBox = new QHBox(this);
  timeBox->setMargin(10);
  QHBox *meterHBox = new QHBox(this);
  meterHBox->setMargin(10);
  QVBox *meterLabelBox = new QVBox(meterHBox);
  QVBox *meterVBox = new QVBox(meterHBox);
  currentFileLabel = new QLabel(fileBox);
  new QWidget(fileBox);
  timeLabel = new QLabel(timeBox);
  new QWidget(timeBox);
  currentFileLabel->setText("File:");
  timeLabel->setText("Time: 0:00:00  ");
  for (l1 = 0; l1 < 2; l1++) {
    QLabel *channelMeterLabel = new QLabel(meterLabelBox);
    if (l1) {
      channelMeterLabel->setText("R ");
    } else {
      channelMeterLabel->setText("L ");
    }
    meter[l1] = new Meter((tickType)l1, recdata, &recdata->max[l1], meterVBox);
  }
//  QHBox *toggleBox = new QHBox(this);
//  toggleBox->setMargin(10);
//  QLabel *captureLabel = new QLabel(toggleBox);
//  captureLabel->setText("Capture");
//  new QWidget(toggleBox);
  if (!recdata->enableJack) {
    captureToggle = new QCheckBox(timeBox);
    captureToggle->setText("Capture");
    QObject::connect(captureToggle, SIGNAL(toggled(bool)), this, SLOT(captureToggled(bool)));
  } else {
    recdata->activateJack();
  }
  QHBox *recStopBox = new QHBox(this);
  recStopBox->setMargin(10);
  new QWidget(recStopBox); 
  QPushButton *recButton = new QPushButton("Record", recStopBox);
  new QWidget(recStopBox);
  QPushButton *pauseButton = new QPushButton("Pause", recStopBox);
  new QWidget(recStopBox);
  QPushButton *stopButton = new QPushButton("Stop", recStopBox);
  new QWidget(recStopBox);
  QObject::connect(recButton, SIGNAL(clicked()), this, SLOT(recordClicked())); 
  QObject::connect(pauseButton, SIGNAL(clicked()), this, SLOT(pauseClicked())); 
  QObject::connect(stopButton, SIGNAL(clicked()), this, SLOT(stopClicked())); 
  QHBox *bufLabelBox = new QHBox(this);
  bufLabelBox->setMargin(3);
  bufLabelBox->setFrameStyle(QFrame::Panel | QFrame::Sunken);
  bufLabel = new QLabel(bufLabelBox);
  bufLabel->setText("Buffer: ");
  new QWidget (bufLabelBox);
  maxBufLabel = new QLabel(bufLabelBox);
  maxBufLabel->setText("Peak: ");
  recdata->doRecord = false;
  timer = new QTimer(this);
  QObject::connect(timer, SIGNAL(timeout()), this, SLOT(timerProc())); 
  if (!recdata->enableJack) {
    capture = new Capture(periodSize, recdata);
  }
  diskwrite = new DiskWrite(recdata);
  maxBuffer = 0;
}

Gui::~Gui() {
  
}

void Gui::displayAbout() {
 
    aboutWidget->about(this, "About qaRecord", aboutText);
    aboutWidget->raise();
}

void Gui::recordClicked() {

  if (recdata->currentFile.isEmpty()) {
    newFile();
  }
  if (recdata->currentFile.isEmpty()) {
    return;
  }
  recdata->doRecord = true;
  if (recdata->enableJack) {
  } else {
    if (!recdata->doCapture) {
      captureToggle->setChecked(true);
    }
    if (!capture->running()) {
      capture->start();
    }
  }
  if (!diskwrite->running()) {
    diskwrite->start();
  }
  if (!recdata->enableJack) {
    captureToggle->setEnabled(false);
  }
  timer->start(200, true);
}

void Gui::stopClicked() {

//  sleep(1); 
  recdata->doRecord = false;
  recdata->pauseFlag = false;
  recdata->maxValidByteCount = 0;
  recdata->waitForData.wakeOne();
  if (!recdata->enableJack) {
    captureToggle->setEnabled(true);
  }
}

void Gui::pauseClicked() {
 
//  sleep(1);
  recdata->doRecord = false;
  recdata->pauseFlag = true;
  recdata->waitForData.wakeOne();
  if (!recdata->enableJack) {
    captureToggle->setEnabled(true);
  }
}

void Gui::newFile() {

  if ((recdata->currentFile = QString(QFileDialog::getSaveFileName(recdata->currentDir, "WAV files (*.wav)")))) {
    currentFileLabel->setText("File: "+recdata->currentFile);    
  }
  recdata->doRecord = false; 
  recdata->pauseFlag = false;
  recdata->maxValidByteCount = 0; 
  timeLabel->setText("Time: 0:00:00  ");
}

void Gui::timerProc() {

  QString qs1, qs2, qs3;
  int seconds, minutes, displaySeconds;

  if (recdata->doRecord) {
    timer->start(200, true);
    seconds = (recdata->wavDataSize / recdata->framesize) / recdata->rate;
    minutes = (seconds % 3600) / 60;
    displaySeconds = seconds % 60;
    qs1.sprintf("%d", seconds / 3600);
    if (minutes < 10) {
      qs2.sprintf("0%d", minutes);
    } else {
      qs2.sprintf("%d", minutes);
    }
    if (displaySeconds < 10) {
      qs3.sprintf("0%d", displaySeconds);
    } else {
      qs3.sprintf("%d", displaySeconds);
    }
    timeLabel->setText("Time: "+qs1+":"+qs2+":"+qs3+"  ");
    qs1.sprintf("%8ld", recdata->validByteCount + WAVBUFSIZE);
    bufLabel->setText("Buffer: "+qs1);
    if (recdata->maxValidByteCount + WAVBUFSIZE < recdata->ringBufSize) {
      qs1.sprintf("%8ld", recdata->maxValidByteCount + WAVBUFSIZE);
      maxBufLabel->setText("Peak: "+qs1);
    } else {
      maxBufLabel->setText("Buffer Overflow");
    }
  }
}

int Gui::initSeqNotifier() {

  int alsaEventFd = 0;  

  struct pollfd pfd[1];
  snd_seq_poll_descriptors(seq_handle, pfd, 1, POLLIN);
  alsaEventFd = pfd[0].fd;
  QSocketNotifier *seqNotifier = new QSocketNotifier(alsaEventFd, QSocketNotifier::Read);
  QObject::connect(seqNotifier, SIGNAL(activated(int)),
                   this, SLOT(midiAction(int)));
  return(0);
}

void Gui::midiAction(int fd) {

  snd_seq_event_t *ev;

  do {
    snd_seq_event_input(seq_handle, &ev);
    if ((ev->type == SND_SEQ_EVENT_NOTEON) && (ev->data.control.channel == recdata->midiChannel)) {
      switch (ev->data.note.note - recdata->midiNote) {
        case 0: 
          recordClicked();
          break;
        case 1:
          pauseClicked();
          break;
        case 2:
          stopClicked();
          break;
      }
    }
    snd_seq_free_event(ev);
  } while (snd_seq_event_input_pending(seq_handle, 0) > 0);
}

int Gui::open_seq(snd_seq_t **seq_handle, int in_ports[], int out_ports[], int num_in, int num_out) {

  int l1;
  char portname[64];

  if (snd_seq_open(seq_handle, "default", SND_SEQ_OPEN_DUPLEX, 0) < 0) {
    fprintf(stderr, "Error opening ALSA sequencer.\n");
    return(-1);
  }
  snd_seq_set_client_name(*seq_handle, "qaRecord");
  for (l1 = 0; l1 < num_in; l1++) {
    sprintf(portname, "QMidiRouter_In_%d", l1);
    if ((in_ports[l1] = snd_seq_create_simple_port(*seq_handle, portname,
              SND_SEQ_PORT_CAP_WRITE|SND_SEQ_PORT_CAP_SUBS_WRITE,
              SND_SEQ_PORT_TYPE_APPLICATION)) < 0) {
      fprintf(stderr, "Error creating sequencer port.\n");
      return(-1);
    }
  }  
  for (l1 = 0; l1 < num_out; l1++) {
    sprintf(portname, "qaRecord_Out_%d", l1);
    if ((out_ports[l1] = snd_seq_create_simple_port(*seq_handle, portname,
              SND_SEQ_PORT_CAP_READ|SND_SEQ_PORT_CAP_SUBS_READ,
              SND_SEQ_PORT_TYPE_APPLICATION)) < 0) {
      fprintf(stderr, "Error creating sequencer port.\n");
      return(-1);
    }
  }  
  return(0);
}

void Gui::captureToggled(bool on) {

  int l1;

  if (on) {
    recdata->doCapture = true;
    if (recdata->enableJack) {
    } else {
      if (!capture->running()) {
        capture->start();
      }
    }
  } else {
    recdata->newMax = false;
    recdata->doCapture = false;
    for (l1 = 0; l1 < 2; l1++) {
      recdata->max[l1] = 0;
      meter[l1]->resetGlobalMax();
    }
  }
}

void Gui::closeGui() {
 
  if (recdata->enableJack) {
    recdata->deactivateJack();
  }
}
