/****************************************************************************
**
** Copyright (C) 2007-2007 Trolltech ASA. All rights reserved.
**
** This file is part of the accessibility project on Trolltech Labs.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://www.trolltech.com/products/qt/opensource.html
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://www.trolltech.com/products/qt/licensing.html or contact the
** sales department at sales@trolltech.com.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#include "accessibletextadaptor.h"
#include "accessibleobject.h"

#include <QtGui/qaccessible.h>
#include <QtGui/qaccessible2.h>

#include <QTextEdit>
#include <QTextCursor>
#include <QAbstractTextDocumentLayout>

AccessibleTextAdaptor::AccessibleTextAdaptor(QObject *parent)
    : QDBusAbstractAdaptor(parent)
{
    Q_ASSERT(qobject_cast<AccessibleObject *>(parent));
}

AccessibleTextAdaptor::~AccessibleTextAdaptor()
{
}

static inline QAccessibleTextInterface *textInterface(QObject *parent)
{
    return static_cast<AccessibleObject *>(parent)->accessibleInterface()->textInterface();
}

int AccessibleTextAdaptor::nCharacters()
{
    return textInterface(parent())->characterCount();
}

int AccessibleTextAdaptor::caretOffset()
{
    return textInterface(parent())->cursorPosition();
}

QString AccessibleTextAdaptor::text(int startOffset, int endOffset)
{
    return textInterface(parent())->text(startOffset, endOffset);
}

bool AccessibleTextAdaptor::setCaretOffset(int offset)
{
    textInterface(parent())->setCursorPosition(offset);
    return caretOffset() == offset;
}

static QAccessible2::BoundaryType translateBoundaryType(int boundaryType)
{
    switch (boundaryType) {
    case IA2_TEXT_BOUNDARY_CHAR:
        return QAccessible2::CharBoundary;
    case IA2_TEXT_BOUNDARY_WORD:
        return QAccessible2::WordBoundary;
    case IA2_TEXT_BOUNDARY_SENTENCE:
        return QAccessible2::SentenceBoundary;
    case IA2_TEXT_BOUNDARY_LINE:
        return QAccessible2::LineBoundary;
    case IA2_TEXT_BOUNDARY_PARAGRAPH:
        return QAccessible2::ParagraphBoundary;
    case IA2_TEXT_BOUNDARY_ALL:
    default:
        qDebug("Unknown IA2 boundary type: %d", boundaryType);
        return QAccessible2::NoBoundary;
    }
}

QString AccessibleTextAdaptor::textAtOffset(int offset, int boundaryType,
            int &startOffset, int &endOffset)
{
    return textInterface(parent())->textAtOffset(offset,
                   translateBoundaryType(boundaryType), &startOffset, &endOffset);
}

int AccessibleTextAdaptor::nSelections()
{
    return textInterface(parent())->selectionCount();
}

static QAccessible2::CoordinateType translateCoordType(int coordType)
{
    switch (coordType) {
    case IA2_COORDTYPE_SCREEN_RELATIVE:
        return QAccessible2::RelativeToScreen;
    case IA2_COORDTYPE_PARENT_RELATIVE:
        return QAccessible2::RelativeToParent;
    default:
        qWarning("translateCoordType: Unknown Coordinate Type %d", coordType);
    }

    return QAccessible2::RelativeToScreen;
}

int AccessibleTextAdaptor::offsetAtPoint(int x, int y, int coordType)
{
    return textInterface(parent())->offsetAtPoint(QPoint(x, y), translateCoordType(coordType));
}

QRect AccessibleTextAdaptor::characterExtents(int offset, int coordType)
{
    return textInterface(parent())->characterRect(offset, translateCoordType(coordType));
}

void AccessibleTextAdaptor::scrollToSubstring(int startIndex, int endIndex)
{
    return textInterface(parent())->scrollToSubstring(startIndex, endIndex);
}

void AccessibleTextAdaptor::addSelection(int startOffset, int endOffset)
{
    textInterface(parent())->addSelection(startOffset, endOffset);
}

QString AccessibleTextAdaptor::attributes(int offset, int &startOffset, int &endOffset)
{
    return textInterface(parent())->attributes(offset, &startOffset, &endOffset);
}

void AccessibleTextAdaptor::removeSelection(int selectionIndex)
{
    textInterface(parent())->removeSelection(selectionIndex);
}

int AccessibleTextAdaptor::selection(int selectionIndex, int &endOffset)
{
    int startOffset;
    textInterface(parent())->selection(selectionIndex, &startOffset, &endOffset);
    return startOffset;
}

void AccessibleTextAdaptor::setSelection(int selectionIndex, int startOffset, int endOffset)
{
    textInterface(parent())->setSelection(selectionIndex, startOffset, endOffset);
}

