/***************************************************************************
 *            filein.c
 *
 *  Tue Sep 26 13:42:04 2006
 *  Copyright  2006-2007  Neil Williams
 *  linux@codehelp.co.uk
 ****************************************************************************/
/** @file filein.c
	@author Copyright 2006-2007  Neil Williams <linux@codehelp.co.uk>
	@author Copyright 1999 Robert Lissner
 */
/*
    This package is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "config.h"

#include <string.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <gtkextra/gtksheet.h>
#include <libgnomevfs/gnome-vfs.h>
#include "types.h"
#include "dialog_initial.h"
#include "dim_list_menu.h"
#include "filein.h"
#include "main.h"
#include "report.h"

#ifndef QOF_UTIL_H
/** \todo remove if quicklist uses QOF directly. */
#define ENUM_BODY(name, value)           \
	name value,

#define DEFINE_ENUM_NON_TYPEDEF(name, list)   \
	enum name {                               \
		list(ENUM_BODY)                       \
	};

#define FROM_STRING_DEC_NON_TYPEDEF(name, list)   \
	void name##fromString                          \
	(const gchar* str, enum name *type);

#define FROM_STRING_CASE_NON_TYPEDEF(name, value) \
	if (strcmp(str, #name) == 0) { *type = name; }

#define FROM_STRING_FUNC_NON_TYPEDEF(name, list)  \
	void name##fromString                          \
	(const gchar* str, enum name *type) {          \
	if(str == NULL) { return; }                    \
	list(FROM_STRING_CASE_NON_TYPEDEF) }

#define AS_STRING_DEC_NON_TYPEDEF(name, list)     \
	const gchar* name##asString(enum name n);

#define AS_STRING_FUNC_NON_TYPEDEF(name, list)    \
	const gchar* name##asString(enum name n) {     \
	   switch (n) {                               \
		   list(AS_STRING_CASE_NON_TYPEDEF)       \
		   default: return ""; } }

#define AS_STRING_CASE_NON_TYPEDEF(name, value)   \
	case name: { return #name; }

#endif

/** \brief Inherited error codes

Previous error codes have been expanded into 
more descriptive enum values that are then mapped
as strings.

\todo rationalise the error codes.
*/
#define QL_ERRORS(_) \
	_(QL_SUCCESS, =0) \
	_(QL_CANNOT_OPEN_FILE, ) \
	_(QL_NO_CONTENT_FOUND, ) \
	_(QL_BAD_QUICKLIST_DATA, ) \
	_(QL_BAD_FIELD_DATA, ) \
	_(QL_NO_FIELD_TYPE, ) \
	_(QL_UNABLE_TO_PARSE_FIELD_TYPE, ) \
	_(QL_UNABLE_TO_PARSE_FIELD, ) \
	_(QL_FIELD_NAME_IS_TOO_LONG, ) \
	_(QL_BAD_SORT_VALUE, ) \
	_(QL_TOO_FEW_SORT_LEVELS, ) \
	_(QL_INVALID_FIELD_NAME, ) \
	_(QL_SORT_NESTING_TOO_DEEP, ) \
	_(QL_BAD_FILTER_VALUE, ) \
	_(QL_CANNOT_PARSE_FILTER_DATA, ) \
	_(QL_INVALID_FILTER_NAME, ) \
	_(QL_TOO_FEW_FILTER_FIELDS, ) \
	_(QL_INVALID_FIELD_TYPE, ) \
	_(QL_TOO_MANY_FILTER_COMPARISONS, =19) \
	_(QL_INVALID_CHAR_IN_FILTER, ) \
	_(QL_FILTER_NESTING_TOO_DEEP, ) \
	_(QL_TOO_MANY_COLUMNS, ) \
	_(QL_INVALID_COLUMN_DATA, ) \
	_(QL_UNRECOGNISED_COLUMN_TYPE, ) \
	_(QL_NO_COLUMNS_IN_REPORT, ) \
	_(QL_INVALID_REPORT_DATA, ) \
	_(QL_REPORT_FIELD_NAME_TOO_LONG, ) \
	_(QL_INVALID_DATA, ) \
	_(QL_TOO_MANY_FIELDS, =30) \
	_(QL_LINE_IS_TOO_BIG, =45)

DEFINE_ENUM_NON_TYPEDEF (QlErrorVal, QL_ERRORS)

AS_STRING_FUNC_NON_TYPEDEF (QlErrorVal, QL_ERRORS)

/** setup a new text field as a standard text field.  */
static void
define_last_field (QlTabData * tab)
{
	gint temp = tab->file->last_field;
	g_snprintf (tab->file->fields[temp].name, MAX_FIELD_NAME, "Column %u",
		temp + 1);
	tab->file->fields[temp].sheet_column = temp;
	tab->file->fields[temp].type = FIELD_TYPE_TEXT;
	tab->file->fields[temp].formatting = 0;
	tab->file->fields[temp].decimal_places = 0;
	tab->file->fields[temp].justification = GTK_JUSTIFY_LEFT;
	tab->file->fields[temp].width = 10;
	tab->file->col_to_field[temp] = temp;
}

/** \brief Add 1 column to sheet.  

 This is necessary for reading unspecified input
  files such as comma, html and tab */
static void
add1field (QlTabData * tab)
{
	gint fieldx = ++tab->file->last_field;
	define_last_field (tab);
	gtk_sheet_add_column (tab->view->sheet, 1);
	gtk_sheet_set_column_width (tab->view->sheet, fieldx, 80);
	gtk_sheet_column_button_add_label (tab->view->sheet, fieldx,
		tab->file->fields[fieldx].name);
	gtk_sheet_set_column_title (tab->view->sheet,
		fieldx, tab->file->fields[fieldx].name);
}								/* end of add1field */


/** The file selection dialog box has returned a file name.
It is supposed to be a comma delimited file (CSV) */
static enum QlErrorVal
read_comma_file (QlTabData * tab)
{
	/** \bug remove need for linebuf 
	using g_strsplit ?
	*/
	gchar linebuf[256];
	gchar *linebufp;
	gchar *beginp;
	gchar *endp;
	gint16 fieldx = 0;
	gint rowx, c;
	gboolean got_data;
	FILE * f;

	/* Make Column 1 in field [0] */
	if (!(f = fopen (tab->file->file_path, "r")))
		return QL_CANNOT_OPEN_FILE;
	build_basic_list_mode (tab);
	define_last_field (tab);
	linebufp = beginp = linebuf;
	rowx = fieldx = 0;
	endp = linebufp + 253;;
	got_data = FALSE;

	while ((c = getc (f)) != EOF)
	{
		if (c >= ' ' && c != ',')
		{
			*linebufp++ = c;
			if (linebufp >= endp)
				return QL_LINE_IS_TOO_BIG;
		}
		else if (c == '\n' || c == '\r' || c == ',')
		{
			if (linebufp != beginp)
			{
				got_data = TRUE;
				*linebufp = '\0';
				while (fieldx > tab->file->last_field)
				{
					if (fieldx >= MAX_FIELDS - 1)
						return QL_TOO_MANY_FIELDS;
					else
						add1field (tab);
				}
				gtk_sheet_set_cell_text 
					(tab->view->sheet, rowx,
					fieldx, linebuf);
				linebufp = beginp;
			}

			fieldx++;
			if (c != ',')
			{
				if (got_data)
				{
					rowx++;
					add1row (tab);
				}
				fieldx = 0;
				got_data = FALSE;
			}
		}
	}
	fclose (f);
	return QL_SUCCESS;
}								/* end of read_comma_file */

/** The file selection dialog box has returned a file name.  Get it,
   open it, open the window.  Build all window
   window components, but do not assemble them.  That is done by
   build_file_win ()

\return 0 on success, otherwise negative.
*/
static enum QlErrorVal
read_ql_file (QlTabData * tab)
{
	gchar linebuf[4096];
	gchar tokbuf[256];
	gchar *tokbufp;
	gchar *linebufp;
	gchar *prevbufp;
	gint16 fieldx = 0;
	gint16 sortx = 0;
	gint16 filterx = 0;
	gint16 subx = 0;
	gint rowx;
	gboolean got_data;
	gint16 reportx = 0;
	gint16 colx = 0;			/* indexes into array */
	gint16 thislen, i, count;
	FILE * f;
	gchar singlechar;
	gint16 temp_field, temp_type;

	if (!(f = fopen (tab->file->file_path, "r")))
		return QL_CANNOT_OPEN_FILE;
	while (fgets (linebuf, sizeof (linebuf), f))
	{
		linebufp = linebuf;
		if (sscanf (linebuf, "%s", tokbuf) != 1)
			return QL_NO_CONTENT_FOUND;

		/* process QUICKLIST header record */
		if (g_ascii_strcasecmp (tokbuf, "QUICKLIST") == 0)
		{
			/** storing width, height, x and y in the file
			is outmoded but needs to be retained for compatibility. */
			if (sscanf (linebuf, "%*s%*s%*s%hu%hu%hu%hu%hu%hu%hu%hu",
					&tab->file->last_field,
					&tab->file->sort_ct,
					&tab->file->filter_ct,
					&tab->file->report_ct,
					&tab->view->width,
					&tab->view->height, 
					&tab->view->x, 
					&tab->view->y) < 6)
				return QL_BAD_QUICKLIST_DATA;
			tab->file->last_field--;
			if (tab->file->last_field < 0
				|| tab->file->last_field >= MAX_FIELDS
				|| tab->file->sort_ct > MAX_SORTS
				|| tab->file->filter_ct > MAX_FILTERS
				|| tab->file->report_ct > MAX_REPORTS)
				return QL_BAD_FIELD_DATA;
		}
		/* process FIELD input header record */
		else if (g_ascii_strcasecmp (tokbuf, "FIELD") == 0)
		{
			guint16 field_type;
			if (fieldx > tab->file->last_field)
				return QL_NO_FIELD_TYPE;
			if (sscanf (linebuf, "%*s%hu%hu%hu%hu%hu%hu%hu%hu%40[^\n\\]",
					&field_type,
					&tab->file->fields[fieldx].formatting,
					&tab->file->fields[fieldx].decimal_places,
					&tab->file->fields[fieldx].justification,
					&tab->file->fields[fieldx].sheet_column,
					&tab->file->fields[fieldx].width,
					&tab->file->fields[fieldx].unused1,
					&tab->file->fields[fieldx].unused2, tokbuf) != 9)
				return QL_UNABLE_TO_PARSE_FIELD_TYPE;
			else
				tab->file->fields[fieldx].type = field_type;
			tokbufp = tokbuf;
			/** remove leading spaces 
			
			\bug replace with glib functions.
			*/
			while (*tokbufp == ' ')
				tokbufp++;
			if (tab->file->fields[fieldx].type < FIELD_TYPE_TEXT
				|| tab->file->fields[fieldx].type > FIELD_TYPE_TIME
				|| tab->file->fields[fieldx].formatting > 12
				|| tab->file->fields[fieldx].decimal_places > 6
				|| tab->file->fields[fieldx].justification > GTK_JUSTIFY_CENTER
				|| tab->file->fields[fieldx].sheet_column > tab->file->last_field
				|| tab->file->fields[fieldx].width > 80)
				return QL_UNABLE_TO_PARSE_FIELD;

			thislen = strlen (tokbufp);
			if (thislen < 1 || thislen > MAX_FIELD_NAME)
				return QL_FIELD_NAME_IS_TOO_LONG;
			strcpy (tab->file->fields[fieldx].name, tokbufp);
			fieldx++;
		}
		/* convert the SORT record to memory */

		else if (g_ascii_strcasecmp (tokbuf, "SORT") == 0)
		{
			if (sortx >= tab->file->sort_ct)
				return QL_BAD_SORT_VALUE;
			count = sscanf (linebuf,
				"%*s%hu%hu%40[^\n\\]%c%hu%u%hu%u%hu%u%hu%u%hu%u%hu%u",
				&tab->file->sorts[sortx].unused1,
				&tab->file->sorts[sortx].unused2,
				tokbuf,
				&singlechar,
				&tab->file->sorts[sortx].line[0].field,
				&tab->file->sorts[sortx].line[0].ascending,
				&tab->file->sorts[sortx].line[1].field,
				&tab->file->sorts[sortx].line[1].ascending,
				&tab->file->sorts[sortx].line[2].field,
				&tab->file->sorts[sortx].line[2].ascending,
				&tab->file->sorts[sortx].line[3].field,
				&tab->file->sorts[sortx].line[3].ascending,
				&tab->file->sorts[sortx].line[4].field,
				&tab->file->sorts[sortx].line[4].ascending,
				&tab->file->sorts[sortx].line[5].field,
				&tab->file->sorts[sortx].line[5].ascending);
			if (count < 6)
				return QL_TOO_FEW_SORT_LEVELS;
			tab->file->sorts[sortx].line_ct = (count - 4) / 2;
			/* number of sort lines read */
			tokbufp = tokbuf;
			/** remove leading spaces
			
			\bug replace with glib functions.
			*/
			while (*tokbufp == ' ')
				tokbufp++;
			thislen = strlen (tokbufp);
			if (thislen < 1
				|| thislen > MAX_FIELD_NAME || singlechar != '\\')
				return QL_INVALID_FIELD_NAME;
			tab->file->sorts[sortx].name = g_strdup(tokbufp);
			for (i = 0; i < MAX_SORT_NESTING; i++)
				if (tab->file->sorts[sortx].line[i].field >=
					tab->file->last_field + 1)
					return QL_SORT_NESTING_TOO_DEEP;
			sortx++;
		}
		/* now store a FILTER record, the most complicated record */
		else if (strcmp (tokbuf, "FILTER") == 0)
		{
			if (filterx >= tab->file->filter_ct)
				return QL_BAD_FILTER_VALUE;
			if (sscanf (linebuf, "%*s%d%d%40[^\n\\]%c%hn",
					&tab->file->filters[filterx].by_and,
					&tab->file->filters[filterx].use_nocase,
					tokbuf, &singlechar, &count) != 4)
			{
				return QL_CANNOT_PARSE_FILTER_DATA;
			}
			tokbufp = tokbuf;
			/** \bug remove leading spaces:
			replace with glib functions.
			*/
			while (*tokbufp == ' ')
				tokbufp++;
			thislen = strlen (tokbufp);
			if (thislen < 1
				|| thislen > MAX_FIELD_NAME || singlechar != '\\')
				return QL_INVALID_FILTER_NAME;
			strcpy (tab->file->filters[filterx].name, tokbufp);
			subx = 0;

			/* now loop through the lines of the filter */
			linebufp = linebuf + count;
			while (TRUE)
			{
				if (sscanf (linebufp, "%hu%hu%hn", &temp_field,
						&temp_type, &count) < 2)
					return QL_TOO_FEW_FILTER_FIELDS;
				linebufp += count;
				tokbuf[0] = '\0';

				while (*linebufp == ' ')
					linebufp++;
				if (*linebufp != '\n' && *linebufp != '\\' &&
					*linebufp != '\0')
				{
					sscanf (linebufp, "%40[^\\\n]%hn", tokbuf, &count);
					linebufp += count;
				}

				if (temp_field > tab->file->last_field || temp_type < 0 || temp_type > 7)	/* read and verify field and type */
					return QL_INVALID_FIELD_TYPE;
				tab->file->filters[filterx].line[subx].field = temp_field;
				tab->file->filters[filterx].line[subx].type = temp_type;
/* what happened to 18 ? */
				tokbufp = tokbuf;
				if (strlen (tokbufp) > MAX_FILTER_COMPARE)
					return QL_TOO_MANY_FILTER_COMPARISONS;
				strcpy (tab->file->filters[filterx].line[subx].compare,
					tokbufp);
				subx++;

				/* now look for \ before next rule, or \n to end it */
				sscanf (linebufp, "%c", &singlechar);
				if (singlechar == '\n' || !singlechar)
					break;		/* no more data */
				if (singlechar != '\\')
					return QL_INVALID_CHAR_IN_FILTER;
				linebufp++;
				if (subx >= MAX_FILTER_NESTING)
					return QL_FILTER_NESTING_TOO_DEEP;

			}					/* End of loop that scans filter lines */

			/* Break comes here */
			tab->file->filters[filterx].line_ct = subx;
			filterx++;
		}

		/* store a report column */
		else if (strcmp (tokbuf, "COLUMN") == 0)
		{
			if (colx >= tab->file->last_field + 1)
				return QL_TOO_MANY_COLUMNS;
			if (sscanf (linebuf,
					"%*s%hu%hu%hu%hu%hu%hu",
					&tab->file->reports[reportx].column[colx].field,
					&tab->file->reports[reportx].column[colx].width,
					&tab->file->reports[reportx].column[colx].group,
					&tab->file->reports[reportx].column[colx].total,
					&tab->file->reports[reportx].column[colx].unused1,
					&tab->file->reports[reportx].column[colx].unused2) < 6)
				return QL_INVALID_COLUMN_DATA;
			if (tab->file->reports[reportx].column[colx].field >=
				tab->file->last_field + 1)
				return QL_UNRECOGNISED_COLUMN_TYPE;
			colx++;
		}
		/* store a report record */
		else if (strcmp (tokbuf, "REPORT") == 0)
		{
			if (reportx >= tab->file->report_ct || colx == 0)	/* never got any column records */
				return QL_NO_COLUMNS_IN_REPORT;
			if (sscanf (linebuf,
					"%*s%hd%hd%hd%hd%40[^\n\\]%c",
					&tab->file->reports[reportx].sort,
					&tab->file->reports[reportx].filter,
					&tab->file->reports[reportx].width,
					&tab->file->reports[reportx].height,
					tokbuf, &singlechar) < 6)
				return QL_INVALID_REPORT_DATA;
			tokbufp = tokbuf;
			while (*tokbufp == ' ')	/* remove leading spaces */
				tokbufp++;
			thislen = strlen (tokbufp);
			if (thislen < 1 || thislen > MAX_FIELD_NAME)
				return QL_REPORT_FIELD_NAME_TOO_LONG;
			strcpy (tab->file->reports[reportx].name, tokbufp);
			tab->file->reports[reportx].last_column = colx - 1;
			reportx++;
			colx = 0;			/* reset the report column indicator */
		}

		else if (strcmp (tokbuf, "DATA") == 0)
			break;
		else
			return QL_LINE_IS_TOO_BIG;
	}

	/** \bug Really ought to check file parameters about right here */
	if (colx != 0
		|| fieldx != tab->file->last_field + 1
		|| sortx != tab->file->sort_ct
		|| filterx != tab->file->filter_ct || reportx != tab->file->report_ct)
		return QL_INVALID_DATA;

	/* fields are all OK so go build the basic window and sheet */
	reset_col_to_field (tab);
	build_basic_list_mode (tab);

	/* The file is already open, now read it in. 
	   Make some effort to not load entirely blank records */
	while (fgets (linebuf, sizeof (linebuf), f))
	{
		fieldx = 0;
		got_data = FALSE;
		linebufp = linebuf;
		prevbufp = linebuf;
		rowx = tab->file->last_row;
		singlechar = ' ';

		while (singlechar != '\n')
		{
			while (*linebufp != '\n' && *linebufp != '\\')
				linebufp++;

			singlechar = *linebufp;
			if (linebufp != prevbufp)
			{					/* ie, not a zero length field */
				if (fieldx > tab->file->last_field)
					return QL_TOO_MANY_FIELDS;

				*linebufp = '\0';	/* to terminate string move */
				got_data = TRUE;
				gtk_sheet_set_cell_text (tab->view->sheet, rowx,
					tab->file->fields[fieldx].sheet_column, prevbufp);
			}
			prevbufp = ++linebufp;
			fieldx++;
		}
		if (got_data)
			add1row (tab);
	}
	fclose (f);
	tab->file->current_file = QL_OLD_QLF;
	return QL_SUCCESS;
}								/* end of read_ql_file */

/** Read tab delimited files */
static enum QlErrorVal
read_tab_file (QlTabData * tab)
{
	gchar linebuf[256];
	gchar *linebufp;
	gchar *beginp;
	gchar *endp;
	gint16 fieldx = 0;
	gint rowx, c;
	gboolean got_data;
	FILE *  f;

	/* Make Column 1 in field [0] */
	if (!(f = fopen (tab->file->file_path, "r")))
		return QL_CANNOT_OPEN_FILE;
	define_last_field (tab);
	build_basic_list_mode (tab);
	linebufp = beginp = linebuf;
	rowx = fieldx = 0;
	endp = linebufp + 253;;
	got_data = FALSE;

	while ((c = getc (f)) != EOF)
	{
		if (c >= ' ')
		{
			*linebufp++ = c;
			if (linebufp >= endp)	/* line is too big */
				return QL_LINE_IS_TOO_BIG;
		}
		else if (c == '\n' || c == '\r' || c == '\t')
		{
			if (linebufp != beginp)
			{					/* ie, there was data */
				got_data = TRUE;
				*linebufp = '\0';
				while (fieldx > tab->file->last_field)
				{
					if (fieldx >= MAX_FIELDS - 1)
						return QL_TOO_MANY_FIELDS;
					else
						add1field (tab);
				}
				gtk_sheet_set_cell_text (tab->view->sheet, rowx,
					fieldx, linebuf);
				linebufp = beginp;
			}					/* end of storing data */

			fieldx++;
			if (c != '\t')
			{					/* process end of line or return */
				if (got_data)
				{
					rowx++;
					add1row (tab);
				}
				fieldx = 0;
				got_data = FALSE;
			}
		}
	}
	fclose (f);
	return QL_SUCCESS;
}								/* end of read_tab_file */

static void
report_error (QlTabData * tab, enum QlErrorVal err_number)
{
	gchar * tmp;

	tmp = g_strconcat (_("Error: "), QlErrorValasString(err_number), 
		". ", _("This file is apparently not the type that you selected."),
		NULL);
	level1_error (tab, tmp);
	g_free (tmp);
}

/** This callback is when someone clicks OK on a 
file selection dialog box */
void
actually_open (const gchar * filename, QlDialogMode G_GNUC_UNUSED open_mode,
			   QlFileType file_type, QlContext * qlc)
{
	QlTabData * tab;
	enum QlErrorVal temp;

	tab = ql_new_tabdata(qlc);
	/* 0,0 is valid, so can't use initial values. */
	tab->view->activate_row = tab->view->activate_col = -5;
	tab->file->file_path = g_strdup (filename);
	tab->file->file_name = g_path_get_basename (filename);

	temp = -1;					/* default error */
	if (file_type == QL_OLD_QLF)	/* .qlf */
		temp = read_ql_file (tab);
	else if (file_type == QL_CSV)	/* .csv */
		temp = read_comma_file (tab);
	else
		temp = read_tab_file (tab);
	if (temp)
	{
		report_error (tab, temp);
		return;
	}
	tab->view->display_mode = DISPLAY_LIST;
	tab->file->changed = FALSE;
	gtk_recent_manager_add_item (qlc->recent_manager, filename);
	dim_all_menus (qlc);
	gtk_widget_show_all (qlc->parent);
	connect_signals (tab);
	tab->view->dialog_mode = MODE_SAVE;
}								/* end of actual open */

/** retrieve path from the filename of 
the current tab, if any. */
static gchar *
check_last_path (QlContext * qlc)
{
	QlFileData * file;

	file = ql_get_filedata (qlc);
	if (!file)
		return NULL;
	return g_path_get_dirname (file->file_path);	
}

/** allow the user to choose the file to open. */
static void
any_open (QlDialogMode open_mode, QlFileType file_type, QlContext * qlc)
{
	GtkWidget * chooser;
	GtkFileFilter * filter;
	gchar * filename, * last_path;
	QlTabData * tab;

	tab = ql_get_tabdata (qlc);
	filename = NULL;
	last_path = check_last_path (qlc);
	filter = gtk_file_filter_new ();
	if (open_mode == MODE_OPEN)
	{
		gtk_file_filter_set_name (filter, _("QuickList files *.qlf"));
		gtk_file_filter_add_mime_type (filter, "application-x-quicklist");
		gtk_file_filter_add_pattern (filter, "*.qlf");
		chooser = gtk_file_chooser_dialog_new
			((_("Open a list")),
			NULL, GTK_FILE_CHOOSER_ACTION_OPEN,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL);
		gtk_file_chooser_add_filter (GTK_FILE_CHOOSER(chooser), 
			filter);
	}
	else
	{
		chooser = gtk_file_chooser_dialog_new
			((_("Import a file")),
			NULL, GTK_FILE_CHOOSER_ACTION_OPEN,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL);
		file_type_menu (tab, chooser);
	}
	filter = gtk_file_filter_new ();
	gtk_file_filter_set_name (filter, _("All files *"));
	gtk_file_filter_add_pattern (filter, "*");
	gtk_file_chooser_add_filter (GTK_FILE_CHOOSER(chooser), 
		filter);
	/* need tab index */
	if (last_path)
		gtk_file_chooser_set_current_folder (
			GTK_FILE_CHOOSER(chooser), last_path);
	if (gtk_dialog_run (GTK_DIALOG (chooser)) == GTK_RESPONSE_ACCEPT)
		filename = gtk_file_chooser_get_filename
			(GTK_FILE_CHOOSER (chooser));
	gtk_widget_destroy (chooser);
	if (filename)
		actually_open (filename, open_mode, file_type, qlc);
}								/* end of any_open callback */

/** window initialiser */
void
build_basic_list_mode (QlTabData * tab)
{
	GtkSheet *sheet;
	GtkWidget *scrolled_window;
	gint colx, fieldx, tbindex;

	colx = 0;
	fieldx = 0;
	tab->view->sheet = GTK_SHEET (gtk_sheet_new (1, tab->file->last_field + 1,
		tab->file->file_name));
	sheet = (tab->view->sheet);
	gtk_sheet_row_button_add_label (sheet, 0, " ");
	gtk_sheet_set_autoscroll (sheet, TRUE);
	gtk_sheet_set_autoresize (sheet, TRUE);

	gtk_sheet_set_row_titles_width (sheet, 20);
	tab->file->last_row = 0;
	gtk_container_set_border_width (GTK_CONTAINER (sheet), 4);

	/* set sensitivity for all column and row buttons */
	gtk_sheet_columns_set_sensitivity (sheet, TRUE);
	gtk_sheet_show_row_titles (sheet);
	gtk_sheet_show_column_titles (sheet);
	gtk_sheet_rows_set_sensitivity (sheet, TRUE);
	for (fieldx = 0; fieldx <= tab->file->last_field; fieldx++)
	{
		colx = tab->file->fields[fieldx].sheet_column;
		gtk_sheet_set_column_width (sheet, colx,
			tab->file->fields[fieldx].width * 8);
		gtk_sheet_column_button_add_label (sheet, colx,
			tab->file->fields[fieldx].name);
		gtk_sheet_column_set_justification (sheet, colx,
			tab->file->fields[fieldx].justification);
	}
	gtk_sheet_show_column_titles (sheet);

	/* create a new scrolled window. */
	scrolled_window = gtk_scrolled_window_new (NULL, NULL);

	gtk_container_set_border_width (GTK_CONTAINER (scrolled_window), 3);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW
		(scrolled_window), GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);

	gtk_container_add (GTK_CONTAINER (scrolled_window),
		GTK_WIDGET (sheet));
	gtk_notebook_append_page(tab->qlc->notebook, 
		GTK_WIDGET(scrolled_window), 
		tab_label_box (tab, tab->file->file_name));
	gtk_widget_show (scrolled_window);
	tbindex = gtk_notebook_get_current_page (tab->qlc->notebook);
	tbindex++;
	gtk_notebook_set_tab_reorderable (tab->qlc->notebook, scrolled_window, TRUE);
	gtk_notebook_set_current_page (tab->qlc->notebook, tbindex);
}								/* end of build_basic_list_mode */

void
file_import (GtkAction G_GNUC_UNUSED * action, gpointer data)
{
	QlContext * qlc;

	qlc = ql_get_context (GTK_WIDGET(data));
	g_return_if_fail (qlc);
	any_open (MODE_IMPORT, QL_CSV, NULL);
}

/** Build the menu button for the file selection box, allowing choice
  of file types.  Works for input and output */
void
file_type_menu (QlTabData * tab, GtkWidget * chooser)
{
	GtkFileFilter * filter;

	if (tab->view->dialog_mode == MODE_EXPORT)
	{
		filter = gtk_file_filter_new ();
		gtk_file_filter_set_name (filter, _("HTML document, *.html"));
		gtk_file_filter_add_pattern (filter, "*.html");
		gtk_file_chooser_add_filter (GTK_FILE_CHOOSER(chooser), filter);
	}
	filter = gtk_file_filter_new ();
	gtk_file_filter_set_name (filter, _("Comma delimited, *.csv"));
	gtk_file_filter_add_pattern (filter, "*.csv");
	gtk_file_chooser_add_filter (GTK_FILE_CHOOSER(chooser), filter);
	filter = gtk_file_filter_new ();
	gtk_file_filter_set_name (filter, _("Tab delimited, *.tsv"));
	gtk_file_filter_add_pattern (filter, "*.csv");
	gtk_file_chooser_add_filter (GTK_FILE_CHOOSER(chooser), filter);
}

/** This is a callback that handles open file from various places */
void
open_file (GtkAction G_GNUC_UNUSED * action, gpointer data)
{
	QlContext * qlc;

	qlc = ql_get_context (GTK_WIDGET(data));
	g_return_if_fail (qlc);
	any_open (MODE_OPEN, QL_OLD_QLF, qlc);
}

void
open_recent (GtkRecentChooser * c, gpointer data)
{
	QlContext * qlc;
	GnomeVFSFileInfo * vfsinfo;
	GnomeVFSResult vfsresult;
	gchar * filename;

	g_return_if_fail (data);
	qlc = (QlContext*)data;
	g_return_if_fail (qlc);
	filename = gtk_recent_chooser_get_current_uri (c);
	if (!filename)
		return;
	vfsinfo = gnome_vfs_file_info_new ();
	vfsresult = gnome_vfs_get_file_info (filename, vfsinfo, 
		GNOME_VFS_FILE_INFO_DEFAULT);
	if ((vfsinfo->type == GNOME_VFS_FILE_TYPE_REGULAR) || (vfsinfo->size > 0))
		actually_open (filename, MODE_OPEN, QL_OLD_QLF, qlc);
	g_free (filename);
}
