/***************************************************************************
                          gaussiirblurdialog.cpp  -  description
                             -------------------
    begin                : Thu Feb 7 2002
    copyright            : (C) 2002 by Michael Herder
    email                : crapsite@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
/*****************************************************************************
 * The algorithm used in this class is based on the Gimp 1.2.0               *
 * Original filename: gauss_iir.c                                            *
 * Original copyright message:                                               *
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis                       *
 *                                                                           *
 * This program is free software; you can redistribute it and/or modify      *
 * it under the terms of the GNU General Public License as published by      *
 * the Free Software Foundation; either version 2 of the License, or         *
 * (at your option) any later version.                                       *
 *                                                                           *
 * This program is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 * GNU General Public License for more details.                              *
 *                                                                           *
 * You should have received a copy of the GNU General Public License         *
 * along with this program; if not, write to the Free Software               *
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.*
 *****************************************************************************/

#include "gaussiirblurdialog.h"
#include "quiteinsane/qxmlconfig.h"
#include "quiteinsane/sliderspin.h"

#include <math.h>

#include <qapplication.h>
#include <qcheckbox.h>
#include <qhbox.h>
#include <qimage.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qnamespace.h>
#include <qpixmap.h>
#include <qvbox.h>

GaussIIRBlurDialog::GaussIIRBlurDialog(int preview_size,QImage* image,QWidget* parent)
                   :ImageFilterDialog(preview_size,image,parent)
{
  mRadiusH = 3.0;
  mRadiusV = 3.0;
  initControls();
}
GaussIIRBlurDialog::~GaussIIRBlurDialog()
{
}
/** No descriptions */
void GaussIIRBlurDialog::initControls()
{
  int v_radius = xmlConfig->intValue("FILTER_GAUSSIIRBLUR_VERTICAL",3);
  int h_radius = xmlConfig->intValue("FILTER_GAUSSIIRBLUR_HORIZONTAL",3);
  bool bind_vals = xmlConfig->boolValue("FILTER_GAUSSIIRBLUR_GREEN",true);
  bool c_up = xmlConfig->boolValue("FILTER_GAUSSIIRBLUR_CONTINOUS_UPDATE",false);

  QVBox* vb = controlsVBox();
  if(!vb)
    return;
  setTitle(tr("Gaussian blur (IIR)"));
  setCaption(tr("Gaussian blur (IIR)"));
  //horizontal radius
  mpSliderSpinH = new SliderSpin(0,100,h_radius,tr("Horizontal radius"),vb);
  mpSliderSpinH->setMinimumWidth(150);
  //vertical radius
  mpSliderSpinV = new SliderSpin(0,100,v_radius,tr("Vertical radius"),vb);
  mpSliderSpinV->setMinimumWidth(150);
  //check box
  mpBindCheckBox = new QCheckBox(tr("Bind Values"),vb);
  mpBindCheckBox->setChecked(bind_vals);

  QWidget* dummy = new QWidget(vb);
  vb->setStretchFactor(dummy,1);
  connect(mpSliderSpinH,SIGNAL(signalValueChanged(int)),
          this,SLOT(slotRadiusHChanged(int)));
  connect(mpSliderSpinV,SIGNAL(signalValueChanged(int)),
          this,SLOT(slotRadiusVChanged(int)));
  connect(mpBindCheckBox,SIGNAL(toggled(bool)),
          this,SLOT(slotBindValues(bool)));
  //set values
  slotRadiusHChanged(h_radius);
  slotRadiusVChanged(v_radius);
  setContinousUpdate(c_up);
  setFixedSize(minimumSizeHint());
}
/** No descriptions */
bool GaussIIRBlurDialog::apply(QImage* image,bool emit_progress)
{
  double horz,vert;
  int width, height;
  int bytes;
  bool has_alpha;
  unsigned char *dest;
  unsigned char *src, *sp_p, *sp_m;
  double n_p[5], n_m[5];
  double d_p[5], d_m[5];
  double bd_p[5], bd_m[5];
  double *val_p, *val_m, *vp, *vm;
  int x1, y1, x2, y2;
  int i, j;
  int row, col, b;
  int terms;
  int progress, progresscnt;
  int initial_p[4];
  int initial_m[4];
  double std_dev;
  int orig_depth;
  int old_p = 0;

  if(image->depth() < 8)
    return false;
  orig_depth = image->depth();

  horz = mRadiusH;
  vert = mRadiusV;

  if((horz < 1.0) && (vert < 1.0))
    return false;

  x1 = 0;
  x2 = image->width();
  y1 = 0;
  y2 = image->height();


  if(image->depth() < 32)
    *image = image->convertDepth(32);

  has_alpha = image->hasAlphaBuffer();

  width = image->width();
  height = image->height();
  if(has_alpha)
    bytes = 4;
  else
    bytes = 3;

  if(width > height)
  {
    val_p = new double [width * bytes];
    val_m = new double [width * bytes];
    src =  new unsigned char [width * bytes];
    dest =  new unsigned char [width * bytes];
  }
  else
  {
    val_p = new double [height * bytes];
    val_m = new double [height * bytes];
    src =  new unsigned char [height * bytes];
    dest =  new unsigned char [height * bytes];
  }

  progresscnt = 0;
  progress = (horz < 1.0 ) ? 0 : width * height * int(horz);
  progress += (vert < 1.0 ) ? 0 : width * height * int(vert);

  /*  First the vertical pass  */
  if (vert >= 1.0)
  {
    vert = fabs (vert) + 1.0;
    std_dev = sqrt (-(vert * vert) / (2 * log (1.0 / 255.0)));

    findConstants (n_p, n_m, d_p, d_m, bd_p, bd_m, std_dev);

    for (col = 0; col < width; col++)
  	{
  	  memset(val_p, 0, height * bytes * sizeof (double));
  	  memset(val_m, 0, height * bytes * sizeof (double));

      for(int n=0;n<image->height();n++)
      {
        QRgb rgb = image->pixel(col + x1,n);
        if(has_alpha)
        {
          src[n*4] = (unsigned char)qRed(rgb);
          src[n*4+1] = (unsigned char)qGreen(rgb);
          src[n*4+2] = (unsigned char)qBlue(rgb);
          src[n*4+3] = (unsigned char)qAlpha(rgb);
        }
        else
        {
          src[n*3] = (unsigned char)qRed(rgb);
          src[n*3+1] = (unsigned char)qGreen(rgb);
          src[n*3+2] = (unsigned char)qBlue(rgb);
        }
      }

	    if (has_alpha)
	      multiplyAlpha (src, height, bytes);

  	  sp_p = src;
  	  sp_m = src + (height - 1) * bytes;
  	  vp = val_p;
  	  vm = val_m + (height - 1) * bytes;

  	  for (i = 0; i < bytes; i++)
	    {
	      initial_p[i] = sp_p[i];
	      initial_m[i] = sp_m[i];
	    }
  	  for (row = 0; row < height; row++)
	    {
	      double *vpptr, *vmptr;
	      terms = (row < 4) ? row : 4;

	      for (b = 0; b < bytes; b++)
	    	{
    		  vpptr = vp + b; vmptr = vm + b;
    		  for (i = 0; i <= terms; i++)
  		    {
	  	      *vpptr += n_p[i] * sp_p[(-i * bytes) + b] -
		                	d_p[i] * vp[(-i * bytes) + b];
		        *vmptr += n_m[i] * sp_m[(i * bytes) + b] -
			                d_m[i] * vm[(i * bytes) + b];
		      }
		      for (j = i; j <= 4; j++)
		      {
		        *vpptr += (n_p[j] - bd_p[j]) * initial_p[b];
		        *vmptr += (n_m[j] - bd_m[j]) * initial_m[b];
		      }
		    }
	      sp_p += bytes;
	      sp_m -= bytes;
	      vp += bytes;
	      vm -= bytes;
	    }
  	  transferPixels (val_p, val_m, dest, bytes, height);

	    if (has_alpha && !horz)
	      separateAlpha (dest, height, bytes);

      for(int n=0;n<image->height();n++)
      {
        int r,g,b,a;
        if(has_alpha)
        {
          r = (int)dest[n*4];
          g = (int)dest[n*4+1];
          b = (int)dest[n*4+2];
          a = (int)dest[n*4+3];
          image->setPixel(col+x1,n,qRgba(r,g,b,a));
        }
        else
        {
          r = (int)dest[n*3];
          g = (int)dest[n*3+1];
          b = (int)dest[n*3+2];
          image->setPixel(col+x1,n,qRgb(r,g,b));
        }
      }
      if(emit_progress)
      {
        if(stopped())
          return false;
        progresscnt += int(double(height) * vert);
        int p = int(100.0*double(progresscnt)/double(progress));
        if((p % 5 == 0) && (p > old_p))
        {
          old_p = p;
          emit signalFilterProgress(p);
          qApp->processEvents();
        }
      }
   	}
  }

  /*  Now the horizontal pass  */
  if (horz >= 1.0)
  {
    horz = fabs (horz) + 1.0;

    if (horz != vert)
  	{
	    std_dev = sqrt (-(horz * horz) / (2 * log (1.0 / 255.0)));

	    findConstants (n_p, n_m, d_p, d_m, bd_p, bd_m, std_dev);
	  }

    for (row = 0; row < height; row++)
	  {
	    memset(val_p, 0, width * bytes * sizeof (double));
	    memset(val_m, 0, width * bytes * sizeof (double));

      for(int n=0;n<image->width();n++)
      {
        QRgb rgb = image->pixel(n,row + y1);
        if(has_alpha)
        {
          src[n*4] = (unsigned char)qRed(rgb);
          src[n*4+1] = (unsigned char)qGreen(rgb);
          src[n*4+2] = (unsigned char)qBlue(rgb);
          src[n*4+3] = (unsigned char)qAlpha(rgb);
        }
        else
        {
          src[n*3] = (unsigned char)qRed(rgb);
          src[n*3+1] = (unsigned char)qGreen(rgb);
          src[n*3+2] = (unsigned char)qBlue(rgb);
        }
      }

	    if (has_alpha && !vert)
	      multiplyAlpha (src, height, bytes);

  	  sp_p = src;
  	  sp_m = src + (width - 1) * bytes;
  	  vp = val_p;
  	  vm = val_m + (width - 1) * bytes;

  	  for (i = 0; i < bytes; i++)
	    {
	      initial_p[i] = sp_p[i];
	      initial_m[i] = sp_m[i];
	    }

	    for (col = 0; col < width; col++)
	    {
	      double *vpptr, *vmptr;
	      terms = (col < 4) ? col : 4;

	      for (b = 0; b < bytes; b++)
		    {
    		  vpptr = vp + b; vmptr = vm + b;
		      for (i = 0; i <= terms; i++)
		      {
		        *vpptr += n_p[i] * sp_p[(-i * bytes) + b] -
		                	d_p[i] * vp[(-i * bytes) + b];
		        *vmptr += n_m[i] * sp_m[(i * bytes) + b] -
			                d_m[i] * vm[(i * bytes) + b];
		      }
		      for (j = i; j <= 4; j++)
		      {
		        *vpptr += (n_p[j] - bd_p[j]) * initial_p[b];
		        *vmptr += (n_m[j] - bd_m[j]) * initial_m[b];
		      }
		    }

	      sp_p += bytes;
	      sp_m -= bytes;
	      vp += bytes;
	      vm -= bytes;
	    }

	    transferPixels (val_p, val_m, dest, bytes, width);

	    if (has_alpha)
	      separateAlpha (dest, width, bytes);

      for(int n=0;n<image->width();n++)
      {
        int r,g,b,a;
        if(has_alpha)
        {
          r = (int)dest[n*4];
          g = (int)dest[n*4+1];
          b = (int)dest[n*4+2];
          a = (int)dest[n*4+3];
          image->setPixel(n,row+y1,qRgba(r,g,b,a));
        }
        else
        {
          r = (int)dest[n*3];
          g = (int)dest[n*3+1];
          b = (int)dest[n*3+2];
          image->setPixel(n,row+y1,qRgb(r,g,b));
        }
      }

      if(emit_progress)
      {
        if(stopped())
          return false;
        progresscnt += int(double(width) * horz);
        int p = int(100.0*double(progresscnt)/double(progress));
        if((p % 5 == 0) && (p > old_p))
        {
          old_p = p;
          emit signalFilterProgress(p);
          qApp->processEvents();
        }
      }
   	}
  }
  if(image->depth() != orig_depth)
    *image = image->convertDepth(orig_depth);
  return true;
}
void GaussIIRBlurDialog::findConstants(double n_p[],double n_m[],
                                   		 double d_p[],double d_m[],
                                   		 double bd_p[],double bd_m[],
                                   		 double std_dev)
{
  int i;
  double constants [8];
  double div;

  div = sqrt(2 * 3.14) * std_dev;
  constants [0] = -1.783 / std_dev;
  constants [1] = -1.723 / std_dev;
  constants [2] = 0.6318 / std_dev;
  constants [3] = 1.997  / std_dev;
  constants [4] = 1.6803 / div;
  constants [5] = 3.735 / div;
  constants [6] = -0.6803 / div;
  constants [7] = -0.2598 / div;

  n_p [0] = constants[4] + constants[6];
  n_p [1] = exp (constants[1]) *
            (constants[7] * sin (constants[3]) -
            (constants[6] + 2 * constants[4]) * cos (constants[3])) +
            exp (constants[0]) *
	          (constants[5] * sin (constants[2]) -
	          (2 * constants[6] + constants[4]) * cos (constants[2]));
  n_p [2] = 2 * exp (constants[0] + constants[1]) *
            ((constants[4] + constants[6]) * cos (constants[3]) * cos (constants[2]) -
            constants[5] * cos (constants[3]) * sin (constants[2]) -
            constants[7] * cos (constants[2]) * sin (constants[3])) +
            constants[6] * exp (2 * constants[0]) +
	          constants[4] * exp (2 * constants[1]);
  n_p [3] = exp (constants[1] + 2 * constants[0]) *
            (constants[7] * sin (constants[3]) - constants[6] * cos (constants[3])) +
            exp (constants[0] + 2 * constants[1]) *
	          (constants[5] * sin (constants[2]) - constants[4] * cos (constants[2]));
  n_p [4] = 0.0;

  d_p [0] = 0.0;
  d_p [1] = -2 * exp (constants[1]) * cos (constants[3]) -
            2 * exp (constants[0]) * cos (constants[2]);
  d_p [2] = 4 * cos (constants[3]) * cos (constants[2]) * exp (constants[0] + constants[1]) +
            exp (2 * constants[1]) + exp (2 * constants[0]);
  d_p [3] = -2 * cos (constants[2]) * exp (constants[0] + 2 * constants[1]) -
            2 * cos (constants[3]) * exp (constants[1] + 2 * constants[0]);
  d_p [4] = exp (2 * constants[0] + 2 * constants[1]);

  for (i = 0; i <= 4; i++)
    d_m [i] = d_p [i];

  n_m[0] = 0.0;
  for (i = 1; i <= 4; i++)
    n_m [i] = n_p[i] - d_p[i] * n_p[0];

  double sum_n_p, sum_n_m, sum_d;
  double a, b;

  sum_n_p = 0.0;
  sum_n_m = 0.0;
  sum_d = 0.0;
  for (i = 0; i <= 4; i++)
  {
	  sum_n_p += n_p[i];
	  sum_n_m += n_m[i];
	  sum_d += d_p[i];
  }

  a = sum_n_p / (1 + sum_d);
  b = sum_n_m / (1 + sum_d);

  for (i = 0; i <= 4; i++)
  {
  	bd_p[i] = d_p[i] * a;
  	bd_m[i] = d_m[i] * b;
  }
}
/**  */
void GaussIIRBlurDialog::separateAlpha(unsigned char* buf,int width,int bytes)
{
  int i, j;
  unsigned char alpha;
  double recip_alpha;
  int new_val;

  for (i = 0; i < width * bytes; i += bytes)
  {
    alpha = buf[i + bytes - 1];
    if (alpha != 0 && alpha != 255)
	  {
	    recip_alpha = int(255.0/double(alpha));
	    for (j = 0; j < bytes - 1; j++)
	    {
	      new_val = int(double(buf[i + j]) * recip_alpha);
        if(255 < new_val)
	        buf[i + j] = 255;
        else
          buf[i + j] = new_val;
	    }
	  }
  }
}
/**  */
void GaussIIRBlurDialog::multiplyAlpha(unsigned char* buf,int width,int bytes)
{
  int i, j;
  double alpha;

  for (i = 0; i < width * bytes; i += bytes)
  {
    alpha = buf[i + bytes - 1] * (1.0 / 255.0);
    for (j = 0; j < bytes - 1; j++)
	    buf[i + j] = (unsigned char)(double(buf[i + j])*alpha);
  }
}
/**  */
void GaussIIRBlurDialog::transferPixels(double* src1,double* src2,unsigned char* dest,
		                                    int bytes,int width)
{
  int b;
  int bend = bytes * width;
  double sum;

  for(b = 0; b < bend; b++)
  {
    sum = *src1++ + *src2++;
    if (sum > 255)
      sum = 255;
    else if(sum < 0) sum = 0;
      *dest++ = (unsigned char) sum;
  }
}
/**  */
void GaussIIRBlurDialog::slotRadiusHChanged(int value)
{
  mRadiusH = double(value);
  if(mpBindCheckBox->isChecked())
    mpSliderSpinV->setValue(value);
  updatePreview();
}
/**  */
void GaussIIRBlurDialog::slotRadiusVChanged(int value)
{
  mRadiusV = double(value);
  if(mpBindCheckBox->isChecked())
    mpSliderSpinH->setValue(value);
  updatePreview();
}
/**  */
void GaussIIRBlurDialog::slotBindValues(bool state)
{
  if(state)
  {
    if(mRadiusH > mRadiusV)
    {
      mpSliderSpinV->setValue(mpSliderSpinH->value());
    }
    else
    {
      mpSliderSpinH->setValue(mpSliderSpinV->value());
    }
  }
}
/** No descriptions */
void GaussIIRBlurDialog::saveConfig()
{
  xmlConfig->setIntValue("FILTER_GAUSSIIRBLUR_VERTICAL",mpSliderSpinV->value());
  xmlConfig->setIntValue("FILTER_GAUSSIIRBLUR_HORIZONTAL",mpSliderSpinH->value());
  xmlConfig->setBoolValue("FILTER_GAUSSIIRBLUR_GREEN",mpBindCheckBox->isChecked());
  xmlConfig->setBoolValue("FILTER_GAUSSIIRBLUR_CONTINOUS_UPDATE",continousUpdate());
}
