#
#  Ordinal: A Library of Ordinal Models
#  Copyright (C) 1998, 1999, 2000, 2001 P.J. Lindsey
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public Licence as published by
#  the Free Software Foundation; either version 2 of the Licence, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public Licence for more details.
#
#  You should have received a copy of the GNU General Public Licence
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#  SYNOPSIS
#
#  lcr(response=NULL,frequencies=NULL,mu=NULL,ccov=NULL,tvcov=NULL,
#      times=F,pcoef=NULL,plink=NULL,distribution=NULL,direction=NULL,
#      transformation=NULL,link=NULL,envir=parent.frame(),
#      steptol=1e-04,iterlim=100)
#
#  DESCRIPTION
#
#    A function to fit various linear categorical regression models.
#

lcr <- function(response=NULL,frequencies=NULL,mu=NULL,ccov=NULL,tvcov=NULL,times=F,pcoef=NULL,plink=NULL,distribution=NULL,direction=NULL,transformation=NULL,
                link=NULL,envir=parent.frame(),steptol=1e-04,iterlim=100) {
  respenv <- exists(deparse(substitute(response)),envir=parent.frame())&&inherits(response,"repeated")&&!inherits(envir,"repeated")
  if(respenv) {
    if(dim(response$response$y)[2]>1)
      stop("Only univariate responses are handled.")
    if(!is.null(response$NAs)&&any(response$NAs))
      stop("Data with NAs cannot be handled.")
    if(response$response$type!="ordinal")
      stop("The response must be of type \'ordinal\'.")
  }
  if(!respenv&&inherits(envir,"repeated")) {
    if(!is.null(envir$NAs)&&any(envir$NAs))
      stop("Data with NAs cannot be handled.")
    cn <- deparse(substitute(response))
    if(length(grep("\"",cn))>0)
      cn <- response
    if(length(cn)>1)
      stop("Only one response variable is allowed.")
    response <- envir
    if(dim(response$response$y)[2]>1) {
      col <- match(cn,colnames(response$response$y))
      if(is.na(col))
        stop("The response variable was not found.")
      if(response$response$type[col]!="ordinal")
        stop("The response must be of type \'ordinal\'.")
      response$response$y <- response$response$y[,col,drop=F]
      rm(col)
    }
    else
      if(response$response$type!="ordinal")
        stop("The response must be of type \'ordinal\'.")
    rm(cn)
  }
  if(respenv||inherits(envir,"repeated")) {
    rm(frequencies,times,ccov,tvcov)
    if(is.null(mu)) {
      response$ccov <- response$tvcov <- NULL
      ncv <- 0
    }
    else
      if(inherits(mu,"formula")) {
        ccname <- rownames(attr(terms(mu,data=response),"factors"))
        if(!is.null(response$response$times))
          ccnames <- c(colnames(response$ccov$ccov),"times",colnames(response$tvcov$tvcov))
        else
          ccnames <- c(colnames(response$ccov$ccov),colnames(response$tvcov$tvcov))
        if(any(is.na(match(ccname,ccnames))))
          stop(paste("Covariate(s) \'",paste(ccname[is.na(match(ccname,ccnames))],collapse=", "),"\' was(were) not found.",sep=""))
        rm(ccname,ccnames)
        tmp <- wr(mu,data=response,expand=T)$design
        response$ccov <- NULL
        response$tvcov$tvcov <- tmp[,-1,drop=F]
        rm(tmp)
        ncv <- dim(response$tvcov$tvcov)[2]
      }
      else
        stop("Covariate(s) must be specified as a W&R formula.")
    rm(mu)
  }
  else {
    mu <- NULL
    if(!inherits(response,"response"))
      response <- restovec(response,times=times,weights=frequencies,type="ordinal")
    rm(frequencies,times)
    if(dim(response$y)[2]>1)
      stop("Multivariate responses are not handled.")
    if(response$type!="ordinal")
      stop("The response must be of type \'ordinal\'.")
    if(!is.null(ccov)) {
      if(!inherits(ccov,"tccov")) {
        ccname <- deparse(substitute(ccov))
        if((is.matrix(ccov)&&is.null(colnames(ccov)))) {
          ccname <- deparse(substitute(ccov))
          if(dim(ccov)[2]>1) {
            tmp <- NULL
            for(i in 1:dim(ccov)[2])
              tmp <- c(tmp,paste(ccname,i,sep=""))
            ccname <- tmp
            rm(tmp)
          }
        }
        ccov <- tcctomat(ccov,names=ccname)
        rm(ccname)
      }
      ncv <- dim(ccov$ccov)[2]
    }
    else
      ncv <- 0
    if(!is.null(tvcov)) {
      if(!inherits(tvcov,"tvcov")) {
        tvcname <- deparse(substitute(tvcov))
        if(is.list(tvcov)&&dim(tvcov[[1]])[2]>1) {
          if(is.null(colnames(tvcov[[1]]))) {
            tvcname <- deparse(substitute(tvcov))
            tmp <- NULL
            for(i in 1:dim(tvcov[[1]])[2])
              tmp <- c(tmp,paste(tvcname,i,sep=""))
            tvcname <- tmp
          }
          else
            tvcname <- colnames(tvcov[[1]])
          rm(tvcname)
        }
        tvcov <- tvctomat(tvcov,names=tvcname)
      }
      ncv <- ncv+dim(tvcov$tvcov)[2]
    }
    response <- rmna(response=response,tvcov=tvcov,ccov=ccov)
    if(!is.null(ccov)) {
      response$ccov$ccov <- apply(response$ccov$ccov,2,function(x) rep(x,nobs(response)))
      if(!is.null(tvcov))
        response$tvcov$tvcov <- cbind(response$ccov$ccov,response$tvcov$tvcov)
      else
        response$tvcov$tvcov <- response$ccov$ccov
      response$ccov <- NULL
    }
    rm(ccov,tvcov)
    if(!is.null(response$response$times)) {
      if(is.null(response$tvcov$tvcov))
        response$tvcov$tvcov <- matrix(response$response$times,ncol=1,dimnames=list(NULL,"times"))
      else
        response$tvcov$tvcov <- cbind(response$tvcov$tvcov,matrix(response$response$times,ncol=1,dimnames=list(NULL,"times")))
      ncv <- ncv+1
    }
  }
  n <- dim(response$response$y)[1]
  nobs <- nobs(response)
  nind <- length(nobs)
  if((inherits(envir,"repeated")&&(length(nobs)!=length(nobs(envir))||any(nobs!=nobs(envir))))||(inherits(envir,"tvcov")&&(length(nobs)!=length(envir$tvcov$nobs)||any(nobs!=envir$tvcov$nobs))))
    stop("Response and envir objects are incompatible.")
  if(response$response$type!="ordinal")
    stop("The response must be of type \'ordinal\'.")
  if(is.null(weights(response))) {
    response$response$wt <- rep(1,n)
    individual <- T
  }
  else
    if(length(unique(weights(response)))==1&&unique(weights(response))==1)
      individual <- T
    else
      individual <- F
  distributions <- c("binary","multinomial","simplified multinomial","continuation-ratio","proportional odds","adjacent categories")
  directions <- c("upwards","downwards")
  transformations <- c("identity","sqrt","log","logit","cloglog","square","exp","loglog")
  if(is.null(transformation))
    transformation <- 1
  else
    transformation <- match(match.arg(transformation,transformations),transformations)
  links <- c("logit","probit","loglog","cloglog","uniform","log-normal","exponential","Pareto","Cauchy","Laplace","Levy","simplex",
             "gamma","Weibull","inverse Gauss","t","chi-square","gen logistic","gen extreme value","Box-Cox","power exponential",
             "Burr","Hjorth","beta","stable","gen gamma","gen Weibull","gen inverse Gauss","F","nc t","nc chi-square","Tukey",
             "nc beta","nc F")
  levels <- sort(unique(response$response$y))
  nc <- length(levels)
  if(nc==2) {
    if(is.null(distribution))
      distribution <- "binary"
    else {
      distribution <- match.arg(distribution,distributions)
      if(distribution!="binary") {
        distribution <- "binary"
        warning("The distribution has been changed to \'binary\' as the response has only 2 levels.")
      }
    }
    if(is.null(pcoef))
      pcoef <- rep(0,(1+ncv))
    else
      if(length(pcoef)!=(1+ncv))
        stop(paste("The initial estimates must be a vector of length ",(1+ncv),".",sep=""))
    if(is.null(link))
      link <- 1
    else
      link <- match(match.arg(link,links),links)
    if(!is.null(direction)) {
      direction <- NULL
      warning("The direction is only required by the \'continuation-ratio\' distribution.")
    }
  }
  else
    if(is.null(distribution)) {
      distribution <- "multinomial"
      if(is.null(pcoef))
        pcoef <- rep(0,(1+ncv)*(nc-1))
      else
        if(length(pcoef)!=(1+ncv)*(nc-1))
          stop(paste("The initial estimates must be a vector of length ",(1+ncv)*(nc-1),".",sep=""))
      if(!is.null(link)&&link!="logit")
        warning("The link argument is only allowed with the \'proportional odds\' and \'binary\' distributions.")
      link <- NULL
      if(!is.null(direction)) {
        direction <- NULL
        warning("The direction is only required by the \'continuation-ratio\' distribution.")
      }
    }
    else {
      distribution <- match.arg(distribution,distributions)
      if(distribution=="binary")
        stop("The option \'distribution\' can only be set to binary if the \'response\' vector has 2 levels.")
      if(distribution=="proportional odds") {
        if(is.null(pcoef))
          pcoef <- c(1:(nc-1),rep(0,ncv))
        else
          if(length(pcoef)!=(nc-1+ncv))
            stop(paste("The initial estimates must be a vector of length ",(nc-1+ncv),".",sep=""))
          else
            if(length(unique(pcoef[1:(nc-1)]))!=(nc-1)) {
              pcoef[1:(nc-1)] <- 1:(nc-1)
              warning("Changed initial estimates for the intercepts as they must be different.")
            }
        if(is.null(link))
          link <- 1
        else
          link <- match(match.arg(link,links),links)
        if(!is.null(direction)) {
          direction <- NULL
          warning("The direction is only required by the \'continuation-ratio\' distribution.")
        }
      }
      else {
        if(!is.null(link)) {
          link <- NULL
          warning("The link argument is only allowed with the \'proportional odds\' and \'binary\' distributions.")
        }
        if(distribution=="multinomial") {
          if(is.null(pcoef))
            pcoef <- rep(rep(0,1+ncv),nc-1)
          else
            if(length(pcoef)!=(1+ncv)*(nc-1))
              stop(paste("The initial estimates must be a vector of length ",(1+ncv)*(nc-1),".",sep=""))
          if(!is.null(direction)) {
            direction <- NULL
            warning("The direction is only required by the \'continuation-ratio\' distribution.")
          }
        }
        else {
          if(is.null(pcoef))
            pcoef <- rep(0,(nc-1+ncv))
          else
            if(length(pcoef)!=(nc-1+ncv))
              stop(paste("The initial estimates must be a vector of length ",(nc-1+ncv),".",sep=""))
          if(distribution=="continuation-ratio")
            if(is.null(direction))
              direction <- "upwards"
            else
              direction <- match.arg(direction,directions)
          else
            if(!is.null(direction)) {
              direction <- NULL
              warning("The direction is only required by the \'continuation-ratio\' distribution.")
            }
        }
      }
    }
  if(distribution=="binary"||distribution=="proportional odds") {
    if(is.null(plink))
      switch(links[link],
             "gamma"=plink <- 2,
             "Weibull"=plink <- 1,
             "inverse Gauss"=plink <- 1,
             "t"=plink <- 1,
             "chi-square"=plink <- 3,
             "gen logistic"=plink <- 4,
             "gen extreme value"=plink <- -2,
             "Box-Cox"=plink <- 1,
             "power exponential"=plink <- 1,
             "Burr"=plink <- 3,
             "Hjorth"=plink <- 1,
             "beta"=plink <- c(2,2),
             "stable"=plink <- c(0.5,1),
             "gen gamma"=plink <- c(2,0.5),
             "gen Weibull"=plink <- c(1,3),
             "gen inverse Gauss"=plink <- c(1,1),
             "F"=plink <- c(10,10),
             "nc t"=plink <- c(1,0),
             "nc chi-square"=plink <- c(3,0),
             "Tukey"=plink <- c(1,2,2),
             "nc beta"=plink <- c(2,2,0),
             "nc F"=plink <- c(10,10,0)
             )
    else
      if(link<=12) {
        plink <- NULL
        warning(paste("No additional parameters are needed for a ",links[link]," link.",sep=""))
      }
      else
        switch(links[link],
               "gamma"=if(length(plink)!=1) {
                 plink <- 2
                 warning("One additional parameter is needed for a \'gamma\' link.")
               }
               else {
                 if(plink<=0) {
                   plink <- 2
                   warning("The additional parameter of the \'gamma\' link (shape) was ignored as it must be greater than 0.")
                 }
               },
               "Weibull"=if(length(plink)!=1) {
                 plink <- 1
                 warning("One additional parameter is needed for a \'Weibull\' link.")
               }
               else {
                 if(plink<=0) {
                   plink <- 1
                   warning("The additional parameter of the \'Weibull\' link (shape) was ignored as it must be greater than 0.")
                 }
               },
               "inverse Gauss"=if(length(plink)!=1) {
                 plink <- 1
                 warning("One additional parameter is needed for a \'inverse Gauss\' link.")
               }
               else {
                 if(plink<0) {
                   plink <- 1
                   warning("The additional parameter of the \'inverse Gauss\' link (shape) was ignored as it must be greater than or equal to 0.")
                 }
               },
               "t"=if(length(plink)!=1) {
                 plink <- 1
                 warning("One additional parameter is needed for a \'t\' link.")
               }
               else {
                 if(plink<=0) {
                   plink <- 1
                   warning("The additional parameter of the \'t\' link (df) was ignored as it must be greater than 0.")
                 }
               },
               "chi-square"=if(length(plink)!=1) {
                 plink <- 3
                 warning("One additional parameter is needed for a \'chi-square\' link.")
               }
               else {
                 if(plink<0) {
                   plink <- 3
                   warning("The first additional parameter of the \'chi-square\' link (df) was ignored as it must be greater than or equal to 0.")
                 }
               },
               "gen logistic"=if(length(plink)!=1) {
                 plink <- 4
                 warning("One additional parameters are needed for a \'gen logistic\' link.")
               }
               else {
                 if(plink[1]<=0) {
                   plink[1] <- 4
                   warning("The additional parameter of the \'gen logistic\' link (family) was ignored as it must be greater than 0.")
                 }
               },
               "gen extreme value"=if(length(plink)!=1) {
                 plink <- -2
                 warning("One additional parameter is needed for a \'gen extreme value\' link.")
               },
               "Box-Cox"=if(length(plink)!=1) {
                 plink <- 1
                 warning("One additional paramete is needed for a \'Box-Cox\' link.")
               },
               "power exponential"=if(length(plink)!=1) {
                 plink <- 1
                 warning("One additional parameter is needed for a \'power exponential\' link.")
               }
               else {
                 if(plink<=0) {
                   plink <- 1
                   warning("The additional parameter of the \'power exponential\' link (family) was ignored as it must be greater than 0.")
                 }
               },
               "Burr"=if(length(plink)!=1) {
                 plink <- 3
                 warning("One additional parameter is needed for a \'Burr\' link.")
               }
               else {
                 if(plink<=0) {
                   plink <- 3
                   warning("The additional parameter of the \'Burr\' link (family) was ignored as it must be greater than 0.")
                 }
               },
               "Hjorth"=if(length(plink)!=1) {
                 plink <- 1
                 warning("One additional parameter is needed for a \'Hjorth\' link.")
               },
               "beta"=if(length(plink)!=2) {
                 plink <- c(2,2)
                 warning("Three additional parameters are needed for a \'beta\' link.")
               }
               else {
                 if(plink[1]<=0) {
                   plink[1] <- 2
                   warning("The first additional parameter of the \'beta\' link (shape1) was ignored as it must be greater than 0.")
                 }
                 if(plink[2]<=0) {
                   plink[2] <- 2
                   warning("The second additional parameter of the \'beta\' link (shape2) was ignored as it must be greater than 0.")
                 }
               },
               "stable"=if(length(plink)!=2) {
                 plink <- c(0.5,1)
                 warning("Two additional parameters are needed for a \'stable\' link.")
               }
               else {
                 if(-1>plink[1]||plink[1]>1) {
                   plink[1] <- 0.5
                   warning("The first additional parameter of the \'stable\' link (skew) was ignored as it must take a value in (-1,1).")
                 }
                 if(plink[2]<=0||plink[2]>2) {
                   plink[2] <- 1
                   warning("The second additional parameter of the \'stable\' link (tail) was ignored as it must take a value in (0,2).")
                 }
               },
               "gen gamma"=if(length(plink)!=2) {
                 plink <- c(2,0.5)
                 warning("Two additional parameters are needed for a \'gen gamma\' link.")
               }
               else {
                 if(plink[1]<=0) {
                   plink[1] <- 2
                   warning("The first additional parameter of the \'gen gamma\' link (shape) was ignored as it must be greater than 0.")
                 }
                 if(plink[2]<=0) {
                   plink[2] <- 0.5
                   warning("The second additional parameter of the \'gen gamma\' link (family) was ignored as it must be greater than 0.")
                 }
               },
               "gen Weibull"=if(length(plink)!=2) {
                 plink <- c(1,3)
                 warning("Two additional parameters are needed for a \'gen Weibull\' link.")
               }
               else {
                 if(plink[1]<=0) {
                   plink[1] <- 1
                   warning("The first additional parameter of the \'gen Weibull\' link (shape) was ignored as it must be greater than 0.")
                 }
                 if(plink[2]<=0) {
                   plink[2] <- 3
                   warning("The second additional parameter of the \'gen Weibull\' link (family) was ignored as it must be greater than 0.")
                 }
               },
               "gen inverse Gauss"=if(length(plink)!=2) {
                 plink <- c(1,1)
                 warning("Two additional parameters are needed for a \'gen inverse Gauss\' link.")
               }
               else {
                 if(plink[1]<=0) {
                   plink[1] <- 1
                   warning("The first additional parameter of the \'gen inverse Gauss\' link (shape) was ignored as it must be greater than 0.")
                 }
               },
               "F"=if(length(plink)!=2) {
                 plink <- c(10,10)
                 warning("Two additional parameters are needed for a \'F\' link.")
               }
               else {
                 if(plink[1]<=0) {
                   plink[1] <- 10
                   warning("The first additional parameter of the \'F\' link (df1) was ignored as it must be greater than 0.")
                 }
                 if(plink[2]<=0) {
                   plink[2] <- 10
                   warning("The second additional parameter of the \'F\' link (df2) was ignored as it must be greater than 0.")
                 }
               },
               "nc t"=if(length(plink)!=2) {
                 plink <- c(1,0)
                 warning("Two additional parameters are needed for a \'nc t\' link.")
               }
               else {
                 if(plink[1]<=0) {
                   plink[1] <- 1
                   warning("The first additional parameter of the \'nc t\' link (df) was ignored as it must be greater than 0.")
                 }
                 if(plink[2]>37.62) {
                   plink[2] <- 0
                   warning("The second additional parameter of the \'nc t\' link (ncp) was ignored as it must be smaller than 37.62.")
                 }
               },
               "nc chi-square"=if(length(plink)!=2) {
                 plink <- c(3,0)
                 warning("Two additional parameters are needed for a \'nc chi-square\' link.")
               }
               else {
                 if(plink[1]<0) {
                   plink[1] <- 3
                   warning("The first additional parameter of the \'nc chi-square\' link (df) was ignored as it must be greater than or equal to 0.")
                 }
                 if(plink[2]<0) {
                   plink[2] <- 0
                   warning("The second additional parameter of the \'nc chi-square\' link (ncp) was ignored as it must be greater than or equal to 0.")
                 }
               },
               "Tukey"=if(length(plink)!=3) {
                 plink <- c(1,2,2)
                 warning("Three additional parameters are needed for a \'Tukey\' link.")
               }
               else {
                 if(plink[1]<1) {
                   plink[1] <- 1
                   warning("The first additional parameter of the \'Tukey\' link (nranges) was ignored as it must be greater than or equal to 1.")
                 }
                 if(plink[2]<2) {
                   plink[2] <- 2
                   warning("The second additional parameter of the \'Tukey\' link (nmeans) was ignored as it must be greater than or equal to 2.")
                 }
                 if(plink[3]<2) {
                   plink[3] <- 2
                   warning("The third additional parameter of the \'Tukey\' link (df) was ignored as it must be greater than or equal to 2.")
                 }
               },
               "nc beta"=if(length(plink)!=3) {
                 plink <- c(2,2,0)
                 warning("Three additional parameters are needed for a \'nc beta\' link.")
               }
               else {
                 if(plink[1]<=0) {
                   plink[1] <- 2
                   warning("The first additional parameter of the \'nc beta\' link (shape1) was ignored as it must be greater than 0.")
                 }
                 if(plink[2]<=0) {
                   plink[2] <- 2
                   warning("The second additional parameter of the \'nc beta\' link (shape2) was ignored as it must be greater than 0.")
                 }
                 if(plink[3]<0) {
                   plink[3] <- 0
                   warning("The third additional parameter of the \'nc beta\' link (ncp) was ignored as it must be greater than or equal to 0.")
                 }
               },
               "nc F"=if(length(plink)!=3) {
                 plink <- c(10,10,0)
                 warning("Three additional parameters are needed for a \'nc F\' link.")
               }
               else {
                 if(plink[1]<=0) {
                   plink[1] <- 10
                   warning("The first additional parameter of the \'nc F\' link (df1) was ignored as it must be greater than 0.")
                 }
                 if(plink[2]<=0) {
                   plink[2] <- 10
                   warning("The second additional parameter of the \'nc F\' link (df2) was ignored as it must be greater than 0.")
                 }
                 if(plink[3]<0) {
                   plink[3] <- 0
                   warning("The third additional parameter of the \'nc F\' link (ncp) was ignored as it must be greater than or equal to 0.")
                 }
               }
               )
  }
  switch(distribution,
         "binary" = z <- .C("lbr",
           as.integer(response$response$y),
           as.integer(response$response$wt),
           as.double(response$tvcov$tvcov),
           as.integer(ncv),
           beta=double(1+ncv),
           as.double(pcoef),
           as.integer(n),
           double(ncv),
           iter=integer(1),
           as.integer(iterlim),
           as.double(steptol),
           double((1+ncv)*(1+ncv)),
           double(1+ncv),
           as.integer(transformation),
           as.integer(link),
           as.double(plink),
           rank=integer(1),
           double(1+ncv),
           as.integer(1:(1+ncv)),
           double(2*(1+ncv)),
           double(1+ncv),
           info=integer(1),
           as.double(diag(1,1+ncv)),
           hessian=double((1+ncv)*(1+ncv)),
           loglik=double(1),
           fit=double(n),
           pred=integer(n),
           cpred=double(n),
           DUP=F,
           PACKAGE="ordinal")[c(5,9,17,22,24:28)],
         "multinomial" = z <- .C("lmr",
           as.integer(response$response$y),
           as.integer(response$response$wt),
           as.double(response$tvcov$tvcov),
           as.integer(nc),
           as.integer(ncv),
           beta=double((nc-1)*(1+ncv)),
           as.double(pcoef),
           as.integer(n),
           integer(nc-1),
           double((nc-1)*(1+ncv)),
           iter=integer(1),
           as.integer(iterlim),
           as.double(steptol),
           double(((nc-1)*(1+ncv))*((nc-1)*(1+ncv))),
           double((nc-1)*(1+ncv)),
           double(nc-1),
           as.integer(transformation),
           double(nc-1),
           rank=integer(1),
           double((nc-1)*(1+ncv)),
           as.integer(1:((nc-1)*(1+ncv))),
           double(2*((nc-1)*(1+ncv))),
           double((nc-1)*(1+ncv)),
           info=integer(1),
           as.double(diag(1,(nc-1)*(1+ncv))),
           hessian=double(((nc-1)*(1+ncv))*((nc-1)*(1+ncv))),
           loglik=double(1),
           fit=double(n),
           pred=integer(n),
           cpred=double(n*(nc-1)),
           DUP=F,
           PACKAGE="ordinal")[c(6,11,19,24,26:30)],
         "simplified multinomial" = z <- .C("lsmr",
           as.integer(response$response$y),
           as.integer(response$response$wt),
           as.double(response$tvcov$tvcov),
           as.integer(nc),
           as.integer(ncv),
           beta=double(nc-1+ncv),
           as.double(pcoef),
           as.integer(n),
           integer(nc-1),
           double(ncv),
           iter=integer(1),
           as.integer(iterlim),
           as.double(steptol),
           double((nc-1+ncv)*(nc-1+ncv)),
           double(nc-1+ncv),
           double(nc-1),
           as.integer(transformation),
           double(nc-1),
           rank=integer(1),
           double(nc-1+ncv),
           as.integer(1:(nc-1+ncv)),
           double(2*(nc-1+ncv)),
           double(nc-1+ncv),
           info=integer(1),
           as.double(diag(1,nc-1+ncv)),
           hessian=double((nc-1+ncv)*(nc-1+ncv)),
           loglik=double(1),
           fit=double(n),
           pred=integer(n),
           cpred=double(n*(nc-1)),
           DUP=F,
           PACKAGE="ordinal")[c(6,11,19,24,26:30)],
         "continuation-ratio" = {
           if(direction=="upwards")
             z <- .C("lucrr",
                     as.integer(response$response$y),
                     as.integer(response$response$wt),
                     as.double(response$tvcov$tvcov),
                     as.integer(nc),
                     as.integer(ncv),
                     beta=double(nc-1+ncv),
                     as.double(pcoef),
                     as.integer(n),
                     integer(nc-1),
                     double(ncv),
                     iter=integer(1),
                     as.integer(iterlim),
                     as.double(steptol),
                     double((nc-1+ncv)*(nc-1+ncv)),
                     double(nc-1+ncv),
                     double(nc-1),
                     as.integer(transformation),
                     double(nc-1),
                     double(nc-1),
                     rank=integer(1),
                     double(nc-1+ncv),
                     as.integer(1:(nc-1+ncv)),
                     double(2*(nc-1+ncv)),
                     double(nc-1+ncv),
                     info=integer(1),
                     as.double(diag(1,nc-1+ncv)),
                     hessian=double((nc-1+ncv)*(nc-1+ncv)),
                     loglik=double(1),
                     fit=double(n),
                     pred=integer(n),
                     cpred=double(n*(nc-1)),
                     DUP=F,
                     PACKAGE="ordinal")[c(6,11,20,25,27:31)]
           else
             z <- .C("ldcrr",
                     as.integer(response$response$y),
                     as.integer(response$response$wt),
                     as.double(response$tvcov$tvcov),
                     as.integer(nc),
                     as.integer(ncv),
                     beta=double(nc-1+ncv),
                     as.double(pcoef),
                     as.integer(n),
                     integer(nc-1),
                     double(ncv),
                     iter=integer(1),
                     as.integer(iterlim),
                     as.double(steptol),
                     double((nc-1+ncv)*(nc-1+ncv)),
                     double(nc-1+ncv),
                     double(nc-1),
                     as.integer(transformation),
                     double(nc-1),
                     double(nc-1),
                     rank=integer(1),
                     double(nc-1+ncv),
                     as.integer(1:(nc-1+ncv)),
                     double(2*(nc-1+ncv)),
                     double(nc-1+ncv),
                     info=integer(1),
                     as.double(diag(1,nc-1+ncv)),
                     hessian=double((nc-1+ncv)*(nc-1+ncv)),
                     loglik=double(1),
                     fit=double(n),
                     pred=integer(n),
                     cpred=double(n*(nc-1)),
                     DUP=F,
                     PACKAGE="ordinal")[c(6,11,20,25,27:31)]
         },
         "proportional odds" = z <- .C("lpor",
           as.integer(response$response$y),
           as.integer(response$response$wt),
           as.double(response$tvcov$tvcov),
           as.integer(nc),
           as.integer(ncv),
           beta=double(nc-1+ncv),
           as.double(pcoef),
           as.integer(n),
           integer(nc-1),
           double(ncv),
           iter=integer(1),
           as.integer(iterlim),
           as.double(steptol),
           double((nc-1+ncv)*(nc-1+ncv)),
           double(nc-1+ncv),
           double(nc-1),
           as.integer(transformation),
           as.integer(link),
           double(nc-1),
           as.double(plink),
           double(nc-1),
           rank=integer(1),
           double(nc-1+ncv),
           as.integer(1:(nc-1+ncv)),
           double(2*(nc-1+ncv)),
           double(nc-1+ncv),
           info=integer(1),
           as.double(diag(1,nc-1+ncv)),
           hessian=double((nc-1+ncv)*(nc-1+ncv)),
           loglik=double(1),
           fit=double(n),
           pred=integer(n),
           cpred=double(n*(nc-1)),
           DUP=F,
           PACKAGE="ordinal")[c(6,11,22,27,29:33)],
         "adjacent categories" = z <- .C("lacr",
           as.integer(response$response$y),
           as.integer(response$response$wt),
           as.double(response$tvcov$tvcov),
           as.integer(nc),
           as.integer(ncv),
           beta=double(nc-1+ncv),
           as.double(pcoef),
           as.integer(n),
           integer(nc-1),
           double(ncv),
           iter=integer(1),
           as.integer(iterlim),
           as.double(steptol),
           double((nc-1+ncv)*(nc-1+ncv)),
           double(nc-1+ncv),
           double(nc-1),
           as.integer(transformation),
           double(nc-1),
           rank=integer(1),
           double(nc-1+ncv),
           as.integer(1:(nc-1+ncv)),
           double(2*(nc-1+ncv)),
           double(nc-1+ncv),
           info=integer(1),
           as.double(diag(1,nc-1+ncv)),
           hessian=double((nc-1+ncv)*(nc-1+ncv)),
           loglik=double(1),
           fit=double(n),
           pred=integer(n),
           cpred=double(n*(nc-1)),
           DUP=F,
           PACKAGE="ordinal")[c(6,11,19,24,26:30)]
         )
  covar <- matrix(-z$hessian,ncol=length(z$beta))
  se <- sqrt(diag(covar))
  corr <- covar/(se%o%se)
  res <- list(call=sys.call(),
              response=response$response,
              tvcov=response$tvcov,
              frequencies=response$response$wt,
              individual=individual,
              distribution=distribution,
              direction=direction,
              transformation=transformations[transformation],
              link=link,
              n=n, # Number of non-empty cells (or number of loops in c-code)
              nc=nc,
              ncv=ncv,
              events=if(individual) length(nobs(response)) else sum(response$response$wt), # Total number of individuals or events
              likelihood=z$loglik,
              coefficients=z$beta,
              se=se,
              covariance=covar,
              correlations=corr,
              fitted=z$fit,
              pred=z$pred,
              cpred=z$cpred,
              iterations=z$iter,
              iterlim=iterlim,
              info=z$info,
              rank=z$rank)
  class(res) <- c("lcr","repeated")
  return(res)
}
