/*************************************************** */
/* Rule Set Based Access Control                     */
/*                                                   */
/* Author and (c) 1999-2005: Amon Ott <ao@rsbac.org> */
/*                                                   */
/* Last modified: 24/Aug/2005                        */
/*************************************************** */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <rsbac/types.h>
#include <rsbac/syscalls.h>
#include <rsbac/error.h>
#include "nls.h"
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

char * progname;
int verbose = 0;
int backup = 0;
int showpass = 0;
rsbac_list_ta_number_t ta_number = 0;

const char   add_prog[] = "rsbac_groupadd";
const char   mod_prog[] = "rsbac_groupmod";

#define ROOM 20

void use(void)
    {
      printf(gettext("%s (RSBAC %s)\n***\n"), progname, VERSION);
      printf(gettext("Use: %s [flags] groupname\n"), progname);
      printf(gettext(" -v = verbose, -a = list all groups\n"));
      printf(gettext(" -l = short list all groups, -b = backup mode\n"));
      printf(gettext(" -p = also show encrypted password\n"));
      printf(gettext(" -N ta = transaction number (default = value of RSBAC_TA, if set, or 0)\n"));
    }

void pass_print(char * pass, u_int len)
  {
    u_int i;

    for(i=0; i<len; i++)
      {
        printf("%02x", pass[i]);
      }
  }

int process(rsbac_gid_t group)
  {
      union rsbac_um_mod_data_t data;
      int res;
      char groupname[RSBAC_MAXNAMELEN];

      res = rsbac_um_get_group_item(ta_number, group, UM_name, &data);
      if(!res)
        {
         if(backup)
           {
             strcpy(groupname, data.string);
             printf("%s -g %u %s\n", add_prog, group, groupname);
           }
         else
           {
             printf("\n");
             printf("Gid: %u\n", group);
             printf("Name: %s\n", data.string);
           }
        }
      else
        {
          fprintf(stderr, gettext("%s: Unknown group %u\n"), progname, group);
          exit(1);
        }
      if(backup)
        {
          printf("%s", mod_prog);
        }
      if(showpass)
        {
          res = rsbac_um_get_group_item(ta_number, group, UM_pass, &data);
          if(!res)
            {
              if(backup)
                {
                  printf(" -Q ");
                  pass_print(data.string, RSBAC_UM_PASS_LEN);
                }
              else
                {
                  printf("Password: ");
                  pass_print(data.string, RSBAC_UM_PASS_LEN);
                  printf("\n");
                }
            }
        }
      res = rsbac_um_get_group_item(ta_number, group, UM_ttl, &data);
      if(!res)
        {
          if(backup)
            {
              if(data.ttl)
                printf(" -T %u", data.ttl + time(NULL));
            }
          else
            printf("Group TTL: %u\n", data.ttl);
        }
      if(backup)
        {
          printf(" %s\n", groupname);
        }
      else
        {
          rsbac_uid_t * user_array = NULL;
          int user_num = 0;
          int i;

          user_num = rsbac_um_get_gm_user_list(ta_number, group, NULL, 0);
          if(user_num > 0)
            {
              user_num += ROOM;
              user_array = malloc(user_num * sizeof(*user_array));
	      if(!user_array)
                {
                  error_exit(-RSBAC_ENOMEM);
                }
              user_num = rsbac_um_get_gm_user_list(ta_number, group, user_array, user_num);
              if(user_num > 0)
                {
                  printf("Group extra members:");
                  for(i=0; i<user_num; i++)
                    if(!i)
                      printf(" %u", user_array[i]);
                    else
                      printf(",%u", user_array[i]);
                  printf("\n");
                }
              free(user_array);
            }
        }
      return 0;
  }

int main(int argc, char ** argv)
{
  int res = 0;
  rsbac_gid_t group = RSBAC_NO_GROUP;
  int err;
  int i;
  int list = 0;

  locale_init();

  progname = argv[0];
  {
    char * env = getenv("RSBAC_TA");

    if(env)
      ta_number = strtoul(env,0,0);
  }
  while((argc > 1) && (argv[1][0] == '-'))
    {
      char * pos = argv[1];
      pos++;
      while(*pos)
        {
          switch(*pos)
            {
              case 'h':
                use();
                return 0;
              case 'v':
                verbose++;
                break;

              case 'b':
                backup = 1;
                break;

              case 'p':
                showpass = 1;
                break;

              case 'l':
                list = 1;
                /* fall through */
              case 'a':
                {
                  rsbac_gid_t * group_array = NULL;
                  int group_num = 0;
                  int i;

                  group_num = rsbac_um_get_group_list(ta_number, NULL, 0);
                  error_exit(group_num);
                  group_num += ROOM;
                  group_array = malloc(group_num * sizeof(*group_array));
	          if(!group_array)
	            {
                      error_exit(-RSBAC_ENOMEM);
	            }
                  group_num = rsbac_um_get_group_list(ta_number, group_array, group_num);
                  error_exit(group_num);
                  if(group_num > 0)
                    {
                      qsort(group_array, group_num, sizeof(*group_array), rsbac_group_compare);
                      if(list)
                        {
                          union rsbac_um_mod_data_t data;

                          for(i=0; i<group_num; i++)
                            {
                              res = rsbac_um_get_group_item(ta_number, group_array[i], UM_name, &data);
                              if(!res)
                                printf("%u %s\n",
                                       group_array[i], data.string);
                            }
                        }
                      else
                        for(i=0; i<group_num; i++)
                          process(group_array[i]);
                    }
                  free(group_array);
                  exit(0);
                }
              case 'N':
                if(argc > 2)
                  {
                    ta_number = strtoul(argv[2], 0, 10);
                    argc--;
                    argv++;
                  }
                else
                  {
                    fprintf(stderr, gettext("%s: missing transaction number value for parameter %c\n"), progname, *pos);
                    exit(1);
                  }
                break;

              default:
                fprintf(stderr, gettext("%s: unknown parameter %c\n"), progname, *pos);
                exit(1);
            }
          pos++;
        }
      argv++;
      argc--;
    }

  if (argc > 1)
    {
      for(i=1; i<argc; i++)
        {
          if(rsbac_um_get_gid(ta_number, argv[i], &group))
            {
              group = strtoul(argv[i],0,0);
              if(!group && strcmp(argv[i],"0"))
                {
                  fprintf(stderr, gettext("%s: Unknown group %s\n"), progname, argv[i]);
                  continue;
                }
            }
          process(group);
        }
      exit(0);
    }
  else
    {
      use();
      return 1;
    }
  return (res);
}
