/* -*-objc-*-
 *
 * RSSKit - A RSS and ATOM Reading Kit for GNUstep
 * Copyright (C) 2004-2006 Guenther Noack
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation;
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#import "DOMParser.h"


//#define DEBUG 1

@implementation XMLText

-(NSString*) contentAndNextContents
{
  return [NSString stringWithFormat: @"%@%@",
		   ( (_content==nil) ? @"" : _content ),
		   ( (_next==nil) ? @"" : [_next contentAndNextContents])];
}

-(NSString*) content
{
  return AUTORELEASE(RETAIN(_content));
}

-(id<XMLTextOrNode>) _setNext: (id<XMLTextOrNode>) node
{
  id<XMLTextOrNode> result;
  
  #ifdef DEBUG
  NSLog(@"%@ setting Next to %@", self, node);
  #endif
  
  // Okay, we do some merging...
  // If the next object is a XMLText, too, we merge
  // both by appending their strings.
  // This may slow down the overall parsing process but it's
  // generally a good idea because I had crashes due to a
  // stack overflow when deallocating certain RSS feeds.
  // feeds with many entities, that were.
  if ([[node class] isSubclassOfClass: [XMLText class]]) {
    
    #ifdef DEBUG
    NSLog(@" merging \"%@\", \"%@\"", _content, [node content]);
    #endif
    
    result = self;
    
    // we try to do it with NSMutableStrings where possible.
    if ([[_content class] isSubclassOfClass: [NSMutableString class]]) {
      [((NSMutableString*)_content) appendString: [node content]];
    } else {
      NSMutableString* str;
      str = [[NSMutableString alloc] initWithString: _content];
      [str appendString: [node content]];
      ASSIGN(_content, str);
      RELEASE(str);
    }
    
    // no getter for it. No, i won't write one.
    _next = ((XMLText*)node)->_next;
    
    #ifdef DEBUG
    NSLog(@" merged  \"%@\"", _content);
    #endif
    
    
  } else {
    // the case in which we can't merge
    ASSIGN(_next, node);
    
    result = node;
  }
  
  return result;
}

-(XMLNode*) nextElement
{
  // !!! If you change this, change it in XMLNode, too!
  // XXX: Write a macro
  
  // we only return *XML elements* here, *not* contents!
  if ([[_next class] isSubclassOfClass: [XMLText class]]) {
    return [_next nextElement];
  } else {
    return AUTORELEASE(RETAIN(_next));
  }  
}

/**
 * @deprecated
 * Please don't call init on XMLText objects. It won't work.
 * Instead, use initWithString:
 */
-(id)init
{
  [self release];
  return nil;
}

-(id)initWithString: (NSString*) str
{
  self = [super init];
  
  if (self != nil) {
    ASSIGN(_content, str);
  }
  
  _next = nil;
  
  return self;
}

-(void)dealloc
{
  #ifdef DEBUG
  NSLog(@"XMLText dealloc: start (%@)", _content);
  #endif
  RELEASE(_next);
  RELEASE(_content);
  
  #ifdef DEBUG
  NSLog(@"XMLText dealloc: end");
  #endif
  [super dealloc];
}

@end


@implementation XMLNode

-(XMLNode*) firstChildElement
{
  if (_child == nil)
    return nil;
  
  if ([[_child class] isSubclassOfClass: [XMLNode class]]) {
    return AUTORELEASE(RETAIN(_child));
  } else {
    return [_child nextElement];
  }
}

-(XMLNode*) nextElement
{
  // !!! If you change this, change it in XMLText, too!
  // XXX: Write a macro
  
  // we only return *XML elements* here, *not* contents!
  if ([[_next class] isSubclassOfClass: [XMLText class]]) {
    return [_next nextElement];
  } else {				    
    return AUTORELEASE(RETAIN(_next));
  }
}

-(NSString*) name
{
  return AUTORELEASE(RETAIN(_name));
}

-(NSString*) contentAndNextContents
{
  NSString* result;
  
  // XXX: attributes are still not shown here! Do we need it?
  
  if (_child == nil) {
    result = [NSString stringWithFormat:
			 @"<%@/>%@", _name,
		       (_next==nil?@"":[_next contentAndNextContents])];
  } else {
    result = [NSString stringWithFormat:
			 @"<%@>%@</%@>%@",
		       _name, [_child contentAndNextContents], _name,
		       (_next==nil?@"":[_next contentAndNextContents])];
  }
  
  return result;
}

-(NSString*) content
{
  NSString* result;
    
  // XXX: attributes are still not shown here! Do we need it?
  
  if (_child == nil) {
    result = @"";
  } else {
    NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
    
    result = [_child contentAndNextContents];
    //This needs to be retained here and autoreleased later to
    //prevent it from being deallocated.
    RETAIN(result);
    
    #ifdef DEBUG
    NSLog(@"autorelease will be called");
    #endif
    [pool release];
    #ifdef DEBUG
    NSLog(@"autorelease called");
    #endif
    
    AUTORELEASE(result);
  }
  
  return result;
}

-(NSDictionary*) attributes
{
  return AUTORELEASE(RETAIN(_attributes));
}

-(NSString*) namespace
{
  return AUTORELEASE(RETAIN(_namespace));
}

-(id) initWithName: (NSString*) name
	 namespace: (NSString*) namespace
	attributes: (NSDictionary*) attributes
	    parent: (XMLNode*) parent;
{
  self = [super init];
  
  if (self != nil) {
    _name = RETAIN(name);
    _namespace = RETAIN(namespace);
    _parent = RETAIN(parent);
    _attributes = RETAIN(attributes);
  }
  
  return self;
}

-(void) dealloc
{
  #ifdef DEBUG
  NSLog(@"XMLNode dealloc: start");
  #endif
  RELEASE(_child);
  RELEASE(_next);
  RELEASE(_namespace);
  RELEASE(_name);
  RELEASE(_current);
  RELEASE(_parent);
  [super dealloc];
  #ifdef DEBUG
  NSLog(@"XMLNode dealloc: end");
  #endif
}

- (id<XMLTextOrNode>) _setNext: (id<XMLTextOrNode>) node
{
  #ifdef DEBUG
  NSLog(@"_setNext: %@ --> %@", self, node);
  #endif

  ASSIGN(_next, node);
  
  return node;
}


- (void) appendTextOrNode: (id<XMLTextOrNode>) aThing
	       fromParser: (NSXMLParser*) aParser
{
  #ifdef DEBUG
  NSLog(@"appendTextOrNode: %@ at: %@", aThing, [self name]);
  #endif
  
  if (_child == nil) {
    _child = RETAIN(aThing);
  }
  
  if (_current == nil) {
    _current = RETAIN(aThing);
  } else {
    id<XMLTextOrNode> newOne = [_current _setNext: aThing];
    
    ASSIGN(_current, newOne);
  }
  
  if ([[aThing class] isSubclassOfClass: [XMLNode class]]) {
    [aParser setDelegate: aThing];
  }
}

@end

@implementation XMLNode (NSXMLParserDelegateEventAdditions)
- (void) parser: (NSXMLParser*)aParser
  didEndElement: (NSString*)anElementName
   namespaceURI: (NSString*)aNamespaceURI
  qualifiedName: (NSString*)aQualifierName
{
  #ifdef DEBUG
  NSLog(@"closing XML node %@", anElementName);
  #endif
  
  if (![anElementName isEqualToString: _name]) {
    NSLog(@"badly nested XML elements!");
  }
  
  if (_parent != nil) {
    [aParser setDelegate: _parent];
    RELEASE(_parent);
    _parent = nil;
  }
}

- (void) parser: (NSXMLParser*)aParser
didStartElement: (NSString*)anElementName
   namespaceURI: (NSString*)aNamespaceURI
  qualifiedName: (NSString*)aQualifierName
     attributes: (NSDictionary*)anAttributeDict
{
  XMLNode* item;
  
  item = [[XMLNode alloc]
	   initWithName: anElementName
	   namespace: aNamespaceURI
	   attributes: anAttributeDict
	   parent: self ];
  
  #ifdef DEBUG
  NSLog(@"starting XML node %@", anElementName);
  #endif
  
  [self appendTextOrNode: item
	fromParser: aParser];
  
  RELEASE(item);
}

- (void)    parser: (NSXMLParser*)aParser
 parseErrorOccured: (NSError*)parseError
{
  NSLog(@"XML-DOM Parser: %@ at line %@, col %@",
	[parseError localizedDescription],
	[aParser lineNumber], [aParser columnNumber]);
}

- (void) parser: (NSXMLParser*)aParser
foundCharacters: (NSString*)aString
{
  XMLText* text;
  text = [[XMLText alloc] initWithString: aString];
  
  [self appendTextOrNode: text
	fromParser: aParser];
  
  RELEASE(text);
}

@end
