/***************************************************************************
 * RT2400 SourceForge Project - http://rt2400.sourceforge.net              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 *                                                                         *
 *   Licensed under the GNU GPL                                            *
 *   Original code supplied under license from RaLink Inc, 2003.           *
 ***************************************************************************/

 /***************************************************************************
 *      Module Name: rtmp.h
 *
 *      Abstract: Miniport generic portion header file
 *
 *      Revision History:
 *      Who             When            What
 *      --------        -----------     -----------------------------
 *      PaulL           1st  Oct 02     Created
 *      MarkW           9th  Feb 04     Baseline of code
 *      PaulL           5th  May 04     Initial Promisc code changes (off)
 *      JerzyK          1st  May 04     Fixes to interrupt handler call
 *      fstanchina      30th May 04     Made interrupt fix 2.4 okay
 *      Ivo 		13th Jul 04	Add registry busy check
 *      MarkW           24th Aug 04     Fixed WRITE call in registry busy
 ***************************************************************************/


#ifndef __RTMP_H__
#define __RTMP_H__

#include "mlme.h"

#ifndef IRQ_HANDLED
/* For 2.6.x compatibility */
typedef void irqreturn_t;
#define IRQ_NONE
#define IRQ_HANDLED
#define IRQ_RETVAL(x)
#endif

#ifndef HAVE_NETDEV_PRIV		/* netdev_priv() */
#define netdev_priv(_dev)	( _dev->priv )
#endif

//
//  MACRO for debugging information
//
extern int    debug;
#ifdef RT2400_DBG
#define DBGPRINT(Level, fmt, args...) 					\
	if(debug){printk(Level DRV_NAME ": " fmt, ## args);}
#else
#define DBGPRINT(Level, fmt, args...)					\
	while(0){}
#endif

//  Assert MACRO to make sure program running
//
#undef  ASSERT
#define ASSERT(x)                                                                           \
{                                                                                           \
    if (!(x))                                                                               \
    {                                                                                       \
        DBGPRINT(RT_DEBUG_ERROR, "Assertion failed: %s:%d %s\n", __FILE__, __LINE__, #x);	\
    }                                                                                       \
}

//
//  Macro for alignment
//
#define ALIGNED(x) __attribute__((__aligned__(x)))

//
//  Macros for flag and ref count operations
//
#define RTMP_SET_FLAG(_M, _F)       ((_M)->Flags |= (_F))
#define RTMP_CLEAR_FLAG(_M, _F)     ((_M)->Flags &= ~(_F))
#define RTMP_CLEAR_FLAGS(_M)        ((_M)->Flags = 0)
#define RTMP_TEST_FLAG(_M, _F)      (((_M)->Flags & (_F)) != 0)
#define RTMP_TEST_FLAGS(_M, _F)     (((_M)->Flags & (_F)) == (_F))

#define RTMP_GET_REF(_A)            ((_A)->RefCount)

#define RTMP_INC_RCV_REF(_A)        ((_A)->RcvRefCount++)
#define RTMP_DEC_RCV_REF(_A)        ((_A)->RcvRefCount--)
#define RTMP_GET_RCV_REF(_A)        ((_A)->RcvRefCount)

#define RTMP_INC_SEND_REF(_A)       ((_A)->SendRefCount++)
#define RTMP_DEC_SEND_REF(_A)       ((_A)->SendRefCount--)
#define RTMP_GET_SEND_REF(_A)       ((_A)->SendRefCount)

//
// MACRO for 32-bit PCI register read / write
//
// Usage : RTMP_IO_READ32(
//              PRTMP_ADAPTER pAd,
//              ULONG Register_Offset,
//              PULONG  pValue)
//
//         RTMP_IO_WRITE32(
//              PRTMP_ADAPTER pAd,
//              ULONG Register_Offset,
//              ULONG Value)
//
#define RTMP_IO_READ32(_A, _R, _pV)	(*_pV = readl( (void*) (_A->CSRBaseAddress + _R) ) )
#define RTMP_IO_WRITE32(_A, _R, _V)	(writel(_V, (void*) (_A->CSRBaseAddress + _R) ) )

//
// BBP & RF are using indirect access. Before write any value into it.
// We have to make sure there is no outstanding command pending via checking busy bit.
//
#define	MAX_BUSY_COUNT	10			// Nunber of retry before failing access BBP & RF indirect register
//
#define	RTMP_BBP_IO_WRITE32(_A, _V)						\
{										\
	BBPCSR_STRUC	Value;							\
	ULONG			BusyCnt = 0;					\
	do {									\
		RTMP_IO_READ32(_A, BBPCSR, &Value.word);			\
		if (Value.field.Busy == IDLE){					\
			RTMP_IO_WRITE32(_A, BBPCSR, _V);			\
			break;							\
		}								\
		BusyCnt++;							\
	}	while (BusyCnt < MAX_BUSY_COUNT);				\
	if (BusyCnt == MAX_BUSY_COUNT){						\
		DBGPRINT(RT_DEBUG_ERROR, "Error - BBPCSR register busy\n");	\
		RTMP_IO_WRITE32(_A, BBPCSR, _V);				\
	}									\
}
//
#define	RTMP_RF_IO_WRITE32(_A, _V)						\
{										\
	RFCSR_STRUC		Value;						\
	ULONG			BusyCnt = 0;					\
	do {									\
		RTMP_IO_READ32(_A, RFCSR, &(Value.word));			\
		if (Value.field.Busy == IDLE)					\
		{								\
			RTMP_IO_WRITE32(_A, RFCSR, _V);				\
			break;							\
		}								\
		BusyCnt++;							\
	}	while (BusyCnt < MAX_BUSY_COUNT);				\
	if (BusyCnt == MAX_BUSY_COUNT){						\
		DBGPRINT(RT_DEBUG_ERROR, "Error - RFCSR register busy\n");	\
		RTMP_IO_WRITE32(_A, RFCSR, _V);					\
	}									\
}
//
#define	RTMP_BBP_IO_READ32(_A, _pV)						\
{										\
	BBPCSR_STRUC	Value;							\
	ULONG			BusyCnt = 0;					\
	RTMP_BBP_IO_WRITE32(_A, *(_pV));					\
	do {									\
		RTMP_IO_READ32(_A, BBPCSR, &Value.word);			\
		if (Value.field.Busy == IDLE){					\
			*(_pV) = Value.field.Value;				\
			break;							\
		}								\
		BusyCnt++;							\
	}	while (BusyCnt < MAX_BUSY_COUNT);				\
	if (BusyCnt == MAX_BUSY_COUNT){						\
		DBGPRINT(RT_DEBUG_ERROR, "Error - BBPCSR register busy\n");	\
		*(_pV) = 0xff;							\
	}									\
}
// Read BBP register by register's ID
#define	RTMP_BBP_IO_READ32_BY_REG_ID(_A, _I, _pV)	\
{												\
	BBPCSR_STRUC	BbpCsr;						\
	BbpCsr.word = 0;								\
	BbpCsr.field.WriteControl = 0;				\
	BbpCsr.field.Busy = 1;						\
	BbpCsr.field.RegNum = _I;					\
	RTMP_BBP_IO_READ32(_A, &BbpCsr.word);		\
	*(_pV) = (UCHAR) BbpCsr.field.Value;			\
}
// Write BBP register by register's ID & value
#define	RTMP_BBP_IO_WRITE32_BY_REG_ID(_A, _I, _V)	\
{												\
	BBPCSR_STRUC	BbpCsr;						\
	BbpCsr.word = 0;								\
    BbpCsr.field.WriteControl = 1;				\
    BbpCsr.field.Busy = 1;						\
    BbpCsr.field.Value = _V;					\
    BbpCsr.field.RegNum = _I;					\
    RTMP_BBP_IO_WRITE32(_A, BbpCsr.word);		\
}

// Acquire resource lock with return status when error occurred
#define	RTMP_LOCK_WITH_RETURN(_A, _pL, _F, _S)	\
{												\
	spin_lock(_pL);					\
	if (RTMP_TEST_FLAG(_A, _F))					\
	{											\
		spin_unlock(_pL);				\
		return _S;								\
	}											\
	else										\
	{											\
		RTMP_SET_FLAG(_A, _F);					\
	}											\
	spin_unlock(_pL);					\
}												\
// Acquire resource lock without return status when error occurred
#define	RTMP_LOCK(_A, _pL, _F)		\
{									\
	spin_lock(_pL);		\
	if (RTMP_TEST_FLAG(_A, _F))		\
	{								\
		spin_unlock(_pL);	\
		return;						\
	}								\
	else							\
	{								\
		RTMP_SET_FLAG(_A, _F);		\
	}								\
	spin_unlock(_pL);		\
}									\
// Release resource lock
#define	RTMP_UNLOCK(_A, _pL, _F)	\
{									\
	spin_lock(_pL);		\
	RTMP_CLEAR_FLAG(_A, _F);		\
	spin_unlock(_pL);		\
}									\

//
//  Some utility macros
//
#ifndef min
#define min(_a, _b)     (((_a) < (_b)) ? (_a) : (_b))
#endif

#ifndef max
#define max(_a, _b)     (((_a) > (_b)) ? (_a) : (_b))
#endif

#define	INFRA_ON(_p)		        (((_p)->PortCfg.Massoc) == TRUE)	// Check Massoc
#define ADHOC_ON(_p)                (((_p)->PortCfg.Mibss) == TRUE)    // check Mibss

// This control buffer owned by you as long as you have the skb queue.
#define	RTMP_SET_PACKET_FRAGMENTS(_p, number)	((_p)->cb[10] = number)
#define RTMP_GET_PACKET_FRAGMENTS(_p)			((_p)->cb[10])
#define	RTMP_SET_PACKET_RTS(_p, number)			((_p)->cb[11] = number)
#define RTMP_GET_PACKET_RTS(_p)					((_p)->cb[11])
//#define	RTMP_GET_PACKET_MR(_p)					(&(_p)->MiniportReserved[0])

#define	MAKE_802_3_HEADER(_p, _pMac1, _pMac2, _pType)								\
do {																				\
	memcpy((_p),                    (_pMac1), MAC_ADDR_LEN);						\
	memcpy((_p) + MAC_ADDR_LEN,     (_pMac2), MAC_ADDR_LEN);						\
	memcpy((_p) + MAC_ADDR_LEN * 2, (_pType), LENGTH_802_3_TYPE);					\
} while (0)

//
// Register set pair for initialzation register set definition
//
typedef	struct	_RTMP_REG_PAIR
{
	ULONG	Register;
	ULONG	Value;
}	RTMP_REG_PAIR, *PRTMP_REG_PAIR;

//
//  Statistic counter structure
//
typedef struct _COUNTER_802_3
{
    // General Stats
	ULONG		GoodTransmits;
	ULONG		GoodReceives;
	ULONG		TxErrors;
	ULONG		RxErrors;
	ULONG		RxNoBuffer;

    // Ethernet Stats
	ULONG		RcvAlignmentErrors;
	ULONG		OneCollision;
	ULONG		MoreCollisions;

}   COUNTER_802_3, *PCOUNTER_802_3;

typedef struct _COUNTER_802_11 {
    LARGE_INTEGER   TransmittedFragmentCount;
    LARGE_INTEGER   MulticastTransmittedFrameCount;
    LARGE_INTEGER   FailedCount;
    LARGE_INTEGER   RetryCount;
    LARGE_INTEGER   MultipleRetryCount;
    LARGE_INTEGER   RTSSuccessCount;
    LARGE_INTEGER   RTSFailureCount;
    LARGE_INTEGER   ACKFailureCount;
    LARGE_INTEGER   FrameDuplicateCount;
    LARGE_INTEGER   ReceivedFragmentCount;
    LARGE_INTEGER   MulticastReceivedFrameCount;
    LARGE_INTEGER   FCSErrorCount;
} COUNTER_802_11, *PCOUNTER_802_11;

typedef struct _COUNTER_RALINK {
    ULONG           TransmittedByteCount;   // both successful and failure, used to calculate TX throughput
    ULONG           ReceivedByteCount;      // both CRC okay and CRC error, used to calculate RX throughput
    ULONG           BeenDisassociatedCount;
    ULONG           BadCQIAutoRecoveryCount;
    ULONG           PoorCQIRoamingCount;
    ULONG           MgmtRingFullCount;
} COUNTER_RALINK, *PCOUNTER_RALINK;

typedef enum _NDIS_MEDIA_STATE
{
	NdisMediaStateConnected,
	NdisMediaStateDisconnected
} NDIS_MEDIA_STATE, *PNDIS_MEDIA_STATE;

typedef enum {PHY_11A, PHY_11B, PHY_11G, PHY_11_TYPE_MAX} PHYMODE;

//
//  Arcfour Structure Added by PaulWu
//
typedef struct  _ARCFOUR
{
	UINT			X;
	UINT			Y;
	UCHAR			STATE[256];
} ALIGNED(4) ARCFOURCONTEXT, *PARCFOURCONTEXT;

// Shared key data structure
typedef	struct	_WEP_KEY {
	UCHAR	KeyLen;						// Key length for each key, 0: entry is invalid
	UCHAR	Key[MAX_LEN_OF_KEY];		// right now we implement 4 keys, 128 bits max
} ALIGNED(4) WEP_KEY, *PWEP_KEY;

// configuration to be used when this STA starts a new ADHOC network
typedef struct _IBSS_CONFIG {
    USHORT    BeaconPeriod;
    USHORT    AtimWin;
    UCHAR     Channel;
    UCHAR     SupportedRates[MAX_LEN_OF_SUPPORTED_RATES];    // Supported rates
    UCHAR     SupportedRatesLen;
}   IBSS_CONFIG, *PIBSS_CONFIG;

typedef struct _LED_CONTROL {
    BOOLEAN				fOdd;
    BOOLEAN				fRxActivity;
    struct timer_list	BlinkTimer; // 50 ms periodic timer
    ULONG				LastLedCsr;
}   LED_CONTROL;

// structure to tune BBP R13 "RX AGC VGC init"
typedef struct _RT_802_11_RX_AGC_VGC_TUNING {
    UCHAR   Duration;           // unit: sec
    UCHAR   FlaseCcaThreshold;  // delta of flase CCA happen in the latest "Duration"
    UCHAR   RxAgcVgcDelta;      // R13 += RxAgcVgcDelta whenever flase CCA >= threshold
    UCHAR   MaxRxAgcVgc;        // max value of R13
} RT_802_11_RX_AGC_VGC_TUNING, *PRT_802_11_RX_AGC_VGC_TUNING;

//  configuration initial parametes.
typedef struct _RTMP_PROFILE_SETTING {
	UCHAR		Profile[32+1];
	UCHAR		SSID[32+1];
	UINT		SsidLen;
	UINT		Channel;
	UINT		Authentication;
	UINT		Encryption;
	UINT		NetworkType;
	UINT		KeyIndex;
	UINT		KeyLength;
	UINT		KeyType;
	UCHAR		Key1[26+1];
	UCHAR		Key2[26+1];
	UCHAR		Key3[26+1];
	UCHAR		Key4[26+1];
	UINT		TransRate;
	UINT		TransPower;
	UINT		RTS;
	UINT		RTSCheck;
	UINT     	FragmentCheck;
	UINT		PSmode;
	UINT		PreamType;
	UINT		AntennaRx;
	UINT		AntennaTx;
	UINT		Fragment;
	UCHAR		CountryRegion;
	struct  _RT_PROFILE_SETTING		*Next;
} RTMP_PROFILE_SETTING, *PRTMP_PROFILE_SETTING;

// PortConfig
typedef struct _PORT_CONFIG {
    // MIB:ieee802dot11.dot11smt(1).dot11StationConfigTable(1)
    ULONG     AuthRspTimeout;       // in units of TU
    USHORT    CapabilityInfo;
    USHORT    Psm;                  // power management mode   (PWR_ACTIVE|PWR_SAVE)
    USHORT    BeaconPeriod;         // in units of TU
    USHORT    DtimPeriod;           // number of BEACONS in between DTIMs
    ULONG     AssocRspTimeout;      // in units of TU

    USHORT    CfpMaxDuration;
    USHORT    CfpDurRemain;
    USHORT    CfpCount;
    USHORT    CfpPeriod;

    USHORT    DisassocReason;
    MACADDR   DisassocSta;
    USHORT    DeauthReason;
    MACADDR   DeauthSta;
    USHORT    AuthFailReason;
    MACADDR   AuthFailSta;

    // MIB:ieee802dot11.dot11smt(1).dot11AuthenticationAlgorithmsTable(2)
    UCHAR     AuthAlgorithm;     // 0-bit:AUTH_OPEN, 1-bit:AUTH_SHARED_KEY (ON/OFF)

    // MIB:ieee802dot11.dot11smt(1).dot11WEPDefaultKeysTable(3)
    WEP_KEY	  SharedKey[SHARE_KEY_NO];
	WEP_KEY	  PeerKey[PEER_KEY_NO];

    // MIB:ieee802dot11.dot11smt(1).dot11WEPKeyMappingsTable(4)
    // not implemented yet

    // MIB:ieee802dot11.dot11smt(1).dot11PrivacyTable(5)
    // 2 counters dot11WEPICVErrorCount and dot11WEPExcludedCount put in g_portsts
    BOOLEAN   PrivacyInvoked;
    UCHAR     DefaultKeyId;
    BOOLEAN   ExcludeUnencrypt;

    // MIB:ieee802dot11.dot11mac(2).dot11OperationTable(1)
    //MACADDR   StaAddr;            // my MAC address
    USHORT    RtsThreshold;       // in units of BYTE
    UCHAR     ShortRetryLimit;    // STA short retry count
    UCHAR     LongRetryLimit;     // STA long retry count
    USHORT    FragmentThreshold;
	BOOLEAN   bFragmentZeroDisable;		// Microsoft use 0 as disable

    // MIB:ieee802dot11.dot11mac(2).dot11CountersTable(2)
    // all statistics counters put in g_portsts

    // MIB:ieee802dot11.dot11mac(2).dot11GroupAddressesTable(3)
    // not implemented yet

    // MIB:ieee802dot11.dot11res(3).dot11resAttribute(1)
    // not implemented yet

    // MIB:ieee802dot11.dot11phy(4).dot11PhyOperationTable(1)
    UCHAR     RadioType;          // always PHYTYPE_DSSS(2)
    UCHAR     TempType;          // TEMPTYPE_0_40(1) or TEMPTYPE_n30_70(2)

    // MIB:ieee802dot11.dot11phy(4).dot11PhyAntennaTable(2)
    UCHAR     CurrentTxAntenna;
    UCHAR     CurrentRxAntenna;
    UCHAR     NumberOfAntenna;

    // MIB:ieee802dot11.dot11phy(4).dot11PhyTxPowerTable(3)
    UCHAR    TxPowerLevel[MAX_NUM_OF_POWER_LEVEL];
    UCHAR     NumOfTxPowerLevel;
    UCHAR     CurrentTxPowerLevelIndex; //default&other value=MaxPower,1=MinPower,2=1*MaxPower/4,3=2*MaxPower/4,4=3*MaxPower/4,
    UCHAR     TxPower;
    UCHAR     TxRate;       // RATE_1, RATE_2, RATE_5_5, RATE_11
    UCHAR     TxRateFixed;

    // MIB:ieee802dot11.dot11phy(4).dot11PhyFHSSTable(4)
    // we don't support FHSS

    // MIB:ieee802dot11.dot11phy(4).dot11PhyDSSSTable(5)
    UCHAR     Channel;        // current (I)BSS channel used in the station
    UCHAR     CountryRegion;	// Enum of country region, 0:FCC, 1:IC, 2:ETSI, 3:SPAIN, 4:France, 5:MKK, 6:MKK1, 7:Israel

    // MIB:ieee802dot11.dot11phy(4).dot11PhyIRTable(6)
    // we don't support InfraRed PHY

    // MIB:ieee802dot11.dot11phy(4).dot11RegisteredDoaminSupportedTable(7)
    // no longer supported, since it was never used

    // MIB:ieee802dot11.dot11phy(4).dot11AntennasListTable(8)
    // no longer supported, since it was never used

    // MIB:ieee802dot11.dot11phy(4).dot11SupportedDataRatesTxTable(9)
    // MIB:ieee802dot11.dot11phy(4).dot11SupportedDataRatesRxTable(10)
    UCHAR     SupportedRates[MAX_LEN_OF_SUPPORTED_RATES];    // Supported rates
    UCHAR     SupportedRatesLen;

    //
    // other parameters not defined in standard MIB
    //
    UCHAR     DesiredRates[MAX_LEN_OF_SUPPORTED_RATES];      // OID_802_11_DESIRED_RATES
    USHORT    CwMin;
    USHORT    CwMax;
    USHORT    RecvDtim;
    UCHAR     DtimCount;
    MACADDR   Bssid;
    MACADDR   Broadcast;            // FF:FF:FF:FF:FF:FF
    USHORT    Pss;                  // current power saving status (PWR_SAVE|PWR_ACTIVE)
    UCHAR     RssiTrigger;
    UCHAR     RssiTriggerMode;      // RSSI_TRIGGERED_UPON_BELOW_THRESHOLD or RSSI_TRIGGERED_UPON_EXCCEED_THRESHOLD
    UCHAR     LastRssi;             // last received BEACON's RSSI
    UCHAR     AvgRssi;              // average of last 2 BEACON's RSSI
    USHORT    AtimWin;              // in kusec; IBSS parameter set element
    USHORT    Aid;                  // association ID
    UCHAR     MaxBasicRate;         // RATE_1, RATE_2, RATE_5_5, RATE_11
    UCHAR     MaxTxRate;            // RATE_1, RATE_2, RATE_5_5, RATE_11
    USHORT    DefaultListenCount;   // default listen count;
    USHORT    CurrListenCount;      // in units of Beacon interval
    UINT      MinChnlTime;          // minimum channel time (in uSEC)
    UINT      MaxChnlTime;          // maximum channel time (in uSEC)
    UCHAR     BssType;              // BSS_INFRA or BSS_INDEP

    UCHAR     SsidLen;               // the actual ssid length in used
    CHAR      Ssid[MAX_LEN_OF_SSID]; // NOT NULL-terminated

    BSS_TABLE BssTab;     // BSS Table
    MAC_TABLE MacTab;     // MAC Table

    // global variables mXXXX used in MAC protocol state machines
    BOOLEAN   Mibss;
    BOOLEAN   Massoc;
    BOOLEAN   Mauth;

    // PHY specification
    PHYMODE   PhyMode;    // PHY_11A, PHY_11G, PHY_11B
    UCHAR     Drxtx;      // RF RX to TX turn around time, in units of usec
    UCHAR     Dswchnl;    // in units of usec, time to switch to a new channel
    UCHAR     Dpreamble;  // in units of usec
    UCHAR     Dplcp;      // in units of usec
    UCHAR     Dslot;      // in units of usec
    UCHAR     Dsifs;      // in units of usec
    UCHAR     Dpifs;      // in units of usec
    UCHAR     Ddifs;      // in units of usec
    UCHAR     Deifs;      // in units of usec, change when BasicRateSet changes
    USHORT    TxPreamble; // Rt802_11PreambleLong, Rt802_11PreambleShort

    BOOLEAN   AutoReconnect;			// Set to TRUE when setting OID_802_11_SSID with no matching BSSID
    ULONG     PacketFilter;				// Packet filter for receiving
    ULONG     WindowsPowerMode;
    ULONG     WindowsBatteryPowerMode;	// Power mode for battery if exists
    BOOLEAN   WindowsACCAMEnable;		// Enable CAM power mode when AC on

    ULONG     WindowsTxPreamble; // Rt802_11PreambleLong, Rt802_11PreambleShort, Rt802_11PreambleAuto

    UCHAR	ChannelTxPower[MAX_LEN_OF_CHANNELS];		// Store Tx power value for all channels.

    // configuration to be used when this STA starts a new ADHOC network
    IBSS_CONFIG IbssConfig;

    ULONG     LastBeaconRxTime; // OS's timestamp of the last BEACON RX time
    ULONG     LastScanTime;		// Record last scan time for issue BSSID_SCAN_LIST
    ULONG     IgnoredScanNumber;	// Ignored BSSID_SCAN_LIST requests
	BOOLEAN	  bSwRadio;			// Software controlled Radio On/Off, TRUE: On
	BOOLEAN	  bHwRadio;			// Hardware controlled Radio On/Off, TRUE: On
    BOOLEAN   bRadio;		        // Radio On/Off, TRUE: On
	BOOLEAN   bHardwareRadio;	// Hardware controlled Radio enabled


    LED_CONTROL LedCntl;
    UCHAR       RfType;         // 0-2420, 1:2421
    UCHAR       LedMode;        // 0-default, 1 TX/RX activity mode
    struct timer_list		RfTuningTimer;
    UCHAR                    RfTuningStep;
    UCHAR                    RfTuningChannel;

    // the following fields are for dynamic BBP "R13: RX AGC VGC INIT" control
    RT_802_11_RX_AGC_VGC_TUNING  RxVgc;
    USHORT                       BbpFalseCca;
    BOOLEAN                      EnableRxAgcVgcTuning;
//  UCHAR                        CurrentRxAgcVgc;

} PORT_CONFIG, *PPORT_CONFIG;

typedef struct _MLME_STRUCT {
    STATE_MACHINE           CntlMachine, AssocMachine, AuthMachine, AuthRspMachine, SyncMachine;
    STATE_MACHINE_FUNC      CntlFunc[CNTL_FUNC_SIZE], AssocFunc[ASSOC_FUNC_SIZE];
    STATE_MACHINE_FUNC      AuthFunc[AUTH_FUNC_SIZE], AuthRspFunc[AUTH_RSP_FUNC_SIZE];
    STATE_MACHINE_FUNC      SyncFunc[SYNC_FUNC_SIZE];

    ASSOC_AUX               AssocAux;
    AUTH_AUX                AuthAux;
    AUTH_RSP_AUX            AuthRspAux;
    SYNC_AUX                SyncAux;
    CNTL_AUX                CntlAux;

    COUNTER_802_11          PrevWlanCounters;
    ULONG                   RoamCqi;  // 0..100, Channel Quality Indication for Roaming
    ULONG                   DrsCqi;   // 0..100, Channel Quality Indication for Dynamic Rate Switching
    ULONG                   TxPER;    // 0..100 %
    ULONG                   RxPER;    // 0..100 %

    BOOLEAN                 Running;
    spinlock_t				TaskLock;
    MLME_QUEUE              Queue;

    UINT                    ShiftReg;
    PSPOLL_FRAME            PsFr;
    MACHDR                  NullFr;

    struct timer_list		PeriodicTimer;
    ULONG                   PeriodicRound;
    ULONG                   PrevTxCnt;
} MLME_STRUCT, *PMLME_STRUCT;

//
// Management ring buffer format
//
typedef	struct	_MGMT_STRUC	{
	BOOLEAN		Valid;
	UCHAR		*pBuffer;
	ULONG		Length;
}	MGMT_STRUC, *PMGMT_STRUC;

//
// P802.11 Frame control field, 16 bit
//
typedef	struct	_FRAME_CONTROL	{
	USHORT		Ver:2;				// Protocol version
	USHORT		Type:2;				// MSDU type
	USHORT		Subtype:4;			// MSDU subtype
	USHORT		ToDs:1;				// To DS indication
	USHORT		FrDs:1;				// From DS indication
	USHORT		MoreFrag:1;			// More fragment bit
	USHORT		Retry:1;			// Retry status bit
	USHORT		PwrMgt:1;			// Power management bit
	USHORT		MoreData:1;			// More data bit
	USHORT		Wep:1;				// Wep data
	USHORT		Order:1;			// Strict order expected
}	FRAME_CONTROL, *PFRAME_CONTROL;

//
// P802.11 intermediate header format
//
typedef	struct	_CONTROL_HEADER	{
	FRAME_CONTROL	Frame;				// Frame control structure
	USHORT			Duration;			// Duration value
	MACADDR			Addr1;				// Address 1 field
	MACADDR			Addr2;				// Address 2 field
}	CONTROL_HEADER, *PCONTROL_HEADER;

//
// P802.11 header format
//
typedef	struct	_HEADER_802_11	{
	CONTROL_HEADER	Controlhead;
	MACADDR			Addr3;				// Address 3 field
	USHORT			Frag:4;				// Fragment number
	USHORT			Sequence:12;		// Sequence number
}	HEADER_802_11, *PHEADER_802_11;

//
// Receive Tuple Cache Format
//
typedef	struct	_TUPLE_CACHE	{
	BOOLEAN			Valid;
	MACADDR			MAC;
	USHORT			Sequence;
	USHORT			Frag;
}	TUPLE_CACHE, *PTUPLE_CACHE;

//
// Fragment Frame structure
//
typedef	struct	_FRAGMENT_FRAME	{
	UCHAR		Header802_3[14];
	UCHAR		Buffer[MAX_FRAME_SIZE];
	ULONG		RxSize;
	USHORT		Sequence;
	USHORT		LastFrag;
}	FRAGMENT_FRAME, *PFRAGMENT_FRAME;

//
// Private / Misc data, counters for driver internal use
//
typedef	struct	__PRIVATE_STRUC	{
	ULONG		SystemResetCnt;			// System reset counter
	ULONG		TxRingFullCnt;			// Tx ring full occurrance number
	ULONG		ResetCountDown;			// Count down before issue reset, patch for RT2430
	ULONG		CCAErrCnt;				// CCA error count, for debug purpose, might move to global counter
	ULONG		PhyRxErrCnt;			// PHY Rx error count, for debug purpose, might move to global counter
	ULONG		PhyTxErrCnt;			// PHY Tx error count, for debug purpose, might move to global counter
	// Variables for WEP encryption / decryption in rtmp_wep.c
	ULONG			FCSCRC32;
	ARCFOURCONTEXT	WEPCONTEXT;
}	PRIVATE_STRUC, *PPRIVATE_STRUC;

//
//	All DMA ring formats
//
struct	ring_desc	{
	// Descriptor size & dma address
	u32			size;
	void		*va_addr;
	dma_addr_t	pa_addr;
	// Dma buffer size and address for real transfer
	u32			data_size;
	void		*va_data_addr;
	dma_addr_t	pa_data_addr;
    UCHAR		FrameType;			// Type of frame in ring buffer
};

//
//  The miniport adapter structure
//
typedef struct _RTMP_ADAPTER
{
	unsigned long			CSRBaseAddress;     // PCI MMIO Base Address, all access will use
                                                // NdisReadRegisterXx or NdisWriteRegisterXx

    // configuration
    UCHAR                   PermanentAddress[MAC_ADDR_LEN];    // Factory default MAC address
    UCHAR                   CurrentAddress[MAC_ADDR_LEN];      // User changed MAC address

    UCHAR					EEPROMAddressNum;		// 93c46=6	93c66=8
    USHORT					EEPROMDefaultValue[NUM_EEPROM_BBP_PARMS];

    // resource for DMA operation
    struct ring_desc		TxRing[TX_RING_SIZE];       // Tx Ring
    struct ring_desc		AtimRing[ATIM_RING_SIZE];   // Atim Ring
    struct ring_desc		PrioRing[PRIO_RING_SIZE];   // Priority Ring
    struct ring_desc		RxRing[RX_RING_SIZE];       // Rx Ring
    struct ring_desc		BeaconRing;       			// Beacon Ring, only one

	MGMT_STRUC				MgmtRing[MGMT_RING_SIZE];	// management ring size

	ULONG					CurRxIndex;					// Next RxD read pointer
	ULONG					CurTxIndex;					// Next TxD write pointer
	ULONG					CurAtimIndex;				// Next AtimD write pointer
	ULONG					CurPrioIndex;				// Next PrioD write pointer
	ULONG					PushMgmtIndex;				// Next SW management ring index
	ULONG					PopMgmtIndex;				// Next SW management ring index
	ULONG					NextTxDoneIndex;
	ULONG					NextAtimDoneIndex;
	ULONG					NextPrioDoneIndex;
	ULONG					NextRxDoneIndex;
	ULONG					MgmtQueueSize;				// Number of Mgmt request stored in MgmtRing

 	// 802.3 multicast support
	ULONG					NumberOfMcAddresses;		// Number of mcast entry exists
	UCHAR					McastTable[MAX_MCAST_LIST_SIZE][MAC_ADDR_LEN];		// Mcast list
   //flags
    ULONG                   Flags;                      // Represent current device status

    // SEND queue list
    struct sk_buff_head     SendTxWaitQueue;

	USHORT					Sequence;					// Current sequence number

	TUPLE_CACHE				TupleCache[MAX_CLIENT];		// Maximum number of tuple caches, only useful in Ad-Hoc
	FRAGMENT_FRAME			FragFrame;					// Frame storage for fragment frame

    // Counters for 802.3 & generic.
    // Add 802.11 specific counters later
    COUNTER_802_3           Counters;                   // 802.3 counters
    COUNTER_802_11          WlanCounters;				// 802.11 MIB counters
    COUNTER_RALINK          RalinkCounters;				// Ralink propriety counters

	NDIS_MEDIA_STATE        MediaState;

	PRIVATE_STRUC			PrivateInfo;				// Private information & counters

    // SpinLocks
	spinlock_t				TxRingLock;					// Tx Ring spinlock
	spinlock_t				PrioRingLock;				// Prio Ring spinlock
	spinlock_t				AtimRingLock;				// Atim Ring spinlock
	spinlock_t				RxRingLock;					// Rx Ring spinlock
	spinlock_t				SendTxWaitQueueLock;		// SendTxWaitQueue spinlock

// Boolean control for packet filter
	BOOLEAN					bAcceptDirect;
	BOOLEAN					bAcceptMulticast;
	BOOLEAN					bAcceptBroadcast;
	BOOLEAN					bAcceptAllMulticast;
	BOOLEAN					bAcceptPromiscuous;

	BOOLEAN					bRaConfig;

    PORT_CONFIG             PortCfg;
    MLME_STRUCT             Mlme;
    UCHAR					SsidLen;                // the actual ssid length in used
    CHAR					Ssid[MAX_LEN_OF_SSID];  // NOT NULL-terminated

    struct pci_dev			*pPci_Dev;
    struct net_device		*net_dev;

    struct timer_list		timer;	// Periodic Media monitoring timer.

#if WIRELESS_EXT >= 12
	struct iw_statistics iw_stats;
#endif
	struct net_device_stats stats;

	char nickn[IW_ESSID_MAX_SIZE+1];      /* nickname, only used in the iwconfig i/f */
	/* Frequently used values: keep some adjacent for cache effect. */
	int chip_id;
}   RTMP_ADAPTER, *PRTMP_ADAPTER;

//
// Enable & Disable NIC interrupt via writing interrupt mask register
// Since it use ADAPTER structure, it have to be put after structure definition.
//

static inline	VOID    NICDisableInterrupt(
    IN  PRTMP_ADAPTER   pAd)
{
    RTMP_IO_WRITE32(pAd, CSR8, 0xFFFF);
    RTMP_CLEAR_FLAG(pAd, fRTMP_ADAPTER_INTERRUPT_ACTIVE);
}

static inline	VOID    NICEnableInterrupt(
    IN  PRTMP_ADAPTER   pAd)
{
	// 0xFF37 : Txdone & Rxdone, 0xFF07: Txdonw, Rxdone, PrioDone, AtimDone,
    RTMP_IO_WRITE32(pAd, CSR8, 0xFF04);
    RTMP_SET_FLAG(pAd, fRTMP_ADAPTER_INTERRUPT_ACTIVE);
}

NDIS_STATUS	NICReadAdapterInfo(
	IN	PRTMP_ADAPTER		pAd);

VOID	NICIssueReset(
	IN	PRTMP_ADAPTER	pAd);

VOID	RTMPRingCleanUp(
	IN	PRTMP_ADAPTER	pAd,
	IN	UCHAR			RingType);

VOID	NICResetFromError(
	IN	PRTMP_ADAPTER	pAd);

BOOLEAN	NICCheckForHang(
	IN	PRTMP_ADAPTER	pAd);


INT		RT2400_close(
	IN	struct net_device *net_dev);

VOID	RT2400_timer(
	IN	unsigned long data);

INT		RT2400_open(
	IN	struct net_device *net_dev);

INT		RT2400_probe(
	IN	struct pci_dev				*pPci_Dev,
	IN	const struct pci_device_id	*ent);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,19)
irqreturn_t RTMPIsr(
       int             irq,
       void            *dev_instance,
       struct pt_regs  *rgs);
#else
irqreturn_t RTMPIsr(
       int             irq,
       void            *dev_instance);
#endif

#if WIRELESS_EXT >= 12
struct iw_statistics *RT2400_get_wireless_stats(
	IN	struct net_device *net_dev);
#endif

struct net_device_stats *RT2400_get_ether_stats(
	IN	struct net_device *net_dev);

VOID	RT2400_set_rx_mode(
	IN	struct net_device *net_dev);

INT	RTMPSendPackets(
	IN	struct sk_buff *skb,
	IN	struct net_device *net_dev);

NDIS_STATUS RTMPAllocDMAMemory(
    IN  PRTMP_ADAPTER   pAd);

VOID	RTMPFreeDMAMemory(
	IN  PRTMP_ADAPTER   pAd);

VOID	NICInitializeAdapter(
    IN  PRTMP_ADAPTER   pAd);

VOID	NICReadEEPROMParameters(
    IN  PRTMP_ADAPTER       pAdapter);

VOID	NICInitAsicFromEEPROM(
	IN  PRTMP_ADAPTER       pAdapter);

VOID	NICInitializeAsic(
	IN	PRTMP_ADAPTER	pAd);

VOID	PortCfgInit(
	IN	PRTMP_ADAPTER	pAd);

VOID	RTMPReadParametersFromFile(
	IN	PRTMP_ADAPTER	pAd);

#define RTMPEqualMemory(p1,p2,n) (memcmp((p1),(p2),(n)) == 0)


//
// Private routines in rtmp_data.c
//
VOID	RTMPHandleRxDoneInterrupt(
	IN	struct net_device *net_dev);

VOID	RTMPHandleTxRingTxDoneInterrupt(
	IN	PRTMP_ADAPTER	pAd);

VOID	RTMPHandlePrioRingTxDoneInterrupt(
	IN	PRTMP_ADAPTER	pAd);

VOID	RTMPHandleAtimRingTxDoneInterrupt(
	IN	PRTMP_ADAPTER	pAd);

VOID	RTMPHandleTbcnInterrupt(
	IN	PRTMP_ADAPTER	pAd);

VOID	RTMPHandleTwakeupInterrupt(
	IN	PRTMP_ADAPTER	pAd);

VOID	RTMPHardTransmitDone(
	IN	PRTMP_ADAPTER	pAd,
	IN	PTXD_STRUC		pTxD,
	IN	UCHAR			FrameType);

NDIS_STATUS	RTMPSendPacket(
	IN	struct net_device *net_dev,
	IN	struct sk_buff *skb);

VOID	RTMPDeQueuePacket(
    IN	PRTMP_ADAPTER	pAd);

NDIS_STATUS	RTMPHardTransmit(
	IN	PRTMP_ADAPTER	pAd,
	IN	struct sk_buff *skb,
	IN	UCHAR			NumberRequired);

NDIS_STATUS	RTMPFreeDescriptorRequest(
	IN		PRTMP_ADAPTER	pAd,
	IN		UCHAR			RingType,
	IN		UCHAR			NumberRequired);

VOID	MlmeHardTransmit(
	IN	PRTMP_ADAPTER	pAd,
	IN	VOID			*pBuffer,
	IN	ULONG			Length);

VOID	RTMPWriteTxDescriptor(
	IN	PTXD_STRUC	pTxD,
	IN	BOOLEAN		Ack,
	IN	BOOLEAN		Fragment,
	IN  BOOLEAN     InsTimestamp,
	IN	UCHAR		RetryMode,
	IN	UCHAR		Ifs,
	IN	UINT		Rate,
	IN	UCHAR		Service,
	IN	UINT		Length,
	IN  USHORT      TxPreamble);

BOOLEAN	RTMPSearchTupleCache(
	IN	PRTMP_ADAPTER	pAd,
	IN	PHEADER_802_11	pHeader);

VOID	RTMPUpdateTupleCache(
	IN	PRTMP_ADAPTER	pAd,
	IN	PHEADER_802_11	pHeader);

VOID    RTMPSuspendMsduTransmission(
   	IN	PRTMP_ADAPTER	pAd);

VOID    RTMPResumeMsduTransmission(
   	IN	PRTMP_ADAPTER	pAd);

VOID	RTMPSendNullFrame(
	IN	PRTMP_ADAPTER	pAd,
	IN	VOID			*pBuffer,
	IN	ULONG			Length);

NDIS_STATUS	RTMPCheckRxDescriptor(
	IN	PRXD_STRUC		pRxD);

NDIS_STATUS	RTMPApplyPacketFilter(
	IN	PRTMP_ADAPTER	pAdapter,
	IN	PRXD_STRUC		pRxD,
	IN	PHEADER_802_11	pHeader);
//
// Private routines in rtmp_wep.c
//
VOID	RTMPInitWepEngine(
	IN	PRTMP_ADAPTER	pAd,
	IN	PUCHAR			pKey,
	IN	UCHAR			KeyId,
	IN	UCHAR			KeyLen,
	IN	PUCHAR			pDest);

VOID	RTMPEncryptData(
	IN	PRTMP_ADAPTER	pAd,
	IN	PUCHAR			pSrc,
	IN	PUCHAR			pDest,
	IN	UINT			Len);

BOOLEAN	RTMPDecryptData(
	IN	PRTMP_ADAPTER	pAd,
	IN	PUCHAR			pSrc,
	IN	UINT			Len);

VOID	RTMPSetICV(
	IN	PRTMP_ADAPTER	pAd,
	IN	PUCHAR			pDest);

VOID	ARCFOUR_INIT(
	IN	PARCFOURCONTEXT	Ctx,
	IN	PUCHAR			pKey,
	IN	UINT			KeyLen);

UCHAR	ARCFOUR_BYTE(
	IN	PARCFOURCONTEXT		Ctx);

VOID	ARCFOUR_DECRYPT(
	IN	PARCFOURCONTEXT	Ctx,
	IN	PUCHAR			pDest,
	IN	PUCHAR			pSrc,
	IN	UINT			Len);

VOID	ARCFOUR_ENCRYPT(
	IN	PARCFOURCONTEXT	Ctx,
	IN	PUCHAR			pDest,
	IN	PUCHAR			pSrc,
	IN	UINT			Len);

ULONG	RTMP_CALC_FCS32(
	IN	ULONG	Fcs,
	IN	PUCHAR	Cp,
	IN	INT		Len);

//
// MLME routines
//
//VOID    Arc4Init(ARC4_CONTEXT *Ctx, UCHAR *Key, ULONG KeyLen);
//UCHAR   Arc4Byte(ARC4_CONTEXT *Ctx);
//VOID    Arc4Cipher(ARC4_CONTEXT *Ctx, UCHAR *Dest, UCHAR *Src, ULONG Len);

// Asic/RF/BBP related functions

VOID    NICInitTransmit(
    IN  PRTMP_ADAPTER   pAd);

VOID    AsicSwitchChannel(
    IN	PRTMP_ADAPTER	pAd,
    ULONG Channel);

VOID    AsicLockChannel(
    IN PRTMP_ADAPTER pAd,
    IN ULONG Channel) ;

VOID SetTxPower(
    IN PRTMP_ADAPTER,
	IN ULONG);

VOID AsicRfTuningExec(
    IN unsigned long data);

VOID    AsicSleepThenAutoWakeup(
    IN	PRTMP_ADAPTER	pAd,
    IN  USHORT TbttNumToNextWakeUp);

VOID    AsicForceSleep(
    IN PRTMP_ADAPTER pAd);

VOID    AsicForceWakeup(
    IN PRTMP_ADAPTER pAd);

VOID    AsicSetBssid(
    IN	PRTMP_ADAPTER	pAd,
    IN  MACADDR *Bssid);

VOID    AsicDisableSync(
    IN	PRTMP_ADAPTER	pAd);

VOID    AsicEnableBssSync(
    IN	PRTMP_ADAPTER	pAd);

VOID    AsicEnableIbssSync(
    IN	PRTMP_ADAPTER	pAd);

VOID    AsicLedPeriodicExec(
    IN unsigned long data);

VOID    MacAddrRandomBssid(
    IN	PRTMP_ADAPTER	pAd,
    OUT PMACADDR Addr);
#if 0
VOID    MacTableEntryInit(
    IN  MAC_TABLE_ENTRY *Entry);

VOID    MacTableInit(
    IN	PRTMP_ADAPTER	pAd);

BOOLEAN MacTableInsertEntry(
    IN	PRTMP_ADAPTER	pAd,
    IN  PMACADDR Addr,
    IN  UCHAR PsMode,
    IN  UCHAR AuthMode,
    IN  UCHAR AssocMode,
    IN  UCHAR Rate,
    OUT ULONG *Index);

BOOLEAN MacTableDeleteEntry(
    IN	PRTMP_ADAPTER	pAd,
    IN  PMACADDR Addr);

ULONG    MacTableLookup(
    IN	PRTMP_ADAPTER	pAd,
    IN  PMACADDR Addr);

VOID    MacTableAgedOut(
    IN PVOID SystemSpecific1,
    IN PVOID FunctionContext,
    IN PVOID SystemSpecific2,
    IN PVOID SystemSpecific3);

BOOLEAN MacTableSetRate(
    IN	PRTMP_ADAPTER	pAd,
    IN  PMACADDR Addr,
    IN  UCHAR Rate);

BOOLEAN MacTableSetTxPower(
    IN	PRTMP_ADAPTER	pAd,
    IN  PMACADDR Addr,
    IN  UCHAR power);

BOOLEAN MacTableGetRateAndPower(
    IN	PRTMP_ADAPTER	pAd,
    IN  PMACADDR Addr,
    OUT UCHAR *Rate,
    OUT UCHAR *Power);

UCHAR*  MacTableGetPeerKey(
    IN	PRTMP_ADAPTER	pAd,
    IN  UCHAR Index,
    OUT UCHAR *Len);

BOOLEAN MacTableSetPeerKey(
    IN	PRTMP_ADAPTER	pAd,
    IN  PMACADDR Addr,
    IN  UCHAR Enable,
    IN  UCHAR KeyLen,
    IN  UCHAR *PeerKey);

BOOLEAN PsInquiry(
    IN	PRTMP_ADAPTER	pAd,
    IN  PMACADDR Addr,
    OUT UCHAR *PsMode,
    OUT UCHAR *KeyLen,
    OUT UCHAR *PeerKey,
    OUT UCHAR *Rate);

BOOLEAN SsInquiry(
    IN	PRTMP_ADAPTER	pAd,
    IN  PMACADDR Addr,
    OUT UCHAR *AuthMode,
    OUT UCHAR *AssocMode,
    OUT UCHAR *Rate,
    OUT UCHAR *Power,
    OUT UCHAR *KeyLen,
    OUT UCHAR **PeerKey);

BOOLEAN PsIndicate(
    IN	PRTMP_ADAPTER	pAd,
    IN  PMACADDR Addr,
    IN  UCHAR Psm);

BOOLEAN StaState(
    IN	PRTMP_ADAPTER	pAd,
    IN  PMACADDR		Addr,
    IN  UCHAR			AoA,
    IN  USHORT			Sst);
#endif

NDIS_STATUS	MiniportMMRequest(
	IN	PRTMP_ADAPTER	pAd,
	IN	VOID			*pBuffer,
	IN	ULONG			Length);

USHORT	RTMPCalcDuration(
	IN	PRTMP_ADAPTER	pAd,
	IN	UCHAR			Rate,
	IN	ULONG			Size);

VOID    MgtMacHeaderInit(
    IN	PRTMP_ADAPTER	pAd,
    IN OUT PMACHDR Hdr,
    IN UCHAR Subtype,
    IN UCHAR ToDs,
//  IN UCHAR AddrType,
    IN PMACADDR Ds,
    IN PMACADDR Bssid);

VOID MlmeRadioOff(
    IN PRTMP_ADAPTER pAd);

VOID MlmeRadioOn(
    IN PRTMP_ADAPTER pAd);

VOID  BssTableInit(
    IN BSS_TABLE *Tab);

ULONG BssTableSearch(
    IN BSS_TABLE *Tab,
    IN PMACADDR Bssid);

VOID BssTableDeleteEntry(
    IN OUT	BSS_TABLE *Tab,
    IN		PMACADDR Bssid);

VOID  BssEntrySet(
    IN	PRTMP_ADAPTER	pAd,
    OUT BSS_ENTRY *Bss,
    IN MACADDR *Bssid,
    IN CHAR Ssid[],
    IN UCHAR SsidLen,
    IN UCHAR BssType,
    IN USHORT BeaconPeriod,
    IN BOOLEAN CfExist,
    IN CF_PARM *CfParm,
    IN USHORT AtimWin,
    IN USHORT CapabilityInfo,
    IN UCHAR Rates[],
    IN UCHAR RatesLen,
    IN UCHAR Channel,
    IN UCHAR Rssi);

ULONG  BssTableSetEntry(
    IN	PRTMP_ADAPTER	pAd,
    OUT BSS_TABLE *Tab,
    IN MACADDR *Bssid,
    IN CHAR Ssid[],
    IN UCHAR SsidLen,
    IN UCHAR BssType,
    IN USHORT BeaconPeriod,
    IN BOOLEAN CfExist,
    IN CF_PARM *CfParm,
    IN USHORT AtimWin,
    IN USHORT CapabilityInfo,
    IN UCHAR Rates[],
    IN UCHAR RatesLen,
    IN UCHAR Channel,
    IN UCHAR Rssi);

VOID  BssTableSsidSort(
    IN  BSS_TABLE *InTab,
    OUT BSS_TABLE *OutTab,
    IN  CHAR Ssid[],
    IN  UCHAR SsidLen,
    IN  UCHAR BssType,
    IN  UCHAR Privacy);

VOID  BssTableSortByRssi(
    IN OUT BSS_TABLE *OutTab);

NDIS_STATUS  MlmeQueueInit(
    IN MLME_QUEUE *Queue);

VOID  MlmeQueueDestroy(
    IN MLME_QUEUE *Queue);

BOOLEAN MlmeEnqueue(
    OUT MLME_QUEUE *Queue,
    IN ULONG Machine,
    IN ULONG MsgType,
    IN ULONG MsgLen,
    IN VOID *Msg);

BOOLEAN MlmeEnqueueForRecv(
    IN PRTMP_ADAPTER	pAd,
    OUT MLME_QUEUE *Queue,
    IN ULONG TimeStampHigh,
    IN ULONG TimeStampLow,
    IN UCHAR Rssi,
    IN ULONG MsgLen,
    IN VOID *Msg);

BOOLEAN MlmeDequeue(
    IN MLME_QUEUE *Queue,
    OUT MLME_QUEUE_ELEM **Elem);

BOOLEAN  MsgTypeSubst(
    IN MACFRAME *Fr,
    OUT INT *Machine,
    OUT INT *MsgType);

VOID StateMachineInit(
    IN STATE_MACHINE *Sm,
    IN STATE_MACHINE_FUNC Trans[],
    IN ULONG StNr,
    IN ULONG MsgNr,
    IN STATE_MACHINE_FUNC DefFunc,
    IN ULONG InitState,
    IN ULONG Base);

VOID StateMachineSetAction(
    IN STATE_MACHINE *S,
    IN ULONG St,
    ULONG Msg,
    IN STATE_MACHINE_FUNC F);

VOID StateMachinePerformAction(
    IN	PRTMP_ADAPTER	pAd,
    IN STATE_MACHINE *S,
    IN MLME_QUEUE_ELEM *Elem);

VOID StateMachineDestroy(
    IN STATE_MACHINE *Sm);

VOID Drop(
    IN	PRTMP_ADAPTER	pAd,
    IN MLME_QUEUE_ELEM *Elem);

VOID  ComposePsPoll(
    IN	PRTMP_ADAPTER	pAd);

VOID  ComposeNullFrame(
    IN  PRTMP_ADAPTER	pAd);

VOID  AssocStateMachineInit(
    IN	PRTMP_ADAPTER	pAd,
    IN  STATE_MACHINE *Sm,
    OUT STATE_MACHINE_FUNC Trans[]);

VOID  ReassocTimeout(
    IN unsigned long data);

VOID  AssocTimeout(
    IN unsigned long data);

VOID  DisassocTimeout(
    IN unsigned long data);

VOID  MlmeDisassocReqAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

//----------------------------------------------

VOID  MlmeAssocReqAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID  MlmeReassocReqAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID  MlmeDisassocReqAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID  PeerAssocRspAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID  PeerReassocRspAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID  PeerDisassocAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID  DisassocTimeoutAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID  AssocTimeoutAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID  ReassocTimeoutAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID  Cls3errAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  PMACADDR pAddr);

VOID  InvalidStateWhenAssoc(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID  InvalidStateWhenReassoc(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID InvalidStateWhenDisassociate(
    IN  PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID  AssocPostProc(
    IN	PRTMP_ADAPTER	pAd,
    IN  MACADDR *Addr2,
    IN  USHORT CapabilityInfo,
    IN  USHORT Aid,
    IN  UCHAR Rates[],
    IN  UCHAR RatesLen);

VOID AuthStateMachineInit(
    IN	PRTMP_ADAPTER		pAd,
    IN PSTATE_MACHINE		sm,
    OUT STATE_MACHINE_FUNC	Trans[]);

VOID AuthTimeout(
    IN unsigned long data);

VOID MlmeAuthReqAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID PeerAuthRspAtSeq2Action(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID PeerAuthRspAtSeq4Action(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID AuthTimeoutAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID Cls2errAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  PMACADDR		pAddr);

VOID MlmeDeauthReqAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID InvalidStateWhenAuth(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

//=============================================

VOID AuthRspStateMachineInit(
    IN	PRTMP_ADAPTER		pAd,
    IN  PSTATE_MACHINE		Sm,
    IN  STATE_MACHINE_FUNC	Trans[]);


VOID AuthRspChallengeTimeout(
    IN unsigned long data);

VOID AuthRspChallengeTimeoutAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID PeerAuthAtAuthRspIdleAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID PeerAuthAtAuthRspWaitAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID PeerDeauthAction(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem);

VOID PeerAuthSimpleRspGenAndSend(
    IN	PRTMP_ADAPTER	pAd,
    IN  PMACHDR Hdr,
    IN  USHORT Alg,
    IN  USHORT Seq,
    IN  USHORT Reason,
    IN  USHORT Status);

//========================================

VOID SyncStateMachineInit(
    IN	PRTMP_ADAPTER	pAd,
    IN  STATE_MACHINE *Sm,
    OUT STATE_MACHINE_FUNC Trans[]);

VOID BeaconTimeout(
    IN unsigned long data);

VOID AtimTimeout(
    IN unsigned long data);

VOID ScanTimeout(
    IN unsigned long data);

VOID MlmeScanReqAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID InvalidStateWhenScan(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID InvalidStateWhenJoin(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID InvalidStateWhenStart(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID PeerBeacon(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

//=========================================

VOID MlmeCntlInit(
    IN	PRTMP_ADAPTER		pAd,
    IN  STATE_MACHINE		*S,
    OUT STATE_MACHINE_FUNC	Trans[]);

VOID MlmeCntlMachinePerformAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  STATE_MACHINE	*S,
    IN  MLME_QUEUE_ELEM *Elem);

VOID CntlIdleProc(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID CntlOidScanProc(
    IN  PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID CntlOidSsidProc(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM	*Elem);

VOID CntlOidRTBssidProc(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM	*Elem);

VOID CntlMlmeRoamingProc(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM	*Elem);

VOID CntlWaitDisassocProc(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID CntlWaitJoinProc(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID CntlWaitReassocProc(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID CntlWaitStartProc(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID CntlWaitAuthProc(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID CntlWaitAuthProc2(
    IN  PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID CntlWaitAssocProc(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID LinkUp(
    IN	PRTMP_ADAPTER	pAd,
    IN  UCHAR			BssType);

VOID LinkDown(
    IN	PRTMP_ADAPTER	pAd);

VOID MlmeCntlConfirm(
    IN	PRTMP_ADAPTER	pAd,
    IN  ULONG 			MsgType,
    IN  USHORT			Msg);

VOID IterateOnBssTab(
    IN	PRTMP_ADAPTER	pAd);

VOID IterateOnBssTab2(
    IN  PRTMP_ADAPTER   pAd);

VOID JoinParmFill(
    IN	PRTMP_ADAPTER	pAd,
    IN  OUT MLME_JOIN_REQ_STRUCT *JoinReq,
    IN  ULONG BssIdx);

VOID AssocParmFill(
    IN	PRTMP_ADAPTER	pAd,
    IN OUT MLME_ASSOC_REQ_STRUCT *AssocReq,
    IN  MACADDR *Addr,
    IN  USHORT CapabilityInfo,
    IN  ULONG Timeout,
    IN  USHORT ListenIntv);

VOID ScanParmFill(
    IN	PRTMP_ADAPTER	pAd,
    IN  OUT MLME_SCAN_REQ_STRUCT *ScanReq,
    IN  CHAR Ssid[],
    IN  UCHAR SsidLen,
    IN  UCHAR BssType,
    IN  UCHAR ScanType);

VOID DisassocParmFill(
    IN	PRTMP_ADAPTER	pAd,
    IN  OUT MLME_DISASSOC_REQ_STRUCT *DisassocReq,
    IN  MACADDR *Addr,
    IN  USHORT Reason);

VOID StartParmFill(
    IN	PRTMP_ADAPTER	pAd,
    IN  OUT MLME_START_REQ_STRUCT *StartReq,
    IN  CHAR Ssid[],
    IN  UCHAR SsidLen);

VOID AuthParmFill(
    IN	PRTMP_ADAPTER	pAd,
    IN  OUT MLME_AUTH_REQ_STRUCT *AuthReq,
    IN  MACADDR *Addr,
    IN  USHORT Alg);

VOID EnqueuePsPoll(
    IN	PRTMP_ADAPTER	pAd);

VOID EnqueueNullFrame(
    IN  PRTMP_ADAPTER pAd);

VOID MlmeJoinReqAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID MlmeScanReqAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID MlmeStartReqAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID ScanTimeoutAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID BeaconTimeoutAtJoinAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID PeerBeaconAtScanAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID PeerBeaconAtJoinAction(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID PeerBeacon(
    IN	PRTMP_ADAPTER	pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID PeerProbeReqAction(
    IN  PRTMP_ADAPTER pAd,
    IN  MLME_QUEUE_ELEM *Elem);

VOID ScanNextChannel(
    IN	PRTMP_ADAPTER	pAd);

ULONG MakeIbssBeacon(
    IN	PRTMP_ADAPTER	pAd);

BOOLEAN ScanReqParmSanity(
    IN	PRTMP_ADAPTER	pAd,
    IN  VOID *Msg,
    IN  ULONG MsgLen,
    OUT UCHAR *BssType,
    OUT CHAR ssid[],
    OUT UCHAR *SsidLen,
    OUT UCHAR *ScanType);

BOOLEAN BeaconAndProbeRspSanity(
    IN	PRTMP_ADAPTER	pAd,
    IN  VOID *Msg,
    IN  ULONG MsgLen,
    OUT MACADDR *Addr2,
    OUT MACADDR *Bssid,
    OUT CHAR Ssid[],
    OUT UCHAR *SsidLen,
    OUT UCHAR *BssType,
    OUT USHORT *BeaconPeriod,
    OUT UCHAR *Channel,
    OUT LARGE_INTEGER *Timestamp,
    OUT BOOLEAN *CfExist,
    OUT CF_PARM *Cf,
    OUT USHORT *AtimWin,
    OUT USHORT *CapabilityInfo,
    OUT UCHAR Rate[],
    OUT UCHAR *RateLen,
    OUT UCHAR *DtimCount,
    OUT UCHAR *DtimPeriod,
    OUT UCHAR *BcastFlag,
    OUT UCHAR *MessageToMe,
    OUT UCHAR *Legacy);

//BOOLEAN JoinParmSanity(
//    IN	PRTMP_ADAPTER	pAd,
//    IN  VOID *Msg,
//    IN  ULONG MsgLen,
//    OUT ULONG *BssIdx,
//    OUT UCHAR SupportedRates[],
//    OUT UCHAR *SupportedRatesLen);

BOOLEAN MlmeAssocReqSanity(
    IN	PRTMP_ADAPTER	pAd,
    IN  VOID *Msg,
    IN  ULONG MsgLen,
    OUT MACADDR *ApAddr,
    OUT USHORT *CapabilityInfo,
    OUT ULONG *Timeout,
    OUT USHORT *ListenIntv);

BOOLEAN AuthReqParmSanity(
    IN	PRTMP_ADAPTER	pAd,
    IN  VOID *Msg,
    IN  ULONG MsgLen,
    OUT MACADDR *Addr,
    OUT ULONG *Timeout,
    OUT USHORT *Alg);

BOOLEAN StartParmSanity(
    IN	PRTMP_ADAPTER	pAd,
    IN  VOID *Msg,
    IN  ULONG MsgLen,
    OUT CHAR Ssid[],
    OUT UCHAR *Ssidlen);

BOOLEAN PeerAuthSanity(
    IN	PRTMP_ADAPTER	pAd,
    IN  VOID *Msg,
    IN  ULONG MsgLen,
    OUT MACADDR *Addr,
    OUT USHORT *Alg,
    OUT USHORT *Seq,
    OUT USHORT *Status,
    OUT CHAR ChlgText[]);

BOOLEAN PeerAssocParmSanity(
    IN	PRTMP_ADAPTER	pAd,
    IN  VOID *Msg,
    IN  ULONG MsgLen,
    OUT MACADDR *Addr2,
    OUT USHORT *CapabilityInfo,
    OUT USHORT *Status,
    OUT USHORT *Aid,
    OUT UCHAR Rates[],
    OUT UCHAR *RatesLen);

BOOLEAN PeerDisassocSanity(
    IN	PRTMP_ADAPTER	pAd,
    IN  VOID *Msg,
    IN  ULONG MsgLen,
    OUT MACADDR *Addr2,
    OUT USHORT *Reason);

BOOLEAN PeerDeauthSanity(
    IN	PRTMP_ADAPTER	pAd,
    IN  VOID *Msg,
    IN  ULONG MsgLen,
    OUT MACADDR *Addr2,
    OUT USHORT *Reason);

BOOLEAN PeerProbeReqSanity(
    IN	PRTMP_ADAPTER	pAd,
    IN  VOID *Msg,
    IN  ULONG MsgLen,
    OUT MACADDR *Addr2,
    OUT CHAR Ssid[],
    OUT UCHAR *SsidLen,
    OUT UCHAR Rates[],
    OUT UCHAR *RatesLen);

BOOLEAN GetTimBit(
    IN  CHAR *Ptr,
    IN  USHORT Aid,
    OUT UCHAR *TimLen,
    OUT UCHAR *BcastFlag,
    OUT UCHAR *DtimCount,
    OUT UCHAR *DtimPeriod,
    OUT UCHAR *MessageToMe);

BOOLEAN GetLegacy(
    IN  CHAR *Ptr,
    OUT UCHAR *Legacy);

ULONG MakeOutgoingFrame(
    OUT CHAR *Buffer,
    OUT ULONG *Length, ...);

VOID  LfsrInit(
    IN	PRTMP_ADAPTER	pAd,
    IN  ULONG Seed);

UCHAR RandomByte(
    IN	PRTMP_ADAPTER	pAd);

VOID  MlmePeriodicExec(
    IN	unsigned long data);

VOID MlmeAutoScan(
    IN PRTMP_ADAPTER pAd);

VOID MlmeAutoRecoverNetwork(
    IN PRTMP_ADAPTER pAd);

VOID MlmeAutoReconnectLastSSID(
    IN PRTMP_ADAPTER pAd);

VOID MlmeCheckForRoaming(
    IN PRTMP_ADAPTER pAd,
    IN ULONG    Now32);

VOID MlmeCheckChannelQuality(
    IN PRTMP_ADAPTER pAd,
    IN ULONG Now);

VOID MlmeCheckForPsmChange(
    IN PRTMP_ADAPTER pAd,
    IN ULONG    Now32);

VOID MlmeSetPsmBit(
    IN PRTMP_ADAPTER pAd,
    IN USHORT psm);

VOID MlmeSetTxPreamble(
    IN PRTMP_ADAPTER pAd,
    IN USHORT TxPreamble);

VOID MlmeUpdateTxRates(
    IN PRTMP_ADAPTER pAd);

VOID MlmeHandler(
    IN	PRTMP_ADAPTER	pAd);

NDIS_STATUS MlmeInit(
    IN	PRTMP_ADAPTER	pAd);

VOID MlmeHalt(
    IN	PRTMP_ADAPTER	pAd);

VOID MlmeConfirm(
    IN	PRTMP_ADAPTER	pAd,
    IN  VOID *Msg,
    IN  INT MsgLen);

UCHAR FirstChannel(
    IN	PRTMP_ADAPTER	pAd);

UCHAR NextChannel(
    IN	PRTMP_ADAPTER	pAd,
    IN  UCHAR channel);

INT RT2400_ioctl(
	IN	struct net_device *net_dev,
	IN	struct ifreq *rq,
	IN	INT cmd);

VOID RaiseClock(
    IN	PRTMP_ADAPTER	pAd,
    IN  ULONG *x);

VOID LowerClock(
    IN	PRTMP_ADAPTER	pAd,
    IN  ULONG *x);

USHORT ShiftInBits(
    IN	PRTMP_ADAPTER	pAd);

VOID ShiftOutBits(
    IN	PRTMP_ADAPTER	pAd,
    IN  USHORT data,
    IN  USHORT count);

VOID EEpromCleanup(
    IN	PRTMP_ADAPTER	pAd);

VOID EWDS(
	IN	PRTMP_ADAPTER	pAd);

VOID EWEN(
	IN	PRTMP_ADAPTER	pAd);

USHORT RTMP_EEPROM_READ16(
    IN	PRTMP_ADAPTER	pAd,
    IN  USHORT Offset);

VOID RTMP_EEPROM_WRITE16(
    IN	PRTMP_ADAPTER	pAd,
    IN  USHORT Offset,
    IN  USHORT Data);

UCHAR ChannelSanity(
    IN PRTMP_ADAPTER pAd,
    IN UCHAR channel);

int rt2400_set_mac_address(struct net_device *net_dev, void *addr);
#endif  // __RTMP_H__
