#  Copyright (C) 1999-2005
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc IMGSVRInit {varname title exec ack update} {
    upvar #0 $varname var
    global $varname

    global ds9

    set w $var(top)

    # AR variables
    set var(status) {}
    set var(current) {}
    set var(sync) 0

    set var(proc,next) IMGSVRServer

    # IMG variables

    set var(proc,exec) $exec
    set var(proc,ack) $ack
    set var(proc,update) $update

    set var(frame) {}
    set var(rgb) {}

    # create the window
    toplevel $w -colormap $ds9(main)
    wm title $w $title
    wm iconname $w $title
    wm group $w $ds9(top)
    wm protocol $w WM_DELETE_WINDOW "IMGSVRDestroy $varname"

    # menu
    $w configure -menu $var(mb)
    menu $var(mb) -tearoff 0

    # dialog
    frame $w.param -relief groove -borderwidth 2
    frame $w.status -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.param $w.status $w.buttons -fill x -expand true -ipadx 4 -ipady 4

    label $w.param.nametitle -text "Name"
    entry $w.param.name -textvariable ${varname}(name) -width 38

    set var(xname) [label $w.param.xtitle -text "" -font {symbol 12} -width 1]
    entry $w.param.x -textvariable ${varname}(x) -width 14
    set var(yname) [label $w.param.ytitle -text "" -font {symbol 12} -width 1]
    entry $w.param.y -textvariable ${varname}(y) -width 14
    label $w.param.system -textvariable ${varname}(sky) \
	-width 10 -relief groove

    grid rowconfigure $w.param 0 -pad 4
    grid rowconfigure $w.param 1 -pad 4
    grid rowconfigure $w.param 2 -pad 4

    grid $w.param.nametitle $w.param.name - - - -padx 4 -pady 1 -sticky w
    grid $w.param.xtitle $w.param.x $w.param.ytitle $w.param.y \
	$w.param.system -padx 4 -pady 1 -sticky w

    label $w.status.item -textvariable ${varname}(status)
    pack $w.status.item -anchor w -pady 4

    set var(apply) [button $w.buttons.apply -text Retrieve \
			-command "IMGSVRApply $varname 0"]
    set var(cancel) [button $w.buttons.cancel -text Cancel \
			-command "ARCancel $varname" -state disabled]
    button $w.buttons.close -text "Close" \
	-command "IMGSVRDestroy $varname"
    pack $w.buttons.apply $w.buttons.cancel $w.buttons.close \
	-side left -padx 10 -expand true

    ARCoord $varname
    ARStatus $varname {}
}

proc IMGSVRFileMenu {varname} {
    upvar #0 $varname var
    global $varname

    global menu

    $var(mb) add cascade -label File -menu $var(mb).file
    menu $var(mb).file -tearoff 0 -selectcolor $menu(selectcolor)
    $var(mb).file add command -label Retrieve -command "IMGSVRApply $varname 0"
    $var(mb).file add command -label Cancel -command "ARCancel $varname"
    $var(mb).file add separator
    $var(mb).file add command -label "Update from Current Frame" \
	-command "IMGSVRUpdate $varname 1"
    $var(mb).file add command -label "Update from Current Crosshair" \
	-command "IMGSVRCrosshair $varname"
    $var(mb).file add separator
    $var(mb).file add command -label Acknowledgement \
	-command "IMGSVRAck $varname"
    $var(mb).file add separator
    $var(mb).file add command -label Close -command "IMGSVRDestroy $varname"
}

proc IMGSVRPrefsMenu {varname} {
    upvar #0 $varname var
    global $varname

    global menu

    $var(mb) add cascade -label Prefs -menu $var(mb).prefs
    menu $var(mb).prefs -tearoff 0 -selectcolor $menu(selectcolor)
    $var(mb).prefs add checkbutton -label "Save FITS on download" \
	-variable ${varname}(save)
    $var(mb).prefs add separator
    $var(mb).prefs add radiobutton -label "New Frame" \
	-variable ${varname}(mode) -value new
    $var(mb).prefs add radiobutton -label "Current Frame" \
	-variable ${varname}(mode) -value current
}

proc IMGSVRDestroy {varname} {
    upvar #0 $varname var
    global $varname

    ARDestroy $varname

    catch {
	unset var(proc,exec)
	unset var(proc,ack)
	unset var(proc,update)

	unset var(frame)
	unset var(rgb)
    }
}

proc IMGSVRApply {varname sync} {
    upvar #0 $varname var
    global $varname

    set var(sync) $sync

    ARStatus $varname {}

    $var(mb).file entryconfig Retrieve -state disabled
    $var(mb).file entryconfig Cancel -state normal

    $var(apply) configure -state disabled
    $var(cancel) configure -state normal

    set var(frame) {}
    set var(rgb) {}

    if {($var(name) != {})} {
	set var(system) wcs
	set var(sky) fk5
	ARCoord $varname

	# remember where we are
	global current
	switch -- $var(mode) {
	    current {
		set var(frame) $current(frame)
		set var(rgb) $current(rgb)
	    }
	    new {}
	}

	NSVRServer $varname
    } else {
	IMGSVRServer $varname
    }
}

proc IMGSVRAck {varname} {
    upvar #0 $varname var
    global $varname

    eval $var(proc,ack)
}

proc IMGSVRUpdate {varname force} {
    upvar #0 $varname var
    global $varname

    global current

    global debug
    if {$debug(tcl,update)} {
	puts "IMGSVRUpdate"
    }

    if {[winfo exist $var(top)]} {
	if {$current(frame) != {} } {
	    set fn [$current(frame) get fits file name]
	    if {($fn != $var(current)) || $force} {
		set var(current) $fn

		if {[$current(frame) has wcs equatorial $var(system)]} {
		    set coord [$current(frame) get cursor \
				   $var(system) $var(sky) sexagesimal]
		    set var(x) [lindex $coord 0]
		    set var(y) [lindex $coord 1]

		    set size [$current(frame) get fits size \
				  $var(system) $var(rformat)]

		    # do any special updating
		    eval $var(proc,update)

		    set var(name) {}
		}
	    }
	}
    }
}

proc IMGSVRCrosshair {varname} {
    upvar #0 $varname var
    global $varname

    global current

    if {$current(frame) != {} } {
	if {[$current(frame) has wcs equatorial $var(system)]} {
	    set coord [$current(frame) get crosshair \
			   $var(system) $var(sky) sexagesimal]
	    set var(x) [lindex $coord 0]
	    set var(y) [lindex $coord 1]

	    set size [$current(frame) get fits size \
			  $var(system) $var(rformat)]

	    # do any special updating
	    eval $var(proc,update)

	    set var(name) {}
	}
    }
}

proc IMGSVRServer {varname} {
    upvar #0 $varname var
    global $varname

    global ds9
    global current

    if {($var(x) != {}) && ($var(y) != {})} {
	switch -- $var(mode) {
	    new {
		set ds9(display,user) tile
		DisplayMode
		CreateFrame
	    }
	    current {}
	}

	# remember where we are
	if {$var(frame) == {}} {
	    set var(frame) $current(frame)
	}
	if {$var(rgb) == {}} {
	    set var(rgb) $current(rgb)
	}

	ARStatus $varname "Contacting $varname Image Server"
	eval $var(proc,exec)
    } else {
	ARStatus $varname "Please specify Coordinates"

	$var(mb).file entryconfig Retrieve -state normal
	$var(mb).file entryconfig Cancel -state disabled

	$var(apply) configure -state normal
	$var(cancel) configure -state disabled
    }
}

proc IMGSVRLoad {varname url query} {
    upvar #0 $varname var
    global $varname

    global http

    StartLoad

    set var(ch) [open "$var(fn)" w]

    if {$var(sync)} {
	set token [http::geturl $url \
		       -channel $var(ch) \
		       -progress [list IMGSVRProgress $varname] \
		       -binary 1 \
		       -headers "[ProxyHTTP]" \
		       -query "$query"]
	set var(state) 1
	set var(token) $token

	IMGSVRLoadFinish $varname $token
    } else {
	set token [http::geturl $url \
		       -channel $var(ch) \
		       -command [list IMGSVRLoadFinish $varname] \
		       -progress [list IMGSVRProgress $varname] \
		       -binary 1 \
		       -headers "[ProxyHTTP]" \
		       -query "$query"]
	set var(state) 1
	set var(token) $token
    }
}

proc IMGSVRProgress {varname token totalsize currentsize} {
    upvar #0 $varname var
    global $varname

    # sometimes we get nothing
    if {$totalsize == {} || $currentsize == {}} {
	ARStatus $varname {}
    } elseif {$totalsize != 0} {
	ARStatus $varname "$currentsize bytes of $totalsize bytes [expr int(double($currentsize)/$totalsize*100)]%"
    } else {
	ARStatus $varname "$currentsize bytes"
    }
}

proc IMGSVRLoadFinish {varname token} {
    upvar #0 $varname var
    global $varname

    global current
    global ds9
    global loadParam

    HTTPLog $token
    if {$var(state)} {
	# check for error
	set code [http::ncode $token]
	if {$code != "200"} {
	    ARError $varname $code
	    return
	}

	# goto to frame
	if {$var(frame) != {}} {
	    if {$current(frame) != $var(frame)} {
		set ds9(next) $var(frame)
		GotoFrame
	    }
	}

	# got to channel
	if {$var(rgb) != {}} {
	    if {$current(rgb) != $var(rgb)} {
		set current(rgb) $var(rgb)
		RGBChannel
	    }
	}

	# alloc it because we are going to delete it after load
	set loadParam(load,type) allocgz
	set loadParam(file,type) fits
	set loadParam(file,mode) {}
	ConvertFile $var(fn)
	ProcessLoad

	catch {close $var(ch)}

	if {!$var(save)} {
	    if {[file exists $var(fn)]} {
		catch {file delete -force $var(fn)}
	    }
	}
	FinishLoad
	ARStatus $varname {Done}
    } else {
	catch {close $var(ch)}

	UnsetWatchCursor
	ARStatus $varname {Cancelled}
    }

    catch {unset var(ch)}

    ARReset $varname
}

