// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "colorscalergb.h"
#include "util.h"

ColorScaleRGB::ColorScaleRGB(int s)
{
  size_ = s;
  psColors_ = new unsigned char[s];

  memset(psColors_, '0', size_);
}

ColorScaleRGB::~ColorScaleRGB()
{
  if (psColors_)
    delete [] psColors_;
}

LinearScaleRGB::LinearScaleRGB(int ii, int s, unsigned char* colorCells, 
			       int count) 
  : ColorScaleRGB(s)
{
  for (int i=0; i<s; i++) {
    double a = (double)i/s;
    int l = (int)(a * count);
    memcpy(psColors_+i, colorCells+l*3+ii,1);
  }
}

LogScaleRGB::LogScaleRGB(int ii, int s, unsigned char* colorCells, int count) 
  : ColorScaleRGB(s)
{
  double x = 1000;
  for (int i=0; i<s; i++) {
    double a = (double)i/s;
    int l = (int)(log10(a*x+1) * (1/log10(x+1)) * count);
    memcpy(psColors_+i, colorCells+l*3+ii,1);
  }
}

SquaredScaleRGB::SquaredScaleRGB(int ii, int s, unsigned char* colorCells, 
				 int count) 
  : ColorScaleRGB(s)
{
  for(int i=0; i<s; i++) {
    double a = (double)i/s;
    int l = (int)(a*a * count);
    memcpy(psColors_+i, colorCells+l*3+ii,1);
  }
}

SqrtScaleRGB::SqrtScaleRGB(int ii, int s, unsigned char* colorCells, int count)
  : ColorScaleRGB(s)
{
  for(int i=0; i<s; i++) {
    double a = (double)i/s;
    int l = (int)(sqrt(a) * count);
    memcpy(psColors_+i, colorCells+l*3+ii,1);
  }
}

HistEquScaleRGB::HistEquScaleRGB(int ii, int s, unsigned char* colorCells, 
				 int count, double* hist, int histsize)
  : ColorScaleRGB(s)
{
  // if no histogram, return linear distribution

  if (!hist)
    for (int i=0; i<s; i++) {
      int l = (int)((double)i/s * count);
      memcpy(psColors_+i, colorCells+l*3+ii,1);
    }
  else
    for (int i=0; i<s; i++) {
      int l = (int)(hist[i*histsize/s]*count);
      memcpy(psColors_+i, colorCells+l*3+ii,1);
    }
}
