// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "framebase.h"
#include "fitsimage.h"

#include "alloc.h"
#include "allocgz.h"
#include "channel.h"
#include "mmap.h"
#include "mmapincr.h"
#include "share.h"
#include "sshare.h"
#include "socket.h"
#include "socketgz.h"
#include "var.h"

// *** Fits ***

void FrameBase::loadFitsAllocCmd(const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsAlloc(this, fn, FitsFile::NOFLUSH);
  load(ALLOC,fn,img);
}

void FrameBase::loadFitsAllocGZCmd(const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsAllocGZ(this, fn, FitsFile::NOFLUSH);
  load(ALLOCGZ,fn,img);
}

void FrameBase::loadFitsChannelCmd(const char* ch, const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsChannel(this, interp, ch, fn, 
					    FitsFile::NOFLUSH);
  load(CHANNEL,fn,img);
}

void FrameBase::loadFitsMMapCmd(const char* fn, LoadMethod lm)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsMMap(this, fn);
  setScanModeIncr(lm);
  load(MMAP,fn,img);
}

void FrameBase::loadFitsSMMapCmd(const char* hdr, const char* fn, 
				 LoadMethod lm)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsSMMap(this, hdr, fn);
  setScanModeIncr(lm);
  load(SMMAP,fn,img);
}

void FrameBase::loadFitsMMapIncrCmd(const char* fn, LoadMethod lm)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsMMapIncr(this, fn);
  setScanModeIncr(lm);
  load(MMAPINCR,fn,img);
}

void FrameBase::loadFitsShareCmd(ShmType type, int id, const char* fn,
				 LoadMethod lm)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsShare(this, type, id, fn);
  setScanModeIncr(lm);
  load(SHARE,fn,img);
}

void FrameBase::loadFitsSShareCmd(ShmType type, int hdr, int id, 
				  const char* fn, LoadMethod lm)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsSShare(this, type, hdr, id, fn);
  setScanModeIncr(lm);
  load(SSHARE,fn,img);
}

void FrameBase::loadFitsSocketCmd(int s, const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsSocket(this, s, fn, FitsFile::FLUSH);
  load(SOCKET,fn,img);
}

void FrameBase::loadFitsSocketGZCmd(int s, const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsSocketGZ(this, s, fn, FitsFile::FLUSH);
  load(SOCKETGZ,fn,img);
}

void FrameBase::loadFitsVarCmd(const char* ch, const char* fn, LoadMethod lm)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsVar(this, interp, ch, fn);
  setScanModeIncr(lm);
  load(VAR,fn,img);
}

// *** Array ***

void FrameBase::loadArrAllocCmd(const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageArrAlloc(this, fn, FitsFile::NOFLUSH);
  load(ALLOC,fn,img);
}

void FrameBase::loadArrAllocGZCmd(const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageArrAllocGZ(this, fn, FitsFile::NOFLUSH);
  load(ALLOCGZ,fn,img);
}

void FrameBase::loadArrChannelCmd(const char* ch, const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageArrChannel(this, interp, ch, fn, 
					   FitsFile::NOFLUSH);
  load(CHANNEL,fn,img);
}

void FrameBase::loadArrMMapCmd(const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageArrMMap(this, fn);
  load(MMAP,fn,img);
}

void FrameBase::loadArrMMapIncrCmd(const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageArrMMapIncr(this, fn);
  load(MMAPINCR,fn,img);
}

void FrameBase::loadArrShareCmd(ShmType type, int id, const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageArrShare(this, type, id, fn);
  load(SHARE,fn,img);
}

void FrameBase::loadArrSocketCmd(int s, const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageArrSocket(this, s, fn, FitsFile::FLUSH);
  load(SOCKET,fn,img);
}

void FrameBase::loadArrSocketGZCmd(int s, const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageArrSocketGZ(this, s, fn, FitsFile::FLUSH);
  load(SOCKETGZ,fn,img);
}

void FrameBase::loadArrVarCmd(const char* ch, const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageArrVar(this, interp, ch, fn);
  load(VAR,fn,img);
}

// *** Fits Slave ***

void FrameBase::loadFitsSlaveCmd(const char* fn, FitsFile* f)
{
  unloadFits();
  FitsImage* img = new FitsImageSlave(this,fn,f);
  load(MMAPINCR,fn,img);
}

// *** Data Cube ***

void FrameBase::loadDataCubeAllocCmd(const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsAlloc(this, fn, FitsFile::NOFLUSH);
  loadDataCube(ALLOC,fn,img);
}

void FrameBase::loadDataCubeAllocGZCmd(const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsAllocGZ(this, fn, FitsFile::NOFLUSH);
  loadDataCube(ALLOCGZ,fn,img);
}

void FrameBase::loadDataCubeChannelCmd(const char* ch, const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsChannel(this, interp, ch, fn, 
					    FitsFile::NOFLUSH);
  loadDataCube(CHANNEL,fn,img);
}

void FrameBase::loadDataCubeMMapCmd(const char* fn, LoadMethod lm)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsMMap(this, fn);
  setScanModeIncr(lm);
  loadDataCube(MMAP,fn,img);
}

void FrameBase::loadDataCubeMMapIncrCmd(const char* fn, LoadMethod lm)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsMMapIncr(this, fn);
  setScanModeIncr(lm);
  loadDataCube(MMAPINCR,fn,img);
}

void FrameBase::loadDataCubeShareCmd(ShmType type, int id, const char* fn,
				 LoadMethod lm)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsShare(this, type, id, fn);
  setScanModeIncr(lm);
  loadDataCube(SHARE,fn,img);
}

void FrameBase::loadDataCubeSocketCmd(int s, const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsSocket(this, s, fn, FitsFile::NOFLUSH);
  loadDataCube(SOCKET,fn,img);
}

void FrameBase::loadDataCubeSocketGZCmd(int s, const char* fn)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsSocketGZ(this, s, fn, FitsFile::NOFLUSH);
  loadDataCube(SOCKETGZ,fn,img);
}

void FrameBase::loadDataCubeVarCmd(const char* ch, const char* fn, LoadMethod lm)
{
  unloadFits();
  FitsImage* img = new FitsImageFitsVar(this, interp, ch, fn);
  setScanModeIncr(lm);
  loadDataCube(VAR,fn,img);
}

// *** Mosaic Image IRAF ***

void FrameBase::loadMosaicImageIRAFAllocCmd(const char* fn)
{
  unloadFits();
  *currentMosaic = IRAF;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageMosaicAlloc(this, fn, FitsFile::NOFLUSH);
  loadMosaicImage(ALLOC,fn,img);
}

void FrameBase::loadMosaicImageIRAFAllocGZCmd(const char* fn)
{
  unloadFits();
  *currentMosaic = IRAF;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageMosaicAllocGZ(this, fn, FitsFile::NOFLUSH);
  loadMosaicImage(ALLOCGZ,fn,img);
}

void FrameBase::loadMosaicImageIRAFChannelCmd(const char* ch, const char* fn)
{
  unloadFits();
  *currentMosaic = IRAF;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageMosaicChannel(this, interp, ch, fn, 
					      FitsFile::NOFLUSH);
  loadMosaicImage(CHANNEL,fn,img);
}

void FrameBase::loadMosaicImageIRAFMMapCmd(const char* fn, LoadMethod lm)
{
  unloadFits();
  *currentMosaic = IRAF;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageMosaicMMap(this, fn);
  setScanModeIncr(lm);
  loadMosaicImage(MMAP,fn,img);
}

void FrameBase::loadMosaicImageIRAFMMapIncrCmd(const char* fn, LoadMethod lm)
{
  unloadFits();
  *currentMosaic = IRAF;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageMosaicMMapIncr(this, fn);
  setScanModeIncr(lm);
  loadMosaicImage(MMAPINCR,fn,img);
}

void FrameBase::loadMosaicImageIRAFShareCmd(ShmType type, int id, 
					    const char* fn, LoadMethod lm)
{
  unloadFits();
  *currentMosaic = IRAF;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageMosaicShare(this, type, id, fn);
  setScanModeIncr(lm);
  loadMosaicImage(SHARE,fn,img);
}

void FrameBase::loadMosaicImageIRAFSocketCmd(int s, const char* fn)
{
  unloadFits();
  *currentMosaic = IRAF;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageMosaicSocket(this, s, fn, FitsFile::NOFLUSH);
  loadMosaicImage(SOCKET,fn,img);
}

void FrameBase::loadMosaicImageIRAFSocketGZCmd(int s, const char* fn)
{
  unloadFits();
  *currentMosaic = IRAF;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageMosaicSocketGZ(this, s, fn, FitsFile::NOFLUSH);
  loadMosaicImage(SOCKETGZ,fn,img);
}

void FrameBase::loadMosaicImageIRAFVarCmd(const char* ch, const char* fn,
					  LoadMethod lm)
{
  unloadFits();
  *currentMosaic = IRAF;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageMosaicVar(this, interp, ch, fn);
  setScanModeIncr(lm);
  loadMosaicImage(VAR,fn,img);
}

// *** Mosaic IRAF ***

void FrameBase::loadMosaicIRAFAllocCmd(const char* fn)
{
  *currentMosaic = IRAF;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageFitsAlloc(this, fn, FitsFile::NOFLUSH);
  loadMosaic(ALLOC,fn,img);
}

void FrameBase::loadMosaicIRAFAllocGZCmd(const char* fn)
{
  *currentMosaic = IRAF;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageFitsAllocGZ(this, fn, FitsFile::NOFLUSH);
  loadMosaic(ALLOCGZ,fn,img);
}

void FrameBase::loadMosaicIRAFChannelCmd(const char* ch, const char* fn)
{
  *currentMosaic = IRAF;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageFitsChannel(this, interp, ch, fn, 
					    FitsFile::NOFLUSH);
  loadMosaic(CHANNEL,fn,img);
}

void FrameBase::loadMosaicIRAFMMapCmd(const char* fn, LoadMethod lm)
{
  *currentMosaic = IRAF;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageFitsMMap(this, fn);
  setScanModeIncr(lm);
  loadMosaic(MMAP,fn,img);
}

void FrameBase::loadMosaicIRAFSMMapCmd(const char* hdr, const char* fn, 
				       LoadMethod lm)
{
  *currentMosaic = IRAF;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageFitsSMMap(this, hdr, fn);
  setScanModeIncr(lm);
  loadMosaic(SMMAP,fn,img);
}

void FrameBase::loadMosaicIRAFMMapIncrCmd(const char* fn, LoadMethod lm)
{
  *currentMosaic = IRAF;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageFitsMMapIncr(this, fn);
  setScanModeIncr(lm);
  loadMosaic(MMAPINCR,fn,img);
}

void FrameBase::loadMosaicIRAFShareCmd(ShmType type, int id, const char* fn,
				       LoadMethod lm)
{
  *currentMosaic = IRAF;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageFitsShare(this, type, id, fn);
  setScanModeIncr(lm);
  loadMosaic(SHARE,fn,img);
}

void FrameBase::loadMosaicIRAFSShareCmd(ShmType type, int hdr, int id,
				       const char* fn, LoadMethod lm)
{
  *currentMosaic = IRAF;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageFitsSShare(this, type, hdr, id, fn);
  setScanModeIncr(lm);
  loadMosaic(SSHARE,fn,img);
}

void FrameBase::loadMosaicIRAFSocketCmd(int s, const char* fn)
{
  *currentMosaic = IRAF;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageFitsSocket(this, s, fn, FitsFile::FLUSH);
  loadMosaic(SOCKET,fn,img);
}

void FrameBase::loadMosaicIRAFSocketGZCmd(int s, const char* fn)
{
  *currentMosaic = IRAF;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageFitsSocketGZ(this, s, fn, FitsFile::FLUSH);
  loadMosaic(SOCKETGZ,fn,img);
}

void FrameBase::loadMosaicIRAFVarCmd(const char* ch, const char* fn,
				     LoadMethod lm)
{
  *currentMosaic = IRAF;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageFitsVar(this, interp, ch, fn);
  setScanModeIncr(lm);
  loadMosaic(VAR,fn,img);
}

// *** Mosaic Image WCS ***

void FrameBase::loadMosaicImageWCSAllocCmd(CoordSystem sys, const char* fn)
{
  unloadFits();
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageMosaicAlloc(this, fn, FitsFile::NOFLUSH);
  loadMosaicImage(ALLOC,fn,img);
}

void FrameBase::loadMosaicImageWCSAllocGZCmd(CoordSystem sys, const char* fn)
{
  unloadFits();
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageMosaicAllocGZ(this, fn, FitsFile::NOFLUSH);
  loadMosaicImage(ALLOCGZ,fn,img);
}

void FrameBase::loadMosaicImageWCSChannelCmd(CoordSystem sys, const char* ch, 
					     const char* fn)
{
  unloadFits();
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageMosaicChannel(this, interp, ch, fn, 
					      FitsFile::NOFLUSH);
  loadMosaicImage(CHANNEL,fn,img);
}

void FrameBase::loadMosaicImageWCSMMapCmd(CoordSystem sys, const char* fn, 
					  LoadMethod lm)
{
  unloadFits();
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageMosaicMMap(this, fn);
  setScanModeIncr(lm);
  loadMosaicImage(MMAP,fn,img);
}

void FrameBase::loadMosaicImageWCSMMapIncrCmd(CoordSystem sys, const char* fn, 
					      LoadMethod lm)
{
  unloadFits();
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageMosaicMMapIncr(this, fn);
  setScanModeIncr(lm);
  loadMosaicImage(MMAPINCR,fn,img);
}

void FrameBase::loadMosaicImageWCSShareCmd(CoordSystem sys, ShmType type, 
					   int id, const char* fn, 
					   LoadMethod lm)
{
  unloadFits();
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageMosaicShare(this, type, id, fn);
  setScanModeIncr(lm);
  loadMosaicImage(SHARE,fn,img);
}

void FrameBase::loadMosaicImageWCSSocketCmd(CoordSystem sys, int s, 
					    const char* fn)
{
  unloadFits();
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageMosaicSocket(this, s, fn, FitsFile::NOFLUSH);
  loadMosaicImage(SOCKET,fn,img);
}

void FrameBase::loadMosaicImageWCSSocketGZCmd(CoordSystem sys, int s, 
					      const char* fn)
{
  unloadFits();
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageMosaicSocketGZ(this, s, fn, FitsFile::NOFLUSH);
  loadMosaicImage(SOCKETGZ,fn,img);
}

void FrameBase::loadMosaicImageWCSVarCmd(CoordSystem sys, const char* ch, 
					 const char* fn, LoadMethod lm)
{
  unloadFits();
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageMosaicVar(this, interp, ch, fn);
  setScanModeIncr(lm);
  loadMosaicImage(VAR,fn,img);
}

// *** Mosaic Image Next WCS ***

void FrameBase::loadMosaicImageNextWCSAllocCmd(CoordSystem sys, const char* fn)
{
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageMosaicAlloc(this, fn, FitsFile::NOFLUSH);
  loadMosaicImageNext(ALLOC,fn,img);
}

void FrameBase::loadMosaicImageNextWCSAllocGZCmd(CoordSystem sys, 
						 const char* fn)
{
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageMosaicAllocGZ(this, fn, FitsFile::NOFLUSH);
  loadMosaicImageNext(ALLOCGZ,fn,img);
}

void FrameBase::loadMosaicImageNextWCSChannelCmd(CoordSystem sys, 
						 const char* ch, const char* fn)
{
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageMosaicChannel(this, interp, ch, fn, 
					      FitsFile::NOFLUSH);
  loadMosaicImageNext(CHANNEL,fn,img);
}

void FrameBase::loadMosaicImageNextWCSMMapCmd(CoordSystem sys, const char* fn, 
					      LoadMethod lm)
{
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageMosaicMMap(this, fn);
  setScanModeIncr(lm);
  loadMosaicImageNext(MMAP,fn,img);
}

void FrameBase::loadMosaicImageNextWCSMMapIncrCmd(CoordSystem sys, 
						  const char* fn, LoadMethod lm)
{
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageMosaicMMapIncr(this, fn);
  setScanModeIncr(lm);
  loadMosaicImageNext(MMAPINCR,fn,img);
}

void FrameBase::loadMosaicImageNextWCSShareCmd(CoordSystem sys, ShmType type, 
					       int id, const char* fn, 
					       LoadMethod lm)
{
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageMosaicShare(this, type, id, fn);
  setScanModeIncr(lm);
  loadMosaicImageNext(SHARE,fn,img);
}

void FrameBase::loadMosaicImageNextWCSSocketCmd(CoordSystem sys, int s, 
						const char* fn)
{
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageMosaicSocket(this, s, fn, FitsFile::NOFLUSH);
  loadMosaicImageNext(SOCKET,fn,img);
}

void FrameBase::loadMosaicImageNextWCSSocketGZCmd(CoordSystem sys, int s, 
						  const char* fn)
{
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageMosaicSocketGZ(this, s, fn, FitsFile::NOFLUSH);
  loadMosaicImageNext(SOCKETGZ,fn,img);
}

void FrameBase::loadMosaicImageNextWCSVarCmd(CoordSystem sys, const char* ch, 
					     const char* fn, LoadMethod lm)
{
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageMosaicVar(this, interp, ch, fn);
  setScanModeIncr(lm);
  loadMosaicImageNext(VAR,fn,img);
}

// *** Mosaic WCS ***

void FrameBase::loadMosaicWCSAllocCmd(CoordSystem sys, const char* fn)
{
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageFitsAlloc(this, fn, FitsFile::NOFLUSH);
  loadMosaic(ALLOC,fn,img);
}

void FrameBase::loadMosaicWCSAllocGZCmd(CoordSystem sys, const char* fn)
{
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageFitsAllocGZ(this, fn, FitsFile::NOFLUSH);
  loadMosaic(ALLOCGZ,fn,img);
}

void FrameBase::loadMosaicWCSChannelCmd(CoordSystem sys, const char* ch, 
					const char* fn)
{
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageFitsChannel(this, interp, ch, fn,
					    FitsFile::NOFLUSH);
  loadMosaic(CHANNEL,fn,img);
}

void FrameBase::loadMosaicWCSMMapCmd(CoordSystem sys, const char* fn, 
				     LoadMethod lm)
{
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageFitsMMap(this, fn);
  setScanModeIncr(lm);
  loadMosaic(SMMAP,fn,img);
}

void FrameBase::loadMosaicWCSSMMapCmd(CoordSystem sys, const char* hdr, 
				      const char* fn, LoadMethod lm)
{
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageFitsSMMap(this, hdr, fn);
  setScanModeIncr(lm);
  loadMosaic(MMAP,fn,img);
}

void FrameBase::loadMosaicWCSMMapIncrCmd(CoordSystem sys, const char* fn, 
					 LoadMethod lm)
{
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageFitsMMapIncr(this, fn);
  setScanModeIncr(lm);
  loadMosaic(MMAPINCR,fn,img);
}

void FrameBase::loadMosaicWCSShareCmd(CoordSystem sys, ShmType type, int id, 
				      const char* fn, LoadMethod lm)
{
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageFitsShare(this, type, id, fn);
  setScanModeIncr(lm);
  loadMosaic(SHARE,fn,img);
}

void FrameBase::loadMosaicWCSSShareCmd(CoordSystem sys, ShmType type, 
				       int hdr, int id, 
				       const char* fn, LoadMethod lm)
{
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageFitsSShare(this, type, hdr, id, fn);
  setScanModeIncr(lm);
  loadMosaic(SSHARE,fn,img);
}

void FrameBase::loadMosaicWCSSocketCmd(CoordSystem sys, int s, const char* fn)
{
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageFitsSocket(this, s, fn, FitsFile::FLUSH);
  loadMosaic(SOCKET,fn,img);
}

void FrameBase::loadMosaicWCSSocketGZCmd(CoordSystem sys, int s, const char* fn)
{
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageFitsSocketGZ(this, s, fn, FitsFile::FLUSH);
  loadMosaic(SOCKETGZ,fn,img);
}

void FrameBase::loadMosaicWCSVarCmd(CoordSystem sys, const char* ch, 
				    const char* fn, LoadMethod lm)
{
  *currentMosaic = WCSMOSAIC;
  *currentMosaicSystem = sys;
  FitsImage* img = new FitsImageFitsVar(this, interp, ch, fn);
  setScanModeIncr(lm);
  loadMosaic(VAR,fn,img);
}

// *** Mosaic Image WFPC2 ***

void FrameBase::loadMosaicImageWFPC2AllocCmd(const char* fn)
{
  unloadFits();
  *currentMosaic = WFPC2;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageFitsAlloc(this, fn, FitsFile::NOFLUSH);
  loadMosaicWFPC2(ALLOC,fn,img);
}

void FrameBase::loadMosaicImageWFPC2AllocGZCmd(const char* fn)
{
  unloadFits();
  *currentMosaic = WFPC2;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageFitsAllocGZ(this, fn, FitsFile::NOFLUSH);
  loadMosaicWFPC2(ALLOCGZ,fn,img);
}

void FrameBase::loadMosaicImageWFPC2ChannelCmd(const char* ch, const char* fn)
{
  unloadFits();
  *currentMosaic = WFPC2;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageFitsChannel(this, interp, ch, fn, 
					    FitsFile::NOFLUSH);
  loadMosaicWFPC2(CHANNEL,fn,img);
}

void FrameBase::loadMosaicImageWFPC2MMapCmd(const char* fn, LoadMethod lm)
{
  unloadFits();
  *currentMosaic = WFPC2;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageFitsMMap(this, fn);
  setScanModeIncr(lm);
  loadMosaicWFPC2(MMAP,fn,img);
}

void FrameBase::loadMosaicImageWFPC2MMapIncrCmd(const char* fn, LoadMethod lm)
{
  unloadFits();
  *currentMosaic = WFPC2;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageFitsMMapIncr(this, fn);
  setScanModeIncr(lm);
  loadMosaicWFPC2(MMAPINCR,fn,img);
}

void FrameBase::loadMosaicImageWFPC2ShareCmd(ShmType type, int id,
					const char* fn, LoadMethod lm)
{
  unloadFits();
  *currentMosaic = WFPC2;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageFitsShare(this, type, id, fn);
  setScanModeIncr(lm);
  loadMosaicWFPC2(SHARE,fn,img);
}

void FrameBase::loadMosaicImageWFPC2SocketCmd(int s, const char* fn)
{
  unloadFits();
  *currentMosaic = WFPC2;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageFitsSocket(this, s, fn, FitsFile::NOFLUSH);
  loadMosaicWFPC2(SOCKET,fn,img);
}

void FrameBase::loadMosaicImageWFPC2SocketGZCmd(int s, const char* fn)
{
  unloadFits();
  *currentMosaic = WFPC2;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageFitsSocketGZ(this, s, fn, FitsFile::NOFLUSH);
  loadMosaicWFPC2(SOCKETGZ,fn,img);
}

void FrameBase::loadMosaicImageWFPC2VarCmd(const char* ch, const char* fn,
				      LoadMethod lm)
{
  unloadFits();
  *currentMosaic = WFPC2;
  *currentMosaicSystem = WCS;
  FitsImage* img = new FitsImageFitsVar(this, interp, ch, fn);
  setScanModeIncr(lm);
  loadMosaicWFPC2(VAR,fn,img);
}

// ***

void FrameBase::load(MemType which, const char* fn, FitsImage* img)
{
  if (!img || !img->isValid()) {
    if (img)
      delete img;
    
    reset();
    resetClip(0);
    update(MATRIX);
    
    Tcl_AppendResult(interp, "Unable to load fits file", NULL);
    result = TCL_ERROR;
    return;
  }

  // Its legal, save it
  slice = 1;
  *channelFits = img;
  *currentMosaicCount = 1;
  *currentSliceCount = 1;
  setCurrentFits(img); // the first is our 'current'

  // get the rest
  // check to see if img is really a binn'd data cube
  if ((*channelFits)->isTable())
    which = HIST;

  FitsImage* ptr = *channelFits;
  for (int i=1; i<(*channelFits)->depth(); i++) {
    FitsImage* next = NULL;
    switch (which) {
    case ALLOC:
      next = new FitsImageFitsNextAlloc(this, fn, ptr->fitsFile());
      break;
    case ALLOCGZ:
      next = new FitsImageFitsNextAllocGZ(this, fn, ptr->fitsFile());
      break;
    case CHANNEL:
      next = new FitsImageFitsNextChannel(this, fn, ptr->fitsFile());
      break;
    case MMAP:
      next = new FitsImageFitsNextMMap(this, fn, ptr->fitsFile());
      break;
    case SMMAP:
      next = new FitsImageFitsNextSMMap(this, fn, ptr->fitsFile());
      break;
    case MMAPINCR:
      next = new FitsImageFitsNextMMapIncr(this, fn, ptr->fitsFile());
      break;
    case SHARE:
      next = new FitsImageFitsNextShare(this, fn, ptr->fitsFile());
      break;
    case SSHARE:
      next = new FitsImageFitsNextSShare(this, fn, ptr->fitsFile());
      break;
    case SOCKET:
      next = new FitsImageFitsNextSocket(this, fn, ptr->fitsFile());
      break;
    case SOCKETGZ:
      next = new FitsImageFitsNextSocketGZ(this, fn, ptr->fitsFile());
      break;
    case VAR:
      next = new FitsImageFitsNextVar(this, fn, ptr->fitsFile());
      break;
    case HIST:
      next = new FitsImageFitsNextHist(this, *channelFits, ptr->imageFile());
      break;
    }

    if (next && next->isValid()) {
	ptr->setNextSlice(next);
	ptr = next;
	(*currentSliceCount)++;
    }
    else {
      if (next)
	delete next;
      break;
    }
  }

  // check to see if img is really a binn'd data cube
  if ((*channelFits)->isTable())
    updateBinFileNames();

  // finish up
  img->close();
  loadFinish();
}

void FrameBase::loadDataCube(MemType which, const char* fn, FitsImage* img)
{
  if (!img || !img->isValid()) {
    if (img)
      delete img;
    
    reset();
    resetClip(0);
    update(MATRIX);
    
    Tcl_AppendResult(interp, "Unable to load fits file", NULL);
    result = TCL_ERROR;
    return;
  }

  // Its legal, save it
  slice = 1;
  *channelFits = img;
  *currentMosaicCount = 1;
  *currentSliceCount = 1;
  setCurrentFits(img); // the first is our 'current'

  // get the rest
  FitsImage* ptr = *channelFits;
  while (1) {
    FitsImage* next = NULL;
    switch (which) {
    case ALLOC:
      next = new FitsImageMosaicNextAlloc(this, fn, ptr->fitsFile(), 
					  FitsFile::NOFLUSH);
      break;
    case ALLOCGZ:
      next = new FitsImageMosaicNextAllocGZ(this,fn,ptr->fitsFile(), 
					  FitsFile::NOFLUSH);
      break;
    case CHANNEL:
      next = new FitsImageMosaicNextChannel(this,fn,ptr->fitsFile(), 
					  FitsFile::NOFLUSH);
      break;
    case MMAP:
      next = new FitsImageMosaicNextMMap(this, fn, ptr->fitsFile());
      break;
    case MMAPINCR:
      next = new FitsImageMosaicNextMMapIncr(this, fn, ptr->fitsFile());
      break;
    case SHARE:
      next = new FitsImageMosaicNextShare(this, fn, ptr->fitsFile());
      break;
    case SOCKET:
      next = new FitsImageMosaicNextSocket(this, fn,ptr->fitsFile(), 
					  FitsFile::FLUSH);
      break;
    case SOCKETGZ:
      next =new FitsImageMosaicNextSocketGZ(this,fn,ptr->fitsFile(), 
					  FitsFile::FLUSH);
      break;
    case VAR:
      next = new FitsImageMosaicNextVar(this, fn, ptr->fitsFile());
      break;
    }

    if (next && next->isValid() && next->isImage()) {
      ptr->setNextSlice(next);
      ptr = next;
      (*currentSliceCount)++;
    }
    else {
      if (next)
	delete next;
      break;
    }
  }

  // check to see if img is really a binn'd data cube
  if ((*channelFits)->isTable())
    updateBinFileNames();

  // finish up
  img->close();
  loadFinish();
}

void FrameBase::loadMosaicImage(MemType which, const char* fn, FitsImage* img)
{
  if (!img || !img->isValid()) {
    if (img)
      delete img;
    
    Tcl_AppendResult(interp, "Unable to load mosaic file", NULL);
    result = TCL_ERROR;
    return;
  }

  // Its legal, save it
  slice = 1;
  *channelFits = img;
  setCurrentFits(img); // the first is our 'current'
  *currentMosaicCount = 1;
  *currentSliceCount = 1;

  // get the rest of slice
  FitsImage* sptr = *channelFits;
  for (int i=1; i<(*channelFits)->depth(); i++) {
    FitsImage* next = NULL;
    switch (which) {
    case ALLOC:
      next = new FitsImageFitsNextAlloc(this, fn, sptr->fitsFile());
      break;
    case ALLOCGZ:
      next = new FitsImageFitsNextAllocGZ(this, fn, sptr->fitsFile());
      break;
    case CHANNEL:
      next = new FitsImageFitsNextChannel(this, fn, sptr->fitsFile());
      break;
    case MMAP:
      next = new FitsImageFitsNextMMap(this, fn, sptr->fitsFile());
      break;
    case SMMAP:
      next = new FitsImageFitsNextSMMap(this, fn, sptr->fitsFile());
      break;
    case MMAPINCR:
      next = new FitsImageFitsNextMMapIncr(this, fn, sptr->fitsFile());
      break;
    case SHARE:
      next = new FitsImageFitsNextShare(this, fn, sptr->fitsFile());
      break;
    case SSHARE:
      next = new FitsImageFitsNextSShare(this, fn, sptr->fitsFile());
      break;
    case SOCKET:
      next = new FitsImageFitsNextSocket(this, fn, sptr->fitsFile());
      break;
    case SOCKETGZ:
      next = new FitsImageFitsNextSocketGZ(this, fn, sptr->fitsFile());
      break;
    case VAR:
      next = new FitsImageFitsNextVar(this, fn, sptr->fitsFile());
      break;
    case HIST:
      next = new FitsImageFitsNextHist(this, *channelFits, sptr->imageFile());
      break;
    }

    if (next && next->isValid()) {
	sptr->setNextSlice(next);
	sptr = next;
	(*currentSliceCount)++;
    }
    else {
      if (next)
	delete next;
      break;
    }
  }

  // get the rest of mosaic
  FitsImage* ptr = *channelFits;
  while (1) {
    FitsImage* next = NULL;
    switch (which) {
    case ALLOC:
      next = new FitsImageMosaicNextAlloc(this, fn, ptr->fitsFile(), 
					  FitsFile::NOFLUSH);
      break;
    case ALLOCGZ:
      next = new FitsImageMosaicNextAllocGZ(this,fn,ptr->fitsFile(), 
					  FitsFile::NOFLUSH);
      break;
    case CHANNEL:
      next = new FitsImageMosaicNextChannel(this,fn,ptr->fitsFile(), 
					  FitsFile::NOFLUSH);
      break;
    case MMAP:
      next = new FitsImageMosaicNextMMap(this, fn, ptr->fitsFile());
      break;
    case MMAPINCR:
      next = new FitsImageMosaicNextMMapIncr(this, fn, ptr->fitsFile());
      break;
    case SHARE:
      next = new FitsImageMosaicNextShare(this, fn, ptr->fitsFile());
      break;
    case SOCKET:
      next = new FitsImageMosaicNextSocket(this, fn,ptr->fitsFile(), 
					  FitsFile::FLUSH);
      break;
    case SOCKETGZ:
      next =new FitsImageMosaicNextSocketGZ(this,fn,ptr->fitsFile(), 
					  FitsFile::FLUSH);
      break;
    case VAR:
      next = new FitsImageMosaicNextVar(this, fn, ptr->fitsFile());
      break;
    }

    if (next && next->isValid() && next->isImage()) {
      ptr->setNextMosaic(next);
      ptr = next;
      (*currentMosaicCount)++;

	for (int i=1; i<(*channelFits)->depth(); i++) {
	  FitsImage* snext = NULL;
	  switch (which) {
	  case ALLOC:
	    snext = new FitsImageFitsNextAlloc(this, fn, next->fitsFile());
	    break;
	  case ALLOCGZ:
	    snext = new FitsImageFitsNextAllocGZ(this, fn,next->fitsFile());
	    break;
	  case CHANNEL:
	    snext = new FitsImageFitsNextChannel(this, fn,next->fitsFile());
	    break;
	  case MMAP:
	    snext = new FitsImageFitsNextMMap(this, fn, next->fitsFile());
	    break;
	  case SMMAP:
	    snext = new FitsImageFitsNextSMMap(this, fn, next->fitsFile());
	    break;
	  case MMAPINCR:
	    snext = new FitsImageFitsNextMMapIncr(this,fn,next->fitsFile());
	    break;
	  case SHARE:
	    snext = new FitsImageFitsNextShare(this, fn, next->fitsFile());
	    break;
	  case SSHARE:
	    snext = new FitsImageFitsNextSShare(this, fn, next->fitsFile());
	    break;
	  case SOCKET:
	    snext = new FitsImageFitsNextSocket(this, fn, next->fitsFile());
	    break;
	  case SOCKETGZ:
	    snext = new FitsImageFitsNextSocketGZ(this,fn,next->fitsFile());
	    break;
	  case VAR:
	    snext = new FitsImageFitsNextVar(this, fn, next->fitsFile());
	    break;
	  case HIST:
	    snext = new FitsImageFitsNextHist(this,*channelFits,
					      next->imageFile());
	    break;
	  }

	  if (snext && snext->isValid()) {
	    next->setNextSlice(snext);
	    next = snext;
	  }
	  else {
	    if (snext)
	      delete snext;
	    break;
	  }
	}
    }
    else {
      if (next)
	delete next;
      break;
    }
  }
  
  // finish up
  img->close();
  loadFinish();
}

void FrameBase::loadMosaicImageNext(MemType which, const char* fn, 
				    FitsImage* img)
{
  if (!img || !img->isValid()) {
    if (img)
      delete img;
    
    Tcl_AppendResult(interp, "Unable to load mosaic file", NULL);
    result = TCL_ERROR;
    return;
  }

  // find last loaded
  FitsImage* ptr = *channelFits;
  while (ptr && ptr->nextMosaic())
    ptr = ptr->nextMosaic();

  // add it
  ptr->setNextMosaic(img);
  ptr = img;
  (*currentMosaicCount)++;

  // find rest
  while (1) {
    FitsImage* next = NULL;
    switch (which) {
    case ALLOC:
      next = new FitsImageMosaicNextAlloc(this, fn, ptr->fitsFile(), 
					  FitsFile::NOFLUSH);
      break;
    case ALLOCGZ:
      next = new FitsImageMosaicNextAllocGZ(this,fn,ptr->fitsFile(), 
					  FitsFile::NOFLUSH);
      break;
    case CHANNEL:
      next = new FitsImageMosaicNextChannel(this,fn,ptr->fitsFile(), 
					  FitsFile::NOFLUSH);
      break;
    case MMAP:
      next = new FitsImageMosaicNextMMap(this, fn, ptr->fitsFile());
      break;
    case MMAPINCR:
      next = new FitsImageMosaicNextMMapIncr(this, fn, ptr->fitsFile());
      break;
    case SHARE:
      next = new FitsImageMosaicNextShare(this, fn, ptr->fitsFile());
      break;
    case SOCKET:
      next = new FitsImageMosaicNextSocket(this, fn,ptr->fitsFile(), 
					  FitsFile::FLUSH);
      break;
    case SOCKETGZ:
      next =new FitsImageMosaicNextSocketGZ(this,fn,ptr->fitsFile(), 
					  FitsFile::FLUSH);
      break;
    case VAR:
      next = new FitsImageMosaicNextVar(this, fn, ptr->fitsFile());
      break;
    }

    if (next && next->isValid()) {
      if (next->isImage()) {
	ptr->setNextMosaic(next);
	ptr = next;
	(*currentMosaicCount)++;
      }
      else
	if (next)
	  delete next;
    }
    else {
      if (next)
	delete next;
      break;
    }
  }

  // finish up
  img->close();
  loadFinish();
}

void FrameBase::loadMosaic(MemType which, const char* fn, FitsImage* img)
{
  if (!img || !img->isValid()) {
    if (img)
      delete img;
    
    Tcl_AppendResult(interp, "Unable to load mosaic file", NULL);
    result = TCL_ERROR;
    return;
  }

  if (*channelFits) {
    FitsImage* ptr = *channelFits;
    while (ptr && ptr->nextMosaic())
      ptr = ptr->nextMosaic();
    ptr->setNextMosaic(img);
    (*currentMosaicCount)++;
  }
  else {
    *channelFits = img;
    setCurrentFits(img); // the first is our 'current'
    *currentMosaicCount = 1;
    *currentSliceCount = img->depth();
  }

  // get the rest of slice
  FitsImage* sptr = img;
  for (int i=1; i<*currentSliceCount; i++) {
    FitsImage* next = NULL;
    switch (which) {
    case ALLOC:
      next = new FitsImageFitsNextAlloc(this, fn, sptr->fitsFile());
      break;
    case ALLOCGZ:
      next = new FitsImageFitsNextAllocGZ(this, fn, sptr->fitsFile());
      break;
    case CHANNEL:
      next = new FitsImageFitsNextChannel(this, fn, sptr->fitsFile());
      break;
    case MMAP:
      next = new FitsImageFitsNextMMap(this, fn, sptr->fitsFile());
      break;
    case SMMAP:
      next = new FitsImageFitsNextSMMap(this, fn, sptr->fitsFile());
      break;
    case MMAPINCR:
      next = new FitsImageFitsNextMMapIncr(this, fn, sptr->fitsFile());
      break;
    case SHARE:
      next = new FitsImageFitsNextShare(this, fn, sptr->fitsFile());
      break;
    case SSHARE:
      next = new FitsImageFitsNextSShare(this, fn, sptr->fitsFile());
      break;
    case SOCKET:
      next = new FitsImageFitsNextSocket(this, fn, sptr->fitsFile());
      break;
    case SOCKETGZ:
      next = new FitsImageFitsNextSocketGZ(this, fn, sptr->fitsFile());
      break;
    case VAR:
      next = new FitsImageFitsNextVar(this, fn, sptr->fitsFile());
      break;
    case HIST:
      next = new FitsImageFitsNextHist(this, *channelFits, sptr->imageFile());
      break;
    }

    if (next && next->isValid()) {
	sptr->setNextSlice(next);
	sptr = next;
    }
    else {
      if (next)
	delete next;
      break;
    }
  }

  // finish up
  img->close();
  loadFinish();
}

void FrameBase::loadMosaicWFPC2(MemType which, const char* fn, FitsImage* img)
{
  // Its legal, save it
  slice = 1;
  *channelFits = img; // the first is our 'current'
  setCurrentFits(img);
  *currentMosaicCount = 1;
  *currentSliceCount = 1;

  if (!img || !img->isValid() || !img->isImage())
    goto error;

  // get the rest
  {
    FitsImage* ptr = *channelFits;
    for (int i=1; i<img->depth(); i++) {
      FitsImage* next = NULL;
      switch (which) {
      case ALLOC:
	next = new FitsImageFitsNextAlloc(this, fn, ptr->fitsFile());
	break;
      case ALLOCGZ:
	next = new FitsImageFitsNextAllocGZ(this, fn, ptr->fitsFile());
	break;
      case CHANNEL:
	next = new FitsImageFitsNextChannel(this, fn, ptr->fitsFile());
	break;
      case MMAP:
	next = new FitsImageFitsNextMMap(this, fn, ptr->fitsFile());
	break;
      case MMAPINCR:
	next = new FitsImageFitsNextMMapIncr(this, fn, ptr->fitsFile());
	break;
      case SHARE:
	next = new FitsImageFitsNextShare(this, fn, ptr->fitsFile());
	break;
      case SOCKET:
	next = new FitsImageFitsNextSocket(this, fn, ptr->fitsFile());
	break;
      case SOCKETGZ:
	next = new FitsImageFitsNextSocketGZ(this, fn, ptr->fitsFile());
	break;
      case VAR:
	next = new FitsImageFitsNextVar(this, fn, ptr->fitsFile());
	break;
      }

      if (next && next->isValid()) {
	ptr->setNextMosaic(next);
	ptr = next;
	(*currentMosaicCount)++;
      }
      else {
	if (next)
	  delete next;
	break;
      }
    }
  }

  // ok, do we have 4 images?
  if (*currentMosaicCount != 4)
    goto error;

  // now, find WCS table
  FitsFile* table;
  switch (which) {
  case ALLOC:
    table = new FitsMosaicNextAlloc((*channelFits)->fitsFile(), 
					  FitsFile::NOFLUSH);
    break;
  case ALLOCGZ:
    table = new FitsMosaicNextAllocGZ((*channelFits)->fitsFile(), 
					  FitsFile::NOFLUSH);
    break;
  case CHANNEL:
    table = new FitsMosaicNextChannel((*channelFits)->fitsFile(), 
					  FitsFile::NOFLUSH);
    break;
  case MMAP:
    table = new FitsMosaicNextMMap((*channelFits)->fitsFile());
    break;
  case MMAPINCR:
    table = new FitsMosaicNextMMapIncr((*channelFits)->fitsFile());
    break;
  case SHARE:
    table = new FitsMosaicNextShare((*channelFits)->fitsFile());
    break;
  case SOCKET:
    table = new FitsMosaicNextSocket((*channelFits)->fitsFile(), 
					  FitsFile::FLUSH);
    break;
  case SOCKETGZ:
    table = new FitsMosaicNextSocketGZ((*channelFits)->fitsFile(), 
					  FitsFile::FLUSH);
    break;
  case VAR:
    table = new FitsMosaicNextVar((*channelFits)->fitsFile());
    break;
  }

  if (!table || !table->isValid() || !table->isTable()) {
    if (table)
      delete table;
    goto error;
  }

  // read WCS from table
  {
    FitsHead* th = table->head();
    if (th->naxes() != 2) {
      if (table)
	delete table;
      goto error;
    }

    FitsTableHDU* thdu = (FitsTableHDU*)th->hdu();

    FitsColumn* crval1 = thdu->find("CRVAL1");
    FitsColumn* crval2 = thdu->find("CRVAL2");
    FitsColumn* crpix1 = thdu->find("CRPIX1");
    FitsColumn* crpix2 = thdu->find("CRPIX2");

    FitsColumn* cd1_1 = thdu->find("CD1_1");
    FitsColumn* cd1_2 = thdu->find("CD1_2");
    FitsColumn* cd2_1 = thdu->find("CD2_1");
    FitsColumn* cd2_2 = thdu->find("CD2_2");

    FitsColumn* ctype1 = thdu->find("CTYPE1");
    FitsColumn* ctype2 = thdu->find("CTYPE2");

    char* tptr = (char*)table->data();
    int rows = thdu->rows();
    int rowlen = thdu->width();

    if (rows != 4)
      goto error;

    FitsImage* ptr = *channelFits;

    // reset count for processKeyWords()
    *currentMosaicCount = 0;
    *currentSliceCount = 0;
    for (int i=0; i<rows; i++, tptr+=rowlen) {
      istringstream istr(ios::in|ios::out);
      ostream ostr(istr.rdbuf());
      ostr << "CRVAL1 = " << crval1->str(tptr) << endl
	   << "CRVAL2 = " << crval2->str(tptr) << endl
	   << "CRPIX1 = " << crpix1->str(tptr) << endl
	   << "CRPIX2 = " << crpix2->str(tptr) << endl
	   << "CD1_1  = " <<  cd1_1->str(tptr) << endl
	   << "CD1_2  = " <<  cd1_2->str(tptr) << endl
	   << "CD2_1  = " <<  cd2_1->str(tptr) << endl
	   << "CD2_2  = " <<  cd2_2->str(tptr) << endl
	   << "CTYPE1 = " << '\'' << ctype1->str(tptr) << '\'' << endl
	   << "CTYPE2 = " << '\'' << ctype2->str(tptr) << '\'' << endl
	   << ends;
      FitsHead* r = parseWCS(istr);

      // fix fitsimage params
      ptr->replaceWCS(r);
      ptr->processKeywordsWCS();

      delete r;
      ptr = ptr->nextMosaic();
      (*currentMosaicCount)++;
    }
  }

  if (table)
    delete table;

  // finish up
  img->close();
  loadFinish();
  return;

 error:
  unloadFits();

  reset();
  resetClip(0);
  update(MATRIX);
    
  Tcl_AppendResult(interp, "Unable to load mosaic file", NULL);
  result = TCL_ERROR;
  return;
}

void FrameBase::loadFinish()
{
  // fill out rest of mosaic/slice ptr matrix}
  if (*currentMosaicCount>1 && *currentSliceCount>1) {
    FitsImage* ptr = *channelFits;
    while (ptr && ptr->nextMosaic()) {
      int jj=0;
      FitsImage* sptr = ptr;
      while (sptr) {
	if (sptr->nextMosaic() == NULL) {
	  // ok, let's figure out next inline
	  FitsImage* mptr = ptr->nextMosaic();
	  for (int nn=0; nn<jj; nn++)
	    mptr = mptr->nextSlice();
	  sptr->setNextMosaic(mptr);
	}
	jj++;
	sptr = sptr->nextSlice();
      }
      ptr = ptr->nextMosaic();
    }
  }

  resetValues();
  resetClip(0);

  update(MATRIX);
}
