// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "point.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

#define POINTSIZE 11

Point::Point(const Point& a) : Marker(a)
{
  strncpy(pointtype,a.pointtype,16);
}

Point::Point(FrameBase* p, const Vector& ctr, 
	     const char* clr, int wth, const char* fnt, const char* txt,
	     unsigned short prop, const char* cmt,
	     const List<Tag>& tg, const List<CallBack>& cb)
  : Marker(p, ctr, 0, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  pointtype[0] = '\0';

  handle = new Vector[4];
  numHandle = 4;

  updateBBox();
}

void Point::updateBBox()
{
  // bound marker

  Vector h = Vector(POINTSIZE, POINTSIZE)/2;
  Vector c = center * parent->refToCanvas;
  bbox = BBox(c-h,c+h);

  // generate handles

  handle[0] = bbox.ll;
  handle[1] = bbox.lr();
  handle[2] = bbox.ur;
  handle[3] = bbox.ul();

  // make room for handles

  bbox.expand(3);

  // calculate overall bbox

  calcAllBBox();
}

void Point::render(Drawable drawable, const Matrix& mx, RenderMode mode)
{
  // remember, bbox is in canvas coords
  if (!(properties & INCLUDE)) {
    BBox b(bbox);
    b.shrink(3);
    Vector ll = (b.ll * parent->getCanvasToRef() * mx).round();
    Vector ur = (b.ur * parent->getCanvasToRef() * mx).round();

    if (mode==SRC)
      XSetForeground(display, gc, parent->getRedColor());

    XDRAWLINE(display, drawable, gc, (int)ll[0], (int)ll[1], 
	      (int)ur[0], (int)ur[1]);    
  }
}

void Point::ps(int mode)
{
  // remember, bbox is in canvas coords
  if (!(properties & INCLUDE)) {
    psColor(mode, "red");

    BBox b(bbox);
    b.shrink(3);

    ostringstream str;
    str << "newpath " 
	<< b.ll.TkCanvasPs(parent->canvas) << "moveto"
	<< b.ur.TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }
}

void Point::setLineDash()
{
  char dlist[] = {4,2};
  XSetLineAttributes(display, gc, lineWidth, 
		     LineOnOffDash, CapButt, JoinMiter);
  XSetDashes(display, gc, 0, dlist, 2);
}

void Point::psLineDash()
{
  ostringstream str;
  str << lineWidth << " setlinewidth" << endl;
  str << "[4 2] 0 setdash" << endl;
  str << ends;
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
}

CirclePoint::CirclePoint(const CirclePoint& a) : Point(a) {}

CirclePoint::CirclePoint(FrameBase* p, const Vector& ctr, 
			 const char* clr, int wth, 
			 const char* fnt, const char* txt,
			 unsigned short prop, const char* cmt,
			 const List<Tag>& tg, const List<CallBack>& cb)
  : Point(p, ctr, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  strcpy(type,"circle point");
  strcpy(pointtype,"circle");
}

void CirclePoint::render(Drawable drawable, const Matrix& mx, RenderMode mode)
{
  setGC(mode);

  Vector v = (center * mx * Translate(-Vector(POINTSIZE,POINTSIZE)/2)).round();
  XDRAWARC(display, drawable, gc, (int)v[0], (int)v[1], 
	   POINTSIZE, POINTSIZE, 0, 360*64);

  Point::render(drawable, mx, mode);
}

void CirclePoint::ps(int mode)
{
  Marker::ps(mode);

  Vector cc = center  * parent->refToCanvas;
  ostringstream str;
  str << "newpath " << cc.TkCanvasPs(parent->canvas) << POINTSIZE/2.
      << " 0 360 arc stroke" << endl << ends;
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);

  Point::ps(mode);
}

BoxPoint::BoxPoint(const BoxPoint& a) : Point(a) {}

BoxPoint::BoxPoint(FrameBase* p, const Vector& ctr, 
		   const char* clr, int wth, 
		   const char* fnt, const char* txt,
		   unsigned short prop, const char* cmt,
		   const List<Tag>& tg, const List<CallBack>& cb)
  : Point(p, ctr, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  strcpy(type, "box point");
  strcpy(pointtype, "box");
}

void BoxPoint::render(Drawable drawable, const Matrix& mx, RenderMode mode)
{
  setGC(mode);

  Vector v = (center * mx * Translate(-Vector(POINTSIZE,POINTSIZE)/2)).round();
  XDRAWRECTANGLE(display, drawable, gc, (int)v[0], (int)v[1], 
		 POINTSIZE, POINTSIZE);

  Point::render(drawable, mx, mode);
}

void BoxPoint::ps(int mode)
{
  Marker::ps(mode);

  Vector v = center*parent->refToCanvas - Vector(POINTSIZE,-POINTSIZE)/2;
  ostringstream str;
  str << "newpath " << v.TkCanvasPs(parent->canvas) 
      << Vector(POINTSIZE,POINTSIZE)
      << " rectstroke" << endl << ends;
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);

  Point::ps(mode);
}

DiamondPoint::DiamondPoint(const DiamondPoint& a) : Point(a) {}

DiamondPoint::DiamondPoint(FrameBase* p, const Vector& ctr, 
			   const char* clr, int wth, 
			   const char* fnt, const char* txt,
			   unsigned short prop, const char* cmt,
			   const List<Tag>& tg, const List<CallBack>& cb)
  : Point(p, ctr, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  strcpy(type, "diamond point");
  strcpy(pointtype, "diamond");
}

void DiamondPoint::render(Drawable drawable, const Matrix& mx, RenderMode mode)

{
  setGC(mode);

  int l = POINTSIZE-1;
  Vector cc = center * mx;
  Vector v1 = (cc - Vector(l,0)/2).round();
  Vector v2 = (cc - Vector(0,l)/2).round();
  Vector v3 = (cc + Vector(l,0)/2).round();
  Vector v4 = (cc + Vector(0,l)/2).round();

  XPoint p[5];
  p[0].x = (short)v1[0];
  p[0].y = (short)v1[1];
  p[1].x = (short)v2[0];
  p[1].y = (short)v2[1];
  p[2].x = (short)v3[0];
  p[2].y = (short)v3[1];
  p[3].x = (short)v4[0];
  p[3].y = (short)v4[1];
  p[4] = p[0];

  XDRAWLINES(display, drawable, gc, p, 5, CoordModeOrigin);

  Point::render(drawable, mx, mode);
}

void DiamondPoint::ps(int mode)
{
  Marker::ps(mode);

  Matrix m = parent->refToCanvas;

  Vector v1 = center*m - Vector(POINTSIZE, 0)/2;
  Vector v2 = center*m - Vector(0, POINTSIZE)/2;
  Vector v3 = center*m + Vector(POINTSIZE, 0)/2;
  Vector v4 = center*m + Vector(0, POINTSIZE)/2;

  ostringstream str;
  str << "newpath " << v1.TkCanvasPs(parent->canvas) << " moveto" << endl
      << v2.TkCanvasPs(parent->canvas) << " lineto" << endl
      << v3.TkCanvasPs(parent->canvas) << " lineto" << endl
      << v4.TkCanvasPs(parent->canvas) << " lineto" << endl
      << "closepath stroke" << endl
      << ends;
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);

  Point::ps(mode);
}

CrossPoint::CrossPoint(const CrossPoint& a) : Point(a) {}

CrossPoint::CrossPoint(FrameBase* p, const Vector& ctr, 
		       const char* clr, int wth, 
		       const char* fnt, const char* txt,
		       unsigned short prop, const char* cmt,
		       const List<Tag>& tg, const List<CallBack>& cb)
  : Point(p, ctr, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  strcpy(type, "cross point");
  strcpy(pointtype, "cross");
}

void CrossPoint::render(Drawable drawable, const Matrix& mx, RenderMode mode)
{
  setGC(mode);

  Vector cc = center * mx;
  Vector v1 = (cc - Vector(POINTSIZE,0)/2).round();
  Vector v2 = (cc + Vector(POINTSIZE,0)/2).round();
  Vector v3 = (cc - Vector(0,POINTSIZE)/2).round();
  Vector v4 = (cc + Vector(0,POINTSIZE)/2).round();

  XDRAWLINE(display, drawable, gc, (int)v1[0], (int)v1[1], 
	    (int)v2[0], (int)v2[1]);
  XDRAWLINE(display, drawable, gc, (int)v3[0], (int)v3[1], 
	    (int)v4[0], (int)v4[1]);

  Point::render(drawable, mx,  mode);
}

void CrossPoint::ps(int mode)
{
  Marker::ps(mode);

  Matrix m = parent->refToCanvas;

  Vector v1 = center*m - Vector(POINTSIZE,0)/2;
  Vector v2 = center*m + Vector(POINTSIZE,0)/2;
  Vector v3 = center*m - Vector(0,POINTSIZE)/2;
  Vector v4 = center*m + Vector(0,POINTSIZE)/2;

  ostringstream str;
  str << "newpath " << v1.TkCanvasPs(parent->canvas) << " moveto" << endl
      << v2.TkCanvasPs(parent->canvas) << " lineto stroke" << endl
      << "newpath " << v3.TkCanvasPs(parent->canvas) << " moveto" << endl
      << v4.TkCanvasPs(parent->canvas) << " lineto stroke" << endl
      << ends;
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);

  Point::ps(mode);
}

ExPoint::ExPoint(const ExPoint& a) : Point(a) {}

ExPoint::ExPoint(FrameBase* p, const Vector& ctr, 
		 const char* clr, int wth, 
		 const char* fnt, const char* txt,
		 unsigned short prop, const char* cmt,
		 const List<Tag>& tg, const List<CallBack>& cb)
  : Point(p, ctr, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  strcpy(type, "x point");
  strcpy(pointtype, "x");
}

void ExPoint::render(Drawable drawable, const Matrix& mx, RenderMode mode)
{
  setGC(mode);

  Vector cc = center * mx;
  Vector v1 = (cc + Vector(-POINTSIZE,-POINTSIZE)/2).round();
  Vector v2 = (cc + Vector( POINTSIZE,-POINTSIZE)/2).round();
  Vector v3 = (cc + Vector( POINTSIZE, POINTSIZE)/2).round();
  Vector v4 = (cc + Vector(-POINTSIZE, POINTSIZE)/2).round();

  XDRAWLINE(display, drawable, gc, (int)v1[0], (int)v1[1], 
	    (int)v3[0], (int)v3[1]);
  XDRAWLINE(display, drawable, gc, (int)v2[0], (int)v2[1], 
	    (int)v4[0], (int)v4[1]);

  Point::render(drawable, mx, mode);
}

void ExPoint::ps(int mode)
{
  Marker::ps(mode);

  Matrix m = parent->refToCanvas;

  Vector v1 = center*m + Vector(-POINTSIZE, -POINTSIZE)/2;
  Vector v2 = center*m + Vector(POINTSIZE, -POINTSIZE)/2;
  Vector v3 = center*m + Vector(POINTSIZE, POINTSIZE)/2;
  Vector v4 = center*m + Vector(-POINTSIZE, POINTSIZE)/2;

  ostringstream str;
  str << "newpath " << v1.TkCanvasPs(parent->canvas) << " moveto" << endl
      << v3.TkCanvasPs(parent->canvas) << " lineto stroke" << endl
      << "newpath " << v2.TkCanvasPs(parent->canvas) << " moveto" << endl
      << v4.TkCanvasPs(parent->canvas) << " lineto stroke" << endl
      << ends;
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);

  Point::ps(mode);
}

ArrowPoint::ArrowPoint(const ArrowPoint& a) : Point(a) {}

ArrowPoint::ArrowPoint(FrameBase* p, const Vector& ctr, 
		       const char* clr, int wth, 
		       const char* fnt, const char* txt,
		       unsigned short prop, const char* cmt,
		       const List<Tag>& tg, const List<CallBack>& cb)
  : Point(p, ctr, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  strcpy(type, "arrow point");
  strcpy(pointtype, "arrow");
}

void ArrowPoint::render(Drawable drawable, const Matrix& mx, RenderMode mode)
{
  setGC(mode);

  Vector cc = center * mx;
  Vector v1 = (cc + Vector(-POINTSIZE, 0)/2).round();
  Vector v2 = (cc + Vector(0, -POINTSIZE)/2).round();
  Vector v3 = (cc - Vector(POINTSIZE, POINTSIZE)/2).round();
  Vector v4 = cc.round();

  XDRAWLINE(display, drawable, gc, (int)v1[0], (int)v1[1], 
	    (int)v4[0], (int)v4[1]);
  XDRAWLINE(display, drawable, gc, (int)v2[0], (int)v2[1], 
	    (int)v4[0], (int)v4[1]);
  XDRAWLINE(display, drawable, gc, (int)v3[0], (int)v3[1], 
	    (int)v4[0], (int)v4[1]);

  Point::render(drawable, mx, mode);
}

void ArrowPoint::ps(int mode)
{
  Marker::ps(mode);

  Matrix m = parent->refToCanvas;

  Vector v1 = center*m + Vector(-POINTSIZE, 0)/2;
  Vector v2 = center*m + Vector(0, -POINTSIZE)/2;
  Vector v3 = center*m - Vector(POINTSIZE, POINTSIZE)/2;
  Vector v4 = center*m;

  ostringstream str;
  str << "newpath " << v1.TkCanvasPs(parent->canvas) << " moveto" << endl
      << v4.TkCanvasPs(parent->canvas) << " lineto stroke" << endl
      << "newpath " << v2.TkCanvasPs(parent->canvas) << " moveto" << endl
      << v4.TkCanvasPs(parent->canvas) << " lineto stroke" << endl
      << "newpath " << v3.TkCanvasPs(parent->canvas) << " moveto" << endl
      << v4.TkCanvasPs(parent->canvas) << " lineto stroke" << endl
      << ends;
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);

  Point::ps(mode);
}

BoxCirclePoint::BoxCirclePoint(const BoxCirclePoint& a) : Point(a) {}

BoxCirclePoint::BoxCirclePoint(FrameBase* p, const Vector& ctr, 
			       const char* clr, int wth, 
			       const char* fnt, const char* txt,
			       unsigned short prop, const char* cmt,
			       const List<Tag>& tg, const List<CallBack>& cb)
  : Point(p, ctr, clr, wth, fnt, txt, prop, cmt, tg, cb)
{
  strcpy(type, "boxcircle point");
  strcpy(pointtype, "boxcircle");
}

void BoxCirclePoint::render(Drawable drawable, const Matrix& m,RenderMode mode)
{
  setGC(mode);

  Vector cc = center * m;
  Vector v1 = (cc * Translate(-Vector(POINTSIZE-2,POINTSIZE-2)/2)).round();
  Vector v2 = (cc * Translate(-Vector(POINTSIZE,POINTSIZE)/2)).round();

  XDRAWARC(display, drawable, gc, (int)v1[0], (int)v1[1], 
	   POINTSIZE-2, POINTSIZE-2, 0, 360*64);
  XDRAWRECTANGLE(display, drawable, gc, (int)v2[0], (int)v2[1], 
		 POINTSIZE, POINTSIZE);

  Point::render(drawable, m, mode);
}

void BoxCirclePoint::ps(int mode)
{
  Marker::ps(mode);

  Vector v1 = center*parent->refToCanvas;
  Vector v2 = center*parent->refToCanvas - Vector(POINTSIZE,-POINTSIZE)/2;

  ostringstream str;
  str << "newpath " << v1.TkCanvasPs(parent->canvas) << (POINTSIZE-2)/2.
      << " 0 360 arc stroke" << endl
      << "newpath " << v2.TkCanvasPs(parent->canvas) 
      << Vector(POINTSIZE,POINTSIZE) 
      << " rectstroke" << endl
      << ends;
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);

  Point::ps(mode);
}

// list

void Point::list(ostream& str, CoordSystem sys, SkyFrame sky, 
		 SkyFormat format, int conj, int strip)
{
  FitsImage* ptr = parent->findFits(center);
  listPre(str, sys, sky, ptr, strip, 0);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,sys);
      str << "point" << '(' << setprecision(8) << v[0] << ',' << v[1] << ')';
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	  {
	    Vector v = ptr->mapFromRef(center,sys,sky);
	    str << "point" << '(' << setprecision(8) 
		<< v[0] << ',' << v[1] << ')';
	  }
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    ptr->mapFromRef(center,sys,sky,format,buf,64);
	    char ra[16];
	    char dec[16];
	    string x(buf);
	    istringstream wcs(x);
	    wcs >> ra >> dec;
	    str << "point" << '(' << ra << ',' << dec << ')';
	  }
	  break;
	}
      }
      else {
	Vector v = ptr->mapFromRef(center,sys);
	str << "point" << '(' << setprecision(8) << v[0] << ',' << v[1] << ')';
      }
    }
  }

  listPost(str, conj, strip);
}

void Point::listPost(ostream& str, int conj, int strip)
{
  // no props for semicolons
  if (!strip) {
    if (conj)
      str << " ||";

    str << " # point=" << pointtype;
    listProperties(str, 0);
  }
  else {
    if (conj)
      str << "||";
    else
      str << ';';
  }
}

void Point::listCiao(ostream& str, CoordSystem sys, int conj, int strip)
{
  FitsImage* ptr = parent->findFits(1);
  listCiaoPre(str);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,PHYSICAL);
      str << "point(" << setprecision(8) << v[0] << ',' << v[1] << ')';
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      char buf[64];
      ptr->mapFromRef(center,sys,FK5,SEXAGESIMAL,buf,64);
      char ra[16];
      char dec[16];
      string x(buf);
      istringstream wcs(x);
      wcs >> ra >> dec;
      str << "point" << '(' << ra << ',' << dec << ')';
    }
    break;
  }

  listCiaoPost(str, conj, strip);
}

void Point::listPros(ostream& str, CoordSystem sys, SkyFrame sky,
		     SkyFormat format, int strip)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case DETECTOR:
  case AMPLIFIER:
    sys = IMAGE;
  case PHYSICAL:
    {
      listProsCoordSystem(str,sys,sky);
      str << "; ";
      Vector v = ptr->mapFromRef(center,sys);
      str << "point " << setprecision(8) << v;
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      listProsCoordSystem(str,sys,sky);
      str << "; ";

      switch (format) {
      case DEGREES:
	{
	  Vector v = ptr->mapFromRef(center,sys,sky);
	  str << "point " << setprecision(8) << v[0] << "d " << v[1] << "d ";
	}
	break;
      case SEXAGESIMAL:
	{
	  char buf[64];
	  ptr->mapFromRef(center,sys,sky,format,buf,64);
	  char ra[16];
	  char dec[16];
	  string x(buf);
	  istringstream wcs(x);
	  wcs >> ra >> dec;
	  if (dec[0]=='+')
	    str << "point " << ra << ' ' << dec+1;
	  else
	    str << "point " << ra << ' ' << dec;
	}
	break;
      }
    }
  }

  listProsPost(str, strip);
}

void Point::listSAOtng(ostream& str, CoordSystem sys, SkyFrame sky,
		       SkyFormat format, int strip)
{
  FitsImage* ptr = parent->findFits(1);
  listSAOtngPre(str, strip);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,IMAGE);
      str << "point(" << setprecision(8) << v[0] << ',' << v[1] << ')';
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      switch (format) {
      case DEGREES:
	{
	  Vector v = ptr->mapFromRef(center,sys,sky);
	  str << "point(" << setprecision(8) << v[0] << ',' << v[1] << ')';
	}
	break;
      case SEXAGESIMAL:
	{
	  char buf[64];
	  ptr->mapFromRef(center,sys,sky,format,buf,64);
	  char ra[16];
	  char dec[16];
	  string x(buf);
	  istringstream wcs(x);
	  wcs >> ra >> dec;
	  str << "point(" << ra << ',' << dec << ')';
	}
	break;
      }
    }
  }

  listSAOtngPost(str,strip);
}

void Point::listSAOimage(ostream& str, int strip)
{
  FitsImage* ptr = parent->findFits(1);
  listSAOimagePre(str);

  // all coords are in image coords

  Vector v = ptr->mapFromRef(center,IMAGE);
  str << "point(" << setprecision(8) << v[0] << ',' << v[1] << ')';

  listSAOimagePost(str, strip);
}
