/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#ifndef SRKMAINWINDOW_H
#define SRKMAINWINDOW_H

/**
 * Main window; implements the menu, statusbar and main accessibility.
 * Try to keep the functionality as minimal as possible in this class.
 * The only things that are handled here are the things that belong in
 * a main window, like actions, statusbar, menu, etc.
 *
 * @author Bram Biesbrouck <b@beligum.org>
 */

#include <string>

#include <kapplication.h>
#include <kmainwindow.h>

#include "srkerror.h"

using namespace std;

#define FILE_DIALOG_FILTER_STRING "*.isd|Instrudeo files"
#define EXPORT_DIALOG_FILTER_STRING "*.flv|flv format\n*.avi|avi format\n*.mpg|mpg format\n*.asf|asf format\n*.m4v|raw MPEG4 video format\n*.mov *.mp4 *.m4a *.3gp *.3g2|QuickTime MPEG4 format\n*.mp4|mp4 format\n*.mpeg|MPEG1 System format\n*.swf|Flash format\n*.vcd|MPEG1 System format (VCD)"
#define ZOOM_OPTIONS_SIZE 7
#define ZOOM_OPTIONS {"auto", "200%", "150%", "100%", "75%", "50%","20%"}

class SRKMainView;
class KComboBox;
class KLanguageButton;
class KPushButton;

class SRKMainWindow : public KMainWindow
{
    Q_OBJECT

 public:
    //-----CONSTRUCTORS-----
    SRKMainWindow();
    
    virtual ~SRKMainWindow();

    //-----METHODS------
    SRKError::SRKErrorCode loadFile(QString fileName);

 public slots:
     /**
      * Use this function to tell the main window a valid file is opened or closed.
      *
      * @param state Open (true) or close(false).
      */
     void setFileOpened(bool state);

     /**
      * Use this function to aggregate back a popup-call to the main window.
      *
      * @param name The name of the menu in the XMLUI file.
      * @param pos The global position of the mouseclick.
      */
     void popupMenu(QString& name, QPoint& pos);

     /**
      * Following two functions can be used to control the contents of the statusbar-language-combobox
      * from classes higher in the chain. The clear-method can be used to clear the whole combobox.
      * 
      * Note: Because of a missing removeItem() method in KLanguageButton, you'll need to clear and rebuild when
      *       you want to delete an item.
      *
      * @param langCode the TwoAlpha code of the language to add/delete.
      */
     void addActiveLanguage(QString langCode);
     void clearActiveLanguages();

     /**
      * If you add a lot of languages, you don't want the canvas to be updated with every addition.
      * This method checks the currently selected language and emits the necessary signals to sync
      * the canvas's commentboxes.
      */
     void syncActiveLanguage();

 protected slots:
    //-----OVERLOADED SLOTS-----
    /**
     * Overloaded from the KMainWindow class and is called when the window is closed.
     * It cleanly closes and/or saves the current recording.
     *
     * @return If false is returned, the closing is aborted.
     */
    bool queryClose();

    //-----FILE MENU SLOTS-----
    void fileNew();

    void fileOpen();

    void fileSave();

    void fileSaveAs();

    void fileMetaInfo();

    void fileTransmitRecording();

    void fileExport();

    void fileClose();

    //-----EDIT MENU SLOTS-----
    void editAddTextBalloonComment();

    void editAddRoundedBoxComment();

    void editAddTextBalloon2DComment();

    void editDeleteComment();

    void editCommentProperties();

    void editEndCommentHere();

    //-----VNC MENU SLOTS-----
    void vncStartServer();
    void vncStopServer();

    //-----OPTIONS MENU SLOTS-----
    void optionsPreferences();

    //-----STATUSBAR SLOTS-----
    void statusZoomChanged(const QString& text);
    void statusLanguageChanged(const QString& lang);
    void statusAddLanguage();

 protected:
    //-----METHODS-----
    /**
     * Change the view zoom factor.
     *
     * @param factor The new zoom factor.
     */
    void viewChangeZoomFactor(int factor);
    
    /**
     * Initialises the different actions that will be coupled with the menu,
     * toolbar and RMB-menus.
     */
    void createActions();
    
    /**
     * Initialises and creates the widgets inside the statusbar, like the zoom combobox.
     */
    void setupStatusBar();
    
    /**
     * Checks if the current file needs to be saved and saves it if necessary.
     * Call this function before loading, closing, ... a new file.
     *
     * @return Returns false if the user cancelled, true otherwise.
     */
    bool maybeSave();
    
    /**
     * Closes the opened file without warnings (as opposed to fileClose()).
     */
    void closeFile();

    //-----VARIABLES-----
    SRKMainView* mainView;
    KComboBox* zoomCombo;
    KLanguageButton* languageCombo;
    KPushButton* addLanguageButton;
    
    bool fileOpened;
    bool newFile;
    QString lastFileName;
};

#endif
