/* config.c  --  The whole configuration in this program, if it wasn't for
                 this file, there wouldn't be much of a program.

   Copyright (C) Kenneth 'Redhead' Nielsen <kn@redhead.dk>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */


/****
 * $Id: config.c,v 1.7 2003/07/22 22:03:42 redhead Exp $
 * $Log: config.c,v $
 * Revision 1.7  2003/07/22 22:03:42  redhead
 * Cleaned up the code, which included rewriting every structure in the program.
 * Fixed some error checking, to make sure theres no undocumentet parts that might
 * produce an error.
 *
 * Revision 1.6  2003/07/18 10:58:10  redhead
 * This is going to be the 0.2.4 release, just fixed some spelling errors in teh documentation
 * fixed the bug with zombie processes of boxes, when using the boxing feature
 * fixed the long time waited sigit.rc(5) error reported by moffe
 * plus some minor changes here and there.
 *
 * Revision 1.5  2003/07/16 12:53:37  redhead
 * Added a conversion char %L to the usual conversions, this one will
 * display the load avarage on the machine.
 * Fixed some error checkign to ensure the right info will be displayed.
 *
 * Revision 1.4  2003/07/05 23:31:24  redhead
 * An update, that will automaticaly check for a working boxes program
 * if none is found, sigit will run as normal, but disable any call to
 * boxes, thus avoiding a possible segfault.
 *
 * Revision 1.3  2003/03/07 20:36:48  redhead
 * Fixed a flaw in the call to boxes, so it now takes several arguments in the
 * boxes_flags setting.
 * Added the functionality of having it giving random boxes, when the boxes_config
 * setting is pointing to the location of your boxes designs.
 *
 * Revision 1.2  2002/10/18 18:25:23  redhead
 * Added the use of boxes to create a surrounding box around the signature
 * This introduced the --with-boxes flag to configure.
 * If that flag isn't used it will default be assumed boxes isn't installed
 * Else the configure script will search for boxes, if none should be found
 * in the search path, you can use --with-boxes=/where to specify it.
 *
 ****/

#include "config.h" /* #include <every_fscking_thing> */

/* Just a small function, to print the needed copyright */

int print_copyright(void)
{
  printf(PACKAGE "-" VERSION " (" CALLSIGN ") [" __DATE__ " " __TIME__ "]\n\n"
	 "Copyright (C) 2000-2003 Kenneth 'Redhead' Nielsen <kn@redhead.dk>\n"
	 "This is free software; see the source for copying conditions.\n"
	 "There is NO warranty; not even for MERCHANTABILITY or FITNESS\n"
	 "FOR A PARTICULAR PURPOSE, to the extent permitted by law.\n"
	 "Latest version can be obtained from: <" HOMEPAGE ">\n\n");
  return SUCCESS;
}

#ifdef sigit

/*
 * Well, since we are running this proram as deamons and what not, 
 * then we need to determain, how we can accomplish an environment 
 * where theres only one instance running for every user.
 *
 * Keeping a pid-lock file for the program, will make this easier.
 * so we can read the PID of the parent thread from that, and decide 
 * to kill that process, if we should be started once more.
 *
 */

int create_pid_file()
{
  int  waiter = -1, xpid = -1, fpid = -1, pid = -1, ppid = -1;
  char host_name1[LINE_SIZE +1]; /* has enyone a hostname, thats longer than 75 chars ? */
  char host_name2[LINE_SIZE +1]; 
  char lockfile[LINE_SIZE +1];   /* should be enough for $HOME the user and PACKAGE */
  char exec_line[2*LINE_SIZE];   /* Since we like to use rsh, theres gotta be room for hostname*/
  FILE* lockfp;
  /* First find out who the user is.. */
  uid_t  original_real_uid = getuid();
  struct passwd* pw = getpwuid(original_real_uid);
  /* If you don't even exist, then why bother */
  if (!pw)
    exit(1);
  if(gethostname(host_name1, LINE_SIZE -1) < 0)
     host_name1[0] = '\0';
  else
    host_name1[LINE_SIZE] = '\0'; /* Just to make sure, if $HOSTNAME is longer than LINE_SIZE */
  /* then test if theres a file called PACKAGE-USER */
  /* if found look into it, and decide the pid of the running program */
  /* and kill it, then start normaly */
#ifdef I__HAVE__SNPRINTF
  snprintf(lockfile, LINE_SIZE, "%s/.%s-%s.lck", getenv("HOME"), PACKAGE, pw->pw_name);
#else
  sprintf(lockfile, "%s/.%s-%s.lck", getenv("HOME"), PACKAGE, pw->pw_name);
#endif
  if((lockfp = fopen(lockfile, "r")) != NULL )
    {
      fscanf(lockfp,"%s %d %d", host_name2, &pid, &ppid);
      /* Better close the file, so theres no disturbing file-pointers around */ 
      fclose(lockfp);
      unlink(lockfile);
      if(!strncmp(host_name1, host_name2, LINE_SIZE) || strlen(host_name2) < 1)
	{
	  /* The program is running on this machine, or the hostname on the orriginal 
	   * machine can't be decided, so just try and kill it 
	   */
	  if(!kill(pid, 15)) 
	    { /* should be a kill-sig thats accepted even if its a suspended process */
	      pid = -1;
	      ppid = -1;
	    }
	  /* If we're unable to kill it, it might be due to the fact that its allready dead */
	}
      else
	{
	  /* the current hostname stored in the PID file isn't the same as this machine,
	     we hope the system has rsh(1) installed, so we can kill it with no fuss */
	  switch (fpid = fork()) 
	    {
	    case -1:
	      unlink(lockfile);
	      return FAIL_FO;
	    case 0:
	      /* child */
	      if (chdir("/tmp") < 0)
		{ 
		  unlink(lockfile);
		  return FAILURE;
		}
	      if (strlen(host_name2) + 20 >= 2*LINE_SIZE) 
		{
		  unlink(lockfile);
		  return EXCEEDED;
		}
#ifdef I__HAVE__SNPRINTF
	      snprintf(exec_line, (2*LINE_SIZE -1), "rsh %s kill -15 %d", host_name2, pid);
#else
	      sprintf(exec_line, "rsh %s kill -15 %d", host_name2, pid);
#endif
	      execl(_PATH_BSHELL, _PATH_BSHELL, "-c", exec_line, NULL);
	      unlink(lockfile);
	      return FAILURE;
	      /*NOTREACHED*/
	    default:
	      /* parent */
	      break;
	    }
	  /* parent */
	  xpid = wait(&waiter);
	  if (xpid != fpid) 
	    {
	      unlink(lockfile);
	      return FAILURE;
	    }
	  if (WIFEXITED(waiter) && WEXITSTATUS(waiter))
	    {
	      unlink(lockfile);
	      return FAILURE;
	    }
	  if (WIFSIGNALED(waiter))
	    {
	      unlink(lockfile);
	      return FAILURE;
	    }
	  unlink(lockfile);
	  return SUCCESS;
	}           
    }

  /* we take a small sleep time, to make sure the NFS part accepts the unlinking */
  sleep(1);
  
  /* else start the program, and create the PID file. */
  pid = getpid();
  ppid = getppid();
  if((lockfp = fopen(lockfile, "w")) != NULL)
    fprintf(lockfp, "%s %d %d", host_name1, pid, ppid);
  fclose(lockfp);
  return SUCCESS;
}


/*
 * Since we can create a useable lock file, then we might aswell give a 
 * gracius way of exiting this program, we can't have lockfiles lying around
 * so that whenever you start the program, it tells you it need to remove a stale lockfile.
 * 
 * So pretty much in the same way as create_pid_file() we simply remove it again
 */

int remove_pid_file()
{
  char lockfile[LINE_SIZE +1]; /* should be enough for the user and /tmp and PACKAGE */
  FILE* lockfp;
  /* First findout who the user is.. */
  uid_t  original_real_uid = getuid();
  struct passwd* pw = getpwuid(original_real_uid);
  if (!pw) /* you don't exist */
    exit(1);
    
  /* then test if theres a file called PACKAGE-USER */
#ifdef I__HAVE__SNPRINTF
  snprintf(lockfile, LINE_SIZE, "%s/.%s-%s.lck", getenv("HOME"),PACKAGE, pw->pw_name);
#else
  sprintf(lockfile, "%s/.%s-%s.lck", getenv("HOME"), PACKAGE, pw->pw_name);
#endif
  if((lockfp = fopen(lockfile, "r")) != NULL )
    {
      fclose(lockfp);	/* not checking should be safe, file mode was "r" */
      unlink(lockfile);
    }
  return SUCCESS;
}

/* 
 * write the debug info to the log file.. 
 * And since this is a way of creating debug info, we won't 
 * do it if the user dosn't want us to.
 *
 * Else its as simple as it can get, open the file in append mode, 
 * write to it, whatever there was of debug info.
 */


int logging(FIFO* fifo, const char* fmt, ...)
{
  va_list ap;
  FILE* debug_file;

  if(fifo->debug_file[0] == '\0' || !fifo->debug)
    return SUCCESS;
  if(!(debug_file = fopen(fifo->debug_file, "a+")))
    return FAILURE;
  
  va_start(ap, fmt);
  vfprintf(debug_file, fmt, ap);
  /* closing the file should flush output-buffer */
  fclose(debug_file);
  va_end(ap);
  return SUCCESS;
}

/* A simple function to check, if the given boxes location is valid.
 * if not it will make sure the boxes needed is disabled to avoid any segfaults
 */
#ifdef BOXING_SUPPORT
int failing_boxes(void)
{
  if (!access(BOXES_PROGRAM, F_OK) && !access(BOXES_PROGRAM, R_OK) && !access(BOXES_PROGRAM, X_OK))
    return 0;
  return -1;
}
#endif

/* 
 * This is a simple config file checking function,
 * It will locate the common used config files, open it
 * and decide if it is a valid file. 
 */

FILE* open_config_file(const char* file_name, int verbose)
{
   FILE* file;
   char name[LINE_SIZE +1];
   if(file_name[0] == '\0')
     {
       /** check for all the default configuration files **/
#ifdef I__HAVE__SNPRINTF
       snprintf(name, LINE_SIZE -1, "%s/.sigitrc", getenv("HOME"));
#else
       sprintf(name, "%s/.sigitrc", getenv("HOME"));
#endif
       
       if (!access(name, F_OK))
	 { /* WOW the user had made his own.. */
	   if(verbose > 0)
	     printf("[open_config_file]\t: The user had made his own .sigitrc file.\n");
	   if (!access(name, R_OK))
	     { /* Lets see if we have read access to it.. */
	       file = fopen (name, "r");
	       if(feof(file))
		 {
		   if(verbose > 0)
		     printf("[open_config_file]\t: But theres nothing to read from it.\n");
		   return NULL;
		 }
	     }	 
	   else
	     {
	       if(verbose > 0)
		 printf("[open_config_file]\t: But there's no read access to it.\n");
	       return NULL;
	     }
	 } 
       else
	 { /* DOH we have to read the lame default settings.. */
	   if(verbose > 0)
	     printf("[open_config_file]\t: The lUSER havn't made a configuration file.\n");
	   if (!access(DEFAULT_CONF_FILE, F_OK))
	     { /* well then we just use the standard one */
	       if(verbose > 0)
		 printf("[open_config_file]\t: So we will have to read from the standard.\n");
	       
	      if (!access(DEFAULT_CONF_FILE, R_OK))
		{
		  /* Lets see if we have read access to it.. */
		  file = fopen (DEFAULT_CONF_FILE, "r");
		  if(feof(file))
		    {
		      if(verbose > 0)
			printf("[open_config_file]\t: That lame SysAdmin havn't made a standard one.\n");
		      return NULL;
		    }
		}
	      else
		{
		  if(verbose > 0)
		    printf("[open_config_file]\t: That lame SysAdmin havn't provided read access to the file.\n");
		  return NULL;
		}
	    }
	  else /* if none of the files exist we must alarm */
	    {
	      if(verbose > 0)
		printf("[open_config_file]\t: That lame SysAdmin havn't made a configuration file.\n");
	      return NULL;
	    }
	 }
     }
   else
     file = fopen(file_name, "r");
   /** make the first check, to see if it's a valid conf file **/
   while(1)
     {
       if(feof(file))
	 {
	   if(verbose > 0)
	     printf("[open_config_file]\t: Ooops.. an unexpected EOF, while reading configuration file.\n");
	   return NULL;
	 }
       if(!fgets(name, LINE_SIZE, file))
	 break;/* let the rest of this function handle it.. */
	   
       if(name[0] == '#')
	 continue; /* just a comment line */
       if(!strncmp(name, "@Sigit@", 7))
	 return file; /* we found the magic word */
    }
   if(verbose > 0)
     printf("[open_config_file]\t: Sorry, but I didn't find the configuration file to be valid for this sigit version.\n");
   return NULL;
}


/*
 * get_config_entry() is pretty simple, handle what ever line is given to it
 * search for a matching '=' and then return what was placed befor the '=' 
 * with any whitespaces removed, and what was placed after the '=' with everything intact.
 * Should there be a comment after the '=' it too will be removed.
 */


int get_config_entry(char line[], char entry[], char value[])
{
  char buffer[LINE_SIZE +1];
  int line_counter = 0, buffer_counter = 0;

  /* first remove all leading spaces or \t */
   while (line[line_counter] == ' ' || line[line_counter] == '\t')
     {/* if we hit a premature newline or an EOL Skip the retrieving. */
       if(line[line_counter] == '\n' || line[line_counter] == '\0')
	 {
	   entry[0] = '\0';
	   value[0] = '\0';
	   return FAILURE;
	 } /* for as long as we are taking spaces or tabs, then we're fine */
       line_counter++;
     }
   /* at this point we _must_ be at the beginning of the entry decission 
    * fetch what it is set to be.. 
    */
   for (; line[line_counter] != '\0'; line_counter++, buffer_counter++)
     {
       if (line[line_counter] == '=')
	 {
	   line[line_counter] = '\0';
	   buffer[buffer_counter] = '\0';
#ifdef I__HAVE__STRNCPY
	   strncpy (entry, buffer, LINE_SIZE);
	   strncpy (value, line+(line_counter+1), LINE_SIZE);
#else
	   strcpy (entry, buffer);
	   strcpy (value, line+(line_counter+1));
#endif
	   if(value[strlen(value) -1] == '\n')
	     value[strlen(value) -1] = '\0';
	   if(value[strlen(value) -1] == '\r')
	     value[strlen(value) -1] = '\0';
	   return SUCCESS;
	 }
       if(line[line_counter] != ' ' || line[line_counter] != '\t')
	 buffer[buffer_counter] = line[line_counter];
     }
   return FAILURE;
}



/*
 * remove_comment() in order to remove any comment in a line like:
 * dash=yes # just to make sure..
 * It will remove any leading spaces/tabs from the setting, till it reaches '#'
 */

int remove_comment(char line[])
{
  int i = 0;
  for (; line[i] != '\0' && line[i] != '\n' && line[i] != '\r'; i++)
    {
      if(line[i] == '#')
	{
	  if(i > 0)
	    {
	      line[i--] = '\0';
	      /* start to rewind this thing, to eliminate all the spaces */
	      for(; i > 0 && (line[i] == ' ' || line[i] == '\t'); i--)
		line[i] = '\0';
	      return SUCCESS;
	    }
	  line[i] = '\0';
	  return SUCCESS;
	}
    }
  return SUCCESS;
}


/*
 * A quite simple configure reading function.. 
 * Basicaly what it does, is read the config file
 * sort out all the settings, and see if the '{' and '}'
 * are balanced correct in the file.
 */

int read_config_fifo(FIFO* fifo, FILE* file)
{
  char entry[LINE_SIZE +1], value[LINE_SIZE +1], buffer[BUFF_SIZE +1];
  int pretty_count = 0, nfs_temp = 0, line_sorter = 0;

  /* 
   * resetting all settinngs retrieved from the config file, 
   * if people havnt specified everyone on every fifo setting 
   */

  fifo->pretty.pretty_setting[0]  = '\0';
  fifo->pretty.top_line[0]        = '\0';
  fifo->pretty.bottom_line[0]     = '\0';
  fifo->pretty.email_address[0]   = '\0';
  fifo->pretty.name_setting[0]    = '\0';
  fifo->pretty.slash              = 0;
  fifo->debug                     = 0;
  fifo->spam_catcher              = 0;
  fifo->nfs_delay                 = 0;
  fifo->seek_indent               = 0;
  fifo->show_sig_ident[0]         = '\0';
  fifo->delete[0]                 = '\0';
#ifdef I__HAVE__SNPRINTF
  snprintf(fifo->fifo_name, LINE_SIZE -1, "%s/%s", getenv("HOME"), DEFAULT_SIG_FILE);
#else	      
  sprintf(fifo->fifo_name, "%s/%s", getenv("HOME"), DEFAULT_SIG_FILE);
#endif
#ifdef I__HAVE__STRNCPY
  strncpy(fifo->data_name, DEFAULT_DATA_FILE, LINE_SIZE -1);
#else
  strcpy(fifo->data_name, DEFAULT_DATA_FILE);
#endif
  fifo->temp_data[0]              = '\0';
  fifo->debug_file[0]             = '\0';
  fifo->limit_sig[0]              = '\0';
  fifo->limit_start               = NULL;
  fifo->limit_end                 = NULL;
#ifdef BOXING_SUPPORT
  fifo->boxes_flags[0]            = '\0';
  fifo->boxes_config[0]           = '\0';
  fifo->boxes_support             = 0;
#endif
  while(!feof(file))
  {
    if(!fgets(buffer, BUFF_SIZE, file))
      return FAILURE; /* premature end of the file */
    if(buffer[0] == '#' || buffer[0] == '\n')/* it's a comment or illigal line, do nothing */
      continue;
 
    if(NULL != strstr(buffer, "pretty{"))
    {
      while (fgets(buffer, LINE_SIZE, file) && !strstr(buffer, "}"))
	{ /* let's hope we wont have a premature EOF right in the pretty setting */
	  if(pretty_count += strlen(buffer) > NUM_LINES*LINE_SIZE)
	    {
	      /* Theres too much in pretty setting abort the pretty reading */
	      /* But since we have to end it graceful, we just fade it out.*/
	      if(fifo->verbose > 1)
		{
		  printf("[read_config_fifo]\t: Error: The pretty setting is {%d} chars, thats more than the allowed {%d} chars\n", 
			 pretty_count, NUM_LINES*LINE_SIZE);
		  printf("[read_config_fifo]\t: Error: refusing to read more of the setting.\n");
		}
	      continue; 
	    }
	  strcat(fifo->pretty.pretty_setting, buffer);
	} 
      if(NULL != strstr(buffer, "}"))
	{ /* its OK lets exit this loop */
	  buffer[0] = '\0'; /* first NULL the whole buffer, just in case */
	  continue;
	}
      else
	{
	  if(fifo->verbose > 0)
	    printf("[read_config_fifo]\t: Error: Reached a premature EOF\n");
	  return FAILURE; /* it must be an error in the fgets()..*/
	}
    }
    /* We better remove any comment later in the line */  
    remove_comment(buffer); 

    if(!get_config_entry(buffer, entry, value))
      {
	/* bahhh.. go through them all...  and copy to FIFO*/
	line_sorter++;
	if (!strcmp(entry, "fifo_file"))
	  {
	    if(fifo->verbose > 1)
	      printf("[read_config_fifo]\t: %d: %s, %s\n", line_sorter, entry, value);
	    if(strlen(value) > 1)
	      {
#ifdef I__HAVE__SNPRINTF
		snprintf(fifo->fifo_name, LINE_SIZE -1, "%s/%s", getenv("HOME"), value);
#else
		sprintf(fifo->fifo_name, "%s/%s", getenv("HOME"), value);
#endif
	      }
	    else
	      {
#ifdef I__HAVE__SNPRINTF
		snprintf(fifo->fifo_name, LINE_SIZE -1, "%s/%s", getenv("HOME"), DEFAULT_SIG_FILE);
#else	      
		sprintf(fifo->fifo_name, "%s/%s", getenv("HOME"), DEFAULT_SIG_FILE);
#endif
	      }
	  }
	if (!strcmp(entry, "data_file"))
	  {
	    if(fifo->verbose > 1)
	      printf("[read_config_fifo]\t: %d: %s, %s\n", line_sorter, entry, value);
	    if(strlen(value) > 1)
	      {
		if(value[0] == '/') /* If its an absolute given path. */
#ifdef I__HAVE__STRNCPY
		  strncpy(fifo->data_name, value, LINE_SIZE -1);
#else
		  strcpy(fifo->data_name, value);
#endif
		else
		  {/* must be a user defined path, meaning taken from $HOME */
#ifdef I__HAVE__SNPRINTF	  
		    snprintf(fifo->data_name, LINE_SIZE -1, "%s/%s", getenv("HOME"), value);
#else
		    sprintf(fifo->data_name, "%s/%s", getenv("HOME"), value);
#endif
		  }
	      }
	    else
#ifdef I__HAVE__STRNCPY
	      strncpy(fifo->data_name, DEFAULT_DATA_FILE, LINE_SIZE -1);
#else
	      strcpy(fifo->data_name, DEFAULT_DATA_FILE);
#endif
	  }
#ifdef BOXING_SUPPORT
	if (!strcmp(entry, "boxes_flags"))
	  {
	    if(fifo->verbose > 1)
	      printf("[read_config_fifo]\t: %d: %s, %s\n", line_sorter, entry, value);
	    if(strlen(value) > 1)
#ifdef I__HAVE__STRNCPY
	      strncpy(fifo->boxes_flags, value, LINE_SIZE);	    
#else
	      strcpy(fifo->boxes_flags, value);	    
#endif
	    else
	      fifo->boxes_flags[0] = '\0';
	  }
	if (!strcmp(entry, "boxes_config"))
	  {
	    if(fifo->verbose > 1)
	      printf("[read_config_fifo]\t: %d: %s, %s\n", line_sorter, entry, value);
	    if(strlen(value) > 1)
	      {
		if(value[0] == '/') /* If its an absolute given path. */
#ifdef I__HAVE__STRNCPY
		  strncpy(fifo->boxes_config, value, LINE_SIZE -1);
#else
		  strcpy(fifo->boxes_config, value);
#endif
		else
		  {/* must be a user defined path, meaning taken from $HOME */
#ifdef I__HAVE__SNPRINTF	  
		    snprintf(fifo->boxes_config, LINE_SIZE -1, "%s/%s", getenv("HOME"), value);
#else
		    sprintf(fifo->boxes_config, "%s/%s", getenv("HOME"), value);
#endif
		  }
	      }
	    else
	      fifo->boxes_config[0] = '\0';
	  }
	if (!strcmp(entry, "boxing"))
	  {
	    if(strstr(value, "yes") || strstr(value, "true"))
	      {
		if(!failing_boxes())
		  fifo->boxes_support=1;
		else
		  {
		    fifo->boxes_support=0;
		    if(fifo->verbose > 0)
		      {
			printf("[open_config_file]\t: Error: Couldn't find a working boxes at [%s]\n", BOXES_PROGRAM);
			printf("[open_config_file]\t: Error: Disabling the use of boxes for now\n");
		      }
		  }
	      }
	    else
	      fifo->boxes_support=0;
	    if(fifo->verbose > 1)
	      printf("[read_config_fifo]\t: %d: %s, %s\n", line_sorter, entry, value);
	  }
#endif /* BOXING_SUPPORT */
	if (!strcmp(entry, "debug_file"))
	  {
	    if(fifo->verbose > 1)
	      printf("[read_config_fifo]\t: %d: %s, %s\n", line_sorter, entry, value);
	    if(strlen(value) > 1)
	      {
		if(value[0] == '/')
#ifdef I__HAVE__STRNCPY
		  strncpy(fifo->debug_file, value, LINE_SIZE -1);
#else
		  strcpy(fifo->debug_file, value);
#endif
		else
		  {
#ifdef I__HAVE__SNPRINTF
		    snprintf(fifo->debug_file, LINE_SIZE, "%s/%s", getenv("HOME"), value);
#else
		    sprintf(fifo->debug_file, "%s/%s", getenv("HOME"), value);
#endif
		  }
	      }
	    else
	      strcpy(fifo->debug_file, "");
	  }
	if (!strcmp(entry, "limit_sig"))
	  {
	    if(fifo->verbose > 1)
	      printf("[read_config_fifo]\t: %d: %s, %s\n", line_sorter, entry, value);
	    if(strlen(value) > 1)
#ifdef I__HAVE__STRNCPY
	      strncpy(fifo->limit_sig, value, LINE_SIZE -1);
#else
	      strcpy(fifo->limit_sig, value);
#endif
	  }
	if (!strcmp(entry, "debug"))
	  {
	    if(strstr(value, "yes") || strstr(value, "true"))
	      fifo->debug=1;
	    else
	      fifo->debug=0;
	    if(fifo->verbose > 1)
	      printf("[read_config_fifo]\t: %d: %s, %s\n", line_sorter, entry, value);
	  }
	if (!strcmp(entry, "top_line"))
	  {
	    if(fifo->verbose > 1)
	      printf("[read_config_fifo]\t: %d: %s, %s\n", line_sorter, entry, value);
#ifdef I__HAVE__STRNCPY
	    strncpy(fifo->pretty.top_line, value, LINE_SIZE);
#else
	    strcpy(fifo->pretty.top_line, value);
#endif
	    /* 
	     * moffe is still babling about a too long lines error in top/bottom_line 
	     * lets see if this will change that.
	     */
	    if(strlen(fifo->pretty.bottom_line) > LINE_SIZE)
	      fifo->pretty.top_line[LINE_SIZE] = '\0';
	    else
	      fifo->pretty.top_line[strlen(fifo->pretty.top_line)] = '\0';
	  }
	if (!strcmp(entry, "bottom_line"))
	  {
	    if(fifo->verbose > 1)
	      printf("[read_config_fifo]\t: %d: %s, %s\n", line_sorter, entry, value);
#ifdef I__HAVE__STRNCPY
	    strncpy(fifo->pretty.bottom_line, value, LINE_SIZE);
#else
	    strcpy(fifo->pretty.bottom_line, value);
#endif
	    /* 
	     * moffe is still babling about a too long lines error in top/bottom_line 
	     * lets see if this will change that.
	     */
	    if(strlen(fifo->pretty.bottom_line) > LINE_SIZE)
	      fifo->pretty.bottom_line[LINE_SIZE] = '\0';
	    else
	      fifo->pretty.bottom_line[strlen(fifo->pretty.bottom_line)] = '\0';
	  }
	if (!strcmp(entry, "email"))
	  {
	    if (fifo->verbose > 1)
	      printf("[read_config_fifo]\t: %d: %s, %s\n", line_sorter, entry, value);
#ifdef I__HAVE__STRNCPY
	    strncpy(fifo->pretty.email_address, value, LINE_SIZE);
#else
	    strcpy(fifo->pretty.email_address, value);
#endif
	  }
	if (!strcmp(entry, "name"))
	  {
	    if(fifo->verbose > 1)
	      printf("[read_config_fifo]\t: %d: %s, %s\n", line_sorter, entry, value);
#ifdef I__HAVE__STRNCPY
	    strncpy(fifo->pretty.name_setting, value, LINE_SIZE);
#else
	    strcpy(fifo->pretty.name_setting, value);
#endif
	  }
	if (!strcmp(entry, "show_ident"))
	  {
	    if(strstr(value, "yes") || strstr(value, "true"))
#ifdef I__HAVE__STRNCPY
	      strncpy(fifo->show_sig_ident, "Statement", LINE_SIZE);
#else
	      strcpy(fifo->show_sig_ident, "Statement");
#endif
	    else
	      if(strlen(value) > 1 && !(strstr(value, "no") || strstr(value, "false")))
#ifdef I__HAVE__STRNCPY
		strncpy(fifo->show_sig_ident, value, LINE_SIZE);
#else
		strcpy(fifo->show_sig_ident, value);
#endif
	      else
		fifo->show_sig_ident[0] = '\0';
	    if(fifo->verbose > 1)
	      printf("[read_config_fifo]\t: %d: %s, %s\n", line_sorter, entry, value);
	  }
	if (!strcmp(entry, "spam_catcher"))
	  {
	    if(strstr(value, "yes") || strstr(value, "true"))
	      fifo->spam_catcher = 1;
	    else
	      fifo->spam_catcher = 0;
	    if(fifo->verbose > 1)
	      printf("[read_config_fifo]\t: %d: %s, %s\n", line_sorter, entry, value);
	  }
	if (!strcmp(entry, "dash"))
	  {
	    if(strstr(value, "yes") || strstr(value, "true"))
	      fifo->pretty.slash = 1;
	    else
	      fifo->pretty.slash = 0;
	    if(fifo->verbose > 1)
	      printf("[read_config_fifo]\t: %d: %s, %s\n", line_sorter, entry, value);
	  }
	if (!strcmp(entry, "nfs_delay"))
	  {
	    if(strstr(value, "yes") || strstr(value, "true"))
	      fifo->nfs_delay = NFS_DELAY;
	    else
	      {
		nfs_temp = atoi(value);
		if(nfs_temp > 0)
		  fifo->nfs_delay = nfs_temp;
		else
		  fifo->nfs_delay = 0;
	      }
	    if(fifo->verbose > 1)
	      printf("[read_config_fifo]\t: %d: %s, %s\n", line_sorter, entry, value);
	  }
      }
    else
      {
	if(strstr(buffer, "#") || strstr(buffer, "}"))
	  ;/* its OK, no need to worry.. */
	else
	  {
	    if(fifo->verbose > 0)
	      {
		/* theres gotta be some error in this configuration file */
		printf("[read_config_fifo]\t: Error: while reading configuration.\n");
		printf("[read_config_fifo]\t: %s\n", buffer);
		printf("[read_config_fifo]\t: dosn't hold any usefull config settings.\n");
	      }
	  }
      }
    if(NULL != strstr(buffer,"}"))
      return SUCCESS; /* we must be at the end of this fifo */
  }
  if(fifo->verbose > 0)
    {
      printf("[read_config_fifo]\t: /* this file must be malformed in its configuration.. */\n");
      printf("[read_config_fifo]\t: /* atleast there must be a missing } at the end of the fifo setting */\n");
    }
  return FAILURE;
}



#endif
