
% imdisplay.sl: Render a stack of images or display an animation. {{{
%
%		The image specifications passed in may be names of files
%		(of various types), GDK pixbufs, or raw S-Lang arrays
%		(2D/greyscale, 3D/RGB, or 3D/RGBA).  If any input image
%		contains an alpha channel then the rendered result will
%		as well.  Input image files may be of any type readable
%		by a GdkPixbuf loader module (including FITS), while
%		output may be saved to any writable GdkPixbuf format
%		(again, including FITS).
%
%		Hint: don't rely upon XV (which is otherwise great!) to
%		judge whether transparency has been correctly preserved
%		in output files.  Most "standard" installations of XV do
%		not really support layering, and my observation,
%		corroborated by others on the WWW, is that it will assign
%		colors to transparent pixels in an inconsistent manner
%		(sometimes white, sometimes black, ...).  Instead, try
%		using a more actively developed tool, and with better RGBA
%		support, such as the GIMP or ImageMagick.
%		
%		This file is part of SLgtk, the S-Lang bindings to Gtk.
%		Copyright (C) 2003-2007 Massachusetts Institute of Technology
%		Copyright (C) 2002 Michael S. Noble (mnoble@space.mit.edu)
% }}}

require("gtk");

private variable Pane = struct {  % {{{
   w, h, resize_w, resize_h,		% image dimensions
   viewport_w, viewport_h,		% starting window dimensions
   win,					% toplevel image display window
   image, drawable, pixmap, gc,		% image widget & related
   pixbuf, resize_pixbuf,		% RGB(A) pixel buffers
   transparency,			% does image have alpha channel?
   title, is_animation,
   ctx,					% parent context
   prev, next
};  % }}}

private variable Context = struct {	% {{{
   wscale, hscale, scale_type,		% resizing along X/Y
   flip, flop,				% mirror along Y/X?
   fill_rule,				% how to grow image when win enlarged
   controller,				% toplevel control window
   size_label,
   transparency,			% does any image have alpha channel?
   layout, panes, last_pane, npanes,
   has_animation
};

private variable EMPTY = "";
private variable version = 304;
private variable version_string = "0.3.4";
define _imdisplay_version_string() { return version_string; }
define _imdisplay_version() { return version; }
% }}}

% Customization options handling {{{

private variable pixbuf_flip = NULL; % {{{
#ifexists gdk_pixbuf_flip
   pixbuf_flip = &gdk_pixbuf_flip;
#endif
if (andelse {pixbuf_flip == NULL}
	    {is_defined( current_namespace + "->gdk_pixbuf_flip")}) {
   eval("&gdk_pixbuf_flip", current_namespace);
   pixbuf_flip = ();
} % }}}

private variable SCALE_PERCENT = 0x1, SCALE_PIXELS  = 0x2;
private variable FILL_OPTION_STRINGS = ["none", "tile", "scale" ],
	FILL_NONE   = 0,
	FILL_TILE  = 1,
	FILL_SCALE = 2;

private variable
	PANE_SINGLE = 0,
	PANE_VERT   = 2,		% corresponds to chain gravity = South
	PANE_HORIZ  = 3,		% corresponds to chain gravity = East
	Current_Layout = PANE_SINGLE;

private variable Current_Context = NULL;
private define default_context()
{
   variable ctx = @Context;
   ctx.fill_rule = FILL_SCALE;
   ctx.layout = PANE_SINGLE;
   ctx.transparency = 0;
   ctx.has_animation = 0;
   ctx.npanes = 0;
   return ctx;
}

private define eprint() % {{{
{
  variable args = __pop_args(_NARGS);
  () = fprintf(stderr,__push_args(args));
  () = fflush(stderr);
} % }}}


private define parse_tuple(tuple, delimiter)  % {{{
{
   variable toks = strtok(tuple, delimiter);
   if (length(toks) == 1)
       toks = [toks, toks];

   return atof(toks[0]), atof(toks[1]);
} % }}}

private define parse_options(ctx, values)  % {{{
{
   if (values == NULL or values == "")
      return;

   variable prev_traceback = _traceback;
   _traceback = -1;

   values = strtok(values, ",");
   foreach(values) {

	variable option = (), value = EMPTY;
	option = array_map(String_Type, &strtrim, strtok(option, "="));
	if (length(option) > 1)
	   value = option[1];

	option[0] = strlow(option[0]);

	switch( option[0] )
	{ case "fill" :

		if (value == EMPTY)
		   continue;

		value = where(FILL_OPTION_STRINGS == strlow(value));
		if (length(value) == 0)
		   continue;

		ctx.fill_rule = value[0];
	}
	{ case "size" :

		if (value == EMPTY)
		   continue;

		(ctx.wscale, ctx.hscale) = parse_tuple(value, "x");

		if (ctx.wscale <= 0 or ctx.hscale <= 0) {
		   eprint("Illegal size option: %s\n", value);
		   continue;
		}

		if (is_substr(value, "%")) {
		   ctx.scale_type = SCALE_PERCENT;
		   ctx.wscale /= 100.0;
		   ctx.hscale /= 100.0;
		}
		else
		   ctx.scale_type = SCALE_PIXELS;
	}
	{ case "flip" or case "flop":

		if (pixbuf_flip == NULL) {
		   eprint("Gtk 2.6 or newer is needed for flip/flop\n");
		   continue;
		}
		set_struct_field(ctx, option[0], TRUE);
	}
	{ case "panes" :

		if (value == EMPTY)
		   continue;

		value = strlow(value);
		switch (value)
		{ case "one" or case "single"      : ctx.layout = PANE_SINGLE;}
		{ case "horiz" or case "horizontal": ctx.layout = PANE_HORIZ; }
		{ case "vert" or case "vertical"   : ctx.layout = PANE_VERT;  }
	}
   }

   _traceback = prev_traceback;
} % }}}

private define scale(pane) % {{{
{
   variable ctx = pane.ctx;

   if (ctx.scale_type == NULL)
      return;

   variable w = pane.w, h = pane.h, pb = pane.pixbuf;  % width, height, pixbuf

   switch(ctx.scale_type)
   { case SCALE_PERCENT: w = int(w*ctx.wscale); h = int(h*ctx.hscale); }
   { case SCALE_PIXELS : w = int(ctx.wscale);   h = int(ctx.hscale);   }

   pb = gdk_pixbuf_scale_simple(pb, w, h, GDK_INTERP_BILINEAR);
   pane.w = w;
   pane.h = h;
   pane.pixbuf = pb;
} % }}}

private define flipflop(pane) % {{{
{
   if (pane.ctx.flip != NULL)
	pane.pixbuf  = (@pixbuf_flip)(pane.pixbuf, FALSE);

   if (pane.ctx.flop != NULL)
	pane.pixbuf  = (@pixbuf_flip)(pane.pixbuf, TRUE);

} % }}}

% }}}

private define pane_align(alpha_pane, delta) % {{{
{
   % Ensure proper alignment of all panes, regardless of WM placement strategy
   if (alpha_pane == NULL) return;

   variable x, y;
   gtk_window_get_position(alpha_pane.win, &x, &y);
   gtk_window_move(alpha_pane.win, x + delta, y + delta);
} % }}}

private define pane_new(ctx, pixbuf, title_info, is_animation) % {{{
{
   variable p = @Pane;

   p.ctx = ctx;
   p.pixbuf = pixbuf;
   p.resize_pixbuf = pixbuf;
   p.title = sprintf("%S", title_info);
   p.is_animation = is_animation;
   ctx.has_animation |= is_animation;
   if (is_animation) {
	p.w = gdk_pixbuf_animation_get_width(pixbuf);
	p.h = gdk_pixbuf_animation_get_height(pixbuf);
	p.transparency = 0;
   }
   else {
	p.w = gdk_pixbuf_get_width(pixbuf),
	p.h = gdk_pixbuf_get_height(pixbuf);
	p.transparency = gdk_pixbuf_get_has_alpha(pixbuf);
   }

   return p;
} % }}}

private define pane_append(ctx, pane) % {{{
{
   if (ctx.panes == NULL) {
	ctx.npanes = 1;
	ctx.panes = pane;
	ctx.last_pane = pane;
   }
   else {
	ctx.last_pane.next = pane;
	pane.prev = ctx.last_pane;
	ctx.last_pane = pane;
	ctx.npanes++;
   }

} % }}}

private define pane_remove(pane, realign) % {{{
{
   variable ctx = pane.ctx;

   if (pane.prev == NULL)
	ctx.panes = pane.next;
   else
	pane.prev.next = pane.next;

   if (pane.next == NULL)
	ctx.last_pane = pane.prev;
   else
	pane.next.prev = pane.prev;

   ctx.npanes--;

   if (realign)
	pane_align(ctx.panes, 0);

} % }}}

private define pane_destroy(pane) % {{{
{
   pane_remove(pane, 1);
   if (pane.ctx.npanes == 0)
	gtk_widget_destroy(pane.ctx.controller);
} % }}}

private define load(image, ctx)	% {{{
{
   variable pb = NULL, animation = 0, info = sprintf("%S", image);

   switch( typeof(image))
   { case String_Type:

	if (stat_file(image) == NULL)
	   return parse_options(ctx, image);

	pb = gdk_pixbuf_animation_new_from_file(image);
  	if (pb != NULL) {
	   info = path_basename(image);
	   if (gdk_pixbuf_animation_is_static_image(pb))
		pb = gdk_pixbuf_animation_get_static_image(pb);
	   else
		animation = 1;
	}
   }
   { case Array_Type:

	variable dims; (dims, ,) = array_info(image);

	if (length(dims) == 1) {
	   % Treat 1D images to NxN if possible
	   variable size = sqrt(dims[0]);
	   if (int(size) != size)
		return eprint("This 1D array cannot be treated as NxN image\n");
	   dims = [ int(size), int(size) ];
	}

	if (length(dims) == 2) {
	   variable rgbbuf = UChar_Type[dims[0], dims[1], 3];
	   image = norm_array(image);
	   rgbbuf[*,*,0] = image;
	   rgbbuf[*,*,1] = image;
	   rgbbuf[*,*,2] = image;
	   image = rgbbuf;
	   info = sprintf("%S x %S", dims[1], dims[0]);
	}
	pb = gdk_pixbuf_new_from_data(image);
	image = sprintf("%S x %S", dims[1], dims[0]);
   }
   { case GdkPixbuf:  pb = image; }

   if (pb == NULL)
	return eprint("Could not create 2D image from: %S\n", image);

   pane_append( ctx, pane_new(ctx, pb, info, animation) );

} % }}}

private define composite(ctx)  % {{{
{
   variable title, pane, transparency = 0;
   variable cpb = NULL, cw, ch;  	% composite pixbuf, width, height

   foreach pane (ctx.panes) {

	if (pane.is_animation)		% When compositing, all non-animated
	   continue;

	pane_remove(pane, 0);		% image panes are collapsed to 1

	variable pb = pane.pixbuf;
	variable w = gdk_pixbuf_get_width(pb), h = gdk_pixbuf_get_height(pb);

	if (cpb == NULL) {
	   cpb = pb; cw = w; ch = h;
	   transparency = pane.transparency;
	   title = pane.title;
	   continue;
	}

	title += " ...";

	if (pane.transparency and not(transparency)) {
	   cpb = gdk_pixbuf_add_alpha(cpb, FALSE, 0, 0, 0);
	   transparency = 1;
	}

	if (w != cw or h != ch) {

	   variable w2 = max([cw, w]), h2 = max ([ch, h]);
	   variable bigger = gdk_pixbuf_new(GDK_COLORSPACE_RGB, TRUE,8,w2,h2);
	   gdk_pixbuf_fill(bigger, 0xFFFFFF00);	% 100% transparent, white

	   % Copy existing image into new pixbuf, then composite new image
	   % Pixels not covered by these two operations remain transparent
	   gdk_pixbuf_copy_area(cpb, 0, 0, cw, ch, bigger, 0, 0);
   	   gdk_pixbuf_composite(pb, bigger, 0, 0, w, h, 0, 0, 1, 1,
						GDK_INTERP_BILINEAR, 255);
	   cpb = bigger;
	   cw = w2; ch = h2;
	   transparency = 1;
   	}
	else
	   gdk_pixbuf_composite(pb, cpb, 0, 0, w, h, 0, 0, 1, 1,
						GDK_INTERP_BILINEAR, 255);
   }

   ctx.transparency = transparency;
   if (cpb != NULL)
   	pane_append( ctx, pane_new(ctx, cpb, title, 0) );
} % }}}

% Realize, Expose, & Focus handling {{{

private define set_tiling(pane) % {{{
{
   if (pane.pixmap == NULL) {
	pane.pixmap = gdk_pixmap_new(pane.drawable, pane.w, pane.h, -1);
	gdk_draw_rectangle(pane.pixmap, pane.gc, 1, 0, 0, pane.w, pane.h);
	gdk_draw_pixbuf(pane.pixmap, pane.gc, pane.pixbuf, 0,0,0,0,
				pane.w, pane.h, GDK_RGB_DITHER_NORMAL, 0, 0);
   }
   % Gdk automatically tiles any applied background pixmap
   gdk_window_set_back_pixmap(pane.drawable, pane.pixmap, 0);
} % }}}

private define image_size(pane) % {{{
{
   if (pane.ctx.fill_rule == FILL_NONE or pane.is_animation) {
   	variable w = pane.w;
	variable h = pane.h;
   }
   else
	gdk_drawable_get_size(pane.drawable, &w, &h);

   variable size = sprintf("%S x %S", w, h);
   if (not pane.is_animation)
      gtk_window_set_title(pane.win, size);
   gtk_label_set_text(pane.ctx.size_label, size);

   return w, h;
} % }}}

private define expose_cb(event, pane) % {{{
{
   variable x, y, w, h, pb, iw, ih;

   (iw, ih) = image_size(pane);

   switch(pane.ctx.fill_rule)
   { case FILL_TILE : return FALSE; }
   { case FILL_NONE : x = 0; y = 0; w = pane.w; h = pane.h; pb = pane.pixbuf; }
   { case FILL_SCALE:

	variable a = event.area;
	if (iw <= pane.w and ih <= pane.h)
	   pb = pane.pixbuf;
	else {
	   if (pane.resize_w != iw or pane.resize_h != ih) {

		pane.resize_pixbuf = gdk_pixbuf_scale_simple (pane.pixbuf,
						iw, ih, GDK_INTERP_BILINEAR);
		pane.resize_w = iw; pane.resize_h = ih;
	   }
	   pb = pane.resize_pixbuf;
	}
	x = a.x; y = a.y; w = a.width; h = a.height;
   }

   gdk_draw_pixbuf (pane.drawable, pane.gc, pb, x, y, x, y, w, h,
						GDK_RGB_DITHER_NORMAL, 0, 0);
   return FALSE;
} % }}}

private define realize_cb(pane)  % {{{
{
   pane.drawable = gtk_widget_get_window(pane.image);
   pane.gc = gdk_gc_new(pane.drawable);
   gtk_widget_modify_bg(pane.image, GTK_STATE_NORMAL, @gdk_white);
   gdk_gc_set_foreground(pane.gc, @gdk_white);
   if (pane.ctx.fill_rule == FILL_TILE)
	set_tiling(pane);
   () = g_signal_connect_swapped(pane.image, "expose_event", &expose_cb, pane);
} % }}}

private define show_size_cb(widget, event, pane)  % {{{
{
  if (pane.image != NULL)
	( , ) = image_size(pane);

  % This callback is used for both signals and events,
  % but needs to return a disposition only for events
  if (typeof(event) == Struct_Type)
	return FALSE;
} % }}}

% }}}

private define save_file(file, kind, pane) % {{{
{
   variable pb, w, h, ctx = pane.ctx;

   if (ctx.fill_rule == FILL_TILE) {
	gdk_drawable_get_size(pane.drawable, &w, &h);
	pb = gdk_pixbuf_get_from_drawable(NULL, pane.drawable,NULL,0,0,0,0,w,h);
	if (pane.transparency)
	   pb = gdk_pixbuf_add_alpha(pb, FALSE, 0, 0, 0);
   }
   else if (ctx.fill_rule == FILL_SCALE)
	pb = pane.resize_pixbuf;
   else
	pb = pane.pixbuf;

   _pixbuf_save (pb, file, kind);
   return FALSE;
} % }}}

private define save_cb(ctx) % {{{
{
   variable name, kind;
   (name, kind) = _image_save_dialog(["png", "jpeg", "fits"]);

   if (name == NULL)
      return;

   name = strtok(name, ".");
   variable stem = name[0], ext = "." + name[1];
   variable pane, num = 1, multiple = (ctx.npanes > 1);

   foreach pane (ctx.panes) {

	% Save the drawn window, rather than pixmap, in order to get the
	% tiling for free.  So, raise it to ensure that it's fully exposed
	gtk_window_present(pane.win);

	name = stem;
	if (multiple) {
	   name += "-" + string(num);
	   num++;
	}
	name += ext;

	() = gtk_timeout_add(750, &save_file, name, kind, pane);
   }
} % }}}

private define resize_cb(fill_button, ctx, fill_rule) % {{{
{
   if (not gtk_toggle_button_get_active (fill_button)) return;
   if (ctx.fill_rule == fill_rule) return;

   foreach (ctx.panes) {

	variable p = ();
	if (p.drawable == NULL)
	   continue;

	ctx.fill_rule = fill_rule;

	switch (fill_rule)
	{ case FILL_TILE:  set_tiling(p); }
	{ gdk_window_set_background(p.drawable, gdk_white); }

	gtk_widget_queue_draw(p.image);
   }
} % }}}

private define revert_cb(revert_button, default_radio_button, ctx) % {{{
{  
   variable p;
   foreach p (ctx.panes) {
	gtk_window_resize(p.win, p.viewport_w, p.viewport_h);
	gtk_toggle_button_set_active(default_radio_button, TRUE);
   }
} % }}}

private define help_cb(button) % {{{
{
   _info_window("Imdisplay Help", _get_slgtk_doc_string("imdisplay"));
} % }}}

private define set_viewport(pane) % {{{
{
   pane.viewport_w = min ([ pane.w, 7 * gdk_screen_width()  / 10]);
   pane.viewport_h = min ([ pane.h, 7 * gdk_screen_height() / 10]);

   % Put unusually large images within a scrolling window
   if (pane.viewport_w < pane.w or pane.viewport_h < pane.h) {
	gtk_window_resize(pane.win, pane.viewport_w, pane.viewport_h);
	variable sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_container_set_border_width (sw, 0);
	gtk_scrolled_window_set_policy(sw, GTK_POLICY_AUTOMATIC,
		 					GTK_POLICY_AUTOMATIC);
	gtk_container_add(pane.win, sw);
	gtk_scrolled_window_add_with_viewport(sw, pane.image);
   }
   else
	gtk_container_add(pane.win, pane.image);
} % }}}

private define make_fill_button(fb, kind, parent, ctx, revert_button) % {{{
{
   variable label = FILL_OPTION_STRINGS[kind];
   label = sprintf("%c%s", label[0]-32, label[[1:]]);	% Upcase first letter

   if (fb == NULL)
	fb = gtk_radio_button_new_with_label(NULL, label);
   else
	fb = gtk_radio_button_new_with_label_from_widget(fb, label);

   if (ctx.fill_rule == kind) 
      () = g_signal_connect(revert_button,"clicked", &revert_cb, fb, ctx);

   gtk_toggle_button_set_active(fb, ctx.fill_rule == kind);
   gtk_box_pack_end(parent, fb, FALSE, FALSE, 0);

   () = g_signal_connect(fb, "clicked", &resize_cb, ctx, kind);
   gtk_widget_unset_flags (fb, GTK_CAN_FOCUS);

   return fb;
} % }}}

private define controller_destroy(ctx) % {{{
{
   variable p;
   foreach p (ctx.panes)
	gtk_widget_destroy(p.win);
   gtk_main_quit();
} % }}}

private define controller_new(ctx, pane, layout) % {{{
{
   variable win = gtk_window_new (GTK_WINDOW_TOPLEVEL);
   gtk_window_set_title(win, sprintf("imdisplay %s", version_string));
   gtk_window_set_resizable(win, FALSE);
   gtk_container_set_border_width(win, 2);
   () = g_signal_connect_swapped(win, "destroy", &controller_destroy, ctx);

   variable vbox = gtk_vbox_new(FALSE, 5);
   gtk_container_add(win, vbox);

   variable hbox = gtk_hbox_new(FALSE, 5);
   gtk_box_pack_end(vbox, hbox, FALSE, FALSE, 0);

   variable button = gtk_button_new_with_label("Done");
   gtk_box_pack_end(hbox, button, FALSE, FALSE, 0);
   gtk_widget_grab_focus(button);
   () = g_signal_connect_swapped(button, "clicked", &gtk_widget_destroy, win);

   variable revert_button = gtk_button_new_with_label("Revert");
   gtk_widget_unset_flags (revert_button, GTK_CAN_FOCUS);
   gtk_box_pack_start(hbox,revert_button, FALSE, FALSE, 0);

   button = gtk_button_new_with_label("Help");
   gtk_widget_unset_flags (button, GTK_CAN_FOCUS);
   gtk_box_pack_start(hbox,button, FALSE, FALSE, 0);
   () = g_signal_connect(button, "clicked", &help_cb);

   ctx.size_label = gtk_label_new(sprintf("%S x %S",pane.w, pane.h));
   gtk_box_pack_start(hbox,ctx.size_label,TRUE,FALSE,0);

   button = gtk_button_new_with_label("Save");
   gtk_widget_unset_flags (button, GTK_CAN_FOCUS);
   gtk_box_pack_end(hbox,button,FALSE,FALSE,0);
   () = g_signal_connect_swapped(button,"clicked", &save_cb, ctx);

   hbox = gtk_hbox_new(FALSE, 0);
   gtk_box_pack_end(vbox,hbox,FALSE,FALSE,0);

   variable frame = gtk_frame_new("Fill Rule");
   gtk_container_add(hbox,frame);

   variable fillbox = gtk_hbox_new(TRUE, 0);
   gtk_container_add(frame, fillbox);

   button = make_fill_button(NULL,   FILL_NONE,  fillbox, ctx, revert_button);
   button = make_fill_button(button, FILL_SCALE, fillbox, ctx, revert_button);
   button = make_fill_button(button, FILL_TILE,  fillbox, ctx, revert_button);

   frame = gtk_frame_new("Transparency");
   gtk_container_add(hbox, frame);
   if (ctx.transparency)
	gtk_container_add(frame, gtk_label_new("Yes"));
   else
	gtk_container_add(frame, gtk_label_new("No"));

   ctx.controller = win;
   gtk_widget_show_all(win);

   % Chain controller to pane
   switch(layout)
   { case PANE_SINGLE: _gtk_window_chain(pane.win, win, 2);}
   { _gtk_window_chain(pane.win, win, layout); }

} % }}}

private define display_panes(ctx) % {{{
{
   if (ctx.npanes == 0)
	verror("No drawable images were found!\n");

   if (ctx.layout == PANE_SINGLE)
	composite(ctx);

   variable p, layout = ctx.layout;

   % See if window placement is broken in window manager; done explicitly
   % here to avoid signal delivery delays with CPU-gobbling animations
   __wm_placement_test();

   foreach p (ctx.panes) {

	p.win = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(p.win, p.title);
	() = g_signal_connect_swapped(p.win, "destroy", &pane_destroy, p);
	() = g_signal_connect(p.win, "focus-in-event", &show_size_cb, p);
	() = g_signal_connect(p.win, "grab-focus", &show_size_cb, 0, p);

	if (p.is_animation) {
	   p.image = gtk_image_new_from_animation(p.pixbuf);
	   gtk_window_set_resizable(p.win, FALSE);
	}
	else {

	   scale(p);
	   flipflop(p);

	   p.image = gtk_drawing_area_new();

	   gtk_widget_add_events(p.image, GDK_ENTER_NOTIFY_MASK);
	   () = g_signal_connect_swapped(p.image, "realize", &realize_cb, p);
	   () = g_signal_connect(p.image,"enter-notify-event", &show_size_cb,p);
	   gtk_widget_set_size_request (p.image, p.w, p.h);
	}

	set_viewport(p);
	gtk_widget_show_all(p.win);

	if (p.prev != NULL)
	   _gtk_window_chain(p.prev.win, p.win, layout);
   }

   if (ctx.npanes > 2) {

	p = ctx.panes;
	loop(ctx.npanes/2)		% Select middle-ish pane 
	   p = p.next;

	if (layout == PANE_HORIZ)
	   layout = PANE_VERT;
	else
	   layout = PANE_HORIZ;
   }

   controller_new(ctx, p, layout);
   pane_align(ctx.panes, 1);
}  % }}}

define imdisplay() % {{{
{
   if (_NARGS == 0)
	usage("imdisplay( FileName_or_ImageArray_or_Option [, ...])");

   % Imaging context
   if (Current_Context == NULL)
	Current_Context = default_context();

   variable ctx = @Current_Context;

   foreach (__pop_args(_NARGS)) {
      	variable arg = ();
	load(arg.value, ctx);
   }

   display_panes(ctx);

   if (gtk_main_level() < 1)
	gtk_main();

} % }}}

define imdisplay_defaults() % {{{
{
   if (_NARGS == 0) {
	Current_Context = default_context();
	return;
   }

   variable setting, settings = __pop_args(_NARGS);
   foreach setting ( settings )
	parse_options(Current_Context, setting.value);
} % }}}

provide("imdisplay");
