package SliMP3::Control;

# SliMP3 Server Copyright (C) 2001 Sean Adams, Slim Devices Inc.
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License, 
# version 2.

# These are the high-level functions for playback and vol/bass/treble

use strict;

use SliMP3::Decoder;
use SliMP3::Misc;
use SliMP3::Timers;
use SliMP3::Stream;

$SliMP3::Control::maxVolume = 100;
$SliMP3::Control::maxTreble = 100;
$SliMP3::Control::minTreble = 0;
$SliMP3::Control::maxBass = 100;
$SliMP3::Control::minBass = 0;

#
# initialize the MAS3507D and tell the client to start a new stream
#
sub play {
	my $client = shift;
	my $paused = shift;
	my $pcm = shift;

	if (!SliMP3::Client::isSliMP3($client)) {
		return 1;
	}

	&volume($client, SliMP3::Prefs::clientGet($client, "volume"));
	SliMP3::Decoder::reset($client, $pcm);
	

	SliMP3::Stream::newStream($client, $paused);
	#
	# We can't start playing until the i2c has all been acked. Ideally
	# something like:
	#	SliMP3::i2c::callback_when_done(SliMP3::Stream::newStream,($client,$paused));
	# For now just kludge it:
	#SliMP3::Timers::setTimer($client, Time::HiRes::time()+2, 
	#	\&SliMP3::Stream::newStream, ($paused));

	return 1;
}

#
# set the MAS3507D volume
#
sub volume {
	my ($client, $volume) = @_;

	if (!SliMP3::Client::isSliMP3($client)) {
		return 1;
	}

	if ($volume > $SliMP3::Control::maxVolume) { $volume = $SliMP3::Control::maxVolume; }
	if ($volume < 0) { $volume = 0; }
	
	# normalize
	$volume = $volume / $SliMP3::Control::maxVolume;

	$::d_control && msg "volume: $volume\n";
 	
	SliMP3::Decoder::volume($client, $volume);
}

#
# set the MAS3507D treble in the range of -1 to 1
#

sub treble {
	my ($client, $treble) = @_;
	if ($treble > $SliMP3::Control::maxTreble) { $treble = $SliMP3::Control::maxTreble; }
	if ($treble < $SliMP3::Control::minTreble) { $treble = $SliMP3::Control::minTreble; }

	SliMP3::Decoder::treble($client, $treble);
}

#
# set the MAS3507D bass in the range of -1 to 1
#

sub bass {
	my ($client, $bass) = @_;
	if ($bass > $SliMP3::Control::maxBass) { $bass = $SliMP3::Control::maxBass; }
	if ($bass < $SliMP3::Control::minBass) { $bass = $SliMP3::Control::minBass; }

	SliMP3::Decoder::bass($client, $bass);
}



# fade the volume up or down
# $fade = number of seconds to fade 100% (positive to fade up, negative to fade down) 
# $callback is function reference to be called when the fade is complete
# FYI 8 to 10 seems to be a good fade value
my %fvolume;  # keep temporary fade volume for each client
sub fade_volume {
	my($client, $fade, $callback, $callbackargs) = @_;

	my $faderate = 20;  # how often do we send updated fade volume commands per second
	
	if (!SliMP3::Client::isSliMP3($client)) {
		return 1;
	}
	SliMP3::Timers::killTimers($client, \&fade_volume);
	
	my $vol = SliMP3::Prefs::clientGet($client, "volume");
	my $mute = SliMP3::Prefs::clientGet($client, "mute");
	if ($vol < 0) {
		# the volume is muted, don't fade.
		$callback && (&$callback(@$callbackargs));
		return;
	}
	
	if ($mute) {
		# Set Target (Negative indicates mute, but still saves old value)
		SliMP3::Prefs::clientSet($client, "volume", $vol * -1);
	}

	# on the first pass, set temporary fade volume
	if(!$fvolume{$client} && $fade > 0) {
		# fading up, start volume at 0
		$fvolume{$client} = 0;
	} elsif(!$fvolume{$client}) {
		# fading down, start volume at current volume
		$fvolume{$client} = $vol;
	}

	$fvolume{$client} += $SliMP3::Control::maxVolume * (1/$faderate) / $fade; # fade volume

	if($fvolume{$client} <= 0 || $fvolume{$client} >= $vol) {
		# done fading
		$::d_ui && msg("fade_volume done.\n");
		$fvolume{$client} = 0; # reset temporary fade volume 
		$callback && (&$callback(@$callbackargs));
	} else {
		$::d_ui && msg("fade_volume - setting volume to $fvolume{$client}\n");
		&volume($client, $fvolume{$client}); # set volume
		SliMP3::Timers::setTimer($client, Time::HiRes::time()+ (1/$faderate), \&fade_volume, ($fade, $callback, $callbackargs));
	}
}

# mute or un-mute volume as necessary
# A negative volume indicates that the player is muted and should be restored 
# to the absolute value when un-muted.
sub mute {
	my $client = shift;
	
	if (!SliMP3::Client::isSliMP3($client)) {
		return 1;
	}
	my $vol = SliMP3::Prefs::clientGet($client, "volume");
	my $mute = SliMP3::Prefs::clientGet($client, "mute");
	
			
	if (($vol < 0) && ($mute)) {
		# mute volume
		# todo: there is actually a hardware mute feature
		# in both decoders. Need to add Decoder::mute
		&volume($client, 0);
	} else {
		# un-mute volume
		$vol *= -1;
		&volume($client, $vol);
	}
	SliMP3::Prefs::clientSet($client, "volume", $vol);
	SliMP3::Display::volumeDisplay($client);
}

#
# tell the client to unpause the decoder
#
sub resume {
	my $client = shift;

	if (!SliMP3::Client::isSliMP3($client)) {
		return 1;
	}

	&volume($client, SliMP3::Prefs::clientGet($client, "volume"));

	SliMP3::Stream::unpause($client);

	return 1;
}

#
# pause
#
sub pause {
	my $client = shift;
	if (!SliMP3::Client::isSliMP3($client)) {
		return 1;
	}
	SliMP3::Stream::pause($client);
	return 1;
}

#
# does the same thing as pause
#
sub stop {
	my $client = shift;

	if (!SliMP3::Client::isSliMP3($client)) {
		return 1;
	}

	SliMP3::Stream::stop($client);
}

#
# playout - play out what's in the buffer
#
sub playout {
	my $client = shift;
	if (!SliMP3::Client::isSliMP3($client)) {
		return 1;
	}
	SliMP3::Stream::playout($client);
	return 1;
}

1;

__END__
