/*
 * ppmfilter.cc -- PPM Filter App
 * Copyright (C) 2003 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <ctype.h>

#include <iostream>
#include <string>
#include <vector>
#include <queue>
#include <stack>
#include <map>
#include <cmath>
#include <strstream>
#include <cassert>
using std::strstream;
using std::map;
using std::vector;
using std::queue;
using std::stack;
using std::string;
using std::cerr;
using std::cin;
using std::endl;

#include <preferences.h>
#include <Diagnostics.h>
#include <PPMDVFileInput.h>
#include <Threader.h>
#include <PPMPump.h>
#include <time.h>
#include <pthread.h>
#include <limits.h>

/** This class reads PPM Frames from stdin.
*/

class PPMPumpProvider : public PPMPump, public Threader
{
	public:
		virtual bool ReadFrame( PPMFrame &frame )
		{
			return frame.ReadImage( );
		}

		void Thread( )
		{
    		bool running = true;

    		while( running && ThreadIsRunning( ) )
    		{
        		if ( GetInputAvailable( ) > 0 )
        		{
            		PPMFrame &frame = GetInputFrame( );
            		if ( ReadFrame( frame ) )
            		{
                		QueueOutputFrame( );
            		}
            		else
            		{
                		Diagnostics::Log( *this, "Input ended" );
                		running = false;
            		}
        		}
    		}

    		ClearPump( );
		}
};

/** The singleton stack is used to communicate frames between filters.
*/

class PPMFrameStack
{
	private:
		stack <PPMFrame *> frame_stack;
		PPMFrame frame_keep;
		bool frame_keep_active;

		PPMFrameStack( ) :
			frame_keep( 100, 100 ),
			frame_keep_active( true )
		{
		}

	public:
		static PPMFrameStack &GetStack( )
		{
			static PPMFrameStack *stack = NULL;
			if ( stack == NULL )
				stack = new PPMFrameStack( );
			assert( stack != NULL );
			return *stack;
		}

		PPMFrame *Pop( )
		{
			if ( frame_stack.size( ) != 0 )
			{
				PPMFrame *frame = frame_stack.top( );
				frame_stack.pop( );
				return frame;
			}
			throw "Stack underflow - aborting.";
			return NULL;
		}

		void Push( PPMFrame &frame )
		{
			Push( new PPMFrame( frame ) );
		}

		void Push( PPMFrame *frame )
		{
			if ( frame != NULL )
				frame_stack.push( frame );
		}

		void Clear( )
		{
			while ( frame_stack.size( ) > 0 ) 
				delete Pop( );
		}

		void SetSeed( PPMFrame &frame )
		{
			frame_keep_active = true;
			frame_keep.Copy( frame );
		}

		void Seed( PPMFrame &frame )
		{
			Clear( );
			if ( frame_keep_active )
			{
				Push( frame_keep );
				frame_keep_active = false;
			}
			else
			{
				Push( frame );
			}
		}
};

/** Utility class.
*/

class EffectUtils
{
	public:
		template<typename ArgType> ArgType clamp(const ArgType A, const ArgType MinVal, const ArgType MaxVal)
		{
        	return std::min(std::max(A, MinVal), MaxVal);
		}

		template<typename ArgType> inline ArgType lerp(const ArgType A, const ArgType B, const double Mix)
		{
        	return static_cast<ArgType>(A * (1.0 - Mix) + B * Mix);
		}

		inline double smoothstep(const double Edge1, const double Edge2, const double A)
		{
        	if(A < Edge1)
                return 0.0;
        	if(A >= Edge2)
                return 1.0;
        	double a = (A - Edge1) / (Edge2 - Edge1);
        	return (a * a * (3 - 2 * a));
		}
};

/** All properties implement this interface.
*/

class PropertyBase : public EffectUtils
{
	public:
		// Parse the input
		virtual bool Parse( string &_value ) = 0;
		// Serialise the value
		virtual string Serialise( ) = 0;
		// Get the name of the property
		virtual string GetName( ) = 0;
		// Get the description of the property
		virtual string GetDescription( ) = 0;
		// Get a format description of the property
		virtual string GetFormat( ) = 0;
		// Summary for --help generation
		string Summary( ) { return GetName( ) + ", " + "<" + GetFormat( ) +">, " + Serialise( ) + ", " + GetDescription( ); }
};

/** All properties are specialisations of this template.
*/

template < typename property > class Property : public PropertyBase
{
	public:
		// type definition
		typedef property property_type;
		// Instance of the property
		property_type value;
		// Virtual destructor
		virtual ~Property( ) { }
		// Get the value at specified position
		virtual property_type GetValue( double = 0.0 ) { return value; }
};

/** Template for property instances.
*/

template < typename property > class PropertyInstance : public property 
{
	private:
		string name;
		string description;

	public:
		// Constructor with no default value
		PropertyInstance( string _name, string _description ) :
			name( _name ),
			description( _description )
		{
		}

		// Constructor with no default string value
		PropertyInstance( string _name, string _description, string initial ) :
			name( _name ),
			description( _description )
		{
			Parse( initial );
		}

		// Constructor with default double value
		PropertyInstance( string _name, string _description, double initial ) :
			name( _name ),
			description( _description )
		{
			char o[ 100 ];
			sprintf( o, "%f", initial );
			string t( o );
			Parse( t );
		}

		// Return the name of the property
		string GetName( ) { return name; }

		// Return the description of the property
		string GetDescription( ) { return description; }
};

/** The boolean property.
*/

class PropertyBool : public Property < bool >
{
	public:
		bool Parse( string &_value ) { value = _value == "true"; return true; }
		string Serialise( ) { return value ? "true" : "false"; }
		string GetFormat( ) { return "bool"; }
};

/** The string property.
*/

class PropertyString : public Property < string >
{
	public:
		bool Parse( string &_value ) { value = _value; return true; }
		string Serialise( ) { return value; }
		string GetFormat( ) { return "string"; }
};

/** The int property.
*/

class PropertyInt : public Property < int >
{
	public:
		bool Parse( string &_value ) { value = atoi( _value.c_str( ) ); return true; }
		string Serialise( ) { char o[ 50 ]; sprintf( o, "%d", value ); return string( o ); }
		string GetFormat( ) { return "int"; }
};

/** The long property.
*/

class PropertyLong : public Property < long >
{
	public:
		bool Parse( string &_value ) { value = atol( _value.c_str( ) ); return true; }
		string Serialise( ) { char o[ 50 ]; sprintf( o, "%ld", value ); return string( o ); }
		string GetFormat( ) { return "long"; }
};

/** The double property.
*/

class PropertyDouble : public Property < double >
{
	public:
		bool Parse( string &_value ) { value = atof( _value.c_str( ) ); return true; }
		string Serialise( ) { char o[ 50 ]; sprintf( o, "%.1f", value ); return string( o ); }
		string GetFormat( ) { return "double"; }
};

/** The rgb colour type.
*/

typedef struct rgb_colour_type
{
	public:
		uint8_t r, g, b, a;
};

/** The colour property.
*/

class PropertyColour : public Property < rgb_colour_type >
{
	public:
		bool Parse( string &_value )
		{
			unsigned int ir, ig, ib, ia = 255;
			if ( sscanf( _value.c_str( ), "0x%2x%2x%2x", &ir, &ig, &ib ) == 3 ||
				 sscanf( _value.c_str( ), "#%2x%2x%2x%2x", &ia, &ir, &ig, &ib ) == 4 )
			{
				value.r = ir;
				value.g = ig;
				value.b = ib;
				value.a = ia;
				return true;
			}
			return false;
		}

		string Serialise( )
		{
			char _value[ 20 ];
			sprintf( _value, "0x%02x%02x%02x", value.r, value.g, value.b );
			return string( _value );
		}

		string GetFormat( )
		{
			return "colour";
		}
};

/** The position type.
*/

typedef struct position_type
{
	double x, y, w, h;
};

/** The position property.
*/

class PropertyPosition : public Property < position_type >
{
	public:
		bool end_value_used;
		position_type end_value;

		PropertyPosition( ) :
			end_value_used( false )
		{
		}

		bool Parse( string &_value )
		{
			if ( _value[ 0 ] != '+' )
			{
				return sscanf( _value.c_str( ), "%lf,%lf:%lfx%lf", &value.x, &value.y, &value.w, &value.h ) == 4;
			}
			else
			{
				end_value_used = true;
				return sscanf( _value.c_str( ) + 1, "%lf,%lf:%lfx%lf", &end_value.x, &end_value.y, &end_value.w, &end_value.h ) == 4;
			}
		}

		string Serialise( )
		{
			char _value[ 100 ];
			sprintf( _value, "%.1f,%.1f:%.1fx%.1f", value.x, value.y, value.w, value.h );
			return string( _value );
		}

		string GetFormat( )
		{
			return "position";
		}

		position_type GetValue( double position )
		{
			if ( end_value_used )
			{
				position_type temp;
				temp.x = lerp( value.x, end_value.x, position );
				temp.y = lerp( value.y, end_value.y, position );
				temp.w = lerp( value.w, end_value.w, position );
				temp.h = lerp( value.h, end_value.h, position );
				return temp;
			}
			else
			{
				return value;
			}
		}
};

/** The size type.
*/

typedef struct size_type
{
	int w, h;
};

/** The size property.
*/

class PropertySize : public Property < size_type >
{
	public:
		bool Parse( string &_value )
		{
			return sscanf( _value.c_str( ), "%dx%d", &value.w, &value.h ) == 2;
		}

		string Serialise( )
		{
			char _value[ 100 ];
			sprintf( _value, "%dx%d", value.w, value.h );
			return string( _value );
		}

		string GetFormat( )
		{
			return "size";
		}
};

/** The file list property.
*/

class PropertyFile : public Property < queue < string > >
{
	public:
		bool Parse( string &_value )
		{
			value.push( _value );
			return true;
		}

		string Serialise( )
		{
			string _value = "";
			if ( value.size( ) )
			{
				_value = value.front( );
				value.pop( );
				value.push( _value );
				for ( int i = value.size( ); i > 1; i -- )
				{
					string t = value.front( );
					_value += "," + t;
					value.pop( );
					value.push( t );
				}
			}
			return _value;
		}

		string GetFormat( )
		{
			return "file";
		}
};

/** Base class for frame filters.
*/

class PPMFrameFilter : public EffectUtils
{
	protected:
		PropertyInstance < PropertyLong > start;
		PropertyInstance < PropertyLong > end;
		PropertyInstance < PropertyLong > cycle;

	public:
		vector < PropertyBase * > properties;

		PPMFrameFilter( ) :
			start( "start", "starting frame number", 0 ),
			end( "end", "ending frame number", LONG_MAX ),
			cycle( "cycle", "cycle after n frames", LONG_MAX )
		{
			AddProperty( &start );
			AddProperty( &end );
			AddProperty( &cycle );
		}

		virtual ~PPMFrameFilter( )
		{
		}

		void AddProperty( PropertyBase *property )
		{
			properties.push_back( property );
		}

		string Summary( )
		{
			string ret = "";
			for ( unsigned int i = 3; i < properties.size( ); i ++ )
				ret += properties[ i ]->Summary( ) + "|";
			return ret;
		}

		// The main filter which assumes that the stack is correctly seeded
		virtual bool Filter( int64_t index )
		{
			PPMFrameStack &stack = PPMFrameStack::GetStack( );
			PPMFrame *frame = stack.Pop( );
			bool ret = Filter( *frame, index );
			stack.Push( frame );
			return ret;
		}

		// This or the previous method must be overriden in the filter classes
		virtual bool Filter( PPMFrame &frame, int64_t index )
		{
			return false;
		}

		virtual bool SetProperty( string name, string value ) 
		{ 
			bool ret = false;
			for ( vector< PropertyBase * >::iterator item = properties.begin( ); 
				  ret == false && item != properties.end( ); 
				  item ++ )
			{
				if ( name == ( *item )->GetName( ) )
					ret = ( *item )->Parse( value );
			}
			return ret; 
		}

		bool Process( int64_t index )
		{
			if ( index % cycle.value >= start.value && index % cycle.value < end.value )
				return Filter( index % cycle.value - start.value );
			else
				return true;
		}
};

/** Base class for frame filter factories.
*/

class PPMFrameFilterFactory
{
	public:
		virtual string GetSwitch( ) = 0;
		virtual string GetDescription( string ) = 0;
		virtual PPMFrameFilter *GetInstance( string ) = 0;
};

/** Stack manipulation filter.
*/

class PPMStack : public PPMFrameFilter
{
	private:
		string action;

	public:
		PPMStack( string _action ) :
			action( _action )
		{
		}

		bool Filter( int64_t index )
		{
			PPMFrameStack &stack = PPMFrameStack::GetStack( );

			if ( action == "--stack-dupe" )
			{
				PPMFrame *f = stack.Pop( );
				if ( f != NULL )
				{
					stack.Push( *f );
					stack.Push( f );
				}
			}
			else if ( action == "--stack-remove" )
			{
				PPMFrame *f = stack.Pop( );
				delete f;
			}
			else if ( action == "--stack-swap" )
			{
				PPMFrame *f = stack.Pop( );
				PPMFrame *g = stack.Pop( );
				stack.Push( f );
				stack.Push( g );
			}
			else if ( action == "--stack-stop" )
			{
				PPMFrame *f = stack.Pop( );
				stack.SetSeed( *f );
				stack.Push( f );
			}

			return true;
		}
};

/** Stack filter factory.
*/

class PPMStackFactory : public PPMFrameFilterFactory
{
	public:
		string GetSwitch( )
		{
			return "--stack-dupe,--stack-remove,--stack-swap,--stack-stop";
		}

		string GetDescription( string _switch )
		{
			if ( _switch == "--stack-dupe" )
				return "Duplicate the current frame on the top of the stack";
			if ( _switch == "--stack-remove" )
				return "Remove the frame at the top of stack";
			if ( _switch == "--stack-swap" )
				return "Swap the top two frames on the stack";
			if ( _switch == "--stack-stop" )
				return "Use the current frame to seed the stack on the next frame";
			return "Unknown operation";
		}

		PPMFrameFilter *GetInstance( string _switch )
		{
			return new PPMStack( _switch );
		}
};

/** Stack image filter.
*/

class PPMStackImage : public PPMFrameFilter
{
	private:
		PPMFrame last_image;
		PropertyInstance < PropertyFile > images;
		PropertyInstance < PropertyColour > colour;
		PropertyInstance < PropertyInt > period;
		PropertyInstance < PropertyBool > scale;
		int position;

	public:
		PPMStackImage( ) :
			last_image( 100, 100 ),
			images( "file", "image to stack" ),
			colour( "colour", "produce a colour frame if no file is given", "0x000000" ),
			period( "period", "number of frames each image lasts", "25" ),
			scale( "scale", "scale image to current frame size", "false" ),
			position( 0 )
		{
			AddProperty( &images );
			AddProperty( &colour );
			AddProperty( &period );
			AddProperty( &scale );
		}

		bool Filter( int64_t index )
		{
			PPMFrameStack &stack = PPMFrameStack::GetStack( );

			if ( position == 0 )
			{
				if ( images.value.size() > 1 )
				{
					string image = images.value.front( );
					images.value.pop( );
					images.value.push( image );
					last_image.Load( image );
				}
				else if ( index == 0 && images.value.size( ) == 1 )
				{
					string image = images.value.front( );
					last_image.Load( image );
				}
				else if ( index == 0 && images.value.size( ) == 0 )
				{
					PPMFrame *f = stack.Pop( );
					last_image.Scale( f->GetWidth( ), f->GetHeight( ) );
					last_image.FillArea( 0, 0, last_image.GetWidth( ), last_image.GetHeight( ), (uint8_t *)&colour.value );
					stack.Push( f );
				}

				if ( scale.value )
				{
					PPMFrame *f = stack.Pop( );
					last_image.Scale( f->GetWidth( ), f->GetHeight( ) );
					stack.Push( f );
				}
			}

			stack.Push( last_image );

			position = ( position + 1 ) % period.value;

			return true;
		}
};

/** Stack image factory.
*/

class PPMStackImageFactory : public PPMFrameFilterFactory
{
	public:
		string GetSwitch( ) { return "--stack-image"; }
		string GetDescription( string _switch ) { return "Place an image from the specified file on the stack"; }
		PPMFrameFilter *GetInstance( string _switch ) { return new PPMStackImage( ); }
};

/** Group manipulation filter.
*/

class PPMGroup : public PPMFrameFilter
{
	private:
		vector < string > properties;

	public:
		bool Filter( int64_t index )
		{
			return true;
		}

		bool SetProperty( string name, string value ) 
		{ 
			string property = name + "=" + value;
			properties.push_back( property );
			return true;
		}

		void CommunicateProperties( PPMFrameFilter *filter )
		{
			for ( unsigned int i = 0; i < properties.size( ); i ++ )
			{
				string property = properties[ i ];
				int split = property.find_first_of( "=" );
				string name = property.substr( 0, split );
				string value = property.substr( split + 1 );
				filter->SetProperty( name, value );
			}
		}
};

/** Group filter factory.
*/

class PPMGroupFactory : public PPMFrameFilterFactory
{
	public:
		string GetSwitch( ) { return "--group"; }
		string GetDescription( string _switch ) { return "Group the following effects"; }
		PPMFrameFilter *GetInstance( string _switch ) { return new PPMGroup( ); }
};

/** Black and white filter.
*/

class PPMBlackAndWhite : public PPMFrameFilter
{
	public:
		virtual bool Filter( PPMFrame &frame, int64_t index )
		{
			uint8_t *p = frame.GetImage( );
		    uint8_t r, g, b;
        	uint8_t *q = p;
        	uint8_t *end = p + frame.GetWidth( ) * frame.GetHeight( ) * 4;
        	while ( p < end )
        	{
           		r = *p ++;
           		g = *p ++;
           		b = *p ++;
           		r = (uint8_t)( 0.299 * r + 0.587 * g + 0.114 * b );
           		*q ++ = r;
           		*q ++ = r;
           		*q ++ = r;
				*q ++ = *p ++;
        	}
			return true;
		}
};

/** Black and white filter factory.
*/

class PPMBlackAndWhiteFactory : public PPMFrameFilterFactory
{
	public:
		string GetSwitch( ) { return "--black-and-white"; }
		string GetDescription( string _switch ) { return "Convert input into black and white"; }
		PPMFrameFilter *GetInstance( string _switch ) { return new PPMBlackAndWhite( ); }
};

/** Mono filter.
*/

class PPMMono : public PPMFrameFilter
{
	public:
		virtual bool Filter( PPMFrame &frame, int64_t index )
		{
			uint8_t *p = frame.GetImage( );
		    uint8_t r, g, b;
        	uint8_t *q = p;
        	uint8_t *end = p + frame.GetWidth( ) * frame.GetHeight( ) * 4;
        	while ( p < end )
        	{
           		r = *p ++;
           		g = *p ++;
           		b = *p ++;
				r = ( ( r + g + b ) / 3 ) > 127 ? 255 : 0;
           		*q ++ = r;
           		*q ++ = r;
           		*q ++ = r;
				*q ++ = *p ++;
        	}
			return true;
		}
};

/** Mono filter factory.
*/

class PPMMonoFactory : public PPMFrameFilterFactory
{
	public:
		string GetSwitch( ) { return "--mono"; }
		string GetDescription( string _switch ) { return "Convert input into mono"; }
		PPMFrameFilter *GetInstance( string _switch ) { return new PPMMono( ); }
};

/** Sepia filter.
*/

class PPMSepia : public PPMFrameFilter
{
	public:
		bool Filter( PPMFrame &frame, int64_t index )
		{
			uint8_t *p = frame.GetImage( );
		    uint8_t r, g, b;
        	uint8_t *q = p;
        	uint8_t *end = p + frame.GetWidth( ) * frame.GetHeight( ) * 4;
        	while ( p < end )
        	{
           		r = *p ++;
           		g = *p ++;
           		b = *p ++;
           		r = (uint8_t)( 0.299 * r + 0.587 * g + 0.114 * b );
           		*q ++ = clamp( r + 15, 0, 255 );
           		*q ++ = r;
           		*q ++ = clamp( r - 15, 0, 255 );
				*q ++ = *p ++;
        	}
			return true;
		}
};

/** Sepia filter factory.
*/

class PPMSepiaFactory : public PPMFrameFilterFactory
{
	public:
		string GetSwitch( ) { return "--sepia"; }
		string GetDescription( string _switch ) { return "Convert input into sepia"; }
		PPMFrameFilter *GetInstance( string _switch ) { return new PPMSepia( ); }
};

/** Xor filter.
*/

class PPMXor : public PPMFrameFilter
{
	public:
		virtual bool Filter( PPMFrame &frame, int64_t index )
		{
			uint8_t *p = frame.GetImage( );
		    uint8_t r, g, b;
        	uint8_t *q = p;
        	uint8_t *end = p + frame.GetWidth( ) * frame.GetHeight( ) * 4;
        	while ( p < end )
        	{
           		r = *p ++ ^ 255;
           		g = *p ++ ^ 255;
           		b = *p ++ ^ 255;
           		*q ++ = r;
           		*q ++ = g;
           		*q ++ = b;
				*q ++ = *p ++;
        	}
			return true;
		}
};

/** Xor filter factory.
*/

class PPMXorFactory : public PPMFrameFilterFactory
{
	public:
		string GetSwitch( ) { return "--xor"; }
		string GetDescription( string _switch ) { return "Negate the colours"; }
		PPMFrameFilter *GetInstance( string _switch ) { return new PPMXor( ); }
};

/** Extract Channel filter.
*/

class PPMExtractChannel : public PPMFrameFilter
{
	private:
		int m_channel;

	public:
		PPMExtractChannel( int channel ) :
			m_channel( channel )
		{
		}

		bool Filter( PPMFrame &frame, int64_t index )
		{
			uint8_t *p = frame.GetImage( );
			uint8_t *q = frame.GetImage( );
        	uint8_t *end = p + frame.GetWidth( ) * frame.GetHeight( ) * 4;
        	while ( p < end )
        	{
				*p ++ = m_channel == 0 ? *q : 0; q ++;
				*p ++ = m_channel == 1 ? *q : 0; q ++;
				*p ++ = m_channel == 2 ? *q : 0; q ++;
				*p ++ = *q ++;
        	}
			return true;
		}
};

/** Extract Channel filter factory.
*/

class PPMExtractChannelFactory : public PPMFrameFilterFactory
{
	public:
		string GetSwitch( ) { return "--extract-red,--extract-green,--extract-blue"; }
		string GetDescription( string _switch ) 
		{ 
			if ( _switch == "--extract-red" )
				return "Extract the red channel"; 
			else if ( _switch == "--extract-green" )
				return "Extract the green channel"; 
			else if ( _switch == "--extract-blue" )
				return "Extract the blue channel"; 
			return "Unknown operation " + _switch;
		}
		PPMFrameFilter *GetInstance( string _switch ) 
		{ 
			if ( _switch == "--extract-red" )
				return new PPMExtractChannel( 0 ); 
			else if ( _switch == "--extract-green" )
				return new PPMExtractChannel( 1 ); 
			else if ( _switch == "--extract-blue" )
				return new PPMExtractChannel( 2 ); 
			return NULL;
		}
};

/** Border filter.
*/

class PPMBorder : public PPMFrameFilter
{
	private:
		PPMFrame border;
		PropertyInstance < PropertyPosition > position;
		PropertyInstance < PropertyColour > colour;

	public:
		PPMBorder( ) :
			position( "position", "border dimensions", "10,10:80x80" ),
			colour( "colour", "colour", "0x0000ff" )
		{
			AddProperty( &position );
			AddProperty( &colour );
		}

		virtual bool Filter( PPMFrame &frame, int64_t index )
		{
			position_type pos = position.GetValue( (double)index / (double)( end.value - start.value ) );

			int width = frame.GetWidth( );
			int height = frame.GetHeight( );

			border.Scale( width, height );
			border.FillArea( 0, 0, width, height, (uint8_t *)&colour.value );

			int new_width = (int)( width * pos.w / 100.0 );
			int new_height = (int)( height * pos.h / 100.0 );
			int new_x = (int)( width * pos.x / 100.0 );
			int new_y = (int)( height * pos.y / 100.0 );

			border.Overlay( frame, new_x, new_y, new_width, new_height );
			frame.Copy( border );

			return true;
		}
};

/** Border filter factory.
*/

class PPMBorderFactory : public PPMFrameFilterFactory
{
	public:
		string GetSwitch( ) { return "--border"; }
		string GetDescription( string _switch ) { return "Add a border of a given width, height and colour"; }
		PPMFrameFilter *GetInstance( string _switch ) { return new PPMBorder( ); }
};

/** Gamma filter.
*/

class PPMGamma : public PPMFrameFilter
{
	private:
		PropertyInstance < PropertyDouble > gamma;
     	uint8_t lookup[256];

	public:
		PPMGamma( ) :
			gamma( "gamma", "gamma correction level", "2.0" )
		{
			AddProperty( &gamma );
		}

		virtual bool Filter( PPMFrame &frame, int64_t index )
		{
     		double exp = 1 / gamma.value;

     		for( int index = 0; index < 256; index ++ )
       			lookup[ index ] = ( uint8_t ) ( pow( ( double ) index / 255.0, exp ) * 255 );

			uint8_t *p = frame.GetImage( );
        	uint8_t *q = p;
        	uint8_t *end = p + frame.GetWidth( ) * frame.GetHeight( ) * 4;

        	while ( p < end )
        	{
				*q ++ = lookup[ *p ++ ];
				*q ++ = lookup[ *p ++ ];
				*q ++ = lookup[ *p ++ ];
				*q ++ = *p ++;
        	}
			return true;
		}
};

/** Gamma factory.
*/

class PPMGammaFactory : public PPMFrameFilterFactory
{
	public:
		string GetSwitch( ) { return "--gamma"; }
		string GetDescription( string _switch ) { return "Gamma correction"; }
		PPMFrameFilter *GetInstance( string _switch ) { return new PPMGamma( ); }
};

/** Obscure filter.
*/

class PPMObscure : public PPMFrameFilter
{
	private:
		PropertyInstance < PropertySize > size;

	public:
		PPMObscure( ) :
			size( "size", "size of pixelation", "4x4" )
		{
			AddProperty( &size );
		}

		void Average( uint8_t *start, int width, int height, int offset )
		{
			double r = (double)*( start );
			double g = (double)*( start + 1 );
			double b = (double)*( start + 2 );

			for ( int y = 0; y < height; y ++ )
			{
				uint8_t *p = start + y * offset;
				for ( int x = 0; x < width; x ++ )
				{
					r = ( r + (double)*( p ++ ) ) / 2;
					g = ( g + (double)*( p ++ ) ) / 2;
					b = ( b + (double)*( p ++ ) ) / 2;
					p ++;
				}
			}

			for ( int y = 0; y < height; y ++ )
			{
				uint8_t *p = start + y * offset;
				for ( int x = 0; x < width; x ++ )
				{
					*p ++ = (uint8_t)r;
					*p ++ = (uint8_t)g;
					*p ++ = (uint8_t)b;
					p ++;
				}
			}
		}

		virtual bool Filter( PPMFrame &frame, int64_t index )
		{
			size_type pixel_size = size.value;
			int rw = pixel_size.w;
			int rh = pixel_size.h;
			uint8_t *io = frame.GetImage( );
			int width = frame.GetWidth( );
			int height = frame.GetHeight( );
			for ( int w = 0; w < width; w += rw )
			{
				for ( int h = 0; h < height; h += rh )
				{
					int aw = w + rw > width ? rw - ( w + rw - width ) : rw;
					int ah = h + rh > height ? rh - ( h + rh - height ) : rh;
					Average( io + h * width * 4 + w * 4, aw, ah, width * 4 );
				}
			}
			return true;
		}
};

/** Obscure factory.
*/

class PPMObscureFactory : public PPMFrameFilterFactory
{
	public:
		string GetSwitch( ) { return "--obscure"; }
		string GetDescription( string _switch ) { return "Obscuring mask"; }
		PPMFrameFilter *GetInstance( string _switch ) { return new PPMObscure( ); }
};

/** Copy filter.
*/

class PPMCopy : public PPMFrameFilter
{
	private:
		PPMFrame border;
		PropertyInstance < PropertyPosition > position;

	public:
		PPMCopy( ) :
			position( "position", "copy dimensions", "10,10:80x80" )
		{
			AddProperty( &position );
		}

		virtual bool Filter( int64_t index )
		{
			PPMFrameStack &stack = PPMFrameStack::GetStack( );

			// Get the image at the top of the stack
			PPMFrame *frame = stack.Pop( );
			stack.Push( frame );

			// Determine the current area to copy
			position_type pos = position.GetValue( (double)index / (double)( end.value - start.value ) );

			int width = frame->GetWidth( );
			int height = frame->GetHeight( );
			int w = (int)( width * pos.w / 100.0 );
			int h = (int)( height * pos.h / 100.0 );
			int x = (int)( width * pos.x / 100.0 );
			int y = (int)( height * pos.y / 100.0 );

			border.Scale( w, h );

			uint8_t *dest = border.GetImage( );
			uint8_t *src = frame->GetImage( );

			for ( int i = 0; i < h; i ++ )
			{
				uint8_t *p = src + ( ( y + i ) * width + x ) * 4;
				memcpy( dest, p, w * 4 );
				dest += w * 4;
			}

			stack.Push( border );

			return true;
		}
};

/** Border filter factory.
*/

class PPMCopyFactory : public PPMFrameFilterFactory
{
	public:
		string GetSwitch( ) { return "--stack-copy"; }
		string GetDescription( string _switch ) { return "Stack a copy of an area of the top of stack"; }
		PPMFrameFilter *GetInstance( string _switch ) { return new PPMCopy( ); }
};

/** Scale filter.
*/

class PPMScale : public PPMFrameFilter
{
	private:
		PropertyInstance < PropertySize > size;
		PropertyInstance < PropertyInt > quality;

	public:
		PPMScale( ) :
			size( "size", "size of image to rescale to", "360x288" ),
			quality( "quality", "quality of scaling algorithm (0 to 3)", 3 )
		{
			AddProperty( &size );
			AddProperty( &quality );
		}

		virtual bool Filter( PPMFrame &frame, int64_t index )
		{
			return frame.Scale( size.value.w, size.value.h, quality.value );
		}
};

/** Scale filter factory.
*/

class PPMScaleFactory : public PPMFrameFilterFactory
{
	public:
		string GetSwitch( ) { return "--scale"; }
		string GetDescription( string _switch ) { return "Scale the frame to a given size and quality"; }
		PPMFrameFilter *GetInstance( string _switch ) { return new PPMScale( ); }
};

/** Overlay filter.
*/

class PPMOverlay : public PPMFrameFilter
{
	private:
		PropertyInstance < PropertyFile > images;
		PropertyInstance < PropertyPosition > position;
		PropertyInstance < PropertyInt > quality;
		PropertyInstance < PropertyDouble > weight;

	public:
		PPMOverlay( ) :
			images( "file", "file to use, or pop the top of the stack if not specifed" ),
			position( "position", "position to overlay the image", "80,10:10x10" ),
			quality( "quality", "quality of scaling algorithm (0 to 3)", 3 ),
			weight( "weight", "weight of transparency", 1 )
		{
			AddProperty( &images );
			AddProperty( &position );
			AddProperty( &quality );
			AddProperty( &weight );
		}

		virtual bool Filter( int64_t index )
		{
			bool ret = true;

			PPMFrameStack &stack = PPMFrameStack::GetStack( );
			PPMFrame *tos = stack.Pop( );
			PPMFrame *frame = tos;
			
			if ( images.value.size( ) == 0 )
				frame = stack.Pop( );

			position_type pos = position.GetValue( (double)index / (double)( end.value - start.value ) );

			int width = frame->GetWidth( );
			int height = frame->GetHeight( );
			int x = (int)( (double)width * pos.x / 100.0 );
			int y = (int)( (double)height * pos.y / 100.0 );
			width = (int)( (double)width * pos.w / 100.0 );
			height = (int)( (double)height * pos.h / 100.0 );

			if ( images.value.size( ) > 0 )
			{
				string image = images.value.front( );
				if ( index == 0 )
				{
					images.value.pop( );
					images.value.push( image );
				}
				ret = frame->Overlay( image, x, y, width, height, weight.value );
			}
			else
			{
				ret = frame->Overlay( *tos, x, y, width, height, weight.value );
				delete tos;
			}

			stack.Push( frame );

			return ret;
		}
};

/** Overlay filter factory.
*/

class PPMOverlayFactory : public PPMFrameFilterFactory
{
	public:
		string GetSwitch( ) { return "--overlay"; }
		string GetDescription( string _switch ) { return "Overlay an image at a given position and size"; }
		PPMFrameFilter *GetInstance( string _switch ) { return new PPMOverlay( ); }
};

/** Charcoal Filter
*/

class PPMCharcoal : public PPMBlackAndWhite
{
	private:
		PPMFrame copy;
		PropertyInstance < PropertyInt > radius;

	public:
		PPMCharcoal( ) :
			radius( "radius", "radius of charcoalisation", 2 )
		{
			AddProperty( &radius );
		}

		int GetPix( uint8_t *pixels, int width, int height, int x, int y )
		{
			if ( x < 0 || x >= width || y < 0 || y >= height )
				return 0;
			else
				return *( pixels + y * width * 4 + x * 4 );
		}

		virtual bool Filter( PPMFrame &frame, int64_t index )
		{
			int width;
			int height;

			PPMBlackAndWhite::Filter( frame, index );

			copy.Copy( frame );

			uint8_t *dest = frame.GetImage( );
			uint8_t *pixels = copy.GetImage( width, height );

			for ( int y = 0; y < height; y ++ )
			{
				for ( int x = 0; x < width; x ++ )
				{
					int pix[ 3 ][ 3 ];

					pix[ 0 ][ 0 ] = GetPix( pixels, width, height, x - radius.value, y - radius.value );
					pix[ 0 ][ 1 ] = GetPix( pixels, width, height, x               , y - radius.value );
					pix[ 0 ][ 2 ] = GetPix( pixels, width, height, x + radius.value, y - radius.value );
					pix[ 1 ][ 0 ] = GetPix( pixels, width, height, x - radius.value, y                );
					pix[ 1 ][ 2 ] = GetPix( pixels, width, height, x + radius.value, y                );
					pix[ 2 ][ 0 ] = GetPix( pixels, width, height, x - radius.value, y + radius.value );
					pix[ 2 ][ 1 ] = GetPix( pixels, width, height, x               , y + radius.value );
					pix[ 2 ][ 2 ] = GetPix( pixels, width, height, x + radius.value, y + radius.value );

		      		double s1 = (pix[2][0] - pix[0][0]) + 2*(pix[2][1] - pix[0][1]) + (pix[2][2] - pix[2][0]);
		      		double s2 = (pix[0][2] - pix[0][0]) + 2*(pix[1][2] - pix[1][0]) + (pix[2][2] - pix[2][0]);
		      		double sum = sqrt( (long) s1 * s1 + (long) s2 * s2 );

					*dest ++ = (uint8_t)( 255 - sum );
					*dest ++ = (uint8_t)( 255 - sum );
					*dest ++ = (uint8_t)( 255 - sum );
					*dest ++ = 255;
				}
			}

			return true;
		}
};

/** Charcoal filter factory.
*/

class PPMCharcoalFactory : public PPMFrameFilterFactory
{
	public:
		string GetSwitch( ) { return "--charcoal"; }
		string GetDescription( string _switch ) { return "Charcoalise the picture"; }
		PPMFrameFilter *GetInstance( string _switch ) { return new PPMCharcoal( ); }
};

/** Motion blur/fade filter.
*/

class PPMMotionBlur : public PPMFrameFilter
{
	private:
		PropertyInstance < PropertyInt > period;
		PropertyInstance < PropertyDouble > multiplier;
		PropertyInstance < PropertyBool > reverse;
		PropertyInstance < PropertyBool > alternate;
		int position;

	public:
		PPMMotionBlur( ) :
			period( "period", "period of blur", 10 ),
			multiplier( "fixed", "fixed multiplier", 0.0 ),
			reverse( "reverse", "reverse the effect", "false" ),
			alternate( "alternate", "alternate between reverse and forward", "false" ),
			position( 0 )
		{
			AddProperty( &period );
			AddProperty( &multiplier );
			AddProperty( &reverse );
			AddProperty( &alternate );
		}

		virtual bool Filter( PPMFrame &a_frame, int64_t index )
		{
			PPMFrameStack &stack = PPMFrameStack::GetStack( );
			PPMFrame *tos = stack.Pop( );

			if ( tos != NULL )
			{
				double weight = multiplier.value != 0.0 ? multiplier.value : ( 1.0 / period.value ) * position;
				int width;
				int height;
				PPMFrame &b_frame = *tos;

				if ( reverse.value )
					weight = 1 - weight;

				uint8_t *dest = a_frame.GetImage( width, height );
				b_frame.Scale( width, height );
				uint8_t *image = b_frame.GetImage( );
				uint8_t *max = image + width * height * 4;
				uint8_t *p = dest;

				while ( image < max )
				{
					*p ++ = (uint8_t)clamp( (int)( *dest ++ * weight + *image ++ * ( 1 - weight ) ), 0, 255 );
					*p ++ = (uint8_t)clamp( (int)( *dest ++ * weight + *image ++ * ( 1 - weight ) ), 0, 255 );
					*p ++ = (uint8_t)clamp( (int)( *dest ++ * weight + *image ++ * ( 1 - weight ) ), 0, 255 );
					*p ++ = (uint8_t)clamp( (int)( *dest ++ * weight + *image ++ * ( 1 - weight ) ), 0, 255 );
				}

				position = ( position + 1 ) % period.value;

				if ( alternate.value && position ==  0 )
					reverse.value = !reverse.value;

				delete tos;
			}

			return true;
		}
};

/** Motion blur filter factory.
*/

class PPMMotionBlurFactory : public PPMFrameFilterFactory
{
	public:
		string GetSwitch( ) { return "--motion-blur"; }
		string GetDescription( string _switch ) { return "Apply a motion blur filter"; }
		PPMFrameFilter *GetInstance( string _switch ) { return new PPMMotionBlur( ); }
};

/** Chroma key filter.
*/

class PPMChromaKey : public PPMFrameFilter
{
	private:
		PropertyInstance < PropertyFile > images;
		PropertyInstance < PropertyColour > chroma;

	protected:
		PPMFrame key_frame;

	public:
		PPMChromaKey( ) :
			images( "file", "file to use, or pop the top of the stack if not specified" ),
			chroma( "colour", "colour to use", "0x303000" )
		{
			AddProperty( &images );
			AddProperty( &chroma );
		}

		virtual bool Filter( int64_t index )
		{
			PPMFrameStack &stack = PPMFrameStack::GetStack( );
			PPMFrame *tos = stack.Pop( );
			PPMFrame *frame = tos;
			
			if ( index == 0 && !images.value.empty( ) )
			{
				if ( images.value.size( ) > 1 ) 
				{
					string image = images.value.front( );
					key_frame.Load( image );
					images.value.pop( );
					images.value.push( image );
				}
				else if ( key_frame.GetImage( ) == NULL )
				{
					string image = images.value.front( );
					key_frame.Load( image );
				}
			}
			else if ( images.value.empty( ) )
			{
				key_frame.Copy( *tos );
				delete tos;
				frame = stack.Pop( );
			}

			int width;
			int height;
			uint8_t *dest = frame->GetImage( width, height );
			uint8_t *key = key_frame.GetImage( );

			if ( key != NULL )
			{
				key_frame.Scale( width, height );
				key = key_frame.GetImage( );

    			uint8_t *k = key;
    			uint8_t *p = dest;
    			uint8_t kr, kg, kb;
				int max = width * height * 4;
    			while ( p < ( dest + max ) ) 
      			{
        			kr = *k;
        			kg = *( k + 1 );
        			kb = *( k + 2 );
        			if ( kb >= chroma.value.b && kr <= chroma.value.r && kg <= chroma.value.g ) 
					{
          				p += 4;
          				k += 4;
        			} 
					else 
					{
          				*p ++ = *k ++; 
          				*p ++ = *k ++; 
          				*p ++ = *k ++; 
          				*p ++ = *k ++; 
        			}
      			}
			}

			stack.Push( frame );

			return true;
		}
};

/** Chroma filter factory.
*/

class PPMChromaKeyFactory : public PPMFrameFilterFactory
{
	public:
		string GetSwitch( ) { return "--chroma"; }
		string GetDescription( string _switch ) { return "Apply a chroma filter from an image file and a colour"; }
		PPMFrameFilter *GetInstance( string _switch ) { return new PPMChromaKey( ); }
};

/** Luma filter.
*/

class PPMLuma : public PPMFrameFilter
{
	private:
		PPMFrame luma;
		PropertyInstance < PropertyFile > images;
		PropertyInstance < PropertyInt > period;
		PropertyInstance < PropertyBool > reverse;
		PropertyInstance < PropertyBool > alternate;
		PropertyInstance < PropertyDouble > smooth;
		PropertyInstance < PropertyDouble > rollfx;
		PropertyInstance < PropertyDouble > rollfy;
		PropertyInstance < PropertyBool > alpha;
		PropertyInstance < PropertyDouble > fixed;
		int position_in_period;

	public:
		PPMLuma( ) :
			images( "file", "image file to use or pop top of stack if not specified" ),
			period( "period", "period of effect", 25 ),
			reverse( "reverse", "reverse the effect", false ),
			alternate( "alternate", "alternate between forward and reverse", false ),
			smooth( "smooth", "smoothness of transition",  0.1 ),
			rollfx( "rollfx", "horizontal luma mask roll fraction",  0.0 ),
			rollfy( "rollfy", "vertical luma mask roll fraction",  0.0 ),
			alpha( "alpha", "apply luma to alpha channel only",  false ),
			fixed( "fixed", "fixed value (between 0 and 1)",  0 ),
			position_in_period( 0 )
		{
			AddProperty( &images );
			AddProperty( &period );
			AddProperty( &reverse );
			AddProperty( &alternate );
			AddProperty( &smooth );
			AddProperty( &rollfx );
			AddProperty( &rollfy );
			AddProperty( &alpha );
			AddProperty( &fixed );
		}

		virtual bool Filter( int64_t index )
		{
			PPMFrameStack &stack = PPMFrameStack::GetStack( );
			PPMFrame *top = stack.Pop( );
			PPMFrame *frame = top;

			if ( position_in_period == 0 && !images.value.empty( ) )
			{
				string image = images.value.front( );
				luma.Load( image );
				images.value.pop( );
				images.value.push( image );
			}
			else if ( images.value.empty( ) )
			{
				luma.Copy( *top );
				delete top;
				frame = stack.Pop( );
			}

			int width = frame->GetWidth( );
			int height = frame->GetHeight( );

			PPMFrame *tos = stack.Pop( );
			PPMFrame &current_frame = *tos;

			if ( luma.GetImage( ) != NULL )
			{
				if ( !alpha.value )
					current_frame.Scale( width, height );

				luma.Scale( width, height );

				uint8_t *a = frame->GetImage( );
				uint8_t *output = a;
				uint8_t *b = current_frame.GetImage( );
				uint8_t *cb = luma.GetImage( );	// image base address
				uint8_t *c = cb;		// current rgba address 
				uint8_t *max = output + width * height * 4;
				double relpip = ( (double)position_in_period ) / (double)period.value;
				double mult = (1.+smooth.value) * ( ( (double)position_in_period + 1 ) / (double)period.value );
				if ( fixed.value != 0.0 ) mult = fixed.value;

    			while ( output < max ) 
      			{
					int cc = ( (c-cb) % ( 4 * width ) ) / 4;  // column of current rgba
					int cr = (c-cb) / ( 4 * width );	  // row of current rgba

					uint8_t *cl = cb + 4 * (		  // roll with periodic BC
					  ( ( cr + (int)ceil(fabs(rollfy.value))*height 
					     - (int)(relpip*rollfy.value*height)) % height ) * width 
					  + ( cc + (int)ceil(fabs(rollfx.value))*width 
					      - (int)(relpip*rollfx.value*width) ) % width );

					double sample = !reverse.value ? (double)*cl / 255.0 : 1 - (double)*cl / 255.0;
					double mix = smoothstep( sample, sample + smooth.value, mult );

					if ( !alpha.value )
					{
						*output ++ = (uint8_t)( ( *a ++ * mix ) + *b ++ * ( 1.0 - mix ) );
						*output ++ = (uint8_t)( ( *a ++ * mix ) + *b ++ * ( 1.0 - mix ) );
						*output ++ = (uint8_t)( ( *a ++ * mix ) + *b ++ * ( 1.0 - mix ) );
						*output ++ = (uint8_t)( ( *a ++ * mix ) + *b ++ * ( 1.0 - mix ) );
					}
					else
					{
						output += 3;
						a += 3;
						*output ++ = (uint8_t)( ( *a ++ * mix ) );
						b += 4;
					}
					c += 4;
     			}
			}

			position_in_period = ( position_in_period + 1 ) % period.value;

			if ( position_in_period == 0 && alternate.value )
				reverse.value = !reverse.value;

			if ( !alpha.value )
				delete tos;
			else
				stack.Push( tos );

			stack.Push( frame );

			return true;
		}
};

/** Luma filter factory.
*/

class PPMLumaFactory : public PPMFrameFilterFactory
{
	public:
		string GetSwitch( ) { return "--luma"; }
		string GetDescription( string _switch ) { return "Apply a luma filter over a period from an image file"; }
		PPMFrameFilter *GetInstance( string _switch ) { return new PPMLuma( ); }
};

/** Extended PPM Frame for assisting in reading from pipes.
*/

class PPMFrameEffect : public PPMFrame
{
	private:
		int pid;
		FILE *reader;
		FILE *writer;

	public:

		PPMFrameEffect( ) :
			pid( -1 )
		{
		}

		virtual ~PPMFrameEffect( )
		{
			Stop( );
		}

		bool IsRunning( )
		{
			return pid != -1;
		}

		bool Start( const char *command )
		{
	        int input[ 2 ];
			int output[ 2 ];

        	pipe( input );
        	pipe( output );

        	this->pid = fork();

        	if ( this->pid == 0 )
        	{
            	dup2( output[ 0 ], STDIN_FILENO );
            	dup2( input[ 1 ], STDOUT_FILENO );

            	close( input[ 0 ] );
            	close( input[ 1 ] );
            	close( output[ 0 ] );
            	close( output[ 1 ] );

            	execl("/bin/sh", "sh", "-c", command, NULL );
            	exit( 255 );
			}
        	else
        	{
            	close( input[ 1 ] );
            	close( output[ 0 ] );

            	reader = fdopen( input[ 0 ], "r" );
            	writer = fdopen( output[ 1 ], "w" );
        	}
			return true;
		}

		void Stop( )
		{
			if ( pid != -1 )
			{
		        fclose( this->reader );
        		fclose( this->writer );
        		waitpid( this->pid, NULL, 0 );
				pid = -1;
			}
		}

		int ReadData( uint8_t *data, int length )
		{
			if ( IsRunning( ) )
				return fread( data, 1, length, reader );
			return 0;
		}

		int WriteData( uint8_t *data, int length )
		{
			if ( IsRunning( ) )
				return fwrite( data, 1, length, writer );
			return 0;
		}

		bool Flush( )
		{
			return fflush( writer ) == 0;
		}
};

/** Effect TV filters.
*/

class PPMEffectTV : public PPMFrameFilter
{
	private:
		int fx;
		PPMFrameEffect effect;

	public:
		PPMEffectTV( int _fx ) :
			fx( _fx )
		{
		}

		bool Filter( PPMFrame &frame, int64_t index )
		{
			if ( index == 0 )
			{
				char command[ 1024 ];
				sprintf( command, "ppmeffectv -e %d", fx );
				effect.Stop( );
				effect.Start( command );
			}

			effect.Copy( frame );
			effect.WriteImage( );
			effect.ReadImage( );
			frame.Copy( effect );

			return true;
		}
};

/** EffectTV filter factory.
*/

class PPMEffectTVFactory : public PPMFrameFilterFactory
{
	private:
		map < string, int > filters;

	public:
		virtual ~PPMEffectTVFactory( ){ }

		void to_lower( char *p )
		{
			char *q = p;
			while( *p )
				*q ++ = tolower( *p ++ );
		}

		string GetSwitch( )
		{
			string ret;
			strstream o;
			int effect;
			char name[ 256 ];
			FILE *p = popen( "ppmeffectv --list", "r" );
			while( fscanf( p, "%d: %s\n", &effect, name ) == 2 )
			{
				to_lower( name );
				filters[ name ] = effect;
				o << "--" << string( name ) << ",";
			}
			pclose( p );
			o >> ret;
			return ret;
		}

		string GetDescription( string _switch )
		{
			return "Run the " + _switch.substr( 2 ) + " effect from ppmeffectv";
		}

		PPMFrameFilter *GetInstance( string _switch )
		{
			return new PPMEffectTV( filters[ _switch.substr( 2 ) ] );
		}
};

/** Plug-in filter.
*/

class PPMPlugin : public PPMFrameFilter
{
	private:
		PPMFrameEffect effect;
		string action;
		PropertyInstance < PropertyString > command;
		PropertyInstance < PropertyBool > alpha;

	public:
		PPMPlugin( string _action ) :
			action( _action ),
			command( "command", "command to execute", "" ),
			alpha( "alpha", "pass PPM+alpha to plug-in", "false" )
		{
			AddProperty( &command );
			AddProperty( &alpha );
		}

		bool Filter( int64_t index )
		{
			PPMFrameStack &stack = PPMFrameStack::GetStack( );

			if ( command.value == "" )
				return false;

			if ( index == 0 || !effect.IsRunning( ) )
			{
				effect.Stop( );
				effect.Start( command.value.c_str( ) );
			}

			if ( action == "--plugin-filter" ) 
			{
				PPMFrame *f = stack.Pop( );
				effect.Copy( *f );
				effect.WriteImage( alpha.value );
				effect.ReadImage( );
				f->Copy( effect );
				stack.Push( f );
			}
			else if ( action == "--plugin-transition" )
			{
				PPMFrame *f = stack.Pop( );
				PPMFrame *g = stack.Pop( );
				effect.Copy( *f );
				effect.WriteImage( alpha.value );
				effect.Copy( *g );
				effect.WriteImage( alpha.value );
				effect.ReadImage( );
				f->Copy( effect );
				stack.Push( f );
				delete g;
			}
			else if ( action == "--plugin-producer" ) 
			{
				effect.ReadImage( );
				stack.Push( effect );
			}
			else if ( action == "--plugin-consumer" ) 
			{
				PPMFrame *f = stack.Pop( );
				effect.Copy( *f );
				effect.WriteImage( alpha.value );
				delete f;
			}

			return true;
		}
};

/** Plug-in factory.
*/

class PPMPluginFactory : public PPMFrameFilterFactory
{
	public:
		string GetSwitch( )
		{
			return "--plugin-filter,--plugin-transition,--plugin-producer,--plugin-consumer";
		}

		string GetDescription( string _switch )
		{
			if ( _switch == "--plugin-filter" )
				return "Use an external filter";
			else if ( _switch == "--plugin-transition" )
				return "Use an external transition";
			else if ( _switch == "--plugin-producer" )
				return "Use an external producer";
			else if ( _switch == "--plugin-consumer" )
				return "Use an external consumer (this will remove the top of stack and return nothing)";
			return "Unknown operation";
		}

		PPMFrameFilter *GetInstance( string _switch )
		{
			return new PPMPlugin( _switch );
		}
};

/** Stack video filter.
*/

class PPMStackVideo : public PPMFrameFilter
{
	private:
		PPMFrameEffect effect;
		PropertyInstance < PropertyFile > images;
		PropertyInstance < PropertyBool > scale;
		PropertyInstance < PropertyBool > ntsc;

	public:
		PPMStackVideo( ) :
			images( "file", "file to use, or pop the top of the stack if not specified" ),
			scale( "scale", "scale images to current frame size", "false" ),
			ntsc( "ntsc", "treat video with NTSC frame rate (default PAL)", "false" )
		{
			AddProperty( &images );
			AddProperty( &scale );
			AddProperty( &ntsc );
		}

		virtual bool Filter( int64_t index )
		{
			PPMFrameStack &stack = PPMFrameStack::GetStack( );

			if ( !effect.ReadImage( ) )
			{
				string image = images.value.front( );
				images.value.pop( );
				images.value.push( image );
				string normalisation = ntsc.value ? "-n" : "";
				string command = "ffmpeg2raw -ppm " + normalisation + " \"" + image + "\"";
				effect.Start( (char *)command.c_str( ) );
				effect.ReadImage( );
			}

			if ( scale.value )
			{
				PPMFrame *frame = stack.Pop( );
				effect.Scale( frame->GetWidth( ), frame->GetHeight( ) );
				stack.Push( frame );
			}

			stack.Push( effect );

			return true;
		}
};

/** Stack filter factory.
*/

class PPMStackVideoFactory : public PPMFrameFilterFactory
{
	public:
		string GetSwitch( ) { return "--stack-video"; }
		string GetDescription( string _switch ) { return "Place images from the specified video on the stack"; }
		PPMFrameFilter *GetInstance( string _switch ) { return new PPMStackVideo( ); }
};

/** The main filter class.
*/

class PPMFilter : public Threader
{
	private:
		int pump_size;
		vector < PPMFrameFilter * > filters;

	public: 
		bool aborted;
	
		PPMFilter( ) : 
			pump_size( 5 ), 
			aborted( false )
		{
		}

		virtual ~PPMFilter( )
		{
		}

		virtual string LogId( )
		{
			return "PPMFilter";
		}

		void SetPumpSize( int _pump_size )
		{
			pump_size = _pump_size;
		}

		void RegisterFilter( PPMFrameFilter *filter )
		{
			filters.push_back( filter );
		}

		bool Filter( PPMFrame &frame, int64_t index )
		{
			PPMFrameStack &stack = PPMFrameStack::GetStack( );

			// Seed the stack with the current frame
			if ( index == 0 )
				stack.Seed( frame );

			// Push the current frame on to the stack
			stack.Push( frame );

			// Iterate through all of the filters
			bool ret = true;
			for ( vector< PPMFrameFilter * >::iterator item = filters.begin( ); 
				  ret == true && item != filters.end( ); 
				  item ++ )
				ret = ( *item )->Process( index );

			PPMFrame *tos = stack.Pop( );
			frame.Copy( *tos );
			delete tos;

			// Seed the stack with the frame produced
			stack.Seed( frame );

			return ret;
		}

		bool SetProperty( string name, string value )
		{
			if ( filters.size( ) > 0 )
				return filters[ filters.size( ) - 1 ]->SetProperty( name, value );
			else
				return false;
		}

		bool SetProperty( string property )
		{
			int split = property.find_first_of( "=" );
			if ( split > 0 )
			{
				string name = property.substr( 0, split );
				string value = property.substr( split + 1 );
				return SetProperty( name, value );
			}
			else if ( filters[ filters.size( ) - 1 ]->properties.size( ) > 2 )
			{
				return SetProperty( filters[ filters.size( ) - 1 ]->properties[ 3 ]->GetName( ), property );
			}
			return false;
		}

	protected:
		void Thread( )
		{
			PPMPumpProvider input;
			input.SetPumpSize( pump_size );

			input.ThreadStart( );

			try
			{
				int64_t index = 0;
				bool running = true;

				while( running && ThreadIsRunning( ) )
				{
					if ( input.GetOutputAvailable( ) > 0 )
					{
						PPMFrame &frame = input.GetOutputFrame( );
						Filter( frame, index ++ );
						frame.WriteImage( );
						input.QueueInputFrame( );
					}
					else
					{
						running = input.ThreadIsRunning( ) && input.PumpIsNotCleared( );
					}
				}
			}
			catch ( string exc )
			{
				cerr << "ppmfilter: " << exc << endl;
			}
			catch ( const char *exc )
			{
				cerr << "ppmfilter: " << exc << endl;
			}

			input.ClearPump( );
			input.ThreadStop( );
		}
};

/** The filter collection.
*/

class PPMFilterCollection
{
	private:
		map < string, PPMFrameFilterFactory * > filters;

	public:
		int split( const string &input, const string &delimiter, vector< string > &items, const bool clean )
		{
			int delimiter_size = delimiter.size();
			int input_size = input.size();

			// Find the first delimiter
			int position = 0;
			int end_position = input.find( delimiter, 0 );

			// While we have a valid position
			while( end_position >= position )
			{
				// Obtain the substr and push if valid
				string s = input.substr( position, end_position - position );
				if( !clean || s.size() > 0 ) items.push_back(s);

				// Find the next delimiter
				position = end_position + delimiter_size;
				end_position = input.find( delimiter, position );
			}

			// Obtain the substr of what's left and push if valid
			string s = input.substr( position, input_size - position );
			if( !clean || s.size() > 0 ) items.push_back(s);

			// Return the number of items found
			return items.size();
		}

		void Add( PPMFrameFilterFactory *factory )
		{
			vector < string > items;
			vector < string >::iterator item;
			split( factory->GetSwitch( ), ",", items, true );
			for ( item = items.begin( ); item != items.end( ); item ++ )
				filters[ *item ] = factory;
		}

		PPMFrameFilterFactory *Find( string _switch )
		{
			PPMFrameFilterFactory *ret = NULL;
			if ( filters.find( _switch ) != filters.end( ) )
				ret = filters[ _switch ];
			return ret;
		}

		void Usage( string _effect = "" )
		{
			cerr << "Usage: ppmfilter [ --effect [ property=value ]* ]*" << endl;
			cerr << "Where effect and properties are:" << endl;

			map < string, PPMFrameFilterFactory * >::iterator item;

			for ( item = filters.begin(); item != filters.end( ); item ++ )
			{
				if ( _effect == "" || _effect == item->first )
				{
					cerr << "      " << item->first << " - " << item->second->GetDescription( item->first ) << endl;
					vector < string > props;
					vector < string >::iterator prop;
					PPMFrameFilter *filter = item->second->GetInstance( item->first );
					split( filter->Summary( ), "|", props, true );
					for ( prop = props.begin( ); prop != props.end( ); prop ++ )
						cerr << "          " << *prop << endl;
					delete filter;
				}
			}

			exit( 0 );
		}

};

/** The main function.
*/

int main( int argc, char **argv )
{
	g_type_init( );

	PPMFilter output;
	PPMFilterCollection filters;

	Diagnostics::SetApp( "ppmfilter" );

	// Register an instance of each factory
	filters.Add( new PPMStackFactory( ) );
	filters.Add( new PPMCopyFactory( ) );
	filters.Add( new PPMPluginFactory( ) );
	filters.Add( new PPMStackImageFactory( ) );
	filters.Add( new PPMStackVideoFactory( ) );
	filters.Add( new PPMGroupFactory( ) );
	filters.Add( new PPMBlackAndWhiteFactory( ) );
	filters.Add( new PPMMonoFactory( ) );
	filters.Add( new PPMSepiaFactory( ) );
	filters.Add( new PPMXorFactory( ) );
	filters.Add( new PPMExtractChannelFactory( ) );
	filters.Add( new PPMBorderFactory( ) );
	filters.Add( new PPMScaleFactory( ) );
	filters.Add( new PPMOverlayFactory( ) );
	filters.Add( new PPMCharcoalFactory( ) );
	filters.Add( new PPMMotionBlurFactory( ) );
	filters.Add( new PPMChromaKeyFactory( ) );
	filters.Add( new PPMLumaFactory( ) );
	filters.Add( new PPMEffectTVFactory( ) );
	filters.Add( new PPMGammaFactory( ) );
	filters.Add( new PPMObscureFactory( ) );

	PPMGroup *group = NULL;

	try
	{
		string last_effect = "";

		for ( int i = 1; i < argc; i ++ )
		{
			if ( !strcmp( argv[ i ], "--help" ) )
			{
				filters.Usage( last_effect );
			}
			else if ( !strncmp( argv[ i ], "--", 2 ) )
			{
				PPMFrameFilterFactory *item = filters.Find( argv[ i ] );

				if ( item != NULL )
				{
					PPMFrameFilter *instance = item->GetInstance( argv[ i ] );
					if ( !strcmp( argv[ i ], "--group" ) )
						group = (PPMGroup *)instance;
					else if ( group != NULL )
						group->CommunicateProperties( instance );
					output.RegisterFilter( instance );
					last_effect = argv[ i ];
				}
				else
				{
					throw string( "Unknown effect requested " ) + string( argv[ i ] );
				}
			}
			else
			{
				if ( !output.SetProperty( argv[ i ] ) )
				{
					filters.Usage( last_effect );
					throw string( "Invalid argument " ) + string( argv[ i ] );
				}
			}
		}

		if ( isatty( fileno( stdin ) ) )
		{
			cerr << "Input must must be obtained from a pipe or a file." << endl;
			filters.Usage( last_effect );
		}

		if ( isatty( fileno( stdout ) ) )
		{
			cerr << "Output be must redirected to a pipe or a file." << endl;
			filters.Usage( last_effect );
		}

		output.ThreadExecute( );
	}
	catch ( string exc )
	{
		cerr << "ppmfilter: " << exc << endl;
	}
	catch ( const char *exc )
	{
		cerr << "ppmfilter: " << exc << endl;
	}

	exit( output.aborted );
}
