/*
 * raw2yuv.cc -- Converts a Raw DV Stream to a YUV4MPEG Stream
 * Copyright (C) 2003 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <unistd.h>

#include <iostream>
#include <string>
#include <cmath>
using std::string;
using std::cerr;
using std::cin;
using std::endl;

#include <preferences.h>
#include <Diagnostics.h>
#include <RawDVFileInput.h>
#include <Threader.h>
#include <extensions.h>
#include <time.h>
#include <pthread.h>

class Raw2YUV : public Threader
{
	private:
		string audio_file;
		int deinterlace;

	public:
		Raw2YUV( ) : audio_file( "" ), deinterlace( 0 )
		{
		}

		virtual ~Raw2YUV( )
		{
		}

		virtual string LogId( )
		{
			return "Raw2YUV";
		}

		void SetDeinterlace( int _deinterlace )
		{
			deinterlace = _deinterlace;
		}

		void SetAudioFile( string _audio_file )
		{
			audio_file = _audio_file;
		}

	protected:
		void Thread( )
		{
			bool first = true;
			bool running = true;

			RawDVFileInput input;
			input.SetPumpSize( 25 );
			input.SetFile( stdin );

			AudioExtractor *audio = AudioExtractor::GetExtractor( audio_file );
			YUV420Extractor *yuv = YUV420Extractor::GetExtractor( deinterlace );

			input.ThreadStart( );

			while( running && ThreadIsRunning( ) )
			{
				if ( input.GetOutputAvailable( ) > 0 )
				{
					Frame &frame = input.GetOutputFrame( );

					if ( first )
					{
						audio->Initialise( frame );
						yuv->Initialise( frame );
						first = false;
					}

					audio->Output( frame );
					yuv->Output( frame );

					//fflush( stdout );

					input.QueueInputFrame( );
				}
				else
				{
					running = input.ThreadIsRunning( ) && input.PumpIsNotCleared( );
				}
			}

			audio->Flush( );
			yuv->Flush( );

			delete yuv;
			delete audio;

			input.ThreadStop( );
		}
};

static void Usage( )
{
	cerr << "Usage: raw2yuv [ options ]" << endl;
	cerr << "Where: options are" << endl;
	cerr << "       -a aufiofile - audio file to write (wav or mp2)" << endl;
	cerr << "       -i type      - deinterlacing type (0 = none, 1 = simple," << endl;
	cerr << "                      2 = 4:1:1 subsampling" << endl;
	exit( 0 );
}

int main( int argc, char **argv )
{
	Raw2YUV output;

	if ( isatty( fileno( stdin ) ) )
	{
		cerr << "Input must must be obtained from a pipe or a file." << endl;
		Usage( );
	}

	if ( isatty( fileno( stdout ) ) )
	{
		cerr << "Output be must redirected to a pipe or a file." << endl;
		Usage( );
	}

	Diagnostics::SetApp( "raw2yuv" );

	try
	{
		for ( int i = 1; i < argc; i ++ )
		{
			if ( !strcmp( argv[ i ], "-a" ) )
				output.SetAudioFile( string( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-i" ) )
				output.SetDeinterlace( atoi( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-v" ) )
				Diagnostics::SetLevel( atoi( argv[ ++ i ] ) );
			else
				Usage( );
		}

		output.ThreadExecute( );
	}
	catch ( string exc )
	{
		cerr << "Exception thrown: " << exc << endl;
	}

	return 0;
}
