/*
 *  smpeg-xmms - (S)MPEG Plugin for XMMS
 *  Copyright (C) 2000-2001 4Front Technologies
 *  http://www.opensound.com
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include <gtk/gtk.h>
#include <smpeg.h>
#include <SDL_syswm.h>
#include <SDL_thread.h>

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <config.h>
#include "smpeg-xmms.h"
#include <xmms/plugin.h>
#include <xmms/xmmsctrl.h>
#include <xmms/configfile.h>
#include <xmms/util.h>
#include <unistd.h>

#include <fcntl.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>

#ifdef __linux
#include <linux/cdrom.h>
#endif

#define CD_BLOCK_OFFSET CD_MSF_OFFSET

static InputPlugin smpeg_ip;

SmpegConfig smpeg_cfg;

static SDL_Thread *thread, *vcd_thread, *audio_thread;
static SDL_mutex *mpeg_mutex;
static SMPEG *mpeg = NULL;
static SDL_Surface *screen = NULL;
static gboolean playing = FALSE;
static gboolean doublesize = FALSE;
static gboolean paused = FALSE;
static gboolean fullscreen = FALSE;
static gboolean maximised = FALSE;
static gboolean is_vcd = FALSE;
static gboolean is_stream = FALSE;
static gboolean rstop = FALSE;
static gboolean bilinear_filtering = FALSE;
static gboolean XMMSAUDIO = FALSE;
static gboolean display = FALSE;
static int streamfd;

static gint vol, nch, srate, bitrate;
static gint size_w, size_h;

#ifdef __linux
static gint vol, vcd_pipe[2], vcd_fd;
struct cdrom_tocentry vcd_toc;
static SDL_mutex *vcd_mutex;
static gint vcdpos;
static gint vcdpos_max;
#endif

static gint smpeg_is_our_file(gchar * filename);
static void smpeg_play_file(gchar * filename);
static void smpeg_stop(void);
static void smpeg_pause(short p);
static void smpeg_seek(gint time);
static gint smpeg_get_time(void);
static void smpeg_get_song_info(gchar * filename, gchar ** title,
				gint * length);
static void smpeg_about(void);
static void smpeg_playback_func(void);
static void smpeg_audio_func(void *arg);

static void smpeg_set_doublesize(gboolean ds);
static void smpeg_set_fullscreen(gboolean fs);

static SDL_Rect maximise_dimensions;

gint ctrlsocket_get_session_id(void);	/* FIXME */

InputPlugin *get_iplugin_info(void)
{
	memset(&smpeg_ip, 0, sizeof (InputPlugin));

	smpeg_ip.description = g_strdup(PACKAGE " " VERSION);
	smpeg_ip.is_our_file = smpeg_is_our_file;
	smpeg_ip.play_file = smpeg_play_file;
	smpeg_ip.stop = smpeg_stop;
	smpeg_ip.pause = smpeg_pause;
	smpeg_ip.seek = smpeg_seek;
	smpeg_ip.get_time = smpeg_get_time;
	smpeg_ip.get_song_info = smpeg_get_song_info;
	smpeg_ip.about = smpeg_about;
	smpeg_ip.configure = smpeg_configure;
	smpeg_ip.add_vis_pcm = NULL;
	smpeg_ip.output = NULL;
	return &smpeg_ip;
}

static gint smpeg_is_our_file(gchar * filename)
{
	gchar *ext;

	ext = strrchr(filename, '.');
	if (!strncasecmp(filename, "vcd:", 4) ||
	    (ext && (!strcasecmp(ext, ".mpg") || !strcasecmp(ext, ".mpeg") ||
	    !strcasecmp(ext, ".bin"))))
	{
		FILE *g = fopen (filename, "r");
		if (!g)
			return FALSE;
		fclose(g);
		return TRUE;
	}
	return FALSE;
}

static void smpeg_set_caption(char *filename, int width, int height, gboolean doublesize)
{
	static char *name;
	char *caption;
	char *filtervalue;

	if (filename && name)
		g_free(name);
	if (filename)
		name = g_strdup(filename);
	if (bilinear_filtering)
		filtervalue="ON";
	else
		filtervalue="OFF";
	if (doublesize)
	{
		width *= 2;
		height *= 2;
	}
	caption = g_strdup_printf("%s (%dx%d) - Filtering is: %s", name, width, height, filtervalue);
	SDL_WM_SetCaption(caption, "XMMS Video");
}

static void smpeg_playback_func(void)
{
	SMPEG_Info info;
	SDL_Event event;
	gint i, ws, hs, scalew, scaleh;
	vol = xmms_remote_get_main_volume(ctrlsocket_get_session_id());
	SDL_mutexP(mpeg_mutex);
	if (!(is_vcd || is_stream))
	{
	 SMPEG_getinfo(mpeg, &info);
	}
	scalew = (info.width * 0.1);
	scaleh = (info.height * 0.1);
	while (SMPEG_status(mpeg) == SMPEG_PLAYING)
	{
		SDL_mutexV(mpeg_mutex);
		while (SDL_WaitEvent(&event))
		{
			switch (event.type)
			{
				case SDL_KEYDOWN:
					switch (event.key.keysym.sym)
					{
						case SDLK_z:
							xmms_remote_playlist_prev(ctrlsocket_get_session_id());
							break;
						case SDLK_x:
							xmms_remote_play(ctrlsocket_get_session_id());
							break;
						case SDLK_c:
							xmms_remote_pause(ctrlsocket_get_session_id());
							break;
						case SDLK_v:
							if(!smpeg_cfg.window_mode && fullscreen)
								smpeg_set_fullscreen(!fullscreen);
							xmms_remote_stop(ctrlsocket_get_session_id());
							break;
						case SDLK_b:
							xmms_remote_playlist_next(ctrlsocket_get_session_id());
							break;
						case SDLK_f:
							SMPEG_getinfo(mpeg, &info);
							size_w = screen->w;
							size_h = screen->h;
							if ( bilinear_filtering ) 
							{
								SMPEG_Filter *filter = SMPEGfilter_null();
								filter = SMPEG_filter( mpeg, filter );
								filter->destroy(filter);
								bilinear_filtering = FALSE;
								smpeg_set_caption(NULL, size_w, size_h, FALSE);
							}
							else
							{
								SMPEG_Filter *filter = SMPEGfilter_bilinear();
								filter = SMPEG_filter( mpeg, filter );
								filter->destroy(filter);
								bilinear_filtering = TRUE;
								smpeg_set_caption(NULL, size_w, size_h, FALSE);
							}
							break;
						case SDLK_PAGEUP:
							xmms_remote_playlist_prev(ctrlsocket_get_session_id());
							break;
						case SDLK_PAGEDOWN:
							xmms_remote_playlist_next(ctrlsocket_get_session_id());
							break;
						case SDLK_RETURN:
							smpeg_set_fullscreen(!fullscreen);
							break;
						case SDLK_ESCAPE:
							rstop = TRUE;
							if(!smpeg_cfg.window_mode && fullscreen)
								smpeg_set_fullscreen(!fullscreen);
							xmms_remote_stop(ctrlsocket_get_session_id());
							break;
						case SDLK_TAB:
							xmms_remote_set_main_volume(ctrlsocket_get_session_id(), vol);
							break;
						case SDLK_p:
							if ( event.key.keysym.mod & KMOD_CTRL ) {
								if(fullscreen)
									smpeg_set_fullscreen(!fullscreen);
								GDK_THREADS_ENTER();
								smpeg_configure();
								GDK_THREADS_LEAVE();
							}
							break;
						/*  this only works for SDL without libXv.a compiled in. */
						/*case SDLK_F12:
							SDL_SaveBMP(screen, "mpeg.bmp");
							break; */
						case SDLK_SPACE:
							smpeg_set_doublesize(!doublesize);
							break;
						case SDLK_PAUSE:
							xmms_remote_pause(ctrlsocket_get_session_id());
							break;
						case SDLK_KP_MINUS: {
							gboolean unpause = FALSE;
							ws = screen->w;
							hs = screen->h;

							SDL_mutexP(mpeg_mutex);
							SMPEG_getinfo(mpeg, &info);
							if (SMPEG_status(mpeg) == SMPEG_PLAYING)
							{
								SMPEG_pause(mpeg);
								unpause = TRUE;
							}
							ws = ws - scalew;
							hs = hs - scaleh;
							if (ws > (0 + scalew) || hs > (0 + scaleh)) {
							screen = SDL_SetVideoMode(ws, hs,
										  screen->format->BitsPerPixel,
										  screen->flags);
							SMPEG_scaleXY(mpeg, ws, hs);
							}
							smpeg_set_caption(NULL, ws, hs, FALSE);
							
							if (unpause)
								SMPEG_pause(mpeg);
							SDL_mutexV(mpeg_mutex);
							break;
						}
						case SDLK_KP_PLUS: {
							gboolean unpause = FALSE;
							ws = screen->w;
							hs = screen->h;
							SDL_mutexP(mpeg_mutex);
							SMPEG_getinfo(mpeg, &info);
							if (SMPEG_status(mpeg) == SMPEG_PLAYING)
							{
								SMPEG_pause(mpeg);
								unpause = TRUE;
							}
							ws = ws + scalew;
							hs = hs + scaleh;
							
							screen = SDL_SetVideoMode(ws, hs,
										  screen->format->BitsPerPixel,
										  screen->flags);
							SMPEG_scaleXY(mpeg, ws, hs);
							smpeg_set_caption(NULL, ws, hs, FALSE);
							
							if (unpause)
								SMPEG_pause(mpeg);
							SDL_mutexV(mpeg_mutex);
							break;
						}
						case SDLK_RIGHT:
							if (!(is_vcd||is_stream))
							{
								SDL_mutexP(mpeg_mutex);
								SMPEG_getinfo(mpeg, &info);
								SDL_mutexV(mpeg_mutex);
								smpeg_seek(MIN((gint)info.total_time,
									       (gint)info.current_time + 5));
							}
							else if(is_vcd)
							{
#ifdef __linux
								SDL_mutexP(vcd_mutex);
								if (vcdpos + (5 * CD_FRAMES) < vcdpos_max)
									vcdpos += 5 * CD_FRAMES;
								else
									vcdpos = vcdpos_max;
								SDL_mutexV(vcd_mutex);
#endif
							}
							break;
						case SDLK_LEFT:
							if (!(is_vcd||is_stream))
							{
								SDL_mutexP(mpeg_mutex);
								SMPEG_getinfo(mpeg, &info);
								SDL_mutexV(mpeg_mutex);
								smpeg_seek(MAX(0, (gint)info.current_time - 5));
							}
							else if(is_vcd)
							{
#ifdef __linux
								SDL_mutexP(vcd_mutex);
								if (vcdpos - (5 * CD_FRAMES) >= 0)
									vcdpos -= 5 * CD_FRAMES;
								else
									vcdpos = 0;
								SDL_mutexV(vcd_mutex);
#endif
							}
							break;
						case SDLK_UP:
							i = xmms_remote_get_main_volume(ctrlsocket_get_session_id());
							i += 5;
							if (i > 100)
								i = 100;
							xmms_remote_set_main_volume(ctrlsocket_get_session_id(), i);
							break;
						case SDLK_DOWN:
							i = xmms_remote_get_main_volume(ctrlsocket_get_session_id());
							i -= 5;
							if (i < 0)
								i = 0;
							xmms_remote_set_main_volume(ctrlsocket_get_session_id(), i);
					}
					break;
				case SDL_MOUSEBUTTONDOWN:
					switch (event.button.button)
					{
						case 1:
							if (fullscreen)
								xmms_remote_pause(ctrlsocket_get_session_id());
							break;
						case 2:
							smpeg_set_doublesize(!doublesize);
							break;
						case 3:
							smpeg_set_fullscreen(!fullscreen);
							break;
						case 4:
							i = xmms_remote_get_main_volume(ctrlsocket_get_session_id());
							i += 5;
							if (i > 100)
								i = 100;
							xmms_remote_set_main_volume(ctrlsocket_get_session_id(), i);
							break;
						case 5:
							i = xmms_remote_get_main_volume(ctrlsocket_get_session_id());
							i -= 5;
							if (i < 0)
								i = 0;
							xmms_remote_set_main_volume(ctrlsocket_get_session_id(), i);
					}
					break;
				case SDL_QUIT: {
					rstop = TRUE;
					xmms_remote_stop(ctrlsocket_get_session_id());
					break;
					}
				case SDL_VIDEORESIZE: {
					gboolean unpause = FALSE;
					gint old_w = 0, old_h = 0;
					gint wg = 0, hg = 0;
					SMPEG_getinfo(mpeg, &info);
					old_w = screen->w;
					old_h = screen->h;
					smpeg_set_fullscreen(FALSE);
					/* doublesize = FALSE; */
					SDL_mutexP(mpeg_mutex);
					if (SMPEG_status(mpeg) == SMPEG_PLAYING)
					{
						SMPEG_pause(mpeg);
						unpause = TRUE;
					}
					wg = ((event.resize.h / (gdouble)info.height) * info.width);
					hg = ((event.resize.w / (gdouble)info.width) * info.height);

					if (!smpeg_cfg.wratio_mode)
					{
						if ((old_w+old_h) < (event.resize.w+event.resize.h))
						{
							hg=((event.resize.h-old_h) > (event.resize.w-old_w) ? event.resize.h : hg );
							wg=((event.resize.w-old_w) > (event.resize.h-old_h) ? event.resize.w : wg );
						}
						else {
							hg=((old_h-event.resize.h) > (old_w-event.resize.w) ? event.resize.h : hg );
							wg=((old_w-event.resize.w) > (old_h-event.resize.h) ? event.resize.w : wg );
						}
						screen = SDL_SetVideoMode(wg, hg, screen->format->BitsPerPixel, screen->flags);
						SMPEG_scaleXY(mpeg, wg, hg);
						smpeg_set_caption(NULL, wg, hg, FALSE);
					}
					else {
						screen = SDL_SetVideoMode(event.resize.w,
									  event.resize.h,
									  screen->format->BitsPerPixel,
									  screen->flags);
						SMPEG_scaleXY(mpeg, event.resize.w, event.resize.h);
						smpeg_set_caption(NULL, event.resize.w, event.resize.h, FALSE);
					}
					if (unpause)
						SMPEG_pause(mpeg);

					SDL_mutexV(mpeg_mutex);
					break;
				}
			}
		}
	}
	SDL_mutexV(mpeg_mutex);
}

static void smpeg_set_doublesize(gboolean ds)
{
	SMPEG_Info info;
	gboolean unpause = FALSE;

	SDL_mutexP(mpeg_mutex);
	SMPEG_getinfo(mpeg, &info);

	if (fullscreen)
		smpeg_set_fullscreen(!fullscreen);
	if (SMPEG_status(mpeg) == SMPEG_PLAYING)
	{
		SMPEG_pause(mpeg);
		unpause = TRUE;
	}
	if (info.has_video)
	{
		if (ds == TRUE)
		{
			screen = SDL_SetVideoMode(info.width * 2, 
						  info.height * 2,
						  screen->format->BitsPerPixel,
						  screen->flags);
		}
		else
		{
			screen = SDL_SetVideoMode(info.width, info.height,
						  screen->format->BitsPerPixel,
						  screen->flags);
		}
		SMPEG_double(mpeg, ds);
		smpeg_set_caption(NULL, info.width, info.height, ds);
		if (unpause)
		{
			SMPEG_pause(mpeg);
		}
	}
	doublesize = ds;
	SDL_mutexV(mpeg_mutex);
}

static void smpeg_set_fullscreen(gboolean fs)
{
	SDL_Rect **modes;
	gint w = 0, h = 0, i, num_modes;
	gint wg = 0, hg = 0;
	static gint old_w = 0, old_h = 0;
	SMPEG_Info info;
	gboolean unpause = FALSE;

	SDL_mutexP(mpeg_mutex);
	SMPEG_getinfo(mpeg, &info);
	SDL_ShowCursor(!fs);

	if (fs != fullscreen)
	{
		if (SMPEG_status(mpeg) == SMPEG_PLAYING)
		{
			SMPEG_pause(mpeg);
			unpause = TRUE;
		}
		fullscreen = fs;
		if (!fullscreen && old_w != 0 && old_h != 0)
		{
			SDL_WM_ToggleFullScreen(screen);
			screen = SDL_SetVideoMode(old_w, old_h,
						  screen->format->BitsPerPixel,
						  screen->flags);
			SMPEG_scaleXY(mpeg, old_w, old_h);
		}
		else if (fullscreen)
		{
			old_w = screen->w;
			old_h = screen->h;
			
			modes = SDL_ListModes(NULL, SDL_FULLSCREEN);
			if ((modes != NULL) && (modes != (SDL_Rect **) - 1))
			{
				for (num_modes = 0; modes[num_modes];
				     num_modes++) ;
				if (smpeg_cfg.samevideo_mode)
				{
					w = modes[0]->w;
					h = modes[0]->h;
				}
				else
				{
					for (i = num_modes - 1; i >= 0; i--)
					{
						if ((modes[i]->w > info.width)
						    && (modes[i]->h > info.height))
						{
							w = modes[i]->w;
							h = modes[i]->h;
							break;
						}
					}
				} 
				if (w != 0 && h != 0)
				wg = ((h / (gdouble)info.height) * info.width);
				hg = ((w / (gdouble)info.width) * info.height);
				
				if (!smpeg_cfg.ratio_mode) {
					if (wg < w) {
 						screen = SDL_SetVideoMode(wg, h,
									  screen->format->BitsPerPixel,
									  screen->flags);
						SMPEG_scaleXY(mpeg, wg, h);
					}
					else {
						screen = SDL_SetVideoMode(w, hg,
									  screen->format->BitsPerPixel,
									  screen->flags);
						SMPEG_scaleXY(mpeg, w, hg);
					}
				}
				else if (smpeg_cfg.ratio_mode) {
					screen = SDL_SetVideoMode(w, h,
								  screen->format->BitsPerPixel,
								  screen->flags);
					SMPEG_scaleXY(mpeg, w, h);
				} 
			}
			SDL_WM_ToggleFullScreen(screen);
		}
		if (unpause)
			SMPEG_pause(mpeg);
	}
	SDL_mutexV(mpeg_mutex);
}


#ifdef __linux

static void set_track_number(int track, char *m, char *s, char *f)
{
  struct cdrom_tocentry vcd_toc_local;

  vcd_toc_local.cdte_track = track;
  vcd_toc_local.cdte_format = CDROM_MSF;
  ioctl(vcd_fd, CDROMREADTOCENTRY, &vcd_toc_local);

  *m = vcd_toc_local.cdte_addr.msf.minute;
  *s = vcd_toc_local.cdte_addr.msf.second;
  *f = vcd_toc_local.cdte_addr.msf.frame; 
}

static int msf_to_lba (int m, int s, int f)
{
  return (((m * CD_SECS) + s) * CD_FRAMES + f) - CD_BLOCK_OFFSET;
}

static void lba_to_msf (int lba, char *m, char *s, char *f)
{
        lba += CD_BLOCK_OFFSET;
        lba &= 0xffffff;  /* negative lbas use only 24 bits */
        *m = lba / (CD_SECS * CD_FRAMES);
        lba %= (CD_SECS * CD_FRAMES);
        *s = lba / CD_FRAMES;
        *f = lba % CD_FRAMES;
}

static int vcd_read(gint fd, gint lba, guchar *buf)
{
    struct cdrom_msf *msf;
    gint    rc;

    msf = (struct cdrom_msf*) buf;
    msf->cdmsf_min0   = (lba + CD_MSF_OFFSET) / CD_FRAMES / CD_SECS; 
    msf->cdmsf_sec0   = (lba + CD_MSF_OFFSET) / CD_FRAMES % CD_SECS;
    msf->cdmsf_frame0 = (lba + CD_MSF_OFFSET) % CD_FRAMES;
    return ioctl(fd, CDROMREADRAW, buf);
}

#endif

#ifdef __linux

static void smpeg_read_vcd_func(void)
{
	struct timeval timeout;
	fd_set fdset;

 	guchar *buffer = g_malloc(CD_FRAMESIZE_RAW);
 	SDL_mutexP(vcd_mutex);
 	for(vcdpos = vcd_toc.cdte_addr.lba; vcd_read(vcd_fd, vcdpos, buffer) >= 0; vcdpos++)	
	{
	SDL_mutexV(vcd_mutex);
		FD_ZERO(&fdset);
		FD_SET(vcd_pipe[1], &fdset);
		timeout.tv_sec = 10;
		timeout.tv_usec = 0;
		if(select(vcd_pipe[1] + 1, NULL, &fdset, NULL, &timeout) <= 0)
			break;
		if(write(vcd_pipe[1], buffer, CD_FRAMESIZE_RAW) < 0)
			break;
	SDL_mutexP(vcd_mutex);
	}
	/* g_message("ioctl failed on reading block %s", vcdpos); */
	g_free(buffer);
}

#endif

static void smpeg_play_file(gchar * filename)
{
	SMPEG_Info info;
	gchar *name, *error;
	Uint32 video_flags;
	struct stat buf;	
	gchar *pip;
	gint track;
	char m, s, f;
	smpeg_read_config();
	XMMSAUDIO = smpeg_cfg.xmmsaudio_mode;

	if(!strncasecmp(filename, "vcd:", 4))
	{
		is_vcd = TRUE;
		track = 2;
		pip = strrchr(filename + 4, ':');
		if(pip)
		{
			*pip = '\0';
			track = atoi(pip + 1);
		}

		if((stat(filename + 4, & buf) < 0) || !S_ISBLK(buf.st_mode))
		{
			if(pip)
			{
				*pip = ':';
				g_message("smpeg_play_file(): %s is not a block device", filename + 4);
				return;
			}
		}
#ifdef __linux
		vcd_fd = open(filename + 4, O_RDONLY, 0);
		if(vcd_fd < 0)
		{
			if(pip)
				*pip = ':';
			g_message("smpeg_play_file(): Unable to open device %s", filename + 4);
		}
		vcd_toc.cdte_track = track;
		vcd_toc.cdte_format = CDROM_LBA;
		if(ioctl(vcd_fd, CDROMREADTOCENTRY, &vcd_toc) < 0)
		{
			g_message("smpeg_play_file(): Couldn't read TOC entry for track %d", track);
			close(vcd_fd);
		}
		if(pipe(vcd_pipe) < 0)
		{
			close(vcd_fd);
			g_message("smpeg_play_file(): Couldn't create pipe for VCD");
			return;
		}
		vcd_mutex = SDL_CreateMutex();
		vcd_thread = SDL_CreateThread((void *)smpeg_read_vcd_func, NULL);
		if(XMMSAUDIO)
			mpeg = SMPEG_new_descr(vcd_pipe[0], &info, FALSE);
		else
			mpeg = SMPEG_new_descr(vcd_pipe[0], &info, TRUE);
#endif
	}
	else
	{
		is_vcd = FALSE;
		if((stat(filename, &buf) < 0) || !S_ISFIFO(buf.st_mode))
		{
			is_stream = FALSE;
			if(XMMSAUDIO)
				mpeg = SMPEG_new(filename, &info, FALSE);
			else
				mpeg = SMPEG_new(filename, &info, TRUE);
		}
		else
		{
			streamfd = open(filename, O_RDONLY, 0);
			is_stream = TRUE;
			if(XMMSAUDIO)
				mpeg = SMPEG_new_descr(streamfd, &info, FALSE);
			else
				mpeg = SMPEG_new_descr(streamfd, &info, TRUE);
		}
	}
	if ((error = SMPEG_error(mpeg)) != NULL)
	{
		g_message("smpeg_play_file(): Unable to load file: %s", error);
		SMPEG_delete(mpeg);
		return;
	}

	if (!is_vcd && access(filename, R_OK) != 0)
		g_error("smpeg_play_file(): File not readable, but SMPEG reported no error");

	if (!smpeg_cfg.audioonly_mode)
	{
		SDL_Init(SDL_INIT_VIDEO);
		video_flags = SDL_SWSURFACE | SDL_ASYNCBLIT | SDL_RESIZABLE;
	}
	mpeg_mutex = SDL_CreateMutex();

	SDL_mutexP(mpeg_mutex);

	if (smpeg_cfg.center_mode)
		putenv("SDL_VIDEO_CENTERED=1");
	else
#ifdef __linux
		unsetenv("SDL_VIDEO_CENTERED");
#else
		putenv("SDL_VIDEO_CENTERED=0");
#endif

	if(!smpeg_cfg.audioonly_mode)
	{
		if (smpeg_cfg.states_mode && smpeg_cfg.windowsize_mode) {
			gint wg = 0, hg = 0;
			SMPEG_getinfo(mpeg, &info);
			wg = ((smpeg_cfg.sizeh_mode / (gdouble)info.height) * info.width);
			hg = ((smpeg_cfg.sizew_mode / (gdouble)info.width) * info.height);
			screen = SDL_SetVideoMode(smpeg_cfg.sizew_mode, smpeg_cfg.sizeh_mode, 0,
						   video_flags);
			if (!smpeg_cfg.wratio_mode) {
				if (wg < smpeg_cfg.sizew_mode)
					SMPEG_scaleXY(mpeg, wg, smpeg_cfg.sizeh_mode);
				else
					SMPEG_scaleXY(mpeg, smpeg_cfg.sizew_mode, hg);
			}
			else if (smpeg_cfg.wratio_mode)
				SMPEG_scaleXY(mpeg, smpeg_cfg.sizew_mode, smpeg_cfg.sizeh_mode);
		}
		else {
			if(smpeg_cfg.double_mode) {
				screen = SDL_SetVideoMode(info.width * 2, info.height * 2, 0,
							  video_flags);
				SMPEG_double(mpeg, smpeg_cfg.double_mode);
			}
			else {
				screen = SDL_SetVideoMode(info.width, info.height, 0,
							  video_flags);
				SMPEG_double(mpeg, smpeg_cfg.double_mode);
			}
		}
		display = TRUE;
	}
	if(!smpeg_cfg.audioonly_mode)
		SMPEG_setdisplay(mpeg, screen, NULL, NULL);
	if(smpeg_cfg.audioonly_mode)
		SMPEG_enablevideo(mpeg, FALSE);
	else
		SMPEG_enablevideo(mpeg, TRUE);
	SMPEG_loop(mpeg, FALSE);
	if(XMMSAUDIO)
		smpeg_ip.output->open_audio(FMT_S16_LE, 44100, 2);
	SMPEG_play(mpeg);
	if (smpeg_cfg.fullscreen_mode)
		smpeg_set_fullscreen(!fullscreen);
	if (smpeg_cfg.filter_mode)
	{
		SMPEG_Filter *filter = SMPEGfilter_bilinear();
		filter = SMPEG_filter( mpeg, filter );
		filter->destroy(filter);
	}
	SDL_mutexV(mpeg_mutex);

	if(is_vcd)
	{
	    name = g_strdup_printf("VCD Track: %d", track);
	}
	else
	{
	    name = g_strdup(g_basename(filename));
	    if (strrchr(name, '.') != NULL)
		*strrchr(name, '.') = '\0';
	}
	if(smpeg_cfg.windowsize_mode)
		smpeg_set_caption(name, smpeg_cfg.sizew_mode,
		smpeg_cfg.sizeh_mode, FALSE);
	else
		smpeg_set_caption(name, info.width, info.height,
		smpeg_cfg.double_mode);
	if (info.has_audio)
	{
		gchar *tmp = strstr(info.audio_string, "kbit/s");
		if (tmp)
		{
			while (isdigit(tmp[-1]))
				tmp--;
			sscanf(tmp, "%dkbit/s", &bitrate);
		}

		tmp = strstr(info.audio_string, "Hz");
		if (tmp)
		{
			while (isdigit(tmp[-1]))
				tmp--;
			sscanf(tmp, "%dHz", &srate);
		}
		if (strstr(info.audio_string, "stereo"))
			nch = 2;
		else if (strstr(info.audio_string, "mono"))
			nch = 1;
	}
#ifdef __linux
	if(is_vcd)
	{
	    set_track_number(CDROM_LEADOUT, &m, &s, &f);
	    vcdpos_max = (((m * CD_SECS) + s) * CD_FRAMES) + f;
	    smpeg_ip.set_info(name, (1000 * vcdpos_max) / CD_FRAMES, bitrate * 1000, srate, nch);
	}
	else
#endif __linux
	    smpeg_ip.set_info(name, info.total_time * 1000, bitrate * 1000, srate, nch);
	g_free(name);

#ifdef __linux
	unsetenv("SDL_VIDEO_CENTERED");
#else
	putenv("SDL_VIDEO_CENTERED=0");
#endif
	playing = TRUE;
	bilinear_filtering = smpeg_cfg.filter_mode;
	if(!smpeg_cfg.audioonly_mode)
		maximise_dimensions = *SDL_ListModes(NULL, SDL_SWSURFACE | SDL_ASYNCBLIT | SDL_FULLSCREEN)[0];
	fullscreen = smpeg_cfg.fullscreen_mode;
	doublesize = smpeg_cfg.double_mode;
	if(XMMSAUDIO)
		audio_thread = SDL_CreateThread((void *) smpeg_audio_func, NULL);
	thread = SDL_CreateThread((void *) smpeg_playback_func, NULL);
}

static void smpeg_audio_func(void *arg)
{
	char data[512*4];
	gint bytes, blk_size;
	SMPEG_Info info;
	SDL_AudioSpec audiofmt;
	audiofmt.format=AUDIO_S16LSB;
	audiofmt.freq=srate;
	audiofmt.size=2048;
	audiofmt.channels=nch;
	SDL_mutexP(mpeg_mutex);
	SMPEG_actualSpec(mpeg, &audiofmt);
	SDL_mutexV(mpeg_mutex);
	bytes = 512*2*nch;
	while(playing && !paused)
	{
		SDL_mutexP(mpeg_mutex);
		SMPEG_playAudio(mpeg, data, bytes);
		SDL_mutexV(mpeg_mutex);
		while(smpeg_ip.output->buffer_free() < bytes && playing)
		{
			xmms_usleep(10000);
		}
		if(SMPEG_status(mpeg) == SMPEG_PLAYING && !paused)
		{
			smpeg_ip.output->write_audio(data, bytes);
			smpeg_ip.add_vis_pcm(smpeg_ip.output->written_time(), FMT_S16_LE, nch, bytes, data);
		}
		memset(data, 0, bytes);
	}
	SDL_mutexV(mpeg_mutex);
}

static gint smpeg_timeout_func(gpointer data)
{
	if(xmms_remote_is_playing(0)) {
		gchar *ext;
		gint pos;
		pos = xmms_remote_get_playlist_pos(0);
		ext = strrchr(xmms_remote_get_playlist_file(0,pos), '.');

		if (!strncasecmp(xmms_remote_get_playlist_file(0,pos), "vcd:", 4) ||
	    	   (ext && (!strcasecmp(ext, ".mpg") || !strcasecmp(ext, ".mpeg")))) {
	    		return FALSE;
		}
		else {
			SDL_Quit();
			return FALSE;
		}
	}
	else {
		SDL_Quit();
		return FALSE;
	}
}

static void smpeg_stop(void)
{
	gint timeout_tag = 0;
	if (!playing)
		return;
	if(!smpeg_cfg.audioonly_mode || display)
	{
		if (smpeg_cfg.states_mode)
		{
			ConfigFile *cfg = xmms_cfg_open_default_file();
			gint size_w, size_h;
			size_w = screen->w;
			size_h = screen->h;
			xmms_cfg_write_boolean(cfg, "smpeg-xmms", "smpeg_double",
					       doublesize);
			xmms_cfg_write_boolean(cfg, "smpeg-xmms", "smpeg_fullscreen",
					       fullscreen);
			if(!fullscreen && smpeg_cfg.windowsize_mode) {
			xmms_cfg_write_int(cfg, "smpeg-xmms", "smpeg_sizew",
					   size_w);
			xmms_cfg_write_int(cfg, "smpeg-xmms", "smpeg_sizeh",
					   size_h);
			}
			xmms_cfg_write_default_file(cfg);
			xmms_cfg_free(cfg);
		}
	}
	SDL_mutexP(mpeg_mutex);
	if(XMMSAUDIO)
	{
		SDL_KillThread(audio_thread);
		smpeg_ip.output->close_audio();
	}
	SDL_KillThread(thread);
	SMPEG_stop(mpeg);
	SMPEG_delete(mpeg);
	if(is_stream)
		close(streamfd);

#ifdef __linux
	if(is_vcd)
	{
		close(vcd_pipe[0]);
		close(vcd_pipe[1]);
		close(vcd_fd);
		SDL_DestroyMutex(vcd_mutex);
	}
#endif
	SDL_mutexV(mpeg_mutex);
	SDL_DestroyMutex(mpeg_mutex);
	if(!smpeg_cfg.audioonly_mode || display)
	{
		SDL_FreeSurface(screen);
		if (smpeg_cfg.audioonly_mode)
			SDL_Quit();
		if(!smpeg_cfg.window_mode && !rstop)
			timeout_tag = gtk_timeout_add(0, smpeg_timeout_func, NULL);
		if(smpeg_cfg.window_mode || fullscreen)
			SDL_Quit();
		if(!smpeg_cfg.window_mode && rstop)
			SDL_Quit();
		display = FALSE;
	}
	playing = FALSE;
	fullscreen = FALSE;
	paused = FALSE;
	is_vcd = FALSE;
	is_stream = FALSE;
	rstop = FALSE;
}

static void smpeg_pause(short p)
{
	if (!playing)
		return;

	SDL_mutexP(mpeg_mutex);
	if (p == paused)
		g_error("Pause confusion");
	SMPEG_pause(mpeg);
	if(XMMSAUDIO)
	{
		smpeg_ip.output->pause(p);
	}
	paused = p;
	SDL_mutexV(mpeg_mutex);
}

static void smpeg_seek(gint time)
{
	SMPEG_Info info;
	SDL_mutexP(mpeg_mutex);
	SMPEG_getinfo(mpeg, &info);

	if(!is_stream)
	{
#ifdef __linux
		if (is_vcd)
		{
			SDL_mutexP(vcd_mutex);
			vcdpos = msf_to_lba(0, (int) time, 0);
			SDL_mutexV(vcd_mutex);
			if(XMMSAUDIO)
			{
				smpeg_ip.output->flush((gint)time * 1000);
			}
		}
		else
#endif 		
			SMPEG_seek(mpeg, (gint) ((gdouble)time / (info.total_time) * info.total_size));
			if(XMMSAUDIO)
			{
				smpeg_ip.output->flush((gint)time*1000);
			}
	}
	SDL_mutexV(mpeg_mutex);
}

static gint smpeg_get_time(void)
{
	SMPEG_Info info;
	char m, s, f;

	if (!playing)
		return -1;

	SDL_mutexP(mpeg_mutex);
	if (SMPEG_status(mpeg) == SMPEG_STOPPED && !paused)
	{
		SDL_mutexV(mpeg_mutex);
		return -1;
	}

#ifdef __linux
	if(is_vcd)
	{
		SDL_mutexV(mpeg_mutex);
		lba_to_msf(vcdpos, &m, &s, &f);
		return (1000 * ((m * CD_SECS) + s) + (f * 1000 / 75));
	}
	else
#endif __linux
	{
		SMPEG_getinfo(mpeg, &info);
		SDL_mutexV(mpeg_mutex);
		if(XMMSAUDIO)
			return smpeg_ip.output->output_time();
		else
			return (info.current_time * 1000);
/*
		return (((info.current_offset / (info.total_size / info.total_time)) * 1000) - (smpeg_ip.output->written_time() - smpeg_ip.output->output_time()));
*/
	}
}

static void smpeg_get_song_info(gchar * filename, gchar ** title, gint * length)
{
	SMPEG *addmpeg;
	SMPEG_Info info;
	gchar *name, *temp;
	gint track = 2;

	*title = NULL;
	*length = 100000;
/*	*length = -1;*/

	if(!strncasecmp(filename, "vcd:", 4))
	{
		temp = strrchr(filename + 4, ':');
		if(temp)
			track = atoi(temp + 1);
		*title = g_strdup_printf("VCD Track: %d", track);
		return;
	}
	
	addmpeg = SMPEG_new(filename, &info, FALSE);
	if ((temp = SMPEG_error(addmpeg)) != NULL)
	{
		SMPEG_delete(addmpeg);
		return;
	}

	if (access(filename, R_OK) != 0)
		g_error("smpeg_get_song_info(): File not readable, but SMPEG reported no error");

	(*length) = info.total_time * 1000;
	SMPEG_delete(addmpeg);

	name = g_strdup(g_basename(filename));
	if ((temp = strrchr(name, '.')) != NULL)
		*temp = '\0';
	(*title) = name;
}

void smpeg_read_config(void)
{

	ConfigFile *cfg;
	
	smpeg_cfg.double_mode = FALSE;
	smpeg_cfg.fullscreen_mode = FALSE;
	smpeg_cfg.states_mode = FALSE;
	smpeg_cfg.windowsize_mode = FALSE;
	smpeg_cfg.ratio_mode = FALSE;
	smpeg_cfg.wratio_mode = FALSE;
	smpeg_cfg.window_mode = FALSE;
	smpeg_cfg.filter_mode = FALSE;
	smpeg_cfg.xmmsaudio_mode = FALSE;
	smpeg_cfg.audioonly_mode = FALSE;
	smpeg_cfg.samevideo_mode = FALSE;
	smpeg_cfg.sizew_mode = 320;
	smpeg_cfg.sizeh_mode = 200;

	cfg = xmms_cfg_open_default_file();

	xmms_cfg_read_boolean(cfg, "smpeg-xmms", "smpeg_double",
			      &smpeg_cfg.double_mode);
	xmms_cfg_read_boolean(cfg, "smpeg-xmms", "smpeg_fullscreen",
			      &smpeg_cfg.fullscreen_mode);
	xmms_cfg_read_boolean(cfg, "smpeg-xmms", "smpeg_center",
			      &smpeg_cfg.center_mode);
	xmms_cfg_read_boolean(cfg, "smpeg-xmms", "smpeg_states",
			      &smpeg_cfg.states_mode);
	xmms_cfg_read_boolean(cfg, "smpeg-xmms", "smpeg_windowsize",
				&smpeg_cfg.windowsize_mode);
	xmms_cfg_read_boolean(cfg, "smpeg-xmms", "smpeg_ratio",
			      &smpeg_cfg.ratio_mode);
	xmms_cfg_read_boolean(cfg, "smpeg-xmms", "smpeg_wratio",
			      &smpeg_cfg.wratio_mode);
	xmms_cfg_read_boolean(cfg, "smpeg-xmms", "smpeg_window",
				&smpeg_cfg.window_mode);
	xmms_cfg_read_boolean(cfg, "smpeg-xmms", "smpeg_filter",
				&smpeg_cfg.filter_mode);
	xmms_cfg_read_boolean(cfg, "smpeg-xmms", "smpeg_xmmsaudio",
				&smpeg_cfg.xmmsaudio_mode);
	xmms_cfg_read_boolean(cfg, "smpeg-xmms", "smpeg_audioonly",
				&smpeg_cfg.audioonly_mode);
	xmms_cfg_read_boolean(cfg, "smpeg-xmms", "smpeg_samevideo",
				&smpeg_cfg.samevideo_mode);
	xmms_cfg_read_int(cfg, "smpeg-xmms", "smpeg_sizew",
				&smpeg_cfg.sizew_mode);
	xmms_cfg_read_int(cfg, "smpeg-xmms", "smpeg_sizeh",
				&smpeg_cfg.sizeh_mode);
	xmms_cfg_free(cfg);
}

static void smpeg_about(void)
{
	static GtkWidget *window = NULL;

	if (window)
		return;

	window = xmms_show_message(
		"About " PACKAGE,
		PACKAGE " " VERSION "\n"
		"Copyright (C) 2000, 4Front Technologies.\n"
		"http://www.opensound.com\n"
		"\n"
		"This library is free software; you can redistribute it and/or\n"
		"modify it under the terms of the GNU Library General Public\n"
		"License as published by the Free Software Foundation; either\n"
		"version 2 of the License, or (at your option) any later version.\n"
		"\n"
		"This library is distributed in the hope that it will be useful,\n"
		"but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
		"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU\n"
		"Library General Public License for more details.\n"
		"\n"
		"You should have received a copy of the GNU Library General Public\n"
		"License along with this library; if not, write to the Free\n"
		"Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.",
		"Ok", FALSE, NULL, NULL);
	
	gtk_signal_connect(GTK_OBJECT(window), "destroy",
			   GTK_SIGNAL_FUNC(gtk_widget_destroyed), &window);

	gtk_widget_show(window);

}
