/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* $Id: smtpguard-daemon.c,v 1.1.1.1 2005/11/04 07:19:35 tkitame Exp $ 
 *
 * Copyright (c) 2005 VA Linux Systems Japan, K.K. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/un.h>
#include <sys/param.h>
#include <unistd.h>
#include <signal.h>
#include <pthread.h>
#include <syslog.h>

#include <glib.h>


#include <smtpguard/smtpguard.h>

#include "smtpguard-daemon.h"
#ifndef FG_DISABLE_DEPRECATED
#include "sg_io.h"
#endif
#include "syslog_names.h"


#define PROGNAME "smtpguard-daemon"

extern int syslog_output;
extern unsigned int sg_log_pri;

int	main_loop;
int	loadconfig;

typedef struct {
    idb_t *idb;
    char *config;
    pthread_t tid;
    int dead;
    int down;
} task_env_t;

#define TASK_RETURN(tp,ret) {tp->dead = 1; return (void *)(ret); }


//
// print usage
static void
usage (void)
{
	fprintf(stderr, "Usage: %s [OPTION]... [config_file]\n", PROGNAME);
	fprintf(stderr, "OPTION:\n");
	fprintf(stderr, "  -h dbhome     directory where the database is stored\n");
	fprintf(stderr, "                (default: %s)\n", DEFAULT_DBHOME);
	fprintf(stderr, "  -n dbname     main database file\n");
	fprintf(stderr, "                (default: %s)\n", DEFAULT_DBNAME);
	fprintf(stderr, "  -m            main daemon only\n");
	fprintf(stderr, "  -e            expire daemon only\n");
	fprintf(stderr, "  -o            overwrite database\n");
	fprintf(stderr, "  -d            set debug level (default: 0)\n");
	fprintf(stderr, "  -i            set info level (default: 0)\n");
	fprintf(stderr, "  -s            log is written to syslog (default: to stderr)\n");
	fprintf(stderr, "  -p            set log priority (default: user.info)\n");
	fprintf(stderr, "NOTE:\n");
	fprintf(stderr, "  config_file is the settings file for smtpguard\n");
	fprintf(stderr, "  it is indispensable except when '-e' is used\n");
	exit(1);
}


//
// expire old data
void *
data_expire_loop(void *arg)
{
	task_env_t *tp = (task_env_t *)arg;
	int	ret = 0;
	unsigned int	now;

	while (main_loop) {
		now = sg_time(NULL);
		if ((ret = idb_expire(tp->idb, now)) != 0) {
			sg_err(0, "data_expire_loop: aborted");
			break;
		}
		sg_sleep(EXPIRE_SPAN);
	}
	TASK_RETURN(tp, ret);
}


//
// signal handler
void
sigterm_catch(int signo)
{
	sg_debug(1, "catch SIGTERM");
	main_loop = 0;
	return;
}

void sighup_catch(int signo)
{
	sg_debug(1, "catch SIGHUP");
	loadconfig = 1;
	return;
}

/* just ignore */
void sigpipe_catch(int signo)
{
	sg_debug(1, "catch SIGPIPE");
	return;
}

//
// main daemon
void *main_daemon(void *arg)
{
	int	fd1, fd2;
	int	len;
	int	ret;
	gchar *xmlbuff; /* should be free */
	fd_set	rwfd;
	struct sockaddr_un	saddr;
	struct sockaddr_un	caddr;
	struct timeval	timeout;
	pid_t	pid;
	task_env_t	*tp = (task_env_t *)arg;
	idb_data_t	*d;
	spam_def_t	*new;
	spam_def_t	*spamdef = NULL;
	spam_context_t	ctx;
	FGSmtpInfo *fsi;

	GError *error = NULL;
	GIOChannel *ioc;

	char *sockdir = SOCK_DIR;
	char *sockname = NULL;
	char sockpath[MAXPATHLEN] = SOCK_NAME;

	spam_errmsg[0] = 0;
	loadconfig = 1;
	pid = getpid();

	if (getenv("SMTPGUARD_SOCK_DIR") && getenv("SMTPGUARD_SOCK_NAME")) {
		sockdir = getenv("SMTPGUARD_SOCK_DIR");
		sockname = getenv("SMTPGUARD_SOCK_NAME");
		snprintf(sockpath, sizeof(MAXPATHLEN), "%s/%s", sockdir, sockname);
	}

	if (mkdir(sockdir, 0700) < 0) {
		if (errno != EEXIST) {
			sg_err(errno, "main_daemon: failed to mkdir %s", sockdir);
			TASK_RETURN(tp, 1);
		}
	}

	if ((fd1 = socket(PF_UNIX, SOCK_STREAM, 0)) < 0) {
		sg_err(errno, "main_daemon: socket");
		TASK_RETURN(tp, 1);
	}

	memset(&saddr, 0, sizeof(struct sockaddr_un));

	saddr.sun_family = AF_UNIX;
	strcpy(saddr.sun_path, sockpath);

	if ((unlink(sockpath)) == -1 && errno != ENOENT) {
		sg_err(errno, "main_daemon: unlink(%s)", sockpath);
		close(fd1);
		TASK_RETURN(tp, 1);
	}

	if (bind(fd1, (struct sockaddr *)&saddr, 
			sizeof(saddr.sun_family) + strlen(sockpath)) < 0) {
		sg_err(errno, "main_daemon: bind(%s)", sockpath);
		close(fd1);
		TASK_RETURN(tp, 1);
	}

	if (listen(fd1, QUEUELIMIT) < 0) {
		sg_err(errno, "main_daemon: listen");
		close(fd1);
		TASK_RETURN(tp, 1);
	}

	while (main_loop) {
		if (loadconfig) {
			sg_info(0, "main_daemon: load config file %s",tp->config);
			// read config file
			if (evalguard_init(tp->config, &ctx, &new) != 0) {
				sg_err(0, "main_daemon: evalguard_init failed.");
				close(fd1);
				TASK_RETURN(tp, 1);
			}

			if (spamdef) spam_free_def(spamdef);
			spamdef = new;
			loadconfig = 0;
		}

		len = sizeof(caddr);

		timeout.tv_sec = 1;
		timeout.tv_usec = 0;
		FD_ZERO(&rwfd);
		FD_SET(fd1, &rwfd);

		ret = select(fd1 + 1, &rwfd, NULL, NULL, &timeout);
		if (ret < 0) {
			if ((errno == EAGAIN) || (errno == EINTR)) {
				continue;
			}
			sg_err(errno, "main_daemon: select");
			break;
		}
		if (FD_ISSET (fd1, &rwfd)) {
			if ((fd2 = accept(fd1, (struct sockaddr *)&caddr,
					  (socklen_t *)&len)) < 0) {
				// log error
				sg_err(errno, "main_daemon: accept");
			}
		} else
			continue;

		// main loop

		ioc = g_io_channel_unix_new (fd2);

		// select loop(read)
		while (main_loop) {
			/* Setup timeout */
			timeout.tv_sec = 1;
			timeout.tv_usec = 0;
			FD_ZERO(&rwfd);
			FD_SET(fd2, &rwfd);
			ret = select(fd2 + 1, &rwfd, NULL, NULL, &timeout);
			if (ret <= 0) {
				if ((errno == EAGAIN) || (errno == EINTR)) {
					continue;
				}
				if (ret != 0)
					sg_err(errno, "main_daemon: select for read");
				break;
			}
			if (FD_ISSET (fd2, &rwfd)) {
				fsi = fg_read_parse (ioc, &error);
				if (error != NULL) {
					sg_err(0, "main_daemon: read_parse: %s", error->message);
					xmlbuff = fg_protocol_error_get_xml (error, NULL);
					g_error_free (error);
					error = NULL;
				} else if ((d = search_db(tp->idb, fsi)) == NULL) {
					sg_err(0, "main_daemon: database search error");
					g_set_error (&error, FG_SMTP_INFO_ERROR,
						     FG_SMTP_INFO_ERROR_DB_SEARCH_FAILED,
						     "database search failed");
					xmlbuff = fg_protocol_error_get_xml (error, NULL);
					g_error_free (error);
					error = NULL;
					fg_smtp_info_free (fsi);
				} else if ((ret = evalguard_calc (&ctx, spamdef, fsi, d)) == -1) {
					sg_err(0, "main_daemon: evalguard_calc error");
					g_set_error (&error, FG_SMTP_INFO_ERROR,
						     FG_SMTP_INFO_ERROR_EVALCALC_FAILED,
						     "evalguard_cal error");
					xmlbuff = fg_protocol_error_get_xml (error, NULL);
					g_error_free (error);
					error = NULL;
					free(d);
					fg_smtp_info_free(fsi);
				} else { /* success */
					disp_smtpdb_data(fsi, d);

					// TIME,PID,SESSION,IP,MAILFROM,RCPTTO,RC,RH
					sg_info(0, "%lu %lu %s %s %s %d", pid,
						fg_smtp_info_get_pid (fsi),
						fg_smtp_info_get_addr (fsi),
						fg_smtp_info_get_mailfrom (fsi),
						fg_smtp_info_get_rcptto (fsi),
						d->P);

					if (ret == 0) ret = 1;
					if ((ret = update_db(tp->idb, d, ret)) != 0) {
						sg_err(0, "main_daemon: update database error");
						g_set_error (&error, FG_SMTP_INFO_ERROR,
							     FG_SMTP_INFO_ERROR_DB_UPDATE_FAILED,
							     "database update failed");
						xmlbuff = fg_protocol_error_get_xml (error, NULL);
						g_error_free (error);
						error = NULL;
					} else {
						xmlbuff = fg_smtp_info_get_xml (fsi, &error);
						if ( error != NULL ) {
							sg_err(0, "main_daemon: data make error: %s", error->message);
							xmlbuff = fg_protocol_error_get_xml (error, NULL);						
							g_error_free (error);
							error = NULL;
						}
					}
					free(d);
					fg_smtp_info_free (fsi);
				} /* parse done xmlbuff should be allocaed */
				/* ready for write */
				while (main_loop) {
					timeout.tv_sec = 1;
					timeout.tv_usec = 0;
					FD_ZERO(&rwfd);
					FD_SET(fd2, &rwfd);
					ret = select(fd2 + 1, NULL, &rwfd, NULL, &timeout);
					if (ret <= 0) {
						if ((errno == EAGAIN) || (errno == EINTR)) {
							continue;
						}
						if (ret != 0)
							sg_err(errno, "main_daemon: select for write");
						break;
					}
					if (FD_ISSET (fd2, &rwfd)) {
						fg_write_protocol (ioc, xmlbuff,
								   strlen(xmlbuff),
								   &error);
						if (error != NULL) {
							sg_err(0, "main_daemon: %s", error->message);
							g_error_free (error);
							error = NULL;
						}
						break;
					}
				} /* while () for write */
				g_free (xmlbuff);
				break;
			} /* FD_ISSET (read) */
		} /* while () for read */

		g_io_channel_unref (ioc);
		close(fd2);
	}

	// when we're done
	if (spamdef) spam_free_def(spamdef);
	close(fd1);

	TASK_RETURN(tp, 0);
}


//
// main program
int main(int argc, char **argv)
{
	int	main_only;
	int	expire_only;
	int	overwrite_db;
	int	ret = 0;
	int	debug_level = 0;
	int	info_level = 0;
	void	*retp;
	char 	c;
	char 	*dbhome = DEFAULT_DBHOME;
	char 	*dbname = DEFAULT_DBNAME;
	idb_t	idb;
	task_env_t expire_env, main_env;

	int	i;
	char	*work;
	char	logpr[32];
	char	logfa[32];
	int	logpr_val;
	int	logfa_val;

	g_type_init ();

	memset(&logpr, 0, sizeof(logpr));
	memset(&logfa, 0, sizeof(logfa));
	logpr_val = 0;
	logfa_val = 0;

	signal(SIGCHLD , mailwarn_handle);

	main_loop = 1;
	main_only = 0;
	expire_only = 0;
	overwrite_db = 0;
	mailfrom = NULL;
	sendmail = NULL;
	expire = 0;

	sg_set_strerror(db_strerror);
	idb_set_errfunc(sg_err);

	if (getenv(SG_DBHOME)) {
		dbhome = getenv(SG_DBHOME);
	}
	if (getenv(SG_DBNAME)) {
		dbname = getenv(SG_DBNAME);
	}
	hostname = getenv("HOSTNAME");

	opterr = 0;
	while ((c = getopt(argc, argv, "d:eh:i:mn:op:s")) != -1) {
		switch(c) {
			case 'd':
				debug_level = strtol(optarg, NULL, 10);
				smtpguard_debug_level = debug_level;
				break;
			case 'h':
				dbhome = strdup(optarg);
				if (dbhome == NULL) {
					sg_err(errno, "strdup for dbhome");
					exit(1);
				}
				break;
			case 'e':
				expire_only = 1;
				break;
			case 'i':
				info_level = strtol(optarg, NULL, 10);
				break;
			case 'm':
				main_only = 1;
				break;
			case 'n':
				dbname = strdup(optarg);
				if (dbname == NULL) {
					sg_err(errno, "strdup for dbname");
					exit(1);
				}
				break;
			case 'o':
				overwrite_db = 1;
				break;
			case 'p':
				work = strdup(optarg);
				if (work == NULL) {
					sg_err(errno, "strdup for dbhome");
					exit(1);
				}
				sscanf(work, "%[^.].%s", logfa, logpr);
				for (i = 0; prioritynames[i].c_name != NULL; i++) {
					if (strcmp(prioritynames[i].c_name, logpr)) continue;
					logpr_val = prioritynames[i].c_val;
					break;
				}
				if (logpr_val == 0) {
					sg_err(errno, "wrong log priority. [%s]", logpr);
					exit(1);
				}
				for (i = 0; facilitynames[i].c_name != NULL; i++) {
					if (strcmp(facilitynames[i].c_name, logfa)) continue;
					logfa_val = facilitynames[i].c_val;
					break;
				}
				if (logfa_val == 0) {
					sg_err(errno, "wrong log facility. [%s]", logfa);
					exit(1);
				}
				sg_log_pri = logpr_val | logfa_val;
				break;
			case 's':
				syslog_output = 1;
				break;
			default:
				usage();
				break;
		}
	}
	if (!expire_only && optind >= argc) {
		sg_err(0, "plz, set config file.");
		usage();
	}

	if (dbhome == NULL || dbname == NULL) {
		sg_err(0, "DB parameter isn't set up.");
		usage();
	}

	if (hostname == NULL) {
		sg_err(0, "hostname is null.");
		usage();
	}

	if (!overwrite_db) {
		if (remove_dir(dbhome) != 0) {
			sg_err(0, "please check DB parameters.");
			exit(1);
		}
		if (remove(dbname) != 0 && errno != ENOENT) {
			sg_err(0, "remove: %s: %s", strerror(errno), dbname);
			sg_err(0, "please check DB parameters.");
			return -1;
		}
	}

	// set log level
	sg_set_info_level(info_level);
	sg_set_debug_level(debug_level);

	// signal handler
	signal(SIGTERM, sigterm_catch);
	signal(SIGHUP, sighup_catch);
	signal(SIGPIPE, sigpipe_catch);

	// open DB
	if ((ret = idb_open(&idb, PROGNAME, dbhome, dbname, "c", 0600)) != 0) {
		sg_err(ret, "main: idb_open failed");
		exit(1);
	}

	// expire_daemon
	memset(&expire_env, 0, sizeof(expire_env));
	expire_env.idb = &idb;
	if (main_only) {
		expire_env.down = 1;
		expire_env.dead = 1;
	}
	else {
		if ((ret = pthread_create(&expire_env.tid, NULL, data_expire_loop,
				   (void *)&expire_env)) != 0) {
			sg_err(errno, "main: start expire daemon");
			goto err;
		}
	}

	// main_daemon
	memset(&main_env, 0, sizeof(main_env));
	main_env.idb = &idb;
	if (expire_only) {
		main_env.down = 1;
		main_env.dead = 1;
	}
	else {
		main_env.config = strdup(argv[optind]);
		if ((ret = pthread_create(&main_env.tid, NULL, main_daemon,
			   	(void *)&main_env)) != 0) {
			sg_err(errno, "main: start main daemon");
			goto err;
		}
	}

	while (!expire_env.down || !main_env.down) {
		sleep(2);
		if (!expire_env.down && expire_env.dead) {
			pthread_join(expire_env.tid, &retp);
			expire_env.down = 1;
			main_loop = 0;
		}
		if (!main_env.down && main_env.dead) {
			pthread_join(main_env.tid, &retp);
			main_env.down = 1;
			main_loop = 0;
		}
	}
	err:
	// close DB
	if (idb_close(&idb) != 0) {
		sg_err(0, "main: Could not close DataBase!");
		exit(1);
	}

	return ret;
}
