/**************************************************************/
/* ********************************************************** */
/* *                                                        * */
/* *              DFG2TPTP                                  * */
/* *                                                        * */
/* *  $Module:    DFG2TPTP                                  * */ 
/* *                                                        * */
/* *  Copyright (C) 1997, 1998, 1999, 2000, 2001            * */
/* *  MPI fuer Informatik                                   * */
/* *                                                        * */
/* *  This program is free software; you can redistribute   * */
/* *  it and/or modify it under the terms of the GNU        * */
/* *  General Public License as published by the Free       * */
/* *  Software Foundation; either version 2 of the License, * */
/* *  or (at your option) any later version.                * */
/* *                                                        * */
/* *  This program is distributed in the hope that it will  * */
/* *  be useful, but WITHOUT ANY WARRANTY; without even     * */
/* *  the implied warranty of MERCHANTABILITY or FITNESS    * */
/* *  FOR A PARTICULAR PURPOSE.  See the GNU General Public * */
/* *  License for more details.                             * */
/* *                                                        * */
/* *  You should have received a copy of the GNU General    * */
/* *  Public License along with this program; if not, write * */
/* *  to the Free Software Foundation, Inc., 59 Temple      * */
/* *  Place, Suite 330, Boston, MA  02111-1307  USA         * */
/* *                                                        * */
/* *                                                        * */
/* $Revision: 0.71 $                                        * */
/* $State: Exp $                                            * */
/* $Date: 2002/02/17 15:15:33 $                             * */
/* $Author: weidenb $                                       * */
/* *                                                        * */
/* *             Contact:                                   * */
/* *             Christoph Weidenbach                       * */
/* *             MPI fuer Informatik                        * */
/* *             Stuhlsatzenhausweg 85                      * */
/* *             66123 Saarbruecken                         * */
/* *             Email: weidenb@mpi-sb.mpg.de               * */
/* *             Germany                                    * */
/* *                                                        * */
/* ********************************************************** */
/**************************************************************/


/* $RCSfile: dfg2tptp.c,v $ */


/**************************************************************/
/* Includes                                                   */
/**************************************************************/

#include "dfg.h"
#include "clause.h"
#include "foldfg.h"

#include <errno.h>

#define DFG2TPTP__VERSION "0.47"

static void TPTPOutput(FILE* File, TERM Term);

static void TPTPSymbolOutput(FILE* File, SYMBOL Symbol)
{
  if (symbol_Equal(Symbol,fol_All())) {
    fputs("\n!", File);
    return;
  }
  if (symbol_Equal(Symbol,fol_Exist())) {
    fputs("\n?", File);
    return;
  }
  if (symbol_Equal(Symbol,fol_Implies())) {
    fputs("=>\n", File);
    return;
  }
  if (symbol_Equal(Symbol,fol_Equiv())) {
    fputs("<=>\n", File);
    return;
  }
  if (symbol_Equal(Symbol,fol_Not())) {
    fputs("~", File);
    return;
  }
  if (symbol_Equal(Symbol,fol_And())) {
    fputs("&\n", File);
    return;
  }
  if (symbol_Equal(Symbol,fol_Or())) {
    fputs("|\n", File);
    return;
  }
  symbol_FPrint(File,Symbol);
  return;
}

static void TPTPListOutput(FILE* File, LIST List)
/**************************************************************
  INPUT:   A list of terms.
  RETURNS: Nothing.
***************************************************************/
{
  for (; !list_Empty(List); List=list_Cdr(List)) {
    TPTPOutput(File,list_Car(List));
    if (!list_Empty(list_Cdr(List)))
      putc(',', File);
  }
}


static void TPTPOutput(FILE* File, TERM Term)
/**************************************************************
  INPUT:   A file and a term.
  RETURNS: Nothing.
  SUMMARY: Prints the term in prefix notation to the file. 
***************************************************************/
{
  SYMBOL Top;
  Top = term_TopSymbol(Term);
  if (term_IsComplex(Term)) {
    if (fol_IsQuantifier(Top)) {
      TPTPSymbolOutput(File,Top);
      fputs(" [", File);
      TPTPListOutput(File,fol_QuantifierVariables(Term));
      fputs("] : ", File);
      TPTPOutput(File, term_SecondArgument(Term));
    }
    else
      if (symbol_Equal(Top,fol_And()) || symbol_Equal(Top,fol_Or()) ||
	  symbol_Equal(Top,fol_Implies()) || symbol_Equal(Top,fol_Equiv())) {
	LIST Scan;
	fputs("(", File);
	for (Scan = term_ArgumentList(Term);!list_Empty(Scan);Scan=list_Cdr(Scan)) {
	  TPTPOutput(File,list_Car(Scan));
	  if (!list_Empty(list_Cdr(Scan))) {
	    putc(' ', File);
	    TPTPSymbolOutput(File,Top);
	    putc(' ', File);
	  }
	}
	putc(')', File);
      }
      else
	{
	  TPTPSymbolOutput(File,Top);
	  putc('(', File);
	  TPTPListOutput(File,term_ArgumentList(Term));
	  putc(')', File);
	}
  }
  else 
    TPTPSymbolOutput(File,Top);
}

void TPTPCommentOutput(FILE *File, const char* String)
/**************************************************************
  INPUT:   A file and a comment string.
  RETURNS: none.
  SUMMARY: Prints the comment string  to the file: if <String>
           string  contains a new line character it is automatically
           followed by a '%' and some space according to TPTP
           standards
***************************************************************/
{
  int i;
  for (i = 0; String[i] != '\0'; i++)
    if (String[i] == 10 && String[i+1] != '%') {
      putc(String[i], File);
      fputs("%            ",File);
    }
    else
      putc(String[i], File);
}

/**************************************************************/
/* Main Function                                              */
/**************************************************************/

int main(int argc, const char* argv[])
{
  LIST       Clauses,Axioms,Conjectures,Sorts,Scan, UserPrecedence;
  FILE       *Output,*Input;
  int        i;
  CLAUSE     Clause;
  const char *Filename,*Outname;
  FLAGSTORE  FlagStore, Precedence;

  memory_Init(memory__UNLIMITED);
  symbol_Init(TRUE);
  stack_Init();
  term_Init();
  flag_Init();

  FlagStore = flag_CreateStore();
  flag_InitStoreByDefaults(FlagStore);
  Precedence = symbol_CreatePrecedence();

  fol_Init(TRUE, Precedence);
  clause_Init();

  if (argc < 3) {
    fputs("\n\t          dfg2tptp Version ", stdout);
    fputs(DFG2TPTP__VERSION, stdout);
    puts("\n\t       Usage: dfg2tptp <input-file> <output-file>\n");
    return EXIT_FAILURE;
  }

  Axioms      = list_Nil();
  Conjectures = list_Nil();
  Sorts       = list_Nil();
  UserPrecedence = list_Nil();

  Filename = *++argv;
  Input    = misc_OpenFile(Filename, "r");
  Clauses  = dfg_DFGParser(Input, FlagStore, Precedence, &Axioms, &Conjectures,
			   &Sorts, &UserPrecedence);
  misc_CloseFile(Input,Filename);

  Axioms = list_Nconc(Axioms, Sorts);
  dfg_StripLabelsFromList(Axioms);
  dfg_StripLabelsFromList(Conjectures);

  symbol_LowerSignature();

  Outname = *++argv;
  Output  = misc_OpenFile(Outname, "w");

  if (!list_Empty(Axioms) || !list_Empty(Conjectures)) {
    fprintf(Output,"%%--------------------------------------------------------------------------");
    fprintf(Output,"\n%% File     : No information");
    fprintf(Output,"\n%% Domain   : No information");
    fprintf(Output,"\n%% Problem  : %s",Filename);
    fprintf(Output,"\n%% Version  : No information");
    fprintf(Output,"\n%% English  : No information");
    fprintf(Output,"\n\n%% Refs     : No information");
    fprintf(Output,"\n%% Source   : No information");
    fprintf(Output,"\n%% Names    : No information");
    fprintf(Output,"\n\n%% Status   : %s",( dfg_ProblemStatus()== DFG_UNSATISFIABLE ? "theorem" :
				  ( dfg_ProblemStatus()== DFG_SATISFIABLE ? "satisfiable" : "unknown")));
    fprintf(Output,"\n%% Rating   : ?");
    fprintf(Output,"\n%% Syntax   : No information");
    fprintf(Output,"\n\n%% Comments : File generated by dfg2tptp Version ");
    fprintf(Output, DFG2TPTP__VERSION);
    fprintf(Output,"\n%%--------------------------------------------------------------------------");
    i=1;
    for (Scan = Axioms;!list_Empty(Scan);Scan=list_Cdr(Scan)) {
      fprintf(Output,"\ninput_formula(ax%d,axiom,(\n\t",i++);
      TPTPOutput(Output,list_Car(Scan));
      fputs(")).", Output);
    }
    i=1;
    for (Scan = Conjectures;!list_Empty(Scan);Scan=list_Cdr(Scan)) {
      fprintf(Output,"\ninput_formula(co%d,conjecture,(\n\t",i++);
      TPTPOutput(Output,list_Car(Scan));
      fputs(")).\n", Output);
    }
    fprintf(Output,"%%--------------------------------------------------------------------------\n");
  }
  else {
    fprintf(Output,"%%--------------------------------------------------------------------------");
    fprintf(Output,"\n%% File     : No information");
    fprintf(Output,"\n%% Domain   : No information");
    fprintf(Output,"\n%% Problem  : %s",Filename);
    fprintf(Output,"\n%% Version  : No information");
    fprintf(Output,"\n%% English  : No information");
    fprintf(Output,"\n\n%% Refs     : No information");
    fprintf(Output,"\n%% Source   : No information");
    fprintf(Output,"\n%% Names    : No information");
    fprintf(Output,"\n\n%% Status   : %s",( dfg_ProblemStatus()== DFG_UNSATISFIABLE ? "unsatisfiable" :
				    ( dfg_ProblemStatus()== DFG_SATISFIABLE ? "satisfiable" : "unknown")));
    fprintf(Output,"\n%% Rating   : ?");
    fprintf(Output,"\n%% Syntax   : No information");
    fprintf(Output,"\n\n%% Comments : File generated by dfg2tptp Version ");
    fprintf(Output, DFG2TPTP__VERSION);
    fprintf(Output,"\n%%--------------------------------------------------------------------------");
    for (Scan = Clauses;!list_Empty(Scan);Scan=list_Cdr(Scan)) {
      Clause = (CLAUSE)list_Car(Scan);
      fprintf(Output,"\ninput_clause(clause%d,%s,\n\t",clause_Number(Clause),
	      (clause_GetFlag(Clause,CONCLAUSE) ? "conjecture" : "axiom"));
      putc('[', Output);
      for (i=clause_FirstLitIndex();i<clause_Length(Clause);i++) {
	if (i <= clause_LastAntecedentLitIndex(Clause))
	  fputs("--", Output);
	else
	  fputs("++", Output);
	TPTPOutput(Output,clause_GetLiteralAtom(Clause,i));
	if (i+1 < clause_Length(Clause))
	  fputs(", ", Output);
      }
      fputs("]).\n", Output);
    }
    fprintf(Output,"%%--------------------------------------------------------------------------\n");
  }

  misc_CloseFile(Output,Outname);

  clause_DeleteClauseList(Clauses);
  term_DeleteTermList(Axioms);
  term_DeleteTermList(Conjectures);

  flag_DeleteStore(FlagStore);
  symbol_DeletePrecedence(Precedence);
  list_Delete(UserPrecedence);

  /*symbol_Dump();*/
  fol_Free();
  symbol_FreeAllSymbols();
  dfg_Free();
#ifdef CHECK
  memory_Print();
#endif
  return 0;
}


