/* 
 * config file parser functions
 *
 * (C) 2000-2003 by Harald Welte <laforge@gnumonks.org>
 * (C) 2004 by Michal Kwiatkowski <ruby@joker.linuxstuff.pl>
 */

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 
 *  as published by the Free Software Foundation
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <conffile/conffile.h>


#ifdef DEBUG
#define DEBUGP(format, args...) do {				\
		fprintf(stderr, "%s:%i:", __FILE__, __LINE__);	\
		fprintf(stderr, format, ## args);		\
	} while (0)
#else
#define DEBUGP(format, args...)
#endif


#define SECTION_NAME_LEN 16
typedef struct section {
	struct section *	next;
	char			name[SECTION_NAME_LEN];
	long			begin;
	long			end;
	long			curr;
	short int		line;
	short int		line_curr;
	short int		flag;
} section_t;


#define BLOCK_NAME_LEN 32
typedef struct block {
	struct block *	next;
	char		name[BLOCK_NAME_LEN];
	short int	line,
			flag; /* set to 1 if block was accessed by config_block() */
	section_t *	sections; /* linked list of sections */
	section_t *	unrelated; /* pointer to unrelated options at the begging of block */
} block_t;


static FILE *fd;
static block_t *blocks;
static block_t *current_block;
static section_t *current_section;


/* 
 * get_word()  - parse a line into words
 * 
 * Arguments:     line   line to parse
 *                delim  possible word delimiters
 *                buf    pointer to buffer where word is returned
 * Return value:         pointer to first char after word
 * 
 * This function can deal with "" quotes.
 *    Written by Harald Welte, modified by Michal Kwiatkowski.
 */
static char *get_word(char *line, char *delim, char *buf)
{
	char *p, *start = NULL, *stop = NULL;
	int inquote = 0;

	for (p = line; *p; p++) {
		if (*p == '"') {
			start  = p + 1;
			inquote = 1;
			break;
		}
		if (!strchr(delim, *p)) {
			start = p;
			break;
		}
	}
	if (!start)
		return NULL;

	/* determine pointer to one char after word */
	for (p = start; *p; p++) {
		if (inquote) {
			if (*p == '"') {
				stop = p;
				break;
			}
		} else {
			if (strchr(delim, *p)) {
				stop = p;
				break;
			}
		}
	}
	if (!stop)
		return NULL;

	if (buf) {
		strncpy(buf, start, (size_t) (stop-start));
		*(buf + (stop-start)) = '\0';
	}

	/* skip quote character */
	if (inquote)
		/* yes, we can return stop + 1. If " was the last 
		 * character in string, it now points to NULL-term */
		return (stop + 1);

	return stop;
}


/*
 * read_line()  - reads one line from file and snips comments
 * 
 * Arguments:     line  buffer to store line
 * Return value:        number of lines read
 */
static int lines_read;
static int read_line(char *line, int len)
{
	char *p;

	if (fgets(line, len, fd) == NULL) {
		return 0;
	}
	lines_read++;

	/* comment should act as new line and string delimiter */
	if ((p = strchr(line, '#')) != NULL) {
		*p = '\n';
		*(p+1) = '\0';
	}

	/* look for line continuation: escape and a newline */
	if ((p = strstr(line, "\\\n")) != NULL) {
		/* recursion - the easiest way */
		*p = '\0';
		len -= strlen(line);
		if (len <= 0)
			return 0;
		if (read_line(p, len) == 0)
			return 0;
	}

	return lines_read;
}


/*
 * find_key()  - looks for config entry with given key name in list
 * 
 * Arguments:     name  name of option we're looking for
 *                list  list of config entries to check
 * Return value:        pointer to config entry or NULL if not found
 */
static config_entry_t *find_key(char *name, config_entry_t *list)
{
	config_entry_t *ce;

	for (ce = list; ce; ce = ce->next) {
		if (!strncmp(name, ce->key, CONFIG_KEY_LEN))
			return ce;
	}

	return NULL;
}


/*
 * mtoi()  - changes memory string into plain integer
 *
 * Arguments:     str   string to parse
 * Return value:        integer value of given string
 */
static int mtoi(char *str)
{
	char number[32];
	char *last;
	int factor = 1;

	strncpy(number, str, 31);
	last = &(number[strlen(number) - 1]);
	switch (*last) {
		case 'M':
			factor *= 1024;
		case 'K':
			factor *= 1024;
			*last = '\0';
		default:
			break;
	}

	return (atoi(number) * factor);
}


/* 
 * config_open()  - opens a file and divide it into blocks
 * 
 * Arguments:     path  points to config file
 * Return value:        number of blocks found
 *
 * Block starts with string and curly bracket '{', while ends with '}'.
 * No nesting is allowed.
 * Section names begin with a colon, and can be only defined inside blocks.
 * '#' is treated as a comment.
 */
int config_open(char *path)
{
	int found = 0, line_ctr = 0, inside_block = 0, inside_section = 0;
	block_t *bl = NULL;
	section_t *se = NULL;

	if ((fd = fopen(path, "r")) == NULL) {
		fprintf(stderr, "Couldn't open config file %s: %s\n",
				path, strerror(errno));
		return 0;
	}

	/* search for blocks */
	do {
		int ret;
		char line[CONFIG_LINE_LEN];
		char id[CONFIG_VALUE_LEN], bracket[CONFIG_VALUE_LEN];
		char *p;

		/* block can end at the same line it started,
		 * so let's check for this first */
		if (inside_block) {
			char *s;
			if ((s = strchr(line, '}')) != NULL) {
				if (get_word(s+1, " \t\n", NULL)) {
					fprintf(stderr, "Config error at line %d: "
						"trailing characters after end of block.\n",
						line_ctr);
					return 0;
				}
				/* end of block means end of last section */
				if (inside_section) {
					se->end = ftell(fd) - strlen(line) + (s - line);
				}
				found++;
				inside_block = inside_section = 0;
			}
			if (strchr(line, '{')) {
				fprintf(stderr, "Config error at line %d: "
						"nesting of blocks forbidden.\n",
						line_ctr);
				return 0;
			}
		}

		lines_read = 0;
		if ((ret = read_line(line, CONFIG_LINE_LEN-1)) == 0)
			break;
		line_ctr += ret;

		/* if we're inside block, just check sections beginings,
		 * validity will be checked during config_read() */
		if (inside_block) {
			p = line;
			if ((p = get_word(p, " \t\n", id)) == NULL)
				continue;
			if (id[0] == ':') {
				/* end previous section first */
				if (inside_section) {
					se->end = ftell(fd) - strlen(line);
				}
				if ((se = (section_t *)malloc(sizeof(section_t))) == NULL) {
					fprintf(stderr, "config_open(): out of memory.\n");
					return 0;
				}
				if (id[1] == '\0') {
					if ((p = get_word(p, " \t\n", id)) == NULL) {
						fprintf(stderr, "Config error at line %d: "
							"section name missing.\n",
							line_ctr);
						return 0;
					}
					strncpy(se->name, id, SECTION_NAME_LEN);
				} else {
					strncpy(se->name, &id[1], SECTION_NAME_LEN);
				}
				se->next = NULL;
				se->curr = se->begin = ftell(fd);
				se->line_curr = se->line = line_ctr+1;
				se->flag = 0;
				if (bl->sections == NULL) {
					bl->sections = se;
				} else {
					section_t *last = bl->sections;
					while (last->next)
						last = last->next;
					last->next = se;
				}
				inside_section = 1;
			} else if (!inside_section) {
				if ((se = (section_t *)malloc(sizeof(section_t))) == NULL) {
					fprintf(stderr, "config_open(): out of memory.\n");
					return 0;
				}
				se->next = NULL;
				se->name[0] = '\0';
				se->curr = se->begin = ftell(fd) - strlen(line);
				se->line_curr = se->line = line_ctr;
				se->flag = 0;
				bl->unrelated = se;
				inside_section = 1;
			}
			continue;
		}

		/* read two words, second should be curly bracket */
		p  = line;
		if ((p = get_word(p, " \t\n", id)) == NULL)
			continue;
		p = get_word(p, " \t\n", bracket);
		if (p == NULL || bracket[0] != '{') {
			fprintf(stderr, "Config error at line %d: "
					"\"%s\" is not a block.\n",
					line_ctr, id);
			return 0;
		}

		for (bl = blocks; bl; bl = bl->next) {
			if (!strcmp(bl->name, id)) {
				fprintf(stderr, "Config error at line %d: "
						"redefinition of block \"%s\".\n",
						line_ctr, id);
				return 0;
			}
		}

		/* create new block_t structure and fill it */
		if ((bl = (block_t *)malloc(sizeof(block_t))) == NULL) {
			fprintf(stderr, "config_open(): out of memory.\n");
			return 0;
		}
		if (!blocks) {
			blocks = bl;
		} else {
			block_t *tmp = blocks;
			while (tmp->next)
				tmp = tmp->next;
			tmp->next = bl;
		}
		bl->next = NULL;
		strncpy(bl->name, id, BLOCK_NAME_LEN);
		bl->line = line_ctr;
		bl->flag = 0;
		inside_block = 1;
		/* clear start of block, so it won't drop into
		 * condition at line 263 */
		*(strchr(line, '{')) = ' ';

	} while (!feof(fd));

	if (inside_block) {
		fprintf(stderr, "Config error at line %d: "
				"unexpected end of file.\n", line_ctr);
		return 0;
	}

	return found;
}


/*
 * set_block_section()  - set given block and section combination as current
 *
 * Arguments:     block     block name
 *                section   section name (or NULL if you want unrelated options)
 *                rewind    if 1 current position will be reset to section's begining
 * Return value:            0 if not found and starting line otherwise
 */
static int set_block_section(const char *block, const char *section, const int rewind)
{
	block_t *p;
	section_t *s;

	for (p = blocks; p; p = p->next) {
		if (!strcmp(p->name, block)) {
			if (section == NULL) {
				if ((s = p->unrelated) == NULL)
					return 0;
				goto found;
			}
			for (s = p->sections; s; s = s->next) {
				if (!strcmp(s->name, section))
					goto found;
			}
		}
	}

	/* not found */
	return 0;

found:
	current_block = p;
	current_section =  s;
	if (rewind) {
		s->curr = s->begin;
		s->line_curr = s->line;
	}
	fseek(fd, s->curr, SEEK_SET);
	p->flag = s->flag = 1;
	return s->line_curr;
}


/* 
 * is_section()  - checks if section in given block exists
 *
 * Arguments:     block     block name
 *                section   section name or NULL for unrelated
 * Return value:            1 if found, 0 if not
 */
int is_section(const char *block, const char *section)
{
	block_t *p;
	section_t *s;

	for (p = blocks; p; p = p->next) {
		if (!strcmp(p->name, block)) {
			if (section == NULL) {
				if (p->unrelated != NULL)
					return 1;
				else
					continue;
			}
			for (s = p->sections; s; s = s->next) {
				if (!strcmp(s->name, section)) {
					return 1;
				}
			}
		}
	}

	return 0;
}


/*
 * is_block()  - checks if given block exists
 *
 * Arguments:     block     block name
 * Return value:            1 if found, 0 if not
 */
int is_block(const char *block)
{
	block_t *p;

	for (p = blocks; p; p = p->next) {
		if (!strcmp(p->name, block)) {
			return 1;
		}
	}

	return 0;
}


/*
 * config_read()  - read options from current block
 *
 * Arguments:     block    block to read options from
 *                section  section to read options from
 *                list     list of config entries to fill
 *                flags    see conffile.h for CONFIG_READ_*
 * Return value:           see conffile.h for CONFIG_RET_*
 *
 * Function normally returns after parsing all options or after reading
 * entry with CONFIG_OPT_MULTI/CONFIG_OPT_ANY set. In the second case
 * function must be invoked many times to read all these options.
 */
int config_read(const char *block, const char *section, config_entry_t *list, const int flags)
{
	int current_line;
	config_entry_t *ce;
	char line[CONFIG_LINE_LEN];
	char wordbuf[CONFIG_VALUE_LEN];
	char *wordend, *p;

	if ((current_line = set_block_section(block, section, flags & CONFIG_READ_REWIND)) == 0)
		return CONFIG_RET_MISSING;

	while (ftell(fd) < current_section->end) {
		int ret;

		lines_read = 0;
		if ((ret = read_line(line, CONFIG_LINE_LEN-1)) == 0)
			break;
		current_line += ret;

		/* terminate at end of block */
		if ((p = strchr(line, '}')) != NULL) {
			*p = '\n';
			*(p+1) = '\0';
		}

		if ((wordend = get_word(line, " \t\n", wordbuf)) == NULL)
			continue;

		if (list->options == CONFIG_OPT_ANY) {
			ce = list;
			strncpy(list->key, wordbuf, sizeof(list->key));
		} else if ((ce = find_key(wordbuf, list)) == NULL) {
			if ((flags & CONFIG_READ_IGNORE)) {
				continue;
			} else {
				fprintf(stderr, "Config error at line %d: "
						"bad option \"%s\".\n",
						current_line, wordbuf);
				return CONFIG_RET_ERROR;
			}
		}

		if (ce->options == CONFIG_OPT_IGNORE)
			continue;

		if (ce->hit) {
			fprintf(stderr, "Config error at line %d: "
					"redefinition of option \"%s\".\n",
					current_line, wordbuf);
			return CONFIG_RET_ERROR;
		}

		if (ce->type != CONFIG_TYPE_BOOLEAN) {
			if ((wordend = get_word(wordend, " \t\n", wordbuf)) == NULL) {
				fprintf(stderr, "Config error at line %d: "
					"no value for option \"%s\".\n",
					current_line, wordbuf);
				return CONFIG_RET_ERROR;
			}
		}

		if (get_word(wordend, " \t\n", wordbuf) != NULL) {
			fprintf(stderr, "Config error at line %d: "
					"trailing characters after option.\n",
					current_line);
			return CONFIG_RET_ERROR;
		}

		switch (ce->type) {
			case CONFIG_TYPE_BOOLEAN:
				ce->u.value = 1;
				break;
			case CONFIG_TYPE_INT:
				ce->u.value = strtol(wordbuf, NULL, 10);
				break;
			case CONFIG_TYPE_HEX:
				ce->u.value = strtol(wordbuf, NULL, 16);
				break;
			case CONFIG_TYPE_STRING:
				strncpy(ce->u.string, wordbuf, CONFIG_VALUE_LEN);
				break;
			case CONFIG_TYPE_MEM:
				ce->u.value = mtoi(wordbuf);
				break;
			default:
				DEBUGP("config_read(): bad config type %i.\n", ce->type);
				break;
		}

		if (ce->options == CONFIG_OPT_MULTI || ce->options == CONFIG_OPT_ANY) {
			current_section->line_curr = current_line;
			current_section->curr = ftell(fd);
			return CONFIG_RET_MORE;
		} else {
			ce->hit = 1;
		}
	}

	for (ce = list; ce; ce = ce->next) {
		if ((ce->options == CONFIG_OPT_MANDATORY) && !ce->hit) {
			fprintf(stderr, "Config error in %s::%s: "
					"mandatory option \"%s\" not set.\n",
					current_block->name, current_section->name,
					ce->key);
			return CONFIG_RET_ERROR;
		}
	}

	current_section->line_curr = current_line;
	current_section->curr = ftell(fd);

	return CONFIG_RET_OK;
}


/*
 * config_close()  - closes config file and frees all block descriptors
 *
 * Arguments:     none
 * Return value:        -1 on error
 */
int config_close(void)
{
	block_t *bl;
	section_t *se;

	if (fclose(fd) == EOF) {
		DEBUGP("config_close(): Couldn't close config file.\n");
		return -1;
	}

	for (bl = blocks; bl; bl = bl->next) {
		if (!bl->flag) {
			fprintf(stderr, "Config warning at line %d: "
					"block \"%s\" unused.\n",
					bl->line, bl->name);
		}
	}

	current_block = blocks;
	while (current_block) {
		bl = current_block->next;

		current_section = current_block->sections;
		while (current_section) {
			se = current_section->next;
			free(current_section);
			current_section = se;
		}
		if (current_block->unrelated) {
			free(current_block->unrelated);
		}

		free(current_block);
		current_block = bl;
	}
	blocks = NULL;
	/* current_block and current_section end up being NULL */

	return 0;
}

