/* specter_HTTP.c
 *
 * specter input plugin for parsing http requests and responses
 *
 * (C) 2004 by Michal Kwiatkowski <ruby@joker.linuxstuff.pl>
 */

/*
 * FIXME
 * 
 * Currently we're not very liberal - dropping packets that
 * don't strictly obey standards. I wasn't aware of broken
 * server/clients when writing this. Some patches will be needed
 * to parse somewhat understood but malformed packets.
 *
 * We don't support multi-line headers, which are discouraged but
 * accepted by rfc.
 */

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 
 *  as published by the Free Software Foundation
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/ip.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <specter/specter.h>
#include "chtons.h"


/* this is storage structure for fixed-length fields */
static struct {
	char	version[12], /* HTTP/x.x */
		content_md5[49], /* bas64/128bit md5 digest */
		date[34],
		expires[34],
		if_modified_since[34],
		if_unmodified_since[34],
		last_modified[34];
} static_values;


#define HTTP_RETS_NUM 64
static specter_iret_t http_rets[HTTP_RETS_NUM] = {
/*  0 */{ .type = SPECTER_IRET_BOOL, .name = "http.type" }, /* 0 for request, 1 for response */

/*  1 */{ .type = SPECTER_IRET_UINT16, .name = "http.code" },
	{ .type = SPECTER_IRET_STRING, .name = "http.method" },
	{ .type = SPECTER_IRET_STRING, .name = "http.reason" },
	{ .type = SPECTER_IRET_STRING, .name = "http.url" },
	{ .type = SPECTER_IRET_STRING, .name = "http.version",
			.value = { .ptr = static_values.version } },

/*  6 */{ .type = SPECTER_IRET_STRING, .name = "http.cache_control" },
	{ .type = SPECTER_IRET_STRING, .name = "http.connection" },
	{ .type = SPECTER_IRET_STRING, .name = "http.date",
			.value = { .ptr = static_values.date } },
	{ .type = SPECTER_IRET_STRING, .name = "http.keep_alive" },
/* 10 */{ .type = SPECTER_IRET_STRING, .name = "http.mime_version" },
	{ .type = SPECTER_IRET_STRING, .name = "http.pragma" },
	{ .type = SPECTER_IRET_STRING, .name = "http.trailer" },
	{ .type = SPECTER_IRET_STRING, .name = "http.transfer_encoding" },
	{ .type = SPECTER_IRET_STRING, .name = "http.upgrade" },
/* 15 */{ .type = SPECTER_IRET_STRING, .name = "http.via" },
	{ .type = SPECTER_IRET_STRING, .name = "http.warning" },

/* 17 */{ .type = SPECTER_IRET_STRING, .name = "http.allow" },
	{ .type = SPECTER_IRET_STRING, .name = "http.content_base" },
	{ .type = SPECTER_IRET_STRING, .name = "http.content_encoding" },
/* 20 */{ .type = SPECTER_IRET_STRING, .name = "http.content_language" },
	{ .type = SPECTER_IRET_UINT32, .name = "http.content_length" },
	{ .type = SPECTER_IRET_STRING, .name = "http.content_location" },
	{ .type = SPECTER_IRET_STRING, .name = "http.content_md5",
			.value = { .ptr = static_values.content_md5 } },
	{ .type = SPECTER_IRET_STRING, .name = "http.content_range" },
/* 25 */{ .type = SPECTER_IRET_STRING, .name = "http.content_type" },
	{ .type = SPECTER_IRET_STRING, .name = "http.content_version" },
	{ .type = SPECTER_IRET_STRING, .name = "http.derived_from" },
	{ .type = SPECTER_IRET_STRING, .name = "http.expires",
			.value = { .ptr = static_values.expires } },
	{ .type = SPECTER_IRET_STRING, .name = "http.last_modified",
			.value = { .ptr = static_values.last_modified } },
/* 30 */{ .type = SPECTER_IRET_STRING, .name = "http.link" },
	{ .type = SPECTER_IRET_STRING, .name = "http.title" },
	{ .type = SPECTER_IRET_STRING, .name = "http.uri" },

/* 33 */{ .type = SPECTER_IRET_STRING, .name = "http.accept" },
	{ .type = SPECTER_IRET_STRING, .name = "http.accept_charset" },
/* 35 */{ .type = SPECTER_IRET_STRING, .name = "http.accept_encoding" },
	{ .type = SPECTER_IRET_STRING, .name = "http.accept_language" },
	{ .type = SPECTER_IRET_STRING, .name = "http.authorization" },
	{ .type = SPECTER_IRET_STRING, .name = "http.expect" },
	{ .type = SPECTER_IRET_STRING, .name = "http.from" },
/* 40 */{ .type = SPECTER_IRET_STRING, .name = "http.host" },
	{ .type = SPECTER_IRET_STRING, .name = "http.if_match" },
	{ .type = SPECTER_IRET_STRING, .name = "http.if_modified_since",
			.value = { .ptr = static_values.if_modified_since } },
	{ .type = SPECTER_IRET_STRING, .name = "http.if_none_match" },
	{ .type = SPECTER_IRET_STRING, .name = "http.if_range" },
/* 45 */{ .type = SPECTER_IRET_STRING, .name = "http.if_unmodified_since",
			.value = { .ptr = static_values.if_unmodified_since } },
	{ .type = SPECTER_IRET_UINT32, .name = "http.max_forwards" },
	{ .type = SPECTER_IRET_STRING, .name = "http.proxy_authorization" },
	{ .type = SPECTER_IRET_STRING, .name = "http.range" },
	{ .type = SPECTER_IRET_STRING, .name = "http.referer" },
/* 50 */{ .type = SPECTER_IRET_STRING, .name = "http.te" },
	{ .type = SPECTER_IRET_STRING, .name = "http.user_agent" },

/* 52 */{ .type = SPECTER_IRET_STRING, .name = "http.accept_ranges" },
	{ .type = SPECTER_IRET_UINT32, .name = "http.age" },
	{ .type = SPECTER_IRET_STRING, .name = "http.alternates" },
/* 55 */{ .type = SPECTER_IRET_STRING, .name = "http.content_disposition" },
	{ .type = SPECTER_IRET_STRING, .name = "http.etag" },
	{ .type = SPECTER_IRET_STRING, .name = "http.location" },
	{ .type = SPECTER_IRET_STRING, .name = "http.proxy_authenticate" },
	{ .type = SPECTER_IRET_STRING, .name = "http.public" },
/* 60 */{ .type = SPECTER_IRET_STRING, .name = "http.retry_after" },
	{ .type = SPECTER_IRET_STRING, .name = "http.server" },
	{ .type = SPECTER_IRET_STRING, .name = "http.vary" },
	{ .type = SPECTER_IRET_STRING, .name = "http.www_authenticate" },
};


#define PORT_HTTP	80
static u_int16_t http_ports[] = {
	__constant_htons(PORT_HTTP),
};

#define HTTP_METHODS_NUM 11
static char http_methods[HTTP_METHODS_NUM][9] = {
	"OPTIONS ",
	"GET ",
	"HEAD ",
	"POST ",
	"PUT ",
	"DELETE ",
	"TRACE ",
	"CONNECT ",
	"PATCH ", /* obsolete */
	"LINK ", /* obsolete */
	"UNLINK ", /* obsolete */
};


/*
 * Hashing function intended for ASCII strings with at least 4 characters.
 * Generates the same output regardless of letter's case.
 */
static inline u_int16_t hash(const char *str, int len)
{
	u_int16_t h;
	int rot;

	for (h = 0x0; len--; str++) {
		h = (h << 4) | (h >> 12);
		h ^= (*str & 0x1e) >> 1;
		rot = ((*str & 0x1) + 1) * 3;
		h = (h << rot) | (h >> (16 - rot));
	}

	return h;
}

enum {
	/* general headers */
	HASH_CACHE_CONTROL	= 0x40f2,
	HASH_CONNECTION		= 0x7125,
	HASH_DATE		= 0xc081,
	HASH_KEEP_ALIVE		= 0xb008, /* obsolete */
	HASH_MIME_VERSION	= 0x8e1a,
	HASH_PRAGMA		= 0x4426,
	HASH_TRAILER		= 0xc2ad,
	HASH_TRANSFER_ENCODING	= 0x2e62,
	HASH_UPGRADE		= 0x1786,
	HASH_VIA		= 0x0584,
	HASH_WARNING		= 0x35e1,

	/* entity headers */
	HASH_ALLOW		= 0x81c6,
	HASH_CONTENT_BASE	= 0x2e9d, /* obsolete */
	HASH_CONTENT_ENCODING	= 0xd38c,
	HASH_CONTENT_LANGUAGE	= 0x0386,
	HASH_CONTENT_LENGTH	= 0xc120,
	HASH_CONTENT_LOCATION	= 0xaa56,
	HASH_CONTENT_MD5	= 0xc40b,
	HASH_CONTENT_RANGE	= 0x0114,
	HASH_CONTENT_TYPE	= 0x4341,
	HASH_CONTENT_VERSION	= 0xeb0c, /* obsolete */
	HASH_DERIVED_FROM	= 0xc717, /* obsolete */
	HASH_EXPIRES		= 0x0acc,
	HASH_LAST_MODIFIED	= 0x0880,
	HASH_LINK		= 0x6341, /* obsolete */
	HASH_TITLE		= 0xc161, /* obsolete */
	HASH_URI		= 0x2401, /* obsolete */

	/* request headers */
	HASH_ACCEPT		= 0x6170,
	HASH_ACCEPT_CHARSET	= 0xbed0, /* where I should go now |-) */
	HASH_ACCEPT_ENCODING	= 0x6934,
	HASH_ACCEPT_LANGUAGE	= 0xd643,
	HASH_AUTHORIZATION	= 0xfc15,
	HASH_EXPECT		= 0x20d3,
	HASH_FROM		= 0xc507,
	HASH_HOST		= 0x23d0,
	HASH_IF_MATCH		= 0x20b8,
	HASH_IF_MODIFIED_SINCE	= 0x6b59,
	HASH_IF_NONE_MATCH	= 0x66b3,
	HASH_IF_RANGE		= 0x0c3b,
	HASH_IF_UNMODIFIED_SINCE= 0x8fe0,
	HASH_MAX_FORWARDS	= 0xdf0f,
	HASH_PROXY_AUTHORIZATION= 0xf293,
	HASH_RANGE		= 0x0a03,
	HASH_REFERER		= 0xc2f9,
	HASH_TE			= 0x4081,
	HASH_USER_AGENT		= 0xeb79,

	/* response headers */
	HASH_ACCEPT_RANGES	= 0x0f18,
	HASH_AGE		= 0x0083,
	HASH_ALTERNATES		= 0x874c, /* obsolete */
	HASH_CONTENT_DISPOSITION= 0x719d,
	HASH_ETAG		= 0x05c4,
	HASH_LOCATION		= 0x120e,
	HASH_PROXY_AUTHENTICATE	= 0x22b2,
	HASH_PUBLIC		= 0x4940, /* obsolete */
	HASH_RETRY_AFTER	= 0x4ce6,
	HASH_SERVER		= 0x88bc,
	HASH_VARY		= 0xe303,
	HASH_WWW_AUTHENTICATE	= 0x58f5,
};


/* 
 * FIXME: macros below are ugly but i first have to get some sleep
 * before I replace them with something more logical
 */

/* are there @x characters in the buffer? */
#define IS_OVERFLOW(x) (x > len - (cur_data - data))


/* allocate memory and copy token into @iret */
#define tokcpy(iret, delim)							\
	do {									\
		size_t size = strcspn(cur_data, delim);				\
										\
		if (!size)							\
			break;							\
										\
		if (IS_OVERFLOW(size))						\
			return 0;						\
										\
		if ((iret.value.ptr = strndup(cur_data, size)) == NULL) {	\
			specter_log(SPECTER_ERROR,				\
				"Couldn't allocate buffer: %s.\n",		\
				strerror(errno));				\
			return -1;						\
		}								\
										\
		iret.flags |= SPECTER_RETF_FREE | SPECTER_RETF_VALID;		\
		cur_data += size;						\
	} while (0)

/* copy token into static array @array */
#define tokcpy_static(iret, delim, array)			\
	do {							\
		size_t size = strcspn(cur_data, delim);		\
								\
		if (!size || size >= sizeof(array))		\
			break;					\
								\
		if (IS_OVERFLOW(size))				\
			return 0;				\
								\
		strncpy(array, cur_data, size);			\
		array[size] = '\0';				\
								\
		iret.flags |= SPECTER_RETF_VALID;		\
		cur_data += size;				\
	} while (0)

/* convert and copy integer into @iret with precision @prec */
#define intcpy(iret, delim, prec)				\
	do {							\
		size_t size = strcspn(cur_data, delim);		\
		char buf[32];					\
								\
		if (!size || size >= sizeof(buf))		\
			break;					\
								\
		if (IS_OVERFLOW(size))				\
			return 0;				\
								\
		strncpy(buf, cur_data, size);			\
		buf[size] = '\0';				\
								\
		iret.value.prec = atoi(buf);			\
		iret.flags |= SPECTER_RETF_VALID;		\
		cur_data += size;				\
	} while (0)


static int parse_request(char *data, const int len)
{
	specter_iret_t *general_ret = &http_rets[6];
	specter_iret_t *entity_ret = &http_rets[17];
	specter_iret_t *request_ret = &http_rets[33];
	char *cur_data = data;
	int i;

	/* check if it's really a http request */
	for (i = 0; i < HTTP_METHODS_NUM; i++) {
		if (!strncmp(data, http_methods[i], strlen(http_methods[i])))
			break;
	}
	if (i == HTTP_METHODS_NUM) {
		return 0;
	} else {
		http_rets[0].value.b = 0;
		http_rets[0].flags |= SPECTER_RETF_VALID;
		/* method */
		http_rets[2].value.ptr = http_methods[i];
		http_rets[2].flags |= SPECTER_RETF_VALID;
		cur_data += strlen(http_methods[i]);
	}

	/* url */
	tokcpy(http_rets[4], " \r");

	/* version - field not present int HTTP/0.9 */
	if (*cur_data == ' ') {
		cur_data++;
		tokcpy_static(http_rets[5], "\r", static_values.version);
	}
	if (!strncmp(cur_data, "\r\n", 2))
		cur_data += 2;
	else
		return 0;

	/* headers - the idea is to compare hash values, but call one strcmp()
	 * on matched data to be sure it's what we wanted */
	do {
		switch (hash(cur_data, strchr(cur_data, ':') - cur_data)) {

			/* general header */
			case HASH_CACHE_CONTROL:
				if (strncasecmp(cur_data, "Cache-Control: ", 15))
					break;
				cur_data += 15;
				tokcpy(general_ret[0], "\r");
				break;
			case HASH_CONNECTION:
				if (strncasecmp(cur_data, "Connection: ", 12))
					break;
				cur_data += 12;
				tokcpy(general_ret[1], "\r");
				break;
			case HASH_DATE:
				if (strncasecmp(cur_data, "Date: ", 6))
					break;
				cur_data += 6;
				tokcpy_static(general_ret[2], "\r", static_values.date);
				break;
			case HASH_KEEP_ALIVE:
				if (strncasecmp(cur_data, "Keep-Alive: ", 12))
					break;
				cur_data += 12;
				tokcpy(general_ret[3], "\r");
				break;
			case HASH_MIME_VERSION:
				if (strncasecmp(cur_data, "MIME-Version: ", 14))
					break;
				cur_data += 14;
				tokcpy(general_ret[4], "\r");
				break;
			case HASH_PRAGMA:
				if (strncasecmp(cur_data, "Pragma: ", 8))
					break;
				cur_data += 8;
				tokcpy(general_ret[5], "\r");
				break;
			case HASH_TRAILER:
				if (strncasecmp(cur_data, "Trailer: ", 9))
					break;
				cur_data += 9;
				tokcpy(general_ret[6], "\r");
				break;
			case HASH_TRANSFER_ENCODING:
				if (strncasecmp(cur_data, "Transfer-Encoding: ", 19))
					break;
				cur_data += 19;
				tokcpy(general_ret[7], "\r");
				break;
			case HASH_UPGRADE:
				if (strncasecmp(cur_data, "Upgrade: ", 9))
					break;
				cur_data += 9;
				tokcpy(general_ret[8], "\r");
				break;
			case HASH_VIA:
				if (strncasecmp(cur_data, "Via: ", 5))
					break;
				cur_data += 5;
				tokcpy(general_ret[9], "\r");
				break;
			case HASH_WARNING:
				if (strncasecmp(cur_data, "Warning: ", 9))
					break;
				cur_data += 9;
				tokcpy(general_ret[10], "\r");
				break;

			/* entity header */
			case HASH_ALLOW:
				if (strncasecmp(cur_data, "Allow: ", 7))
					break;
				cur_data += 7;
				tokcpy(entity_ret[0], "\r");
				break;
			case HASH_CONTENT_BASE:
				if (strncasecmp(cur_data, "Content-Base: ", 14))
					break;
				cur_data += 14;
				tokcpy(entity_ret[1], "\r");
				break;
			case HASH_CONTENT_ENCODING:
				if (strncasecmp(cur_data, "Content-Encoding: ", 18))
					break;
				cur_data += 18;
				tokcpy(entity_ret[2], "\r");
				break;
			case HASH_CONTENT_LANGUAGE:
				if (strncasecmp(cur_data, "Content-Language: ", 18))
					break;
				cur_data += 18;
				tokcpy(entity_ret[3], "\r");
				break;
			case HASH_CONTENT_LENGTH:
				if (strncasecmp(cur_data, "Content-Length: ", 16))
					break;
				cur_data += 16;
				intcpy(entity_ret[4], "\r", ui32);
				break;
			case HASH_CONTENT_LOCATION:
				if (strncasecmp(cur_data, "Content-Location: ", 18))
					break;
				cur_data += 18;
				tokcpy(entity_ret[5], "\r");
				break;
			case HASH_CONTENT_MD5:
				if (strncasecmp(cur_data, "Content-MD5: ", 13))
					break;
				cur_data += 13;
				tokcpy_static(entity_ret[6], "\r", static_values.content_md5);
				break;
			case HASH_CONTENT_RANGE:
				if (strncasecmp(cur_data, "Content-Range: ", 15))
					break;
				cur_data += 15;
				tokcpy(entity_ret[7], "\r");
				break;
			case HASH_CONTENT_TYPE:
				if (strncasecmp(cur_data, "Content-Type: ", 14))
					break;
				cur_data += 14;
				tokcpy(entity_ret[8], "\r");
				break;
			case HASH_CONTENT_VERSION:
				if (strncasecmp(cur_data, "Content-Version: ", 17))
					break;
				cur_data += 17;
				tokcpy(entity_ret[9], "\r");
				break;
			case HASH_DERIVED_FROM:
				if (strncasecmp(cur_data, "Derived-From: ", 14))
					break;
				cur_data += 14;
				tokcpy(entity_ret[10], "\r");
				break;
			case HASH_EXPIRES:
				if (strncasecmp(cur_data, "Expires: ", 9))
					break;
				cur_data += 9;
				tokcpy_static(entity_ret[11], "\r", static_values.expires);
				break;
			case HASH_LAST_MODIFIED:
				if (strncasecmp(cur_data, "Last-Modified: ", 15))
					break;
				cur_data += 15;
				tokcpy_static(entity_ret[12], "\r", static_values.last_modified);
				break;
			case HASH_LINK:
				if (strncasecmp(cur_data, "Link: ", 6))
					break;
				cur_data += 6;
				tokcpy(entity_ret[13], "\r");
				break;
			case HASH_TITLE:
				if (strncasecmp(cur_data, "Title: ", 7))
					break;
				cur_data += 7;
				tokcpy(entity_ret[14], "\r");
				break;
			case HASH_URI:
				if (strncasecmp(cur_data, "URI: ", 5))
					break;
				cur_data += 5;
				tokcpy(entity_ret[15], "\r");
				break;

			/* request header */
			case HASH_ACCEPT:
				if (strncasecmp(cur_data, "Accept: ", 8))
					break;
				cur_data += 8;
				tokcpy(request_ret[0], "\r");
				break;
			case HASH_ACCEPT_CHARSET:
				if (strncasecmp(cur_data, "Accept-Charset: ", 16))
					break;
				cur_data += 16;
				tokcpy(request_ret[1], "\r");
				break;
			case HASH_ACCEPT_ENCODING:
				if (strncasecmp(cur_data, "Accept-Encoding: ", 17))
					break;
				cur_data += 17;
				tokcpy(request_ret[2], "\r");
				break;
			case HASH_ACCEPT_LANGUAGE:
				if (strncasecmp(cur_data, "Accept-Language: ", 17))
					break;
				cur_data += 17;
				tokcpy(request_ret[3], "\r");
				break;
			case HASH_AUTHORIZATION:
				if (strncasecmp(cur_data, "Authorization: ", 15))
					break;
				cur_data += 15;
				tokcpy(request_ret[4], "\r");
				break;
			case HASH_EXPECT:
				if (strncasecmp(cur_data, "Expect: ", 8))
					break;
				cur_data += 8;
				tokcpy(request_ret[5], "\r");
				break;
			case HASH_FROM:
				if (strncasecmp(cur_data, "From: ", 6))
					break;
				cur_data += 6;
				tokcpy(request_ret[6], "\r");
				break;
			case HASH_HOST:
				if (strncasecmp(cur_data, "Host: ", 6))
					break;
				cur_data += 6;
				tokcpy(request_ret[7], "\r");
				break;
			case HASH_IF_MATCH:
				if (strncasecmp(cur_data, "If-Match: ", 10))
					break;
				cur_data += 10;
				tokcpy(request_ret[8], "\r");
				break;
			case HASH_IF_MODIFIED_SINCE:
				if (strncasecmp(cur_data, "If-Modified-Since: ", 19))
					break;
				cur_data += 19;
				tokcpy_static(request_ret[9], "\r", static_values.if_modified_since);
				break;
			case HASH_IF_NONE_MATCH:
				if (strncasecmp(cur_data, "If-None-Match: ", 15))
					break;
				cur_data += 15;
				tokcpy(request_ret[10], "\r");
				break;
			case HASH_IF_RANGE:
				if (strncasecmp(cur_data, "If-Range: ", 10))
					break;
				cur_data += 10;
				tokcpy(request_ret[11], "\r");
				break;
			case HASH_IF_UNMODIFIED_SINCE:
				if (strncasecmp(cur_data, "If-Unmodified-Since: ", 21))
					break;
				cur_data += 21;
				tokcpy_static(request_ret[12], "\r", static_values.if_unmodified_since);
				break;
			case HASH_MAX_FORWARDS:
				if (strncasecmp(cur_data, "Max-Forwards: ", 14))
					break;
				cur_data += 14;
				intcpy(request_ret[13], "\r", ui32);
				break;
			case HASH_PROXY_AUTHORIZATION:
				if (strncasecmp(cur_data, "Proxy-Authorization: ", 21))
					break;
				cur_data += 21;
				tokcpy(request_ret[14], "\r");
				break;
			case HASH_RANGE:
				if (strncasecmp(cur_data, "Range: ", 7))
					break;
				cur_data += 7;
				tokcpy(request_ret[15], "\r");
				break;
			case HASH_REFERER:
				if (strncasecmp(cur_data, "Referer: ", 9))
					break;
				cur_data += 9;
				tokcpy(request_ret[16], "\r");
				break;
			case HASH_TE:
				if (strncasecmp(cur_data, "TE: ", 4))
					break;
				cur_data += 4;
				tokcpy(request_ret[17], "\r");
				break;
			case HASH_USER_AGENT:
				if (strncasecmp(cur_data, "User-Agent: ", 12))
					break;
				cur_data += 12;
				tokcpy(request_ret[18], "\r");
				break;

			default:
				/* unsupported header - ignore */
				while (!IS_OVERFLOW(2) && strncmp(cur_data, "\r\n", 2))
					cur_data++;
				break;
		}

		if (*(cur_data+1) != '\n')
			return 0;
		cur_data += 2;

	} while (!IS_OVERFLOW(2) && strncmp(cur_data, "\r\n", 2));

	return 0;
}


static int parse_response(char *data, int len)
{
	specter_iret_t *general_ret = &http_rets[6];
	specter_iret_t *entity_ret = &http_rets[17];
	specter_iret_t *response_ret = &http_rets[52];
	char *cur_data = data;

	/* check if it's really a http response */
	if (IS_OVERFLOW(5) || strncmp(data, "HTTP/", 5)) {
		return 0;
	} else {
		http_rets[0].value.b = 1;
		http_rets[0].flags |= SPECTER_RETF_VALID;
	}

	/* version */
	tokcpy(http_rets[5], " ");
	if (*cur_data++ != ' ')
		return 0;

	/* code */
	intcpy(http_rets[1], " ", ui16);
	if (*cur_data++ != ' ')
		return 0;

	/* reason */
	tokcpy(http_rets[3], "\r");
	if (strncmp(cur_data, "\r\n", 2))
		return 0;

	/* headers */
	do {
		switch (hash(cur_data, strchr(cur_data, ':') - cur_data)) {

			/* general header */
			case HASH_CACHE_CONTROL:
				if (strncasecmp(cur_data, "Cache-Control: ", 15))
					break;
				cur_data += 15;
				tokcpy(general_ret[0], "\r");
				break;
			case HASH_CONNECTION:
				if (strncasecmp(cur_data, "Connection: ", 12))
					break;
				cur_data += 12;
				tokcpy(general_ret[1], "\r");
				break;
			case HASH_DATE:
				if (strncasecmp(cur_data, "Date: ", 6))
					break;
				cur_data += 6;
				tokcpy_static(general_ret[2], "\r", static_values.date);
				break;
			case HASH_KEEP_ALIVE:
				if (strncasecmp(cur_data, "Keep-Alive: ", 12))
					break;
				cur_data += 12;
				tokcpy(general_ret[3], "\r");
				break;
			case HASH_MIME_VERSION:
				if (strncasecmp(cur_data, "MIME-Version: ", 14))
					break;
				cur_data += 14;
				tokcpy(general_ret[4], "\r");
				break;
			case HASH_PRAGMA:
				if (strncasecmp(cur_data, "Pragma: ", 8))
					break;
				cur_data += 8;
				tokcpy(general_ret[5], "\r");
				break;
			case HASH_TRAILER:
				if (strncasecmp(cur_data, "Trailer: ", 9))
					break;
				cur_data += 9;
				tokcpy(general_ret[6], "\r");
				break;
			case HASH_TRANSFER_ENCODING:
				if (strncasecmp(cur_data, "Transfer-Encoding: ", 19))
					break;
				cur_data += 19;
				tokcpy(general_ret[7], "\r");
				break;
			case HASH_UPGRADE:
				if (strncasecmp(cur_data, "Upgrade: ", 9))
					break;
				cur_data += 9;
				tokcpy(general_ret[8], "\r");
				break;
			case HASH_VIA:
				if (strncasecmp(cur_data, "Via: ", 5))
					break;
				cur_data += 5;
				tokcpy(general_ret[9], "\r");
				break;
			case HASH_WARNING:
				if (strncasecmp(cur_data, "Warning: ", 9))
					break;
				cur_data += 9;
				tokcpy(general_ret[10], "\r");
				break;

			/* entity header */
			case HASH_ALLOW:
				if (strncasecmp(cur_data, "Allow: ", 7))
					break;
				cur_data += 7;
				tokcpy(entity_ret[0], "\r");
				break;
			case HASH_CONTENT_BASE:
				if (strncasecmp(cur_data, "Content-Base: ", 14))
					break;
				cur_data += 14;
				tokcpy(entity_ret[1], "\r");
				break;
			case HASH_CONTENT_ENCODING:
				if (strncasecmp(cur_data, "Content-Encoding: ", 18))
					break;
				cur_data += 18;
				tokcpy(entity_ret[2], "\r");
				break;
			case HASH_CONTENT_LANGUAGE:
				if (strncasecmp(cur_data, "Content-Language: ", 18))
					break;
				cur_data += 18;
				tokcpy(entity_ret[3], "\r");
				break;
			case HASH_CONTENT_LENGTH:
				if (strncasecmp(cur_data, "Content-Length: ", 16))
					break;
				cur_data += 16;
				intcpy(entity_ret[4], "\r", ui32);
				break;
			case HASH_CONTENT_LOCATION:
				if (strncasecmp(cur_data, "Content-Location: ", 18))
					break;
				cur_data += 18;
				tokcpy(entity_ret[5], "\r");
				break;
			case HASH_CONTENT_MD5:
				if (strncasecmp(cur_data, "Content-MD5: ", 13))
					break;
				cur_data += 13;
				tokcpy_static(entity_ret[6], "\r", static_values.content_md5);
				break;
			case HASH_CONTENT_RANGE:
				if (strncasecmp(cur_data, "Content-Range: ", 15))
					break;
				cur_data += 15;
				tokcpy(entity_ret[7], "\r");
				break;
			case HASH_CONTENT_TYPE:
				if (strncasecmp(cur_data, "Content-Type: ", 14))
					break;
				cur_data += 14;
				tokcpy(entity_ret[8], "\r");
				break;
			case HASH_CONTENT_VERSION:
				if (strncasecmp(cur_data, "Content-Version: ", 17))
					break;
				cur_data += 17;
				tokcpy(entity_ret[9], "\r");
				break;
			case HASH_DERIVED_FROM:
				if (strncasecmp(cur_data, "Derived-From: ", 14))
					break;
				cur_data += 14;
				tokcpy(entity_ret[10], "\r");
				break;
			case HASH_EXPIRES:
				if (strncasecmp(cur_data, "Expires: ", 9))
					break;
				cur_data += 9;
				tokcpy_static(entity_ret[11], "\r", static_values.expires);
				break;
			case HASH_LAST_MODIFIED:
				if (strncasecmp(cur_data, "Last-Modified: ", 15))
					break;
				cur_data += 15;
				tokcpy_static(entity_ret[12], "\r", static_values.last_modified);
				break;
			case HASH_LINK:
				if (strncasecmp(cur_data, "Link: ", 6))
					break;
				cur_data += 6;
				tokcpy(entity_ret[13], "\r");
				break;
			case HASH_TITLE:
				if (strncasecmp(cur_data, "Title: ", 7))
					break;
				cur_data += 7;
				tokcpy(entity_ret[14], "\r");
				break;
			case HASH_URI:
				if (strncasecmp(cur_data, "URI: ", 5))
					break;
				cur_data += 5;
				tokcpy(entity_ret[15], "\r");

			/* response header */
			case HASH_ACCEPT_RANGES:
				if (strncasecmp(cur_data, "Accept-Ranges: ", 15))
					break;
				cur_data += 15;
				tokcpy(response_ret[0], "\r");
				break;
			case HASH_AGE:
				if (strncasecmp(cur_data, "Age: ", 5))
					break;
				cur_data += 5;
				intcpy(response_ret[1], "\r", ui32);
				break;
			case HASH_ALTERNATES:
				if (strncasecmp(cur_data, "Alternates: ", 12))
					break;
				cur_data += 12;
				tokcpy(response_ret[2], "\r");
				break;
			case HASH_CONTENT_DISPOSITION:
				if (strncasecmp(cur_data, "Content-Disposition: ", 21))
					break;
				cur_data += 21;
				tokcpy(response_ret[3], "\r");
				break;
			case HASH_ETAG:
				if (strncasecmp(cur_data, "ETag: ", 6))
					break;
				cur_data += 6;
				tokcpy(response_ret[4], "\r");
				break;
			case HASH_LOCATION:
				if (strncasecmp(cur_data, "Location: ", 10))
					break;
				cur_data += 10;
				tokcpy(response_ret[5], "\r");
				break;
			case HASH_PROXY_AUTHENTICATE:
				if (strncasecmp(cur_data, "Proxy-Authenticate: ", 20))
					break;
				cur_data += 20;
				tokcpy(response_ret[6], "\r");
				break;
			case HASH_PUBLIC:
				if (strncasecmp(cur_data, "Public: ", 8))
					break;
				cur_data += 8;
				tokcpy(response_ret[7], "\r");
				break;
			case HASH_RETRY_AFTER:
				if (strncasecmp(cur_data, "Retry-After: ", 13))
					break;
				cur_data += 13;
				tokcpy(response_ret[8], "\r");
				break;
			case HASH_SERVER:
				if (strncasecmp(cur_data, "Server: ", 8))
					break;
				cur_data += 8;
				tokcpy(response_ret[9], "\r");
				break;
			case HASH_VARY:
				if (strncasecmp(cur_data, "Vary: ", 6))
					break;
				cur_data += 6;
				tokcpy(response_ret[10], "\r");
				break;
			case HASH_WWW_AUTHENTICATE:
				if (strncasecmp(cur_data, "WWW-Authenticate: ", 18))
					break;
				cur_data += 18;
				tokcpy(response_ret[11], "\r");
				break;

			default:
				/* unsupported header - ignore */
				while (!IS_OVERFLOW(2) && strncmp(cur_data, "\r\n", 2))
					cur_data++;
				break;
		}

		if (*(cur_data+1) != '\n')
			return 0;
		cur_data += 2;

	} while (!IS_OVERFLOW(2) && strncmp(cur_data, "\r\n", 2));

	return 0;
}


static int http_input(ulog_packet_msg_t *pkt)
{
	struct iphdr *iph = (struct iphdr *) pkt->payload;
	struct tcphdr *tcph = (struct tcphdr *)((u_int32_t *)iph + iph->ihl);
	char *tcpdata = (char *)tcph + 4 * tcph->doff;
	int datalen = ntohs(iph->tot_len) - 4 * (iph->ihl + tcph->doff);
	int ctr;

	if (iph->protocol != IPPROTO_TCP || !datalen)
		return 0;

	for (ctr = 0; ctr < sizeof(http_ports)/sizeof(u_int16_t); ctr++) {
		if (tcph->source == http_ports[ctr])
			return parse_response(tcpdata, datalen);
		else if (tcph->dest == http_ports[ctr])
			return parse_request(tcpdata, datalen);
	}

	return 0;
}

static specter_input_t http_ip = { 
	.name = "http",
	.input = &http_input
};

void _init(void)
{
	if (register_input(&http_ip, http_rets, HTTP_RETS_NUM, 0) == -1) {
		specter_log(SPECTER_FATAL, "Couldn't register.\n");
		exit(EXIT_FAILURE);
	}
}

