#!/usr/local/bin/perl
#
# This represents a view of hosts by host groups. 
# a single host, or a list of hosts by FDQN name.
#
#   groupnames     - a list of string names of all of the groups
#   group_hash      - a hash of HostList objects keyed by group name
#
# + new()         - constructor (sets instance vars to arguments passed in)
# + gets/sets()   - magical set/get functions (autoloaded based on func name)
# + display()     - output format and view
#
## + display_problems() - shows information only about hosts that have services
##                        that indicate a problem
#
# + group         - used to get at a specific HostList by name
# + groups        - used to get back a generic list of groups that are stored
# + add           - adds a specific host group to the instance
#

# $Id: HostGroups.pm,v 1.11 2001/01/10 16:29:41 sljohnson Exp $

use Spong::Host;
use Spong::HostList;

package Spong::HostGroups;

my %COLORS = ( 'blue', 4, 'red', 5, 'yellow', 3, 'green', 1, 'purple', 2, 'clear', 0 );

# This constructor expects one of four different types of options.  Either an
# empty string specifying that no actual hostlist will be loaded, the string
# "ALL" to load all the host grups, the string containing a group name that
# represents a list of hosts, or a reference to an array that contains a list
# of host groups 
# Note: "ALL" is used to signify all host groups, becuase there is a host group
#       called "all" that is already predefined.

sub new {
   my( $proto, $grps ) = @_;
   my( $class ) = ref($proto) || $proto;
   my( %groups, $groups, $group ) = ();
   my $self = {};

   if( $grps eq "" ) {
      %groups = ();
   } elsif( ref( $group ) eq "ARRAY" ) {
      foreach $group ( @$grps ) {
	 my $object = Spong::HostList->new( $group );
	 if( $object ) { $groups{$group} = $object; }
      }
   } elsif ( $grps eq "ALL" ) {
      foreach $group ( keys %main::GROUPS ) {
         next if $group eq 'all' && ! $main::HOSTGROUPSALL;
         next if ( defined $main::GROUPS{$group}->{'display'} &&
                   $main::GROUPS{$group}->{'display'} == 0);
         my $object = Spong::HostList->new( $group );
         if( $object ) { $groups{$group} = $object; }
      }
   } else {
     my $object = Spong::HostList->new( $grps );
     if( $object ) { $groups{$grps} = $object; }
   }

   $self->{'group_hash'}      = \%groups;

   bless ($self,$class);
   return $self;
}


# Get/Set methods, hostnames() is a little fancy in that it return a list
# rather then just the list reference (easier for others to deal with).  The
# hosthash function returns a reference however.

sub group_hash { my $var = 'group_hash';
   if( defined $_[1] ) { $_[0]->{$var} = $_[1]; } return $_[0]->{$var}; }


# Some specific functions that get at and manipulate the data in the instance
# vars in more convenient ways

sub group { 
   return $_[0]->{'group_hash'}->{$_[1]}; }

sub group_names { 
   my( @grps, @ret );
   # Try to use GROUPS_LIST list order, else use sorted group names
   if ( defined @main::GROUPS_LIST ) { @grps = @main::GROUPS_LIST; }
   else { @grps = (sort keys %main::GROUPS ); }
   # Go thru list order, in order to sort the group names
   foreach my $grp ( @grps ) {
      if ( defined %{$_[0]->{'group_hash'}}->{$grp} ) {
         push @ret,$grp;
      }
   }
   return (@ret);
}

sub groups { 
   my $self = shift;
   my( @tmp );

   foreach( $self->group_names() ) { push( @tmp, $self->group( $_ ) ); }
   return @tmp;
}

sub add {
   my( $self, $hostlist ) = @_;
   my $name = $hostlist->name();

   $self->{'group_hash'}->{$name} = $hostlist;
}


# Display routines.  Does both text and html, does rely on both the Host and
# Service objects for some help.
#
# brief       Hostname and "one" color reflecting overall status of the host
# standard    Table showing hosts vs services and the state of each service
# full        Records for each host showing detailed info about its state

sub display {
   my( $self, $type, $view ) = @_;

   $self->display_text( $view ) if $type eq "text";
   $self->display_html( $view ) if $type eq "html";
   $self->display_wml( $view ) if $type eq "wml";
}

# This displays a summary of all the hosts in this list in a text format 
# suitable for displaying on dumb ascii terminals

sub display_text {
   my( $self, $format ) = @_;

   if( $format eq "brief" ) {
      my( %services, $host, $service, @names );

      # Compute the total list of services running on the various hosts, and
      # sort them alphabetically (except always put ping first).

#      foreach $host ( $self->hosts() ) {
#	 foreach $service ( $host->service_names() ) { $services{$service}++;}}

      if( grep( /^ping$/, keys %main::SERVICES ) ) { push( @names, "ping" ); }
      foreach $service ( sort keys %main::SERVICES ) {
	 push( @names, $service ) unless $service eq "ping"; }
     
      # Print the horizontal axis of the table (names of the services)

      print "-"x30, "-"x(($#names+1)*3), "\n";
      foreach my $cnt ( 0..5 ) {
	 if( $cnt == 0 ) {  print "Key: . = green,  ? = purple   "; }
	 if( $cnt == 1 ) {  print "     Y = yellow, R = red      "; }
	 if( $cnt == 2 ) {  print "     B = blue,   o = clear    "; }
	 if( $cnt == 3 ) { print " "x30; }
	 if( $cnt == 4 ) { print " "x30; }
	 if( $cnt == 5 ) { print "Group:", " "x24; }
	 print substr( "status", (length("status") - 6 + $cnt), 1 ),"  ";
	 print "\n";
      }
      print "-"x30, "-"x(($#names+1)*3), "\n";

      # Now go through each host, and fill in the table.

      foreach $group ( $self->groups() ) {
	 print substr( $group->name(), 0, 29 );
	 print " "x(30-length(substr($group->name(), 0, 29)));

         my ($color);
	 foreach $service ( @names ) {
            my ($servicecolor) = $group->services_color( $service );
            $color = $servicecolor if ($COLORS{$servicecolor} > $COLORS{$color});
         }

         if( $color ) {
            if( $color eq "green" )  { print ".  "; }
            if( $color eq "purple" ) { print "?  "; }
            if( $color eq "yellow" ) { print "Y  "; }
            if( $color eq "red" )    { print "R  "; }
            if( $color eq "blue" )   { print "B  "; }
            if( $color eq "clear" )  { print "o  "; }
         } else {
	    print "   ";
	 }
	 print "\n";
      }
      print "\n";

   } elsif( $format eq "standard" ) {
      my( %services, $host, $service, @names );

      # Compute the total list of services running on the various hosts, and
      # sort them alphabetically (except always put ping first).

#      foreach $host ( $self->hosts() ) {
#	 foreach $service ( $host->service_names() ) { $services{$service}++;}}

      if( grep( /^ping$/, keys %main::SERVICES ) ) { push( @names, "ping" ); }
      foreach $service ( sort keys %main::SERVICES ) {
	 push( @names, $service ) unless $service eq "ping"; }
     
      # Print the horizontal axis of the table (names of the services)

      print "-"x30, "-"x(($#names+1)*3), "\n";
      foreach my $cnt ( 0..4 ) {
	 if( $cnt == 0 ) {  print "Key: . = green,  ? = purple   "; }
	 if( $cnt == 1 ) {  print "     Y = yellow, R = red      "; }
	 if( $cnt == 2 ) {  print "     B = blue,   o = clear    "; }
	 if( $cnt == 3 ) { print " "x30; }
	 if( $cnt == 4 ) { print "Group:", " "x24; }
	 foreach $service ( @names ) { 
	    if( length($service) - 5 + $cnt >= 0 ) {
	       print substr( $service, (length($service) - 5 + $cnt), 1 ),"  ";
	    } else {
	       print "   ";
	    }
	 }
	 print "\n";
      }
      print "-"x30, "-"x(($#names+1)*3), "\n";

      # Now go through each host, and fill in the table.

      foreach $group ( $self->groups() ) {
	 print substr( $group->name(), 0, 29 );
	 print " "x(30-length(substr($group->name(), 0, 29)));

	 foreach my $service ( @names ) {
            my ($color) = $group->services_color( $service );
            if( $color ) {
               if( $color eq "green" )  { print ".  "; }
               if( $color eq "purple" ) { print "?  "; }
               if( $color eq "yellow" ) { print "Y  "; }
               if( $color eq "red" )    { print "R  "; }
               if( $color eq "blue" )   { print "B  "; }
               if( $color eq "clear" )  { print "o  "; }
            } else {
	       print "   ";
	    }
         }
	 print "\n";
      }
      print "\n";

   } elsif( $format eq "full" ) {
      my( $service, @names );

      # Compute the total list of services running on the various hosts, and
      # sort them alphabetically (except always put ping first).

      if( grep( /^ping$/, keys %main::SERVICES ) ) { push( @names, "ping" ); }
      foreach $service ( sort keys %main::SERVICES ) {
	 push( @names, $service ) unless $service eq "ping"; }
     
      # Print the horizontal axis of the table (names of the services)

      # Now go through each host, and fill in the table.

      foreach $group ( $self->groups() ) {
         print "-"x30, "-"x(($#names+1)*3), "\n";
	 print "Group: " . substr( $group->name(), 0, 29 ) . "\n";

         $group->display_text("standard");
      }
      print "\n";


   }
}

# This displays a summary of all the hosts in this list in an HTML format
# suitable for displaying on web clients capable of displaying tables.

sub display_html { 
   my( $self, $format ) = @_;

   if( $format eq "brief" ) {
      my (%services, $group, $serivce, @names );
      foreach $service ( sort keys %main::SERVICES ) {
         push( @names, $service );
      }

      # Print the header of the table.

      print "<table border=1 cellspacing=0 cellpadding=1";
      print " bgcolor=" . $main::WWW_TITLE_COLOR if $main::WWW_TITLE_COLOR;
      print ">\n  <tr>\n";
      print "    <td align=center>Host Group</td>\n";
      print "    <td>Status</td>\n";
      print "  </tr>\n";
 
      # Now for each group, go through and find out overall status of that
      # host.   

      foreach $group ( $self->groups() ) {
         my $name = $group->name();
         my $grname = $main::GROUPS{$name}->{'name'};

         print "  <tr>\n";
         print "    <th align=left bgcolor=#ffffff nowrap><a href=\"!!WWWSPONG!!/group/$name\" target=\"view\">$grname</a></th>\n";
   
         my ($color);
         foreach $service ( @names ) {
           my ($servicecolor) = $group->services_color( $service );
           $color = $servicecolor if ($COLORS{$servicecolor} > $COLORS{$color});
         }

         if( $color ) {
            if( $main::WWW_USE_IMAGES == 1 ) {
               print "    <td align=center";
               print " bgcolor=" . $main::WWW_CELL_COLOR
                  if $main::WWW_CELL_COLOR;
               print ">";
               my $alt = "";
               print "<img src=\"!!WWWGIFS!!/$color.gif\" alt=\"$alt\" border=0>";
               print "";
            } else {
               print "    <td align=center bgcolor=\"";
               print $main::WWW_COLOR{$color} . "\" width=25>";
               print "<font color=\"" . $main::WWW_COLOR{$color} . "\">";
               print "___</font>";
            }
            print "</td>\n";
         } else {
            if( $main::WWW_USE_IMAGES == 1 ) {
               print "    <td align=center width=25";
               print " bgcolor=" . $main::WWW_CELL_COLOR
                   if $main::WWW_CELL_COLOR;
               print "> - </td>\n";
            } else {
               print "    <td align=center width=25";
               print " bgcolor=" . $main::WWW_CELL_COLOR
                   if $main::WWW_CELL_COLOR;
               print ">&nbsp;</td>\n";
            }
         }
         print "  </tr>\n";
      }
      print "</table>\n";
         
   } elsif( $format eq "standard" ) {
      my( %services, $group, $service, @names );

      # Compute the total list of services running on the various hosts, and
      # sort them alphabetically (except always put ping first).
      
      if( grep( /^ping$/, keys %main::SERVICES ) ) { push( @names, "ping" ); }
      foreach $service ( sort keys %main::SERVICES ) {
	 push( @names, $service ) unless $service eq "ping"; }
      
      # Print the horizontal axis of the table (names of the services)
      
      print "<table border=1 cellspacing=0 cellpadding=1";
      print " bgcolor=" . $main::WWW_TITLE_COLOR if $main::WWW_TITLE_COLOR;
      print "><tr>";
      print "<td align=center width=80 nowrap><b>Host Group</b></td>\n";
      foreach $service ( @names ) { 
	 print "<td align=center valign=bottom width=25>\n";
	 print "<font size=-1><a href=\"!!WWWSPONG!!/help/$service\">";
	 print "$service</a></font></td>\n"; 
      }
      
      print "</tr>\n\n";
      
      # Now go through each host group, and fill in the table.
      
      foreach $group ( $self->groups() ) { 
         my $name = $group->name();
         my $grname = $main::GROUPS{$name}->{'name'};
	 
	 print "<tr><td align=left bgcolor=#ffffff nowrap>\n";
	 print "<a href=\"!!WWWSPONG!!/group/$name\" target=\"view\">$grname</a></td>\n";
	 
	 foreach $service ( @names ) {
            my $color = $group->services_color( $service );

	    if( $color ) {
	       if( $main::WWW_USE_IMAGES == 1 ) {
		  print "<td align=center";
                  print " bgcolor=" . $main::WWW_CELL_COLOR 
                     if $main::WWW_CELL_COLOR;
                  print ">";
                  my $alt = "";
		  print "<img src=\"!!WWWGIFS!!/$color.gif\" alt=\"$alt\" border=0>";
		  print "";
	       } else {
		  print "<td align=center bgcolor=\"";
		  print $main::WWW_COLOR{$color} . "\" width=25>";
		  print "<font color=\"" . $main::WWW_COLOR{$color} . "\">";
		  print "___</font>";
	       }
	       print "</td>";
	    } else {
	       if( $main::WWW_USE_IMAGES == 1 ) {
		  print "<td align=center width=25";
                  print " bgcolor=" . $main::WWW_CELL_COLOR
                      if $main::WWW_CELL_COLOR;
                  print "> - </td>\n"; 
	       } else {
		  print "<td align=center width=25";
                  print " bgcolor=" . $main::WWW_CELL_COLOR
                      if $main::WWW_CELL_COLOR;
                  print ">&nbsp;</td>\n";
	       }
	    }
	 }
      }	
      print "</tr></table>";

   } elsif( $format eq "full" ) {

      # Compute the total list of services running on the various hosts, and
      # sort them alphabetically (except always put ping first).
      
      if( grep( /^ping$/, keys %main::SERVICES ) ) { push( @names, "ping" ); }
      foreach $service ( sort keys %main::SERVICES ) {
	 push( @names, $service ) unless $service eq "ping"; }

      # Run though each group and print the hosts for each group
      foreach $group ( $self->groups() ) { 
         my $name = $group->name();
         my $grname = $main::GROUPS{$name}->{'name'};

         # Display the host group table
#         print "<a href=\"$main::WWWSPONG/group/$name\">";
         print "<strong>Group - $grname</strong>";
#         print "</a>\n";
         $group->display_html("standard");
       
         print "\n<HR>\n";
      }
   }
}


# These methods all display summary information about only those machines
# that are currently registering a problem, or that have services that have
# not been updated (showing purple) - it also relays information about 
# services that have been acknowledged.
#
# brief       Hostname and "one" color reflecting overall status of the host
# standard    Table showing hosts vs services and the state of each service
# full        Records for each host showing detailed info about the problems

sub display_problems {
   my( $self, $type, $format ) = @_; 
   my( $purple, $problem, $problems );
  
   $purple = $problems = 0;
   $badgroups = new Spong::HostGroups("");
   foreach my $group ( $self->groups() ) {
      $problem = 0;
      my $badhosts = new Spong::HostList("");
      foreach my $host ( $group->hosts() ) {
         if ( $host->has_problem() ) {
            $problem = $problems = 1;
            $badhosts->add( $host );
         } elsif ( $host->has_color( "purple" )) {
            $purple = 1;
        }
      }
      if ( $problem ) {
          $badhosts->name($group->name());
          $badgroups->add($badhosts);
      }
   }

   if ( $problems ) {
      foreach my $group ( $badgroups->groups() ) {
         my $name = $main::GROUPS{$group->name()}->{'name'};
         print "<b>" 
            if $type eq "html";
         print "<strong>Group: $name</strong>\n";
         print "</b><p>"                   if $type eq "html";
         $group->display( $type, $format );
         print "<hr align=left width=25%>\n";
      }
   } else {
      if( $format eq "full" ) {
         if( $purple ) {
            print "<p><font color=green><b>"       if( $type eq "html" );
            print "No known problems.\n";
            print "</b></font>"                    if( $type eq "html" );
            print "<p><font size=-1 color=purple>" if( $type eq "html" );
            print "Although some information is out of date which might ";
            print "indicate a problem.\n";
            print "</font>"                        if( $type eq "html" );
         } else {
            print "<p><font color=green><b>"       if( $type eq "html" );
            print "No current problems.\n";
            print "</b></font>"                    if( $type eq "html" );
         }
      }
   }

}


# This displays a summary of all the hosts in this list in an WML format
# suitable for displaying on WAP enabled devices

sub display_wml { 
   my( $self, $format ) = @_;

   if( $format eq "brief" ) {

      # Go through each HostList
      foreach $hostlist ( $self->groups() ) {
         my $color = substr($hostlist->summary_color(),0,2);
         $color =~ tr/a-z/A-Z/;
         my $name = $hostlist->name();
         print "<b><anchor title=\"$name\">$color";
         print "<go href=\"!!WAPSPONG!!/grp-summary/$name\"/></anchor></b> ";
         print $main::GROUPS{$hostlist->name()}->{'name'}, "<br/>\n";
      }

   }
   elsif( $format eq "standard" ) {

      # Go through each HostList
      foreach $group ( $self->groups() ) {
         print "<b> ",$main::GROUPS{$group->name()}->{'name'}, " </b><br/><br/>\n";
         $group->display_wml("standard");
         print "<br/>\n";
      }
   }
}


1;
