/* ====================================================================
 * Copyright (c) 2003-2006, Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "ListWidget.h"
#include "RepositoryModel.h"
#include "WcModel.h"
#include "ScModel.h"
#include "ProjectLvi.h"
#include "ProjectFoldersWidget.h"
#include "RepositoryFilesWidget.h"
#include "WorkingCopyFilesWidget.h"
#include "ActionStorage.h"
#include "events/CustomEvents.h"
#include "sublib/SplitLayout.h"
#include "sublib/Utility.h"
#include "sublib/settings/LayoutSettings.h"

// qt
#include <qlayout.h>
#include <qwidgetstack.h>
#include <qaction.h>
#include <qtextedit.h>

// sys
#include <assert.h>



///////////////////////////////////////////////////////////////////////////////

enum Stack
{
  sEmpty,
  sRepository,
  sWorkingCopy
};

static QString StartupText(
"<qt>"
" <u>repository & working copy navigation</u><br>"
" <ul>"
"  <li>"
"   The project folder view shows the repository and working copy bookmarks"
"   of your project. All bookmark details are displayed in a repository or"
"   a working copy view. Either as a flat view or as a tree view."
"  </li>"
"  <li>"
"   Double clicking a folder in the repository or working copy view makes the"
"   current folder the new root item."
"   Double clicking the current folder (ie. '.') will change the root back to"
"   the current folders parent."
"  </li>"
"  <li>"
"   Opening/Closing a folder item by clicking on the tree decoration will"
"   open/close it without changing the root folder."
"  </li>"
" </ul>"
" <u>caching</u><br>"
" <ul>"
"  <li>"
"   subcommander caches the working copy status and repository list information."
"   To refresh the information use the 'reload' button in the toolbar."
"  </li>"
" </ul>"
" <u>help</u><br>"
" <ul>"
"  <li>"
"   pressing SHIFT-F1 in the working copy view will display a detailed"
"   description of the the columns and their possible values."
"  </li>"
" </ul>"
" <u>internationalization (i18n)</u><br>"
" <ul>"
"  <li>"
"   Subcommander is internationalized (using the GNU gettext package). If "
"   you like to run subcommander in your native language you can help with "
"   the translation. Write to dev@subcommander.tigris.org."
"  </li>"
" </ul>"
" <u>tips/notes</u><br>"
" <ul>"
"  <li>"
"   you can simply drop an url from a web browser or a path from a file browser"
"   on a project to add it to the project."
"  </li>"
"  <li>"
"   you don't have to follow the typical trunk/branches/tags layout in your "
"   repository to use subcommanders branch and tag feature. You can set any "
"   repository folder as branch or tag folder. subcommander needs only to "
"   know WHERE you keep your branches and tags. It does NOT require that they "
"   are named 'branches' or 'tags'."
"  </li>"
" </ul>"
"</qt>"
);


ListWidget::ListWidget( ScModel* model, QWidget *parent, const char *name )
: super(parent,name), _model(model)
{
  _rpModel = new RepositoryModel(_model);
  _wcModel = new WcModel(_model);

  QGridLayout* mwl = new QGridLayout( this, 1, 1, 0, 0 );
  {
    QWidget*      hs  = new QWidget(this);
    HSplitLayout* hsl = new HSplitLayout(hs);
    _split = hsl;
    mwl->addWidget(hs,0,0);
    mwl->setRowStretch(0,1);
    {
      _subvn = new ProjectFoldersWidget( _rpModel, _wcModel, hs );
      hsl->addWidgetOne( _subvn, false, 1 );

      _stackFiles = new QWidgetStack(hs);
      hsl->addWidgetTwo( _stackFiles, false, 3 );

      _filesRp = new RepositoryFilesWidget( _rpModel, _subvn, _stackFiles );
      _filesWc = new WorkingCopyFilesWidget( _wcModel, _subvn, _stackFiles );

      QTextEdit* text = new QTextEdit(_stackFiles);
      text->setText( StartupText );
      _stackFiles->addWidget( text, sEmpty );
      _stackFiles->addWidget( _filesRp, sRepository );
      _stackFiles->addWidget( _filesWc, sWorkingCopy );

      _stackFiles->raiseWidget( sEmpty );
      //_stackFiles->setEnabled(false);
    }
  }

  connect( _subvn, SIGNAL(wcItemSelected()), this, SLOT(showWorkingCopyFiles()) );
  connect( _subvn, SIGNAL(wcStatusReset()), _filesWc, SLOT(clear()) );
  connect( _subvn, SIGNAL(wcStatusChanged()), _filesWc, SLOT(updateView()) );

  connect( _subvn, SIGNAL(rpItemSelected()), this, SLOT(showRepositoryFiles()) );
  connect( _subvn, SIGNAL(rpEntryReset()), _filesRp, SLOT(clear()) );
  connect( _subvn, SIGNAL(rpEntryChanged()), _filesRp, SLOT(updateView()) );

  connect( _subvn, SIGNAL(prjItemSelected()), this, SLOT(showProject()) );
  
  connect( _filesRp, SIGNAL(prjChanged(Project*)), _subvn, SLOT(prjChanged(Project*)) );

  // init projects...
  const Projects& projects = _model->getProjects();

  for( Projects::const_iterator it = projects.begin(); it != projects.end(); it++ )
  {
    addProject( *it );
  }
  

  LayoutSettings* ls = _model->getLayoutSettings();
  ls->getHeaderColumns( "ProjectFolderView", _subvn->header() );
  ls->getHeaderColumns( "RepositoryFilesView", _filesRp->header() );
  ls->getHeaderColumns( "WorkingCopyFilesView", _filesWc->header() );
  _split->setHandlePos( ls->getPosition( "ListWidget.split" ) );
}

ListWidget::~ListWidget()
{
  LayoutSettings* ls = _model->getLayoutSettings();
  ls->setHeaderColumns( "ProjectFolderView", _subvn->header() );
  ls->setHeaderColumns( "RepositoryFilesView", _filesRp->header() );
  ls->setHeaderColumns( "WorkingCopyFilesView", _filesWc->header() );
  ls->setPosition( "ListWidget.split", _split->getHandlePos() );

  delete _wcModel;
  delete _rpModel;
}

QPopupMenu* ListWidget::getRepositoryFilesMenu()
{
  return _filesRp->getMenu();
}

QPopupMenu* ListWidget::getWorkingCopyFilesMenu()
{
  return _filesWc->getMenu();
}

QPopupMenu* ListWidget::getWorkingCopyFolderMenu()
{
  return _subvn->getWorkingCopyMenu(); 
}

QPopupMenu* ListWidget::getRepositoryFolderMenu()
{
  return _subvn->getRepositoryMenu(); 
}

QPopupMenu* ListWidget::getProjectMenu()
{
  return _subvn->getProjectMenu();
}

// todo move to ProjectFoldersWidget
void ListWidget::addProject( Project* prj )
{
  ProjectLvi* prjLvi = new ProjectLvi( _subvn, _rpModel, _wcModel, prj );
  prjLvi->setExpandable(true);
}

void ListWidget::refreshProjects()
{
  _subvn->refreshProjects();
}

void ListWidget::installChildEventFilter( const QObject* filter )
{
  _subvn->installEventFilter(filter);
}

void ListWidget::showProject()
{
  _stackFiles->setEnabled(false);
}

void ListWidget::showRepositoryFiles()
{
  _stackFiles->setEnabled(true);

  if( _stackFiles->visibleWidget() != _filesRp )
  {
    _stackFiles->raiseWidget( sRepository );
  }
}

void ListWidget::showWorkingCopyFiles()
{
  _stackFiles->setEnabled(true);

  if( _stackFiles->visibleWidget() != _filesWc )
  {
    _stackFiles->raiseWidget( sWorkingCopy );
  }
}

void ListWidget::clearWorkingCopyFiles()
{
  _filesWc->clear();
}

void ListWidget::showEmptyFiles()
{
  _stackFiles->raiseWidget( sEmpty );
}

void ListWidget::customEvent( QCustomEvent* ce )
{
#if 0
  switch( ce->type() )
  {
  default:
    {
      printf( "ListWidget: unknown custom event type %d!\n", ce->type() );
    }
  }
#endif
}

ScModel* ListWidget::getModel() const
{
  return _model;
}

void ListWidget::reload()
{
  // hmmm
  if( _stackFiles->visibleWidget() == _filesWc )
  {
    sc::String path = _wcModel->getCurrentPath();
    if( path.isEmpty() )
    {
      _subvn->reload();
    }
    else
    {
      _filesWc->reload();
    }
  }
  else if( _stackFiles->visibleWidget() == _filesRp )
  {
    _filesRp->reload();
  }
}

ProjectFoldersWidget* ListWidget::getProjectFoldersWidget()
{
  return _subvn;
}
