/* ====================================================================
 * Copyright (c) 2003-2006, Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "config.h"
#include "SingleTextWidget.h"
#include "sublib/TextModel.h"
#include "sublib/NullTextModel.h"
#include "util/String.h"

// sc gui
#include "TextWidget.h"
#include "TextLineNrWidget.h"

// qt
#include <qgrid.h>
#include <qhbox.h>
#include <qvbox.h>
#include <qlayout.h>
#include <qscrollbar.h>
#include <qsplitter.h>
#include <qmultilineedit.h>
#include <qpushbutton.h>
#include <qlineedit.h>
#include <qlabel.h>
#include <qtooltip.h>

// sys
#include <stdio.h>
#include <algorithm>
#include <assert.h>



static NullTextModel NullText;

SingleTextWidget::SingleTextWidget( QWidget *parent, const char *name )
: super( parent, name )
{
  // base class layout
  QGridLayout* gl = (QGridLayout*)layout();
  {
    QWidget*     iw  = new QWidget(this);
    QGridLayout* mgl = new QGridLayout( iw, 3, 1 );
    mgl->setMargin( 2 );
    mgl->setSpacing( 2 );
    gl->addWidget( iw, 0, 0 );
    {
      // first row
      _label = new QLineEdit(iw);
      _label->setText( _q("empty...") );
      _label->setFrameStyle( QFrame::Panel | QFrame::Sunken );
      _label->setLineWidth( 1 );
      _label->setReadOnly(true);
      _label->setMargin(1);
      _label->setBackgroundMode( Qt::PaletteButton );
      _label->setSizePolicy( QSizePolicy(QSizePolicy::Expanding,QSizePolicy::Fixed) );
      mgl->addWidget( _label, 0, 0 );

      // second
      QFrame* f = new QFrame(iw);
      f->setFrameStyle( QFrame::HLine | QFrame::Sunken );
      f->setLineWidth(1);
      mgl->addWidget( f, 1, 0 );

      // third
      QHBox* mltb = new QHBox(iw);
      mltb->setBackgroundMode( Qt::NoBackground );
      mltb->setSizePolicy( QSizePolicy(QSizePolicy::Expanding,QSizePolicy::Expanding) );
      mltb->setMargin(0);
      mltb->setSpacing(0);
      mgl->addWidget( mltb, 2, 0 );
      {
        _line = new TextLineNrWidget(mltb);
        _text = new TextWidget(mltb);
      }
    }
  }
  connect( _text, SIGNAL(xChanged(int)), SLOT(xChange(int)) );
  connect( _text, SIGNAL(yChanged(int)), SLOT(yChange(int)) );
  connect( _text, SIGNAL(sizeChanged()), SLOT(sizeChanged()) );
}

SingleTextWidget::~SingleTextWidget()
{
}

void SingleTextWidget::setModel( TextModel* model )
{
  _text->setModel( model, &NullText );
  _line->setModel( model );
  _label->setText( model->getSourceName().getStr() );
  QToolTip::add( _label, model->getSourceName().getStr() );
}

void SingleTextWidget::setModel( DiffInfoModel* info )
{
  assert(false);
}

void SingleTextWidget::setLabel( const sc::String& l )
{
  if( l.getCharCnt() > 0 )
  {
    _label->setText( (const char*)l );
  }
}

void SingleTextWidget::enableSelection( bool enable )
{
  _text->enableSelection(enable);
}

void SingleTextWidget::setEditable( bool editable )
{
  _text->setEditable(editable);
}

void SingleTextWidget::vsbChange(int y)
{
  _text->setScrollPosY(y);
  _line->setScrollPosY(y);
}

void SingleTextWidget::hsbChange(int x)
{
  _text->setScrollPosX(x);
}

void SingleTextWidget::xChange(int x)
{
  if( _hsb->value() == x )
  {
    return;
  }

  _hsb->setValue(x);
}

void SingleTextWidget::yChange(int y)
{
  if( _vsb->value() == y )
  {
    return;
  }

  _vsb->setValue(y);
}

void SingleTextWidget::sizeChanged()
{
  _line->update();
}

QSize SingleTextWidget::getScrollSizeHint()
{
  return _text->sizeHint();
}

QSize SingleTextWidget::getScrollSizeCurr()
{
  return _text->size();
}

TextWidget* SingleTextWidget::getText() const
{
  return _text;
}

void SingleTextWidget::jumpToLine( int line )
{
  int ypos = _text->calcLineY(line,true);

  if( _vsb->value() == ypos )
  {
    return;
  }

  _vsb->setValue(ypos);
}

void SingleTextWidget::jumpToBlock( int block )
{
  BlockInfo bi = _text->getModel()->getBlockInfo( block );

  if( bi.isEmpty() )
  {
    return;
  }

  jumpToLine( bi.getStart() );
}

void SingleTextWidget::update()
{
  _text->update();
  _line->update();
  super::update();
}

void SingleTextWidget::setAcceptDrops(bool on)
{
  _text->setAcceptDrops(true);
}
