/* swfc.c
   Compiles swf code (.sc) files into .swf files.

   Part of the swftools package.

   Copyright (c) 2001 Matthias Kramm <kramm@quiss.org>
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <memory.h>
#include <errno.h>
#include <math.h>
#include "../config.h"
#include "../lib/rfxswf.h"
#include "../lib/drawer.h"
#include "../lib/log.h"
#include "../lib/args.h"
#include "../lib/q.h"
#include "../lib/mp3.h"
#include "../lib/wav.h"
#include "parser.h"
#include "../lib/png.h"

//#define DEBUG

static char * filename = 0;
static char * outputname = "output.swf";
static int verbose = 2;
static int optimize = 0;
static int override_outputname = 0;
static int do_cgi = 0;

static struct options_t options[] = {
{"h", "help"},
{"V", "version"},
{"C", "cgi"},
{"v", "verbose"},
{"o", "output"},
{0,0}
};
    
int args_callback_option(char*name,char*val)
{
    if(!strcmp(name, "V")) {
        printf("swfc - part of %s %s\n", PACKAGE, VERSION);
        exit(0);
    }
    else if(!strcmp(name, "o")) {
	outputname = val;
	override_outputname = 1;
	return 1;
    }
    else if(!strcmp(name, "O")) {
	optimize = 1;
	return 0;
    }
    else if(!strcmp(name, "C")) {
	do_cgi = 1;
	return 0;
    }
    else if(!strcmp(name, "v")) {
	verbose ++;
	return 0;
    }
    else {
        printf("Unknown option: -%s\n", name);
	exit(1);
    }
    return 0;
}
int args_callback_longoption(char*name,char*val)
{
    return args_long2shortoption(options, name, val);
}
void args_callback_usage(char *name)
{
    printf("\n");
    printf("Usage: %s [-o file.swf] file.sc\n", name);
    printf("\n");
    printf("-h , --help                    Print short help message and exit\n");
    printf("-V , --version                 Print version info and exit\n");
    printf("-C , --cgi                     Output to stdout (for use in CGI environments)\n");
    printf("-v , --verbose                 Increase verbosity. \n");
    printf("-o , --output <filename>       Set output file to <filename>.\n");
    printf("\n");
}
int args_callback_command(char*name,char*val)
{
    if(filename) {
        fprintf(stderr, "Only one file allowed. You supplied at least two. (%s and %s)\n",
                 filename, name);
    }
    filename = name;
    return 0;
}

static struct token_t* file;

static int pos;
static char*text;
static int textlen;
static int type;
static int line;
static int column;

static void syntaxerror(char*format, ...)
{
    char buf[1024];
    va_list arglist;
    va_start(arglist, format);
    vsprintf(buf, format, arglist);
    va_end(arglist);
    fprintf(stderr, "\"%s\", line %d column %d: error- %s\n", filename, line, column, buf);
    exit(1);
}

static void warning(char*format, ...)
{
    char buf[1024];
    va_list arglist;
    va_start(arglist, format);
    vsprintf(buf, format, arglist);
    va_end(arglist);
    fprintf(stderr, "\"%s\", line %d column %d: warning- %s\n", filename, line, column, buf);
}

static void readToken()
{
    type = file[pos].type;
    if(type == END) {
	syntaxerror("unexpected end of file");
    }
    text = file[pos].text;
    textlen = strlen(text);
    line = file[pos].line;
    column = file[pos].column;
    pos++;
    //printf("---> %d(%s) %s\n", type, type_names[type], text);
}

static void pushBack()
{
    int p;
    if(!pos) syntaxerror("internal error 3");
    pos--;
    p = pos;
    if(p) p--;
    text = file[p].text;
    textlen = strlen(text);
    type = file[p].type;
    line = file[p].line;
    column = file[p].column;
}

static int noMoreTokens()
{
    if(file[pos].type == END)
	return 1;
    return 0;
}

// ------------------------------ swf routines ----------------------------
struct _character;
static struct level
{
   int type; //0=swf, 1=sprite, 2=clip, 3=button

   /* for swf (0): */
   SWF*swf;
   char*filename; 

   /* for sprites (1): */
   TAG*tag;
   U16 id;
   char*name;
   U16 olddepth;
   int oldframe;
   dictionary_t oldinstances;
   SRECT oldrect;
   TAG* cut;

} stack[256];
static int stackpos = 0;

static dictionary_t characters;
static dictionary_t images;
static dictionary_t textures;
static dictionary_t outlines;
static dictionary_t gradients;
static dictionary_t filters;
static char idmap[65536];
static TAG*tag = 0; //current tag

static int id; //current character id
static int currentframe; //current frame in current level
static SRECT currentrect; //current bounding box in current level
static U16 currentdepth;
static dictionary_t instances;
static dictionary_t fonts;
static dictionary_t sounds;

typedef struct _parameters {
    int x,y; 
    float scalex, scaley; 
    CXFORM cxform;
    float rotate;
    float shear;
    SPOINT pivot;
    SPOINT pin;
    U8 blendmode; //not interpolated
    FILTER*filter;
} parameters_t;

typedef struct _character {
    TAG*definingTag;
    U16 id;
    SRECT size;
} character_t;

typedef struct _instance {
    character_t*character;
    U16 depth;
    parameters_t parameters;
    TAG* lastTag; //last tag which set the object
    U16 lastFrame; //frame lastTag is in
} instance_t;

typedef struct _outline {
    SHAPE* shape;
    SRECT bbox;
} outline_t;

typedef struct _gradient {
    GRADIENT gradient;
    char radial;
    int rotate;
} gradient_t;

typedef struct _filter {
    FILTER filter;
} filter_t;

typedef struct _texture {
    FILLSTYLE fs;
} texture_t;

static void character_init(character_t*c)
{
    memset(c, 0, sizeof(character_t));
}
static character_t* character_new()
{
    character_t*c;
    c = (character_t*)malloc(sizeof(character_t));
    character_init(c);
    return c;
}
static void instance_init(instance_t*i)
{
    memset(i, 0, sizeof(instance_t));
}
static instance_t* instance_new()
{
    instance_t*c;
    c = (instance_t*)malloc(sizeof(instance_t));
    instance_init(c);
    return c;
}

static void incrementid()
{
    while(idmap[++id]) {
	if(id==65535)
	    syntaxerror("Out of character ids.");
    }
    idmap[id] = 1;
}

static void s_addcharacter(char*name, U16 id, TAG*ctag, SRECT r)
{
    character_t* c = character_new();
    
    c->definingTag = ctag;
    c->id = id;
    c->size = r;
    if(dictionary_lookup(&characters, name))
	syntaxerror("character %s defined twice", name);
    dictionary_put2(&characters, name, c);

    tag = swf_InsertTag(tag, ST_NAMECHARACTER);
    swf_SetU16(tag, id);
    swf_SetString(tag, name);
    tag = swf_InsertTag(tag, ST_EXPORTASSETS);
    swf_SetU16(tag, 1);
    swf_SetU16(tag, id);
    swf_SetString(tag, name);
}
static void s_addimage(char*name, U16 id, TAG*ctag, SRECT r)
{
    character_t* c = character_new();
    c->definingTag = ctag;
    c->id = id;
    c->size = r;

    if(dictionary_lookup(&images, name))
	syntaxerror("image %s defined twice", name);
    dictionary_put2(&images, name, c);
}
static instance_t* s_addinstance(char*name, character_t*c, U16 depth)
{
    instance_t* i = instance_new();
    i->character = c;
    i->depth = depth;
    //swf_GetMatrix(0, &i->matrix);
    if(dictionary_lookup(&instances, name))
	syntaxerror("object %s defined twice", name);
    dictionary_put2(&instances, name, i);
    return i;
}

static void parameters_clear(parameters_t*p)
{
    p->x = 0; p->y = 0; 
    p->scalex = 1.0; p->scaley = 1.0;
    p->pin.x = 0;  //1??
    p->pin.y = 0;
    p->pivot.x = 0; p->pivot.y = 0;
    p->rotate = 0; 
    p->shear = 0; 
    p->blendmode = 0;
    p->filter = 0;
    swf_GetCXForm(0, &p->cxform, 1);
}

static void makeMatrix(MATRIX*m, parameters_t*p)
{
    SPOINT h;
    float sx,r1,r0,sy;

    /*	      /sx r1\ /x\ 
     *	      \r0 sy/ \y/
     */

    sx =  p->scalex*cos(p->rotate/360*2*3.14159265358979);
    r1 = -p->scalex*sin(p->rotate/360*2*3.14159265358979)+sx*p->shear;
    r0 =  p->scaley*sin(p->rotate/360*2*3.14159265358979);
    sy =  p->scaley*cos(p->rotate/360*2*3.14159265358979)+r0*p->shear;

    m->sx = (int)(sx*65536+0.5);
    m->r1 = (int)(r1*65536+0.5);
    m->r0 = (int)(r0*65536+0.5);
    m->sy = (int)(sy*65536+0.5);

    m->tx = m->ty = 0;
   
    h = swf_TurnPoint(p->pin, m);
    m->tx = p->x - h.x;
    m->ty = p->y - h.y;
}

static MATRIX s_instancepos(SRECT rect, parameters_t*p)
{
    MATRIX m;
    SRECT r;
    makeMatrix(&m, p);
    r = swf_TurnRect(rect, &m);
    if(currentrect.xmin == 0 && currentrect.ymin == 0 && 
       currentrect.xmax == 0 && currentrect.ymax == 0)
	currentrect = r;
    else
	swf_ExpandRect2(&currentrect, &r);
    return m;
}

void s_swf(char*name, SRECT r, int version, int fps, int compress, RGBA background)
{
    SWF*swf = (SWF*)malloc(sizeof(SWF));

    if(stackpos)
	syntaxerror(".swf blocks can't be nested");

    memset(swf, 0, sizeof(swf));
    swf->fileVersion = version;
    swf->movieSize = r;
    swf->frameRate = fps;
    swf->firstTag = tag = swf_InsertTag(0, ST_SETBACKGROUNDCOLOR);
    swf->compressed = compress;
    swf_SetRGB(tag,&background);

    if(stackpos==sizeof(stack)/sizeof(stack[0]))
	syntaxerror("too many levels of recursion");
    
    dictionary_init(&characters);
    dictionary_init(&images);
    dictionary_init(&textures);
    dictionary_init(&outlines);
    dictionary_init(&gradients);
    dictionary_init(&filters);
    dictionary_init(&instances);
    dictionary_init(&fonts);
    dictionary_init(&sounds);

    memset(&stack[stackpos], 0, sizeof(stack[0]));
    stack[stackpos].type = 0;
    stack[stackpos].filename = strdup(name);
    stack[stackpos].swf = swf;
    stack[stackpos].oldframe = -1;
    stackpos++;
    id = 0;

    currentframe = 0;
    memset(&currentrect, 0, sizeof(currentrect));
    currentdepth = 1;
    
    memset(idmap, 0, sizeof(idmap));
    incrementid();
}

void s_sprite(char*name)
{
    tag = swf_InsertTag(tag, ST_DEFINESPRITE);
    swf_SetU16(tag, id); //id
    swf_SetU16(tag, 0); //frames

    memset(&stack[stackpos], 0, sizeof(stack[0]));
    stack[stackpos].type = 1;
    stack[stackpos].oldframe = currentframe;
    stack[stackpos].olddepth = currentdepth;
    stack[stackpos].oldrect = currentrect;
    stack[stackpos].oldinstances = instances;
    stack[stackpos].tag = tag;
    stack[stackpos].id = id;
    stack[stackpos].name = strdup(name);
   
    /* FIXME: those four fields should be bundled together */
    dictionary_init(&instances);
    currentframe = 0;
    currentdepth = 1;
    memset(&currentrect, 0, sizeof(currentrect));

    stackpos++;
    incrementid();
}

typedef struct _buttonrecord
{
    U16 id;
    MATRIX matrix;
    CXFORM cxform;
    char set;
} buttonrecord_t;

typedef struct _button
{
    int endofshapes;
    int nr_actions;
    buttonrecord_t records[4];
} button_t;

static button_t mybutton;

void s_button(char*name)
{
    tag = swf_InsertTag(tag, ST_DEFINEBUTTON2);
    swf_SetU16(tag, id); //id
    swf_ButtonSetFlags(tag, 0); //menu=no

    memset(&mybutton, 0, sizeof(mybutton));

    memset(&stack[stackpos], 0, sizeof(stack[0]));
    stack[stackpos].type = 3;
    stack[stackpos].tag = tag;
    stack[stackpos].id = id;
    stack[stackpos].name = strdup(name);
    stack[stackpos].oldrect = currentrect;
    memset(&currentrect, 0, sizeof(currentrect));

    stackpos++;
    incrementid();
}
void s_buttonput(char*character, char*as, parameters_t p)
{
    character_t* c = dictionary_lookup(&characters, character);
    MATRIX m;
    int flags = 0;
    char*o = as,*s = as;
    buttonrecord_t r;
    if(!stackpos || (stack[stackpos-1].type != 3))  {
	syntaxerror(".show may only appear in .button");
    }
    if(!c) {
	syntaxerror("character %s not known (in .shape %s)", character, character);
    }
    if(mybutton.endofshapes) {
	syntaxerror("a .do may not precede a .show", character, character);
    }
   
    m = s_instancepos(c->size, &p);

    r.id = c->id;
    r.matrix = m;
    r.cxform = p.cxform;
    r.set = 1;

    while(1) {
	if(*s==',' || *s==0) {
	    if(!strncmp(o,"idle",s-o)) {mybutton.records[0]=r;o=s+1;}
	    else if(!strncmp(o,"shape",s-o)) {mybutton.records[0]=r;o=s+1;}
	    else if(!strncmp(o,"hover",s-o)) {mybutton.records[1]=r;o=s+1;}
	    else if(!strncmp(o,"pressed",s-o)) {mybutton.records[2]=r;o=s+1;}
	    else if(!strncmp(o,"area",s-o)) {mybutton.records[3]=r;o=s+1;}
	    else syntaxerror("unknown \"as\" argument: \"%s\"", strdup_n(o,s-o));
	}
	if(!*s)
	    break;
	s++;
    }
}
static void setbuttonrecords(TAG*tag)
{
    int flags[] = {BS_UP,BS_OVER,BS_DOWN,BS_HIT};
    if(!mybutton.endofshapes) {
	int t;
		
	if(!mybutton.records[3].set) {
	    memcpy(&mybutton.records[3], &mybutton.records[0], sizeof(buttonrecord_t));
	}

	for(t=0;t<4;t++) {
	    if(mybutton.records[t].set) {
		swf_ButtonSetRecord(tag,flags[t],mybutton.records[t].id,1,&mybutton.records[t].matrix,&mybutton.records[t].cxform);
	    }
	}
	swf_SetU8(tag,0); // end of button records
	mybutton.endofshapes = 1;
    }
}

void s_buttonaction(int flags, char*action)
{
    ActionTAG* a = 0;
    if(flags==0) {
	return;
    }
    setbuttonrecords(stack[stackpos-1].tag);
    
    a = swf_ActionCompile(text, stack[0].swf->fileVersion);
    if(!a) {
	syntaxerror("Couldn't compile ActionScript");
    }

    swf_ButtonSetCondition(stack[stackpos-1].tag, flags);
    swf_ActionSet(stack[stackpos-1].tag, a);
    mybutton.nr_actions++;

    swf_ActionFree(a);
}

static void setactionend(TAG*tag)
{
    if(!mybutton.nr_actions) {
	/* no actions means we didn't have an actionoffset,
	   which means we can't signal the end of the
	   buttonaction records, so, *sigh*, we have
	   to insert a dummy record */
	swf_SetU16(tag, 0); //offset
	swf_SetU16(tag, 0); //condition
	swf_SetU8(tag, 0); //action
    }
}

static void s_endButton()
{
    SRECT r;
    setbuttonrecords(stack[stackpos-1].tag);
    setactionend(stack[stackpos-1].tag);
    stackpos--;
      
    swf_ButtonPostProcess(stack[stackpos].tag, mybutton.nr_actions);

    r = currentrect;

    tag = stack[stackpos].tag;
    currentrect = stack[stackpos].oldrect;

    s_addcharacter(stack[stackpos].name, stack[stackpos].id, stack[stackpos].tag, r);
    free(stack[stackpos].name);
}

TAG* removeFromTo(TAG*from, TAG*to)
{
    TAG*save = from->prev;
    while(from!=to) {
	TAG*next = from->next;
	swf_DeleteTag(from);
	from = next;
    }
    save->next = 0;
    return save;
}

static void s_endSprite()
{
    SRECT r = currentrect;
    
    if(stack[stackpos].cut)
	tag = removeFromTo(stack[stackpos].cut, tag);

    stackpos--;
   
    /* TODO: before clearing, prepend "<spritename>." to names and
             copy into old instances dict */
    dictionary_clear(&instances);

    currentframe = stack[stackpos].oldframe;
    currentrect = stack[stackpos].oldrect;
    currentdepth = stack[stackpos].olddepth;
    instances = stack[stackpos].oldinstances;

    tag = swf_InsertTag(tag, ST_SHOWFRAME);
    tag = swf_InsertTag(tag, ST_END);

    tag = stack[stackpos].tag;
    swf_FoldSprite(tag);
    if(tag->next != 0)
	syntaxerror("internal error(7)");

    s_addcharacter(stack[stackpos].name, stack[stackpos].id, stack[stackpos].tag, r);
    free(stack[stackpos].name);
}

static void s_endSWF()
{
    int fi;
    SWF* swf;
    char*filename;
    
    if(stack[stackpos].cut)
	tag = removeFromTo(stack[stackpos].cut, tag);

    stackpos--;
   
    swf = stack[stackpos].swf;
    filename = stack[stackpos].filename;
  
    //if(tag->prev && tag->prev->id != ST_SHOWFRAME)
    //    tag = swf_InsertTag(tag, ST_SHOWFRAME);
    tag = swf_InsertTag(tag, ST_SHOWFRAME);

    tag = swf_InsertTag(tag, ST_END);

    swf_OptimizeTagOrder(swf);
   
    if(optimize) {
	swf_Optimize(swf);
    }
    
    if(!(swf->movieSize.xmax-swf->movieSize.xmin) || !(swf->movieSize.ymax-swf->movieSize.ymin)) {
	swf->movieSize = currentrect; /* "autocrop" */
    }
    
    if(!(swf->movieSize.xmax-swf->movieSize.xmin) || !(swf->movieSize.ymax-swf->movieSize.ymin)) {
	swf->movieSize.xmax += 20; /* 1 by 1 pixels */
	swf->movieSize.ymax += 20;
	warning("Empty bounding box for movie");
    }
    
    if(do_cgi || !strcmp(filename, "-"))
	fi = fileno(stdout);
    else
	fi = open(filename, O_WRONLY|O_CREAT|O_TRUNC|O_BINARY, 0644);
    if(fi<0) {
	syntaxerror("couldn't create output file %s", filename);
    }
    if(do_cgi)
	{if(swf_WriteCGI(swf)<0) syntaxerror("WriteCGI() failed.\n");}
    else if(swf->compressed) 
	{if(swf_WriteSWC(fi, swf)<0) syntaxerror("WriteSWC() failed.\n");}
    else
	{if(swf_WriteSWF(fi, swf)<0) syntaxerror("WriteSWF() failed.\n");}

    close(fi);
    
    dictionary_clear(&instances);
    dictionary_clear(&characters);
    dictionary_clear(&images);
    dictionary_clear(&textures);
    dictionary_clear(&outlines);
    dictionary_clear(&gradients); // mem leak
    dictionary_clear(&filters);
    dictionary_clear(&fonts);
    dictionary_clear(&sounds);

    swf_FreeTags(swf);
    free(swf);
    free(filename);
}

void s_close()
{
    if(stackpos) {
	if(stack[stackpos-1].type == 0)
	    syntaxerror("End of file encountered in .flash block");
	if(stack[stackpos-1].type == 1)
	    syntaxerror("End of file encountered in .sprite block");
	if(stack[stackpos-1].type == 2)
	    syntaxerror("End of file encountered in .clip block");
    }
}

int s_getframe()
{
    return currentframe+1;
}

void s_frame(int nr, int cut, char*name, char anchor)
{
    int t;
    TAG*now = tag;

    if(nr<1) 
	syntaxerror("Illegal frame number");
    nr--; // internally, frame 1 is frame 0

    for(t=currentframe;t<nr;t++) {
	tag = swf_InsertTag(tag, ST_SHOWFRAME);
	if(t==nr-1 && name && *name) {
	    tag = swf_InsertTag(tag, ST_FRAMELABEL);
	    swf_SetString(tag, name);
	    if(anchor)
		swf_SetU8(tag, 1); //make this an anchor
	}
    }
    if(nr == 0 && currentframe == 0 && name && *name) {
        tag = swf_InsertTag(tag, ST_FRAMELABEL);
        swf_SetString(tag, name);
	if(anchor)
	    swf_SetU8(tag, 1); //make this an anchor
    }

    if(cut) {
	if(now == tag) {
	    syntaxerror("Can't cut, frame empty");
	}
	stack[stackpos].cut = tag;
    }

    currentframe = nr;
}

int parseColor2(char*str, RGBA*color);

int addFillStyle(SHAPE*s, SRECT*r, char*name)
{
    RGBA color;
    character_t*image;
    gradient_t*gradient;
    texture_t*texture;
    if(name[0] == '#') {
	parseColor2(name, &color);
	return swf_ShapeAddSolidFillStyle(s, &color);
    } else if ((texture = dictionary_lookup(&textures, name))) {
	return swf_ShapeAddFillStyle2(s, &texture->fs);
    } else if((image = dictionary_lookup(&images, name))) {
	MATRIX m;
	swf_GetMatrix(0, &m);
	m.sx = 65536.0*20.0*(r->xmax - r->xmin)/image->size.xmax;
	m.sy = 65536.0*20.0*(r->ymax - r->ymin)/image->size.ymax;
	m.tx = r->xmin;
	m.ty = r->ymin;
	return swf_ShapeAddBitmapFillStyle(s, &m, image->id, 0);
    }  else if ((gradient = dictionary_lookup(&gradients, name))) {
	SRECT r2;
	MATRIX rot,m;
	double ccos,csin;
	swf_GetMatrix(0, &rot);
	ccos = cos(-gradient->rotate*2*3.14159265358979/360);
	csin = sin(-gradient->rotate*2*3.14159265358979/360);
	rot.sx =  ccos*65536;
	rot.r1 = -csin*65536;
	rot.r0 =  csin*65536;
	rot.sy =  ccos*65536;
	r2 = swf_TurnRect(*r, &rot);
	swf_GetMatrix(0, &m);
	m.sx =  (r2.xmax - r2.xmin)*2*ccos;
	m.r1 = -(r2.xmax - r2.xmin)*2*csin;
	m.r0 =  (r2.ymax - r2.ymin)*2*csin;
	m.sy =  (r2.ymax - r2.ymin)*2*ccos;
	m.tx = r->xmin + (r->xmax - r->xmin)/2;
	m.ty = r->ymin + (r->ymax - r->ymin)/2;
	return swf_ShapeAddGradientFillStyle(s, &m, &gradient->gradient, gradient->radial);
    }  else if (parseColor2(name, &color)) {
	return swf_ShapeAddSolidFillStyle(s, &color);
    } else {
	syntaxerror("not a color/fillstyle: %s", name);
	return 0;
    }
}
	
RGBA black={r:0,g:0,b:0,a:0};
void s_box(char*name, int width, int height, RGBA color, int linewidth, char*texture)
{
    SRECT r,r2;
    SHAPE* s;
    int ls1=0,fs1=0;
    r2.xmin = 0;
    r2.ymin = 0;
    r2.xmax = width;
    r2.ymax = height;
    tag = swf_InsertTag(tag, ST_DEFINESHAPE3);
    swf_ShapeNew(&s);
    if(linewidth) {
	linewidth = linewidth>=20?linewidth-20:0;
        ls1 = swf_ShapeAddLineStyle(s,linewidth,&color);
    }
    if(texture)
	fs1 = addFillStyle(s, &r2, texture);

    swf_SetU16(tag,id);
    r.xmin = r2.xmin-linewidth/2;
    r.ymin = r2.ymin-linewidth/2;
    r.xmax = r2.xmax+linewidth/2;
    r.ymax = r2.ymax+linewidth/2;
    swf_SetRect(tag,&r);
    swf_SetShapeHeader(tag,s);
    swf_ShapeSetAll(tag,s,0,0,ls1,fs1,0);
    swf_ShapeSetLine(tag,s,width,0);
    swf_ShapeSetLine(tag,s,0,height);
    swf_ShapeSetLine(tag,s,-width,0);
    swf_ShapeSetLine(tag,s,0,-height);
    swf_ShapeSetEnd(tag);
    swf_ShapeFree(s);
   
    s_addcharacter(name, id, tag, r);
    incrementid();
}

void s_filled(char*name, char*outlinename, RGBA color, int linewidth, char*texture)
{
    SRECT rect,r2;
    SHAPE* s;
    int ls1,fs1=0;
    outline_t* outline;
    outline = dictionary_lookup(&outlines, outlinename);
    if(!outline) {
	syntaxerror("outline %s not defined", outlinename);
    }
    r2 = outline->bbox;

    tag = swf_InsertTag(tag, ST_DEFINESHAPE3);
    swf_ShapeNew(&s);
    if(linewidth) {
	linewidth = linewidth>=20?linewidth-20:0;
        ls1 = swf_ShapeAddLineStyle(s,linewidth,&color);
    }
    if(texture)
	fs1 = addFillStyle(s, &r2, texture);
    
    swf_SetU16(tag,id);
    rect.xmin = r2.xmin-linewidth/2;
    rect.ymin = r2.ymin-linewidth/2;
    rect.xmax = r2.xmax+linewidth/2;
    rect.ymax = r2.ymax+linewidth/2;

    swf_SetRect(tag,&rect);
    swf_SetShapeStyles(tag, s);
    swf_ShapeCountBits(s,0,0);
    swf_RecodeShapeData(outline->shape->data, outline->shape->bitlen, outline->shape->bits.fill, outline->shape->bits.line, 
                        &s->data,             &s->bitlen,             s->bits.fill,              s->bits.line);
    swf_SetShapeBits(tag, s);
    swf_SetBlock(tag, s->data, (s->bitlen+7)/8);
    swf_ShapeFree(s);

    s_addcharacter(name, id, tag, rect);
    incrementid();
}

void s_circle(char*name, int r, RGBA color, int linewidth, char*texture)
{
    SRECT rect,r2;
    SHAPE* s;
    int ls1=0,fs1=0;
    r2.xmin = r2.ymin = 0;
    r2.xmax = 2*r;
    r2.ymax = 2*r;

    tag = swf_InsertTag(tag, ST_DEFINESHAPE3);
    swf_ShapeNew(&s);
    if(linewidth) {
	linewidth = linewidth>=20?linewidth-20:0;
        ls1 = swf_ShapeAddLineStyle(s,linewidth,&color);
    }
    if(texture)
	fs1 = addFillStyle(s, &r2, texture);
    swf_SetU16(tag,id);
    rect.xmin = r2.xmin-linewidth/2;
    rect.ymin = r2.ymin-linewidth/2;
    rect.xmax = r2.xmax+linewidth/2;
    rect.ymax = r2.ymax+linewidth/2;

    swf_SetRect(tag,&rect);
    swf_SetShapeHeader(tag,s);
    swf_ShapeSetAll(tag,s,0,0,ls1,fs1,0);
    swf_ShapeSetCircle(tag, s, r,r,r,r);
    swf_ShapeSetEnd(tag);
    swf_ShapeFree(s);
   
    s_addcharacter(name, id, tag, rect);
    incrementid();
}

void s_textshape(char*name, char*fontname, float size, char*_text)
{
    int g;
    U8*text = (U8*)_text;
    outline_t* outline;

    SWFFONT*font;
    font = dictionary_lookup(&fonts, fontname);
    if(!font)
	syntaxerror("font \"%s\" not known!", fontname);
    
    if(text[0] >= font->maxascii || font->ascii2glyph[text[0]]<0) {
	warning("no character 0x%02x (%c) in font \"%s\"", text[0], text[0], fontname);
	s_box(name, 0, 0, black, 20, 0);
	return;
    }
    g = font->ascii2glyph[text[0]];

    outline = malloc(sizeof(outline_t));
    memset(outline, 0, sizeof(outline_t));
    outline->shape = font->glyph[g].shape;
    outline->bbox = font->layout->bounds[g];
    
    {
	drawer_t draw;
	swf_Shape11DrawerInit(&draw, 0);
	swf_DrawText(&draw, font, (int)(size*100), _text);
	draw.finish(&draw);
	outline->shape = swf_ShapeDrawerToShape(&draw);
	outline->bbox = swf_ShapeDrawerGetBBox(&draw);
	draw.dealloc(&draw);
    }

    if(dictionary_lookup(&outlines, name))
	syntaxerror("outline %s defined twice", name);
    dictionary_put2(&outlines, name, outline);
}

void s_text(char*name, char*fontname, char*text, int size, RGBA color)
{
    SRECT r;
    MATRIX _m,*m=0;
    SWFFONT*font;
    font = dictionary_lookup(&fonts, fontname);
    if(!font)
	syntaxerror("font \"%s\" not known!", fontname);
    
    tag = swf_InsertTag(tag, ST_DEFINETEXT2);
    swf_SetU16(tag, id);
    if(!font->numchars) {
	s_box(name, 0, 0, black, 20, 0);
	return;
    }
    r = swf_SetDefineText(tag, font, &color, text, size);
   
    s_addcharacter(name, id, tag, r);
    incrementid();
}

void s_quicktime(char*name, char*url)
{
    SRECT r;
    MATRIX _m,*m=0;

    memset(&r, 0, sizeof(r));
    
    tag = swf_InsertTag(tag, ST_DEFINEMOVIE);
    swf_SetU16(tag, id);
    swf_SetString(tag, url);
    
    s_addcharacter(name, id, tag, r);
    incrementid();
}

void s_edittext(char*name, char*fontname, int size, int width, int height, char*text, RGBA*color, int maxlength, char*variable, int flags, int align)
{
    SWFFONT*font = 0;
    EditTextLayout layout;
    SRECT r;

    if(fontname && *fontname) {
	flags |= ET_USEOUTLINES;
	font = dictionary_lookup(&fonts, fontname);
	if(!font)
	    syntaxerror("font \"%s\" not known!", fontname);
    }
    tag = swf_InsertTag(tag, ST_DEFINEEDITTEXT);
    swf_SetU16(tag, id);
    layout.align = align;
    layout.leftmargin = 0;
    layout.rightmargin = 0;
    layout.indent = 0;
    layout.leading = 0;
    r.xmin = 0;
    r.ymin = 0;
    r.xmax = width;
    r.ymax = height;
    
    swf_SetEditText(tag, flags, r, text, color, maxlength, font?font->id:0, size, &layout, variable);

    s_addcharacter(name, id, tag, r);
    incrementid();
}

/* type: either "jpeg" or "png"
 */
void s_image(char*name, char*type, char*filename, int quality)
{
    /* an image is actually two folded: 1st bitmap, 2nd character.
       Both of them can be used separately */
    
    /* step 1: the bitmap */
    SRECT r;
    int imageID = id;
    int width, height;
    if(!strcmp(type,"jpeg")) {
#ifndef HAVE_JPEGLIB
	warning("no jpeg support compiled in");
	s_box(name, 0, 0, black, 20, 0);
	return;
#else
	tag = swf_InsertTag(tag, ST_DEFINEBITSJPEG2);
	swf_SetU16(tag, imageID);

	if(swf_SetJPEGBits(tag, filename, quality) < 0) {
	    syntaxerror("Image \"%s\" not found, or contains errors", filename);
	}

	swf_GetJPEGSize(filename, &width, &height);

	r.xmin = 0;
	r.ymin = 0;
	r.xmax = width*20;
	r.ymax = height*20;

	s_addimage(name, id, tag, r);
	incrementid();
#endif
    } else if(!strcmp(type,"png")) {
	RGBA*data = 0;
	swf_SetU16(tag, imageID);

	getPNG(filename, &width, &height, (unsigned char**)&data);

	if(!data) {
	    syntaxerror("Image \"%s\" not found, or contains errors", filename);
	}

	/*tag = swf_AddImage(tag, imageID, data, width, height, quality)*/
	tag = swf_InsertTag(tag, ST_DEFINEBITSLOSSLESS);
	swf_SetU16(tag, imageID);
	swf_SetLosslessImage(tag, data, width, height);

	r.xmin = 0;
	r.ymin = 0;
	r.xmax = width*20;
	r.ymax = height*20;
	s_addimage(name, id, tag, r);
	incrementid();
    } else {
	warning("image type \"%s\" not supported yet!", type);
	s_box(name, 0, 0, black, 20, 0);
	return;
    }

    /* step 2: the character */
    tag = swf_InsertTag(tag, ST_DEFINESHAPE); // todo: should be defineshape2/3 once images can be transparent.(?)
    swf_SetU16(tag, id);
    swf_ShapeSetBitmapRect(tag, imageID, width, height);

    s_addcharacter(name, id, tag, r);
    incrementid();
}

void s_getBitmapSize(char*name, int*width, int*height)
{
    character_t* image = dictionary_lookup(&images, name);
    gradient_t* gradient = dictionary_lookup(&gradients,name);
    if(image) {
	*width = image->size.xmax;
	*height = image->size.ymax;
	return;
    }
    if(gradient) {
	/* internal SWF gradient size */
	if(gradient->radial) {
	    *width = 16384;
	    *height = 16384;
	} else {
	    *width = 32768;
	    *height = 32768;
	}
	return;
    }
    syntaxerror("No such bitmap/gradient: %s", name);
}

void s_texture(char*name, char*object, int x, int y, float scalex, float scaley, float rotate, float shear)
{
    gradient_t* gradient = dictionary_lookup(&gradients, object);
    character_t* bitmap = dictionary_lookup(&images, object);
    texture_t* texture = (texture_t*)rfx_calloc(sizeof(texture_t));
    parameters_t p;
    FILLSTYLE*fs = &texture->fs;

    memset(&p, 0, sizeof(parameters_t));

    if(bitmap) {
	fs->type = FILL_TILED;
	fs->id_bitmap = bitmap->id;
    } else if(gradient) {
	fs->type = gradient->radial?FILL_RADIAL:FILL_LINEAR;
	fs->gradient = gradient->gradient;
    }
    p.x = x;p.y = y;p.scalex = scalex;p.scaley = scaley;p.rotate=rotate;p.shear=shear;
    makeMatrix(&fs->m, &p);
    if(gradient && !gradient->radial) {
	MATRIX m = fs->m;
	SPOINT p1,p2;
	m.tx = 0;
	m.ty = 0;
	p1.x = 16384;
	p1.y = 16384;
	p2 = swf_TurnPoint(p1, &m);
	fs->m.tx += p2.x;
	fs->m.ty += p2.y;
    }
    if(bitmap) {
	fs->m.sx *= 20;
	fs->m.sy *= 20;
    }


    if(dictionary_lookup(&textures, name))
	syntaxerror("texture %s defined twice", name);
    dictionary_put2(&textures, name, texture);
}

void dumpSWF(SWF*swf)
{
    TAG* tag = swf->firstTag;
    printf("vvvvvvvvvvvvvvvvvvvvv\n");
    while(tag) {
	printf("%8d %s\n", tag->len, swf_TagGetName(tag));
	tag = tag->next;
    }
    printf("^^^^^^^^^^^^^^^^^^^^^\n");
}
    
void s_font(char*name, char*filename)
{
    SWFFONT* font;
    font = swf_LoadFont(filename);
   
    if(font == 0) {
	warning("Couldn't open font file \"%s\"", filename);
	font = (SWFFONT*)malloc(sizeof(SWFFONT));
	memset(font, 0, sizeof(SWFFONT));
	dictionary_put2(&fonts, name, font);
	return;
    }

    if(0)
    {
	/* fix the layout. Only needed for old fonts */
	int t;
	for(t=0;t<font->numchars;t++) {
	    font->glyph[t].advance = 0;
	}
	font->layout = 0;
	swf_FontCreateLayout(font);
    }
    /* just in case this thing is used in .edittext later on */
    swf_FontPrepareForEditText(font);

    font->id = id;
    tag = swf_InsertTag(tag, ST_DEFINEFONT2);
    swf_FontSetDefine2(tag, font);
    tag = swf_InsertTag(tag, ST_EXPORTASSETS);
    swf_SetU16(tag, 1);
    swf_SetU16(tag, id);
    swf_SetString(tag, name);
    incrementid();

    if(dictionary_lookup(&fonts, name))
	syntaxerror("font %s defined twice", name);
    dictionary_put2(&fonts, name, font);
}



typedef struct _sound_t
{
    U16 id;
    TAG*tag;
} sound_t;

void s_sound(char*name, char*filename)
{
    struct WAV wav, wav2;
    struct MP3 mp3;
    sound_t* sound;
    U16*samples = NULL;
    unsigned numsamples;
    unsigned blocksize = 1152;
    int is_mp3 = 0;

    if(wav_read(&wav, filename)) {
        int t;
	wav_convert2mono(&wav, &wav2, 44100);
	samples = (U16*)wav2.data;
	numsamples = wav2.size/2;
	free(wav.data);
#ifdef WORDS_BIGENDIAN
	/* swap bytes */
	for(t=0;t<numsamples;t++) {
	    samples[t] = (samples[t]>>8)&0xff | (samples[t]<<8)&0xff00;
	}
#endif
    } else if(mp3_read(&mp3, filename)) {
        fprintf(stderr, "\"%s\" seems to work as a MP3 file...\n", filename);
        blocksize = 1;
        is_mp3 = 1;
    }
    else
    {
	warning("Couldn't read WAV/MP3 file \"%s\"", filename);
	samples = 0;
	numsamples = 0;
    }
    
    if(numsamples%blocksize != 0)
    {
	// apply padding, so that block is a multiple of blocksize
	int numblocks = (numsamples+blocksize-1)/blocksize;
	int numsamples2;
	U16* samples2;
	numsamples2 = numblocks * blocksize;
	samples2 = malloc(sizeof(U16)*numsamples2);
	memcpy(samples2, samples, numsamples*sizeof(U16));
	memset(&samples2[numsamples], 0, sizeof(U16)*(numsamples2 - numsamples));
	numsamples = numsamples2;
	samples = samples2;
    }

    tag = swf_InsertTag(tag, ST_DEFINESOUND);
    swf_SetU16(tag, id); //id
    if(is_mp3)
    {
        swf_SetSoundDefineMP3(
                tag, mp3.data, mp3.size,
                mp3.SampRate,
                mp3.Channels,
                mp3.NumFrames);
	mp3_clear(&mp3);
    }
    else
    {
        swf_SetSoundDefine(tag, samples, numsamples);
    }
    
    tag = swf_InsertTag(tag, ST_NAMECHARACTER);
    swf_SetU16(tag, id);
    swf_SetString(tag, name);
    tag = swf_InsertTag(tag, ST_EXPORTASSETS);
    swf_SetU16(tag, 1);
    swf_SetU16(tag, id);
    swf_SetString(tag, name);

    sound = (sound_t*)malloc(sizeof(sound_t)); /* mem leak */
    sound->tag = tag;
    sound->id = id;

    if(dictionary_lookup(&sounds, name))
	syntaxerror("sound %s defined twice", name);
    dictionary_put2(&sounds, name, sound);
    
    incrementid();

    if(samples)
	free(samples);
}

static char* gradient_getToken(const char**p)
{
    const char*start;
    char*result;
    while(**p && strchr(" \t\n\r", **p)) {
	(*p)++;
    } 
    start = *p;
    while(**p && !strchr(" \t\n\r", **p)) {
	(*p)++;
    }
    result = malloc((*p)-start+1);
    memcpy(result,start,(*p)-start+1);
    result[(*p)-start] = 0;
    return result;
}

float parsePercent(char*str);
RGBA parseColor(char*str);

GRADIENT parseGradient(const char*str)
{
    GRADIENT gradient;
    int lastpos = -1;
    const char* p = str;
    memset(&gradient, 0, sizeof(GRADIENT));
    gradient.ratios = rfx_calloc(16*sizeof(U8));
    gradient.rgba = rfx_calloc(16*sizeof(RGBA));
    while(*p) {
	char*posstr,*colorstr;
	int pos;
	RGBA color;
	posstr = gradient_getToken(&p);
	if(!*posstr)
	    break;
	pos = (int)(parsePercent(posstr)*255.0);
	if(pos == lastpos)
	    pos++;
	if(!*p) syntaxerror("Error in shape data: Color expected after %s", posstr);
	colorstr = gradient_getToken(&p);
	color = parseColor(colorstr);
	if(gradient.num == 16) {
	    warning("gradient record too big- max size is 16, rest ignored");
	    break;
	}
	gradient.ratios[gradient.num] = pos;
	gradient.rgba[gradient.num] = color;
	gradient.num++;
	free(posstr);
	free(colorstr);
	lastpos = pos;
    }
    return gradient;
}

void s_gradient(char*name, const char*text, int radial, int rotate)
{
    gradient_t* gradient;
    gradient = malloc(sizeof(gradient_t));
    memset(gradient, 0, sizeof(gradient_t));
    gradient->gradient = parseGradient(text);
    gradient->radial = radial;
    gradient->rotate = rotate;

    if(dictionary_lookup(&gradients, name))
	syntaxerror("gradient %s defined twice", name);
    dictionary_put2(&gradients, name, gradient);
}
    
void s_gradientglow(char*name, char*gradient, float blurx, float blury, 
		    float angle, float distance, float strength, char innershadow, 
		    char knockout, char composite, char ontop, int passes)
{
    gradient_t* g = dictionary_lookup(&gradients, gradient);

    composite = 1;

    if(!g)
	syntaxerror("unknown gradient %s", gradient);
    FILTER_GRADIENTGLOW* filter = rfx_calloc(sizeof(FILTER_GRADIENTGLOW));
    filter->type = FILTERTYPE_GRADIENTGLOW;
    filter->gradient = &g->gradient;
    filter->blurx = blurx;
    filter->blury = blury;
    filter->strength = strength;
    filter->angle = angle;
    filter->distance = distance;
    filter->innershadow = innershadow;
    filter->knockout = knockout;
    filter->composite = composite;
    filter->ontop = ontop;
    filter->passes = passes;

    if(dictionary_lookup(&filters, name))
	syntaxerror("filter %s defined twice", name);
    dictionary_put2(&filters, name, filter);
}

void s_dropshadow(char*name, RGBA color, double blurx, double blury, double angle, double distance, double strength, char innershadow, char knockout, char composite, int passes)
{
    composite = 1;
    FILTER_DROPSHADOW* filter = rfx_calloc(sizeof(FILTER_DROPSHADOW));
    filter->type = FILTERTYPE_DROPSHADOW;
    filter->color= color;
    filter->blurx = blurx;
    filter->blury = blury;
    filter->strength = strength;
    filter->angle = angle;
    filter->distance = distance;
    filter->innershadow = innershadow;
    filter->knockout = knockout;
    filter->composite = composite;
    filter->passes = passes;

    if(dictionary_lookup(&filters, name))
	syntaxerror("filter %s defined twice", name);
    dictionary_put2(&filters, name, filter);
}

void s_bevel(char*name, RGBA shadow, RGBA highlight, double blurx, double blury, double angle, double distance, double strength, char innershadow, char knockout, char composite, char ontop, int passes)
{
    composite = 1;
    FILTER_BEVEL* filter = rfx_calloc(sizeof(FILTER_BEVEL));
    filter->type = FILTERTYPE_BEVEL;
    filter->shadow = shadow;
    filter->highlight = highlight;
    filter->blurx = blurx;
    filter->blury = blury;
    filter->strength = strength;
    filter->angle = angle;
    filter->distance = distance;
    filter->innershadow = innershadow;
    filter->knockout = knockout;
    filter->composite = composite;
    filter->ontop = ontop;
    filter->passes = passes;

    if(dictionary_lookup(&filters, name))
	syntaxerror("filter %s defined twice", name);
    dictionary_put2(&filters, name, filter);
}

void s_blur(char*name, double blurx, double blury, int passes)
{
    FILTER_BLUR* filter = rfx_calloc(sizeof(FILTER_BLUR));
    filter->type = FILTERTYPE_BLUR;
    filter->blurx = blurx;
    filter->blury = blury;
    filter->passes = passes;

    if(dictionary_lookup(&filters, name))
	syntaxerror("filter %s defined twice", name);
    dictionary_put2(&filters, name, filter);
}

void s_action(const char*text)
{
    ActionTAG* a = 0;
    a = swf_ActionCompile(text, stack[0].swf->fileVersion);
    if(!a) {
	syntaxerror("Couldn't compile ActionScript");
    }

    tag = swf_InsertTag(tag, ST_DOACTION);

    swf_ActionSet(tag, a);

    swf_ActionFree(a);
}

void s_initaction(const char*character, const char*text)
{
    ActionTAG* a = 0;
    character_t*c = 0;
    a = swf_ActionCompile(text, stack[0].swf->fileVersion);
    if(!a) {
	syntaxerror("Couldn't compile ActionScript");
    }

    c = (character_t*)dictionary_lookup(&characters, character);

    tag = swf_InsertTag(tag, ST_DOINITACTION);
    swf_SetU16(tag, c->id);
    swf_ActionSet(tag, a);

    swf_ActionFree(a);
}

int s_swf3action(char*name, char*action)
{
    ActionTAG* a = 0;
    instance_t* object = 0;
    if(name) 
	object = (instance_t*)dictionary_lookup(&instances, name);
    if(!object && name && *name) {
	/* we have a name, but couldn't find it. Abort. */
	return 0;
    }
    a = action_SetTarget(0, name);
    if(!strcmp(action, "nextframe")) a = action_NextFrame(a);
    else if(!strcmp(action, "previousframe")) a = action_PreviousFrame(a);
    else if(!strcmp(action, "stop")) a = action_Stop(a);
    else if(!strcmp(action, "play")) a = action_Play(a);
    a = action_SetTarget(a, "");
    a = action_End(a);

    tag = swf_InsertTag(tag, ST_DOACTION);
    swf_ActionSet(tag, a);
    swf_ActionFree(a);
    return 1;
}

void s_outline(char*name, char*format, char*source)
{
    outline_t* outline;

    drawer_t draw;
    SHAPE* shape;
    SHAPE2* shape2;
    SRECT bounds;
    
    //swf_Shape10DrawerInit(&draw, 0);
    swf_Shape11DrawerInit(&draw, 0);

    draw_string(&draw, source);
    draw.finish(&draw);
    shape = swf_ShapeDrawerToShape(&draw);
    bounds = swf_ShapeDrawerGetBBox(&draw);
    draw.dealloc(&draw);
    
    outline = (outline_t*)rfx_calloc(sizeof(outline_t));
    outline->shape = shape;
    outline->bbox = bounds;
    
    if(dictionary_lookup(&outlines, name))
	syntaxerror("outline %s defined twice", name);
    dictionary_put2(&outlines, name, outline);
}

int s_playsound(char*name, int loops, int nomultiple, int stop)
{
    sound_t* sound;
    SOUNDINFO info;
    if(!name)
	return 0;
    sound = dictionary_lookup(&sounds, name);
    if(!sound)
	return 0;

    tag = swf_InsertTag(tag, ST_STARTSOUND);
    swf_SetU16(tag, sound->id); //id
    memset(&info, 0, sizeof(info));
    info.stop = stop;
    info.loops = loops;
    info.nomultiple = nomultiple;
    swf_SetSoundInfo(tag, &info);
    return 1;
}

void s_includeswf(char*name, char*filename)
{
    int f;
    SWF swf;
    TAG* ftag;
    SRECT r;
    TAG* s;
    int level = 0;
    U16 cutout[] = {ST_SETBACKGROUNDCOLOR, ST_PROTECT, ST_FREEALL, ST_REFLEX};
    f = open(filename,O_RDONLY|O_BINARY);
    if (f<0) { 
	warning("Couldn't open file \"%s\": %s", filename, strerror(errno));
	s_box(name, 0, 0, black, 20, 0);
	return;
    }
    if (swf_ReadSWF(f,&swf)<0) { 
	warning("Only SWF files supported in .shape for now. File \"%s\" wasn't SWF.", filename);
	s_box(name, 0, 0, black, 20, 0);
	return;
    }
    close(f);

    /* FIXME: The following sets the bounding Box for the character. 
              It is wrong for two reasons:
	      a) It may be too small (in case objects in the movie clip at the borders)
	      b) it may be too big (because the poor movie never got autocropped)
    */
    r = swf.movieSize;
    
    s = tag = swf_InsertTag(tag, ST_DEFINESPRITE);
    swf_SetU16(tag, id);
    swf_SetU16(tag, swf.frameCount);

    swf_Relocate(&swf, idmap);

    ftag = swf.firstTag;
    level = 1;
    while(ftag) {
	int t;
	for(t=0;t<sizeof(cutout)/sizeof(cutout[0]);t++)
	    if(cutout[t] == ftag->id) {
		ftag = ftag->next;
		continue;
	    }
	if(ftag->id == ST_DEFINESPRITE && !swf_IsFolded(ftag))
	    level++;
	if(ftag->id == ST_END)
	    level--;
	if(!level)
	    break;

	if(ftag->id != ST_SETBACKGROUNDCOLOR) {
	    /* We simply dump all tags right after the sprite
	       header, relying on the fact that swf_OptimizeTagOrder() will
	       sort things out for us later. 
	       We also rely on the fact that the imported SWF is well-formed.
	     */
	    tag = swf_InsertTag(tag, ftag->id);
	    swf_SetBlock(tag, ftag->data, ftag->len);
	}

	ftag = ftag->next;
    }
    if(!ftag)
	syntaxerror("Included file %s contains errors", filename);
    tag = swf_InsertTag(tag, ST_END);

    swf_FreeTags(&swf);

    s_addcharacter(name, id, tag, r);
    incrementid();
}
SRECT s_getCharBBox(char*name)
{
    character_t* c = dictionary_lookup(&characters, name);
    if(!c) syntaxerror("character '%s' unknown(2)", name);
    return c->size;
}
SRECT s_getInstanceBBox(char*name)
{
    instance_t * i = dictionary_lookup(&instances, name);
    character_t * c;
    if(!i) syntaxerror("instance '%s' unknown(4)", name);
    c = i->character;
    if(!c) syntaxerror("internal error(5)");
    return c->size;
}
parameters_t s_getParameters(char*name)
{
    instance_t * i = dictionary_lookup(&instances, name);
    if(!i) syntaxerror("instance '%s' unknown(10)", name);
    return i->parameters;
}
void s_startclip(char*instance, char*character, parameters_t p)
{
    character_t* c = dictionary_lookup(&characters, character);
    instance_t* i;
    MATRIX m;
    if(!c) {
	syntaxerror("character %s not known", character);
    }
    i = s_addinstance(instance, c, currentdepth);
    i->parameters = p;
    m = s_instancepos(i->character->size, &p);
    
    tag = swf_InsertTag(tag, ST_PLACEOBJECT2);
    /* TODO: should be ObjectPlaceClip, with clipdepth backpatched */
    swf_ObjectPlace(tag, c->id, currentdepth, &m, &p.cxform, instance);
    i->lastTag = tag;
    i->lastFrame= currentframe;

    stack[stackpos].tag = tag;
    stack[stackpos].type = 2;
    stackpos++;

    currentdepth++;
}
void s_endClip()
{
    SWFPLACEOBJECT p;
    stackpos--;
    swf_SetTagPos(stack[stackpos].tag, 0);
    swf_GetPlaceObject(stack[stackpos].tag, &p);
    p.clipdepth = currentdepth;
    p.name = 0;
    swf_ClearTag(stack[stackpos].tag);
    swf_SetPlaceObject(stack[stackpos].tag, &p);
    currentdepth++;
}

void setPlacement(TAG*tag, U16 id, U16 depth, MATRIX m, char*name, parameters_t*p, char move)
{
    SWFPLACEOBJECT po;
    FILTERLIST flist;
    swf_GetPlaceObject(NULL, &po);
    po.id = id;
    po.depth = depth;
    po.matrix = m;
    po.cxform = p->cxform;
    po.name = name;
    po.move = move;
    if(move)
	po.id = 0;
    if(p->blendmode) {
	po.blendmode = p->blendmode;
    }
    if(p->filter) {
	flist.num = 1;
	flist.filter[0] = p->filter;
	po.filters = &flist;
    }
    swf_SetPlaceObject(tag, &po);
}

void s_put(char*instance, char*character, parameters_t p)
{
    character_t* c = dictionary_lookup(&characters, character);
    instance_t* i;
    MATRIX m;
    if(!c) {
	syntaxerror("character %s not known (in .put %s=%s)", character, instance, character);
    }
    
    i = s_addinstance(instance, c, currentdepth);
    i->parameters = p;
    m = s_instancepos(i->character->size, &p);
   
    if(p.blendmode || p.filter) {
	tag = swf_InsertTag(tag, ST_PLACEOBJECT3);
    } else {
	tag = swf_InsertTag(tag, ST_PLACEOBJECT2);
    }
    setPlacement(tag, c->id, currentdepth, m, instance, &p, 0);
    
    i->lastTag = tag;
    i->lastFrame = currentframe;
    currentdepth++;
    
}

void s_jump(char*instance, parameters_t p)
{
    instance_t* i = dictionary_lookup(&instances, instance);
    MATRIX m;
    if(!i) {
	syntaxerror("instance %s not known", instance);
    }

    i->parameters = p;
    m = s_instancepos(i->character->size, &p);

    if(p.blendmode || p.filter) {
	tag = swf_InsertTag(tag, ST_PLACEOBJECT3);
    } else {
	tag = swf_InsertTag(tag, ST_PLACEOBJECT2);
    }
    setPlacement(tag, 0, i->depth, m, 0, &p, 1);

    i->lastTag = tag;
    i->lastFrame = currentframe;
}

RGBA interpolateColor(RGBA c1, RGBA c2, float ratio)
{
    RGBA c;
    c.r = c1.r * (1-ratio) + c2.r * ratio;
    c.g = c1.g * (1-ratio) + c2.g * ratio;
    c.b = c1.b * (1-ratio) + c2.b * ratio;
    c.a = c1.a * (1-ratio) + c2.a * ratio;
    return c;
}

FILTER* interpolateFilter(FILTER*filter1,FILTER*filter2, float ratio)
{
    if(!filter1 && !filter2)
	return 0;
    if(!filter1)
	return interpolateFilter(filter2,filter1,1-ratio);

    if(filter2 && filter2->type != filter1->type)
	syntaxerror("can't interpolate between %s and %s filters yet", filtername[filter1->type], filtername[filter2->type]);
   
    if(filter1->type == FILTERTYPE_BLUR) {
	FILTER_BLUR*f1 = (FILTER_BLUR*)filter1;
	FILTER_BLUR*f2 = (FILTER_BLUR*)filter2;
	if(f2 && f1->blurx == f2->blurx && f1->blury == f2->blury)
	    return 0;
	FILTER_BLUR*f = (FILTER_BLUR*)swf_NewFilter(FILTERTYPE_BLUR);
	f->blurx= (f1->blurx)*(1-ratio) + (f2?f2->blurx:0)*ratio;
	f->blury= (f1->blury)*(1-ratio) + (f2?f2->blury:0)*ratio;
	f->passes= (f1->passes)*(1-ratio) + (f2?f2->passes:0)*ratio;
	return (FILTER*)f;
    } else if (filter1->type == FILTERTYPE_DROPSHADOW) {
	FILTER_DROPSHADOW*f1 = (FILTER_DROPSHADOW*)filter1;
	FILTER_DROPSHADOW*f2 = (FILTER_DROPSHADOW*)filter2;
	if(f2 && !memcmp(&f1->color,&f2->color,sizeof(RGBA)) && f1->strength == f2->strength && 
	   f1->blurx == f2->blurx && f1->blury == f2->blury && 
	   f1->angle == f2->angle && f1->distance == f2->distance)
	    return 0;
	FILTER_DROPSHADOW*f = (FILTER_DROPSHADOW*)swf_NewFilter(FILTERTYPE_DROPSHADOW);
	memcpy(f, f1, sizeof(FILTER_DROPSHADOW));
	f->color = interpolateColor(f1->color, f2->color, ratio);
	f->blurx= (f1->blurx)*(1-ratio) + (f2?f2->blurx:0)*ratio;
	f->blury= (f1->blury)*(1-ratio) + (f2?f2->blury:0)*ratio;
	f->passes= (f1->passes)*(1-ratio) + (f2?f2->passes:0)*ratio;
	f->angle= (f1->angle)*(1-ratio) + (f2?f2->angle:0)*ratio;
	f->distance= (f1->distance)*(1-ratio) + (f2?f2->distance:0)*ratio;
	f->strength= (f1->strength)*(1-ratio) + (f2?f2->strength:0)*ratio;
	return (FILTER*)f;
    } else if (filter1->type == FILTERTYPE_BEVEL) {
	FILTER_BEVEL*f1 = (FILTER_BEVEL*)filter1;
	FILTER_BEVEL*f2 = (FILTER_BEVEL*)filter2;
	if(f2 && !memcmp(&f1->shadow,&f2->shadow,sizeof(RGBA)) && 
	   !memcmp(&f1->highlight,&f2->highlight,sizeof(RGBA)) && 
	   f1->blurx == f2->blurx && f1->blury == f2->blury && f1->angle == f2->angle && f1->strength == f2->strength && f1->distance == f2->distance)
	    return 0;
	FILTER_BEVEL*f = (FILTER_BEVEL*)swf_NewFilter(FILTERTYPE_BEVEL);
	memcpy(f, f1, sizeof(FILTER_BEVEL));
	f->shadow = interpolateColor(f1->shadow, f2->shadow, ratio);
	f->highlight = interpolateColor(f1->highlight, f2->highlight, ratio);
	f->blurx= (f1->blurx)*(1-ratio) + (f2?f2->blurx:0)*ratio;
	f->blury= (f1->blury)*(1-ratio) + (f2?f2->blury:0)*ratio;
	f->passes= (f1->passes)*(1-ratio) + (f2?f2->passes:0)*ratio;
	f->angle= (f1->angle)*(1-ratio) + (f2?f2->angle:0)*ratio;
	f->distance= (f1->distance)*(1-ratio) + (f2?f2->distance:0)*ratio;
	f->strength= (f1->strength)*(1-ratio) + (f2?f2->strength:0)*ratio;
	return (FILTER*)f;
    } /*else if (filter1->type == FILTERTYPE_GRADIENTGLOW) {
	FILTER_GRADIENTGLOW*f = (FILTER_GRADIENTGLOW*)swf_NewFilter(FILTERTYPE_GRADIENTGLOW);
	// can't interpolate gradients
	memcpy(f, filter1, sizeof(FILTER_GRADIENTGLOW));
	return (FILTER*)f;
    }*/ else {
	syntaxerror("can't interpolate %s filters yet", filtername[filter1->type]);
    }
    return 0;
}

parameters_t s_interpolate(parameters_t*p1, parameters_t*p2, int pos, int num)
{
    parameters_t p;
    float ratio;
    if(num==0 || num==1)
	return *p1;
    ratio = (float)pos/(float)num;
    
    p.x = (p2->x-p1->x)*ratio + p1->x;
    p.y = (p2->y-p1->y)*ratio + p1->y;
    p.scalex = (p2->scalex-p1->scalex)*ratio + p1->scalex;
    p.scaley = (p2->scaley-p1->scaley)*ratio + p1->scaley;
    p.rotate = (p2->rotate-p1->rotate)*ratio + p1->rotate;
    p.shear = (p2->shear-p1->shear)*ratio + p1->shear;

    p.cxform.r0 = ((float)p2->cxform.r0-(float)p1->cxform.r0)*ratio + p1->cxform.r0;
    p.cxform.g0 = ((float)p2->cxform.g0-(float)p1->cxform.g0)*ratio + p1->cxform.g0;
    p.cxform.b0 = ((float)p2->cxform.b0-(float)p1->cxform.b0)*ratio + p1->cxform.b0;
    p.cxform.a0 = ((float)p2->cxform.a0-(float)p1->cxform.a0)*ratio + p1->cxform.a0;

    p.cxform.r1 = (p2->cxform.r1-p1->cxform.r1)*ratio + p1->cxform.r1;
    p.cxform.g1 = (p2->cxform.g1-p1->cxform.g1)*ratio + p1->cxform.g1;
    p.cxform.b1 = (p2->cxform.b1-p1->cxform.b1)*ratio + p1->cxform.b1;
    p.cxform.a1 = (p2->cxform.a1-p1->cxform.a1)*ratio + p1->cxform.a1;

    p.pivot.x = (p2->pivot.x-p1->pivot.x)*ratio + p1->pivot.x;
    p.pivot.y = (p2->pivot.y-p1->pivot.y)*ratio + p1->pivot.y;
    p.pin.x = (p2->pin.x-p1->pin.x)*ratio + p1->pin.x;
    p.pin.y = (p2->pin.y-p1->pin.y)*ratio + p1->pin.y;

    p.filter = interpolateFilter(p1->filter, p2->filter, ratio);
    return p;
}

void s_change(char*instance, parameters_t p2)
{
    instance_t* i = dictionary_lookup(&instances, instance);
    MATRIX m;
    parameters_t p1;
    TAG*t;
    int frame, allframes;
    if(!i) {
	syntaxerror("instance %s not known", instance);
    }
    p1 = i->parameters;
    
    allframes = currentframe - i->lastFrame - 1;
    if(allframes < 0) {
	warning(".change ignored. can only .put/.change an object once per frame.");
	return;
    }
    
    m = s_instancepos(i->character->size, &p2);
    if(p2.blendmode || p2.filter) {
	tag = swf_InsertTag(tag, ST_PLACEOBJECT3);
    } else {
	tag = swf_InsertTag(tag, ST_PLACEOBJECT2);
    }
    setPlacement(tag, 0, i->depth, m, 0, &p2, 1);
    i->parameters = p2;

    /* o.k., we got the start and end point set. Now iterate though all the
       tags in between, inserting object changes after each new frame */
    t = i->lastTag;
    i->lastTag = tag;
    if(!t) syntaxerror("internal error(6)");
    frame = 0;
    while(frame < allframes) {
	if(t->id == ST_SHOWFRAME) {
	    parameters_t p;
	    MATRIX m;
	    TAG*lt;
	    frame ++;
	    p = s_interpolate(&p1, &p2, frame, allframes);
	    m = s_instancepos(i->character->size, &p); //needed?

	    i->lastFrame = currentframe;
	    if(p.blendmode || p.filter) {
		lt = swf_InsertTag(t, ST_PLACEOBJECT3);
	    } else {
		lt = swf_InsertTag(t, ST_PLACEOBJECT2);
	    }
	    setPlacement(lt, 0, i->depth, m, 0, &p, 1);
	    t = lt;
	    if(frame == allframes)
		break;
	}
	t = t->next;
	if(!t) 
	    syntaxerror("internal error(8) (frame=%d/%d)", frame, allframes);
    }
}

void s_delinstance(char*instance)
{
    instance_t* i = dictionary_lookup(&instances, instance);
    if(!i) {
	syntaxerror("instance %s not known", instance);
    }
    tag = swf_InsertTag(tag, ST_REMOVEOBJECT2);
    swf_SetU16(tag, i->depth);
    dictionary_del(&instances, instance);
}

void s_qchange(char*instance, parameters_t p)
{
}

void s_end()
{
    if(!stackpos)
	syntaxerror(".end unexpected");
    if(stack[stackpos-1].type == 0)
	s_endSWF();
    else if(stack[stackpos-1].type == 1)
	s_endSprite();
    else if(stack[stackpos-1].type == 2)
	s_endClip();
    else if(stack[stackpos-1].type == 3)
	s_endButton();
    else syntaxerror("internal error 1");
}

// ------------------------------------------------------------------------

typedef int command_func_t(map_t*args);

SRECT parseBox(char*str)
{
    SRECT r = {0,0,0,0};
    float xmin, xmax, ymin, ymax;
    char*x = strchr(str, 'x');
    char*d1=0,*d2=0;
    if(!strcmp(str, "autocrop")) {
	r.xmin = r.ymin = r.xmax = r.ymax = 0;
	return r;
    }
    if(!x) goto error;
    d1 = strchr(x+1, ':');
    if(d1)
	d2 = strchr(d1+1, ':');
    if(!d1) {
	if(sscanf(str, "%fx%f", &xmax, &ymax) < 2)
	    goto error;
	xmin = ymin = 0;
    }
    else if(d1 && !d2) {
	if(sscanf(str, "%fx%f:%f", &xmax, &ymax, &xmin) < 3)
	    goto error;
	xmax += xmin;
	ymin = 0;
    }
    else {
	if(sscanf(str, "%fx%f:%f:%f", &xmax, &ymax, &xmin, &ymin) < 4)
	    goto error;
	xmax += xmin;
	ymax += ymin;
    }
    r.xmin = (SCOORD)(xmin*20);
    r.ymin = (SCOORD)(ymin*20);
    r.xmax = (SCOORD)(xmax*20);
    r.ymax = (SCOORD)(ymax*20);
    return r;
error:
    syntaxerror("expression %s is not a valid bound Box.\nE.g. 1024x768 or 1024x768:30:30 would have been valid bounding Boxes.", str);
    return r;
}
float parseFloat(char*str)
{
    return atof(str);
}
int parseInt(char*str)
{
    int t;
    int l=strlen(str);
    int s=0;
    if(str[0]=='+' || str[0]=='-')
	s++;

    for(t=s;t<l;t++)
	if(str[t]<'0' || str[t]>'9')
	    syntaxerror("Not an Integer: \"%s\"", str);
    return atoi(str);
}
int parseTwip(char*str)
{
    char*dot;
    int sign=1;
    if(str[0]=='+' || str[0]=='-') {
	if(str[0]=='-')
	    sign = -1;
	str++;
    }
    dot = strchr(str, '.');
    if(!dot) {
	int l=strlen(str);
	int t;
	return sign*parseInt(str)*20;
    } else {
	char* old = strdup(str);
	int l=strlen(dot+1);
	char*s;
	*dot++ = 0;
	for(s=str;s<dot-1;s++)
	    if(*s<'0' || *s>'9')
		syntaxerror("Not a coordinate: \"%s\"", str);
	for(s=dot;*s;s++) {
	    if(*s<'0' || *s>'9')
		syntaxerror("Not a coordinate: \"%s\"", str);
	}
	if(l>2 || (l==2 && (dot[1]!='0' && dot[1]!='5'))) {
	    dot[1] = ((dot[1]-0x30)/5)*5 + 0x30;
	    dot[2] = 0;
	    l=2;
	    warning("precision loss: %s converted to twip: %s.%s", old, str, dot);
	}
	free(old);
	if(l==0)
	    return sign*atoi(str)*20;
	if(l==1)
	    return sign*atoi(str)*20+atoi(dot)*2;
	if(l==2)
	    return sign*atoi(str)*20+atoi(dot)/5;
    }
    return 0;
}

int isPoint(char*str)
{
    if(strchr(str, '('))
	return 1;
    else
	return 0;
}

SPOINT parsePoint(char*str)
{
    SPOINT p;
    char tmp[80];
    int l = strlen(str);
    char*comma = strchr(str, ',');
    if(str[0]!='(' || str[l-1]!=')' || !comma || l>70)
	syntaxerror("\"%s\" is not a valid point of the form (x,y)", str);
    strncpy(tmp, str+1, comma-(str+1));tmp[comma-(str+1)]=0;
    p.x = parseTwip(tmp);
    strncpy(tmp, comma+1, l-1-(comma+1-str));tmp[l-1-(comma+1-str)]=0;
    p.y = parseTwip(tmp);
    return p;
}

int parseColor2(char*str, RGBA*color)
{
    int l = strlen(str);
    int r,g,b,a;
    int t;

    struct {unsigned char r,g,b;char*name;} colors[] =
    {{255,250,250,"snow"},{220,220,220,"gainsboro"},{250,240,230,"linen"},{255,228,196,"bisque"},
    {255,228,181,"moccasin"},{255,248,220,"cornsilk"},{255,255,240,"ivory"},{255,245,238,"seashell"},
    {240,255,240,"honeydew"},{240,255,255,"azure"},{230,230,250,"lavender"},{255,255,255,"white"},
    {0,0,0,"black"},{190,190,190,"gray"},{190,190,190,"grey"},{0,0,128,"navy"},{0,0,255,"blue"},
    {64,224,208,"turquoise"},{0,255,255,"cyan"},{127,255,212,"aquamarine"}, {0,255,0,"green"},
    {127,255,0,"chartreuse"},{240,230,140,"khaki"},{255,255,0,"yellow"},
    {255,215,0,"gold"},{218,165,32,"goldenrod"},{160,82,45,"sienna"},{205,133,63,"peru"},
    {222,184,135,"burlywood"},{245,245,220,"beige"},{245,222,179,"wheat"},{210,180,140,"tan"},
    {210,105,30,"chocolate"},{178,34,34,"firebrick"},{165,42,42,"brown"},{250,128,114,"salmon"},
    {255,165,0,"orange"},{255,127,80,"coral"},{255,99,71,"tomato"},{255,0,0,"red"},
    {255,192,203,"pink"},{176,48,96,"maroon"},{255,0,255,"magenta"},{238,130,238,"violet"},
    {221,160,221,"plum"},{218,112,214,"orchid"},{160,32,240,"purple"},{216,191,216,"thistle"}};

    a=255;r=g=b=0;

    if(str[0]=='#' && (l==7 || l==9)) {
	if(l == 7 && !(sscanf(str, "#%02x%02x%02x", &r, &g, &b)))
	    return 0;
	if(l == 9 && !(sscanf(str, "#%02x%02x%02x%02x", &r, &g, &b, &a)))
	    return 0;
	color->r = r; color->g = g; color->b = b; color->a = a;
	return 1;
    }
    int len=strlen(str);
    U8 alpha = 255;
    if(strchr(str, '/')) {
	len = strchr(str, '/')-str;
	sscanf(str+len+1,"%02x", &alpha);
    }
    for(t=0;t<sizeof(colors)/sizeof(colors[0]);t++)
	if(!strncmp(str, colors[t].name, len)) {
	    r = colors[t].r;
	    g = colors[t].g;
	    b = colors[t].b;
	    a = alpha;
	    color->r = r; color->g = g; color->b = b; color->a = a;
	    return 1;
	}
    return 0;

}
RGBA parseColor(char*str)
{
    RGBA c;
    if(!parseColor2(str, &c))
	syntaxerror("Expression '%s' is not a color", str);
    return c;
}

typedef struct _muladd {
    S16 mul;
    S16 add;
} MULADD;

MULADD parseMulAdd(char*str)
{
    float add, mul;
    char* str2 = (char*)malloc(strlen(str)+5);
    int i;
    MULADD m;
    strcpy(str2, str);
    strcat(str2, " 0");
    add = 0;
    mul = 1.0;
    if(sscanf(str2, "%f%f %d", &mul, &add, &i)==2+1) { add/=256.0; i=1;}
    else if(sscanf(str2, "%f%%%f %d", &mul, &add, &i)==2+1) { mul/=100.0; add/=256.0; i=2;}
    else if(sscanf(str2, "%f%f%% %d", &mul, &add, &i)==2+1) { add/=100.0; i=3;}
    else if(sscanf(str2, "%f%%%f%% %d", &mul, &add, &i)==2+1) { mul/=100.0; add/=100.0; i=4;}
    else if(sscanf(str2, "+%f%% %d", &add, &i)==1+1) { mul=1.0;add/=100.0; i=5;}
    else if(sscanf(str2, "+%f %d", &add, &i)==1+1) { mul=1.0;add/=256.0; i=6;}
    else if(sscanf(str2, "-%f%% %d", &add, &i)==1+1) { mul=1.0;add/=-100.0; i=7;}
    else if(sscanf(str2, "-%f %d", &add, &i)==1+1) { mul=1.0;add/=-256.0; i=8;}
    else if(sscanf(str2, "%f%% %d", &mul, &i)==1+1) { mul/=100.0;add=0; i=9;}
    else if(sscanf(str2, "%f %d", &mul, &i)==1+1) { add=0; i=10;}
    else {
	syntaxerror("'%s' is not a valid color transform expression", str);
    }
    m.add = (int)(add*256);
    m.mul = (int)(mul*256);
    free(str2);
    return m;
}

MULADD mergeMulAdd(MULADD m1, MULADD m2)
{
    int a = (int)((double)m2.add+((double)m1.add*(double)m2.mul)/256.0);
    double m = ((double)m1.mul*(double)m2.mul)/256.0;
    MULADD r;
    if(a<-32768) a=-32768;
    if(a>32767) a=32767;
    if(m<-32768) m=-32768;
    if(m>32767) m=32767;
    r.add = a;
    r.mul = (int)m;
    return r;
}

float parsePxOrPercent(char*fontname, char*str)
{
    int l = strlen(str);
    if(strchr(str, '%'))
	return parsePercent(str);
    if(l>2 && str[l-2]=='p' && str[l-1]=='t') {
	float p = atof(str);
	return p/64.0; /*64 = FT_SUBPIXELS- see ../lib/modules/swffont.c */
    }
    syntaxerror("Expression '%s' is neither a point size (?pt) nor a percentage (?%)", str);
    return 0;
}

float parsePercent(char*str)
{
    int l = strlen(str);
    if(!l)
	return 1.0;
    if(str[l-1]=='%') {
	return atoi(str)/100.0;
    }
    syntaxerror("Expression '%s' is not a percentage", str);
    return 0;
}
int isPercent(char*str)
{
    return str[strlen(str)-1]=='%';
}
int parseNewSize(char*str, int size)
{
    if(isPercent(str))
	return parsePercent(str)*size;
    else
	return (int)(atof(str)*20);
}

int isColor(char*str)
{
    RGBA c;
    return parseColor2(str, &c);
}

static char* lu(map_t* args, char*name)
{
    char* value = map_lookup(args, name);
    if(!value) {
	map_dump(args, stdout, "");
	syntaxerror("internal error 2: value %s should be set", name);
    }
    return value;
}

static int c_flash(map_t*args) 
{
    char* filename = map_lookup(args, "filename");
    char* compressstr = lu(args, "compress");
    SRECT bbox = parseBox(lu(args, "bbox"));
    int version = parseInt(lu(args, "version"));
    int fps = (int)(parseFloat(lu(args, "fps"))*256);
    int compress = 0;
    RGBA color = parseColor(lu(args, "background"));

    if(!filename || !*filename) {
	/* for compatibility */
	filename = map_lookup(args, "name");
	if(!filename || !*filename) {
	    filename = 0;
	} else {
	    //msg("<warning> line %d: .flash name=... is deprecated, use .flash filename=...", line);
	    msg("<notice> line %d: .flash name=... is deprecated, use .flash filename=...", line);
	}
    }

    if(!filename || override_outputname)
	filename = outputname;
    
    if(!strcmp(compressstr, "default"))
	compress = version>=6;
    else if(!strcmp(compressstr, "yes") || !strcmp(compressstr, "compress"))
	compress = 1;
    else if(!strcmp(compressstr, "no"))
	compress = 0;
    else syntaxerror("value \"%s\" not supported for the compress argument", compressstr);

    s_swf(filename, bbox, version, fps, compress, color);
    return 0;
}
int isRelative(char*str)
{
    return !strncmp(str, "<plus>", 6) ||
	   !strncmp(str, "<minus>", 7);
}
char* getOffset(char*str)
{
    if(!strncmp(str, "<plus>", 6))
	return str+6;
    if(!strncmp(str, "<minus>", 7))
	return str+7;
    syntaxerror("internal error (347)");
    return 0;
}
int getSign(char*str)
{
    if(!strncmp(str, "<plus>", 6))
	return 1;
    if(!strncmp(str, "<minus>", 7))
	return -1;
    syntaxerror("internal error (348)");
    return 0;
}
static dictionary_t points;
static mem_t mpoints;
int points_initialized = 0;

SPOINT getPoint(SRECT r, char*name)
{
    int l=0;
    if(!strcmp(name, "center")) {
	SPOINT p;
	p.x = (r.xmin + r.xmax)/2;
	p.y = (r.ymin + r.ymax)/2;
	return p;
    }

    if(points_initialized)
	l = (int)dictionary_lookup(&points, name);
    if(l==0) {
	syntaxerror("Invalid point: \"%s\".", name);
    }
    l--;
    return *(SPOINT*)&mpoints.buffer[l];
}

static int texture2(char*name, char*object, map_t*args, int errors) 
{
    SPOINT pos,size;
    char*xstr = map_lookup(args, "x");
    char*ystr = map_lookup(args, "y");
    char*widthstr = map_lookup(args, "width");
    char*heightstr = map_lookup(args, "height");
    char*scalestr = map_lookup(args, "scale");
    char*scalexstr = map_lookup(args, "scalex");
    char*scaleystr = map_lookup(args, "scaley");
    char*rotatestr = map_lookup(args, "rotate");
    char* shearstr = map_lookup(args, "shear");
    char* radiusstr = map_lookup(args, "r");
    float x=0,y=0;
    float scalex = 1.0, scaley = 1.0;
    float rotate=0, shear=0;
    float r = 0;
    if(!*xstr && !*ystr) {
	if(errors)
	    syntaxerror("x and y must be set");
	return 0;
    }
    if(*scalestr && (*scalexstr || *scaleystr)) {
	syntaxerror("scale and scalex/scaley can't both be set");
	return 0;
    }
    if((*widthstr || *heightstr) && *radiusstr) {
	syntaxerror("width/height and radius can't both be set");
    }
    if(*radiusstr) {
	widthstr = radiusstr;
	heightstr = radiusstr;
    }
    if(!*xstr) xstr="0";
    if(!*ystr) ystr="0";
    if(!*rotatestr) rotatestr="0";
    if(!*shearstr) shearstr="0";

    if(*scalestr) {
	scalex = scaley = parsePercent(scalestr);
    } else if(*scalexstr || *scaleystr) {
	if(scalexstr) scalex = parsePercent(scalexstr);
	if(scaleystr) scaley = parsePercent(scaleystr);
    } else if(*widthstr || *heightstr) {
	int width=0;
	int height=0;
	s_getBitmapSize(object, &width, &height);
	if(*widthstr)
	    scalex = (float)parseTwip(widthstr)/(float)width;
	if(*heightstr)
	    scaley = (float)parseTwip(heightstr)/(float)height;
    }
    x = parseTwip(xstr);
    y = parseTwip(ystr);
    rotate = parseFloat(rotatestr);
    shear = parseFloat(shearstr);

    s_texture(name, object, x,y,scalex,scaley,rotate, shear);

    return 0;
}

static int c_texture(map_t*args) 
{
    char*name = lu(args, "instance");
    char*object = lu(args, "character");
    return texture2(name, object, args, 1);
}

static int c_gradient(map_t*args) 
{
    char*name = lu(args, "name");
    int radial= strcmp(lu(args, "radial"), "radial")?0:1;
    int rotate = parseInt(lu(args, "rotate"));

    readToken();
    if(type != RAWDATA)
	syntaxerror("colon (:) expected");

    s_gradient(name, text, radial, rotate);

    /* check whether we also have placement information,
       which would make this a positioned gradient.
       If there is placement information, texture2() will
       add a texture, which has priority over the gradient.
     */
    texture2(name, name, args, 0);
    return 0;
}

static int c_blur(map_t*args) 
{
    char*name = lu(args, "name");
    char*blurstr = lu(args, "blur");
    char*blurxstr = lu(args, "blurx");
    char*blurystr = lu(args, "blury");
    float blurx=1.0, blury=1.0;
    if(blurstr[0]) {
	blurx = parseFloat(blurstr);
	blury = parseFloat(blurstr);
    } 
    if(blurxstr[0])
	blurx = parseFloat(blurxstr);
    if(blurystr[0])
	blury = parseFloat(blurystr);
    int passes = parseInt(lu(args, "passes"));
    s_blur(name, blurx, blury, passes);
    return 0;
}

static int c_gradientglow(map_t*args) 
{
    char*name = lu(args, "name");
    char*gradient = lu(args, "gradient");
    char*blurstr = lu(args, "blur");
    char*blurxstr = lu(args, "blurx");
    char*blurystr = lu(args, "blury");
    float blurx=1.0, blury=1.0;
    if(blurstr[0]) {
	blurx = parseFloat(blurstr);
	blury = parseFloat(blurstr);
    } 
    if(blurxstr[0])
	blurx = parseFloat(blurxstr);
    if(blurystr[0])
	blury = parseFloat(blurystr);

    float angle = parseFloat(lu(args, "angle")) / 180 * M_PI;
    float distance = parseFloat(lu(args, "distance"));
    float strength = parseFloat(lu(args, "strength"));
    char innershadow = strcmp(lu(args, "innershadow"),"innershadow")?0:1;
    char knockout = strcmp(lu(args, "knockout"),"knockout")?0:1;
    char composite = strcmp(lu(args, "composite"),"composite")?0:1;
    char ontop = strcmp(lu(args, "ontop"),"ontop")?0:1;
    int passes = parseInt(lu(args, "passes"));

    s_gradientglow(name, gradient, blurx, blury, angle, distance, strength, innershadow, knockout, composite, ontop, passes);
    return 0;
}

static int c_dropshadow(map_t*args) 
{
    char*name = lu(args, "name");
    RGBA color = parseColor(lu(args, "color"));
    char*blurstr = lu(args, "blur");
    char*blurxstr = lu(args, "blurx");
    char*blurystr = lu(args, "blury");
    float blurx=1.0, blury=1.0;
    if(blurstr[0]) {
	blurx = parseFloat(blurstr);
	blury = parseFloat(blurstr);
    } 
    if(blurxstr[0])
	blurx = parseFloat(blurxstr);
    if(blurystr[0])
	blury = parseFloat(blurystr);

    float angle = parseFloat(lu(args, "angle")) / 180 * M_PI;
    float distance = parseFloat(lu(args, "distance"));
    float strength = parseFloat(lu(args, "strength"));
    char innershadow = strcmp(lu(args, "innershadow"),"innershadow")?0:1;
    char knockout = strcmp(lu(args, "knockout"),"knockout")?0:1;
    char composite = strcmp(lu(args, "composite"),"composite")?0:1;
    int passes = parseInt(lu(args, "passes"));

    s_dropshadow(name, color, blurx, blury, angle, distance, strength, innershadow, knockout, composite, passes);
    return 0;
}

static int c_bevel(map_t*args) 
{
    char*name = lu(args, "name");
    RGBA shadow = parseColor(lu(args, "shadow"));
    RGBA highlight = parseColor(lu(args, "highlight"));
    char*blurstr = lu(args, "blur");
    char*blurxstr = lu(args, "blurx");
    char*blurystr = lu(args, "blury");
    float blurx=1.0, blury=1.0;
    if(blurstr[0]) {
	blurx = parseFloat(blurstr);
	blury = parseFloat(blurstr);
    } 
    if(blurxstr[0])
	blurx = parseFloat(blurxstr);
    if(blurystr[0])
	blury = parseFloat(blurystr);

    float angle = parseFloat(lu(args, "angle")) / 180 * M_PI;
    float distance = parseFloat(lu(args, "distance"));
    float strength = parseFloat(lu(args, "strength"));
    char innershadow = strcmp(lu(args, "innershadow"),"innershadow")?0:1;
    char knockout = strcmp(lu(args, "knockout"),"knockout")?0:1;
    char composite = strcmp(lu(args, "composite"),"composite")?0:1;
    char ontop = strcmp(lu(args, "ontop"),"ontop")?0:1;
    int passes = parseInt(lu(args, "passes"));

    s_bevel(name, shadow, highlight, blurx, blury, angle, distance, strength, innershadow, knockout, composite, ontop, passes);
    return 0;
}

static int c_point(map_t*args) 
{
    char*name = lu(args, "name");
    int pos;
    string_t s1;
    SPOINT p;
    if(!points_initialized) {
	dictionary_init(&points);
	mem_init(&mpoints);
	points_initialized = 1;
    }
    p.x = parseTwip(lu(args, "x"));
    p.y = parseTwip(lu(args, "y"));
    pos = mem_put(&mpoints, &p, sizeof(p));
    string_set(&s1, name);
    pos++;
    dictionary_put(&points, s1, (void*)pos);
    return 0;
}
static int c_play(map_t*args) 
{
    char*name = lu(args, "name");
    char*loop = lu(args, "loop");
    char*nomultiple = lu(args, "nomultiple");
    int nm = 0;
    if(!strcmp(nomultiple, "nomultiple"))
	nm = 1;
    else
	nm = parseInt(nomultiple);

    if(s_playsound(name, parseInt(loop), nm, 0)) {
	return 0;
    } else if(s_swf3action(name, "play")) {
	return 0;
    }
    return 0;
}

static int c_stop(map_t*args) 
{
    char*name = map_lookup(args, "name");

    if(s_playsound(name, 0,0,1)) {
	return 0;
    } else if(s_swf3action(name, "stop")) {
	return 0;
    }
    syntaxerror("I don't know anything about sound/movie \"%s\"", name);
    return 0;
}

static int c_nextframe(map_t*args) 
{
    char*name = lu(args, "name");

    if(s_swf3action(name, "nextframe")) {
	return 0;
    }
    syntaxerror("I don't know anything about movie \"%s\"", name);
    return 0;
}

static int c_previousframe(map_t*args) 
{
    char*name = lu(args, "name");

    if(s_swf3action(name, "previousframe")) {
	return 0;
    }
    syntaxerror("I don't know anything about movie \"%s\"", name);
    return 0;
}

static int c_placement(map_t*args, int type)
{
    char*instance = lu(args, (type==0||type==4)?"instance":"name");
    char*character = 0;

    char* luminancestr = lu(args, "luminance");
    char* scalestr = lu(args, "scale");
    char* scalexstr = lu(args, "scalex");
    char* scaleystr = lu(args, "scaley");
    char* rotatestr = lu(args, "rotate");
    char* shearstr = lu(args, "shear");
    char* xstr="", *pivotstr="";
    char* ystr="", *anglestr="";
    char*above = lu(args, "above"); /*FIXME*/
    char*below = lu(args, "below");
    char* rstr = lu(args, "red");
    char* gstr = lu(args, "green");
    char* bstr = lu(args, "blue");
    char* astr = lu(args, "alpha");
    char* pinstr = lu(args, "pin");
    char* as = map_lookup(args, "as");
    char* blendmode = lu(args, "blend");
    char*filterstr = lu(args, "filter");
    U8 blend;
    MULADD r,g,b,a;
    float oldwidth;
    float oldheight;
    SRECT oldbbox;
    MULADD luminance;
    parameters_t p;

    if(type==9) { // (?) .rotate  or .arcchange
	pivotstr = lu(args, "pivot");
	anglestr = lu(args, "angle");
    } else {
	xstr = lu(args, "x");
	ystr = lu(args, "y");
    }
    if(luminancestr[0])
	luminance = parseMulAdd(luminancestr);
    else {
	luminance.add = 0;
	luminance.mul = 256;
    }

    if(scalestr[0]) {
	if(scalexstr[0]||scaleystr[0])
	    syntaxerror("scalex/scaley and scale cannot both be set");
	scalexstr = scaleystr = scalestr;
    }
    
    if(type == 0 || type == 4)  {
	// put or startclip
	character = lu(args, "character");
	parameters_clear(&p);
    } else if (type == 5) {
	character = lu(args, "name");
	parameters_clear(&p);
	// button's show
    } else {
	p = s_getParameters(instance);
    }

    /* x,y position */
    if(xstr[0]) {
	if(isRelative(xstr)) {
	    if(type == 0 || type == 4)
		syntaxerror("relative x values not allowed for initial put or startclip");
	    p.x += parseTwip(getOffset(xstr))*getSign(xstr);
	} else {
	    p.x = parseTwip(xstr);
	}
    }
    if(ystr[0]) {
	if(isRelative(ystr)) {
	    if(type == 0 || type == 4)
		syntaxerror("relative y values not allowed for initial put or startclip");
	    p.y += parseTwip(getOffset(ystr))*getSign(ystr);
	} else {
	    p.y = parseTwip(ystr);
	}
    }

    /* scale, scalex, scaley */
    if(character) {
	oldbbox = s_getCharBBox(character);
    } else {
	oldbbox = s_getInstanceBBox(instance);
    }
    oldwidth = oldbbox.xmax - oldbbox.xmin;
    oldheight = oldbbox.ymax - oldbbox.ymin;
    if(scalexstr[0]) {
	if(oldwidth==0) p.scalex = 1.0;
	else {	    
	    if(scalexstr[0])
		p.scalex = (float)(parseNewSize(scalexstr, oldwidth))/oldwidth;
	}
    }
    if(scaleystr[0]) {
	if(oldheight==0) p.scaley = 1.0;
	else {
	    if(scaleystr[0])
		p.scaley = (float)(parseNewSize(scaleystr, oldheight))/oldheight;
	}
    }
   
    /* rotation */
    if(rotatestr[0]) {
	if(isRelative(rotatestr)) {
	    p.rotate += parseFloat(getOffset(rotatestr))*getSign(rotatestr);
	} else {
	    p.rotate = parseFloat(rotatestr);
	}
    }

    /* shearing */
    if(shearstr[0]) {
	if(isRelative(shearstr)) {
	    p.shear += parseFloat(getOffset(shearstr))*getSign(shearstr);
	} else {
	    p.shear = parseFloat(shearstr);
	}
    }

    if(pivotstr[0]) {
	if(isPoint(pivotstr)) 
	    p.pivot = parsePoint(pivotstr);
	else 
	    p.pivot = getPoint(oldbbox, pivotstr);
    }
    if(pinstr[0]) {
	if(isPoint(pinstr))
	    p.pin = parsePoint(pinstr);
	else
	    p.pin = getPoint(oldbbox, pinstr);
    }
	
    /* color transform */

    if(rstr[0] || luminancestr[0]) {
	MULADD r;
	if(rstr[0])
	    r = parseMulAdd(rstr);
	else {
	    r.add = p.cxform.r0;
	    r.mul = p.cxform.r1;
	}
	r = mergeMulAdd(r, luminance);
	p.cxform.r0 = r.mul;p.cxform.r1 = r.add;
    }
    if(gstr[0] || luminancestr[0]) {
	MULADD g;
	if(gstr[0])
 	    g = parseMulAdd(gstr);
	else {
	    g.add = p.cxform.g0;
	    g.mul = p.cxform.g1;
	}
	g = mergeMulAdd(g, luminance);
	p.cxform.g0 = g.mul;p.cxform.g1 = g.add;
    }
    if(bstr[0] || luminancestr[0]) {
	MULADD b;
	if(bstr[0])
 	    b = parseMulAdd(bstr);
	else {
	    b.add = p.cxform.b0;
	    b.mul = p.cxform.b1;
	}
	b = mergeMulAdd(b, luminance);
	p.cxform.b0 = b.mul;p.cxform.b1 = b.add;
    }
    if(astr[0]) {
	MULADD a = parseMulAdd(astr);
	p.cxform.a0 = a.mul;p.cxform.a1 = a.add;
    }

    if(blendmode[0]) {
	int t;
	blend = 255;
	for(t=0;blendModeNames[t];t++) {
	    if(!strcmp(blendModeNames[t], blendmode)) {
		blend = t;
		break;
	    }
	}
	if(blend == 255) {
	    syntaxerror("unknown blend mode: '%s'", blendmode);
	}
	p.blendmode = blend;
    }
    
    if(filterstr[0]) {
	FILTER*f = dictionary_lookup(&filters, filterstr);
	if(!f) 
	    syntaxerror("Unknown filter %s", filterstr);
	p.filter = f;
    }


    if(type == 0)
	s_put(instance, character, p);
    else if(type == 1)
	s_change(instance, p);
    else if(type == 2)
	s_qchange(instance, p);
    else if(type == 3)
	s_jump(instance, p);
    else if(type == 4)
	s_startclip(instance, character, p);
    else if(type == 5) {
	if(as && as[0]) {
	    s_buttonput(character, as, p);
	} else {
	    s_buttonput(character, "shape", p);
	}
    }
    return 0;
}
static int c_put(map_t*args) 
{
    c_placement(args, 0);
    return 0;
}
static int c_change(map_t*args) 
{
    c_placement(args, 1);
    return 0;
}
static int c_qchange(map_t*args) 
{
    c_placement(args, 2);
    return 0;
}
static int c_arcchange(map_t*args) 
{
    c_placement(args, 2);
    return 0;
}
static int c_jump(map_t*args) 
{
    c_placement(args, 3);
    return 0;
}
static int c_startclip(map_t*args) 
{
    c_placement(args, 4);
    return 0;
}
static int c_show(map_t*args) 
{
    c_placement(args, 5);
    return 0;
}
static int c_del(map_t*args) 
{
    char*instance = lu(args, "name");
    s_delinstance(instance);
    return 0;
}
static int c_end(map_t*args) 
{
    s_end();
    return 0;
}
static int c_sprite(map_t*args) 
{
    char* name = lu(args, "name");
    s_sprite(name);
    return 0;
}
static int c_frame(map_t*args) 
{
    char*framestr = lu(args, "n");
    char*cutstr = lu(args, "cut");
    
    char*name = lu(args, "name");
    char*anchor = lu(args, "anchor");
    char buf[40];

    if(!strcmp(anchor, "anchor") && !*name)
	name = framestr;

    int frame;
    int cut = 0;
    if(strcmp(cutstr, "no"))
	cut = 1;
    if(isRelative(framestr)) {
	frame = s_getframe();
	if(getSign(framestr)<0)
	    syntaxerror("relative frame expressions must be positive");
	frame += parseInt(getOffset(framestr));
    }
    else {
	frame = parseInt(framestr);
	if(s_getframe() >= frame
		&& !(frame==1 && s_getframe()==frame)) // equality is o.k. for frame 0
	    syntaxerror("frame expression must be >%d (is:%s)", s_getframe(), framestr);
    }
    s_frame(frame, cut, name, !strcmp(anchor, "anchor"));
    return 0;
}
static int c_primitive(map_t*args) 
{
    char*name = lu(args, "name");
    char*command = lu(args, "commandname");
    int width=0, height=0, r=0;
    int linewidth = parseTwip(lu(args, "line"));
    char*colorstr = lu(args, "color");
    RGBA color = parseColor(colorstr);
    char*fillstr = lu(args, "fill");
    int dofill = 1;
    int type=0;
    char* font;
    char* text;
    char* outline=0;
    RGBA fill;
    if(!strcmp(command, "circle"))
	type = 1;
    else if(!strcmp(command, "filled"))
	type = 2;
   
    if(type==0) {
	width = parseTwip(lu(args, "width"));
	height = parseTwip(lu(args, "height"));
    } else if (type==1) {
	r = parseTwip(lu(args, "r"));
    } else if (type==2) {
	outline = lu(args, "outline");
    }

    if(!strcmp(fillstr, "fill"))
	fillstr = colorstr;
    if(!strcmp(fillstr, "none"))
	fillstr = 0;
    if(width<0 || height<0 || linewidth<0 || r<0)
	syntaxerror("values width, height, line, r must be positive");
    
    if(type == 0) s_box(name, width, height, color, linewidth, fillstr);
    else if(type==1) s_circle(name, r, color, linewidth, fillstr);
    else if(type==2) s_filled(name, outline, color, linewidth, fillstr);
    return 0;
}

static int c_textshape(map_t*args) 
{
    char*name = lu(args, "name");
    char*text = lu(args, "text");
    char*font = lu(args, "font");
    float size = parsePxOrPercent(font, lu(args, "size"));

    s_textshape(name, font, size, text);
    return 0;
}

static int c_swf(map_t*args) 
{
    char*name = lu(args, "name");
    char*filename = lu(args, "filename");
    char*command = lu(args, "commandname");
    if(!strcmp(command, "shape"))
	warning("Please use .swf instead of .shape");
    s_includeswf(name, filename);
    return 0;
}

static int c_font(map_t*args) 
{
    char*name = lu(args, "name");
    char*filename = lu(args, "filename");
    s_font(name, filename);
    return 0;
}

static int c_sound(map_t*args) 
{
    char*name = lu(args, "name");
    char*filename = lu(args, "filename");
    s_sound(name, filename);
    return 0;
}

static int c_text(map_t*args) 
{
    char*name = lu(args, "name");
    char*text = lu(args, "text");
    char*font = lu(args, "font");
    float size = parsePxOrPercent(font, lu(args, "size"));
    RGBA color = parseColor(lu(args, "color"));
    s_text(name, font, text, (int)(size*100), color);
    return 0;
}

static int c_soundtrack(map_t*args) 
{
    return 0;
}

static int c_quicktime(map_t*args) 
{
    char*name = lu(args, "name");
    char*url = lu(args, "url");
    s_quicktime(name, url);
    return 0;
}

static int c_image(map_t*args) 
{
    char*command = lu(args, "commandname");
    char*name = lu(args, "name");
    char*filename = lu(args, "filename");
    if(!strcmp(command,"jpeg")) {
	int quality = (int)(parsePercent(lu(args, "quality"))*100);
	s_image(name, "jpeg", filename, quality);
    } else {
	s_image(name, "png", filename, 0);
    }
    return 0;
}

static int c_outline(map_t*args) 
{
    char*name = lu(args, "name");
    char*format = lu(args, "format");

    readToken();
    if(type != RAWDATA)
	syntaxerror("colon (:) expected");

    s_outline(name, format, text);
    return 0;
}

int fakechar(map_t*args)
{
    char*name = lu(args, "name");
    s_box(name, 0, 0, black, 20, 0);
    return 0;
}

static int c_egon(map_t*args) {return fakechar(args);}
static int c_button(map_t*args) {
    char*name = lu(args, "name");
    s_button(name);
    return 0;
}
static int current_button_flags = 0;
static int c_on_press(map_t*args)
{
    char*position = lu(args, "position");
    char*action = "";
    if(!strcmp(position, "inside")) {
	current_button_flags |= BC_OVERUP_OVERDOWN;
    } else if(!strcmp(position, "outside")) {
	//current_button_flags |= BC_IDLE_OUTDOWN;
	syntaxerror("IDLE_OVERDOWN not supported by SWF");
    } else if(!strcmp(position, "anywhere")) {
	current_button_flags |= /*BC_IDLE_OUTDOWN|*/BC_OVERUP_OVERDOWN|BC_IDLE_OVERDOWN;
    }
    readToken();
    if(type == RAWDATA) {
	action = text;
	s_buttonaction(current_button_flags, action);
	current_button_flags = 0;
    }
    else
	pushBack();
    return 0;
}
static int c_on_release(map_t*args)
{
    char*position = lu(args, "position");
    char*action = "";
    if(!strcmp(position, "inside")) {
	current_button_flags |= BC_OVERDOWN_OVERUP;
    } else if(!strcmp(position, "outside")) {
	current_button_flags |= BC_OUTDOWN_IDLE;
    } else if(!strcmp(position, "anywhere")) {
	current_button_flags |= BC_OVERDOWN_OVERUP|BC_OUTDOWN_IDLE|BC_OVERDOWN_IDLE;
    }
    readToken();
    if(type == RAWDATA) {
	action = text;
	s_buttonaction(current_button_flags, action);
	current_button_flags = 0;
    }
    else
	pushBack();
    return 0;
}
static int c_on_move_in(map_t*args)
{
    char*position = lu(args, "state");
    char*action = "";
    if(!strcmp(position, "pressed")) {
	current_button_flags |= BC_OUTDOWN_OVERDOWN;
    } else if(!strcmp(position, "not_pressed")) {
	current_button_flags |= BC_IDLE_OVERUP;
    } else if(!strcmp(position, "any")) {
	current_button_flags |= BC_OUTDOWN_OVERDOWN|BC_IDLE_OVERUP|BC_IDLE_OVERDOWN;
    }
    readToken();
    if(type == RAWDATA) {
	action = text;
	s_buttonaction(current_button_flags, action);
	current_button_flags = 0;
    }
    else
	pushBack();
    return 0;
}
static int c_on_move_out(map_t*args)
{
    char*position = lu(args, "state");
    char*action = "";
    if(!strcmp(position, "pressed")) {
	current_button_flags |= BC_OVERDOWN_OUTDOWN;
    } else if(!strcmp(position, "not_pressed")) {
	current_button_flags |= BC_OVERUP_IDLE;
    } else if(!strcmp(position, "any")) {
	current_button_flags |= BC_OVERDOWN_OUTDOWN|BC_OVERUP_IDLE|BC_OVERDOWN_IDLE;
    }
    readToken();
    if(type == RAWDATA) {
	action = text;
	s_buttonaction(current_button_flags, action);
	current_button_flags = 0;
    }
    else
	pushBack();
    return 0;
}
static int c_on_key(map_t*args)
{
    char*key = lu(args, "key");
    char*action = "";
    if(strlen(key)==1) {
	/* ascii */
	if(key[0]>=32) {
	    current_button_flags |= 0x4000 + (key[0]*0x200);
	} else {
	    syntaxerror("invalid character: %c"+key[0]);
	    return 1;
	}
    } else {
	/* TODO: 
	   <ctrl-x> = 0x200*(x-'a')
	   esc = = 0x3600
	   space = = 0x4000;
	*/
	syntaxerror("invalid key: %s",key);
    }
    readToken();
    if(type == RAWDATA) {
	action = text;
	s_buttonaction(current_button_flags, action);
	current_button_flags = 0;
    }
    else
	pushBack();
    return 0;
}

static int c_edittext(map_t*args) 
{
 //"name font size width height text="" color=black maxlength=0 variable="" @password=0 @wordwrap=0 @multiline=0 @html=0 @noselect=0 @readonly=0 @autosize=0"},
    char*name = lu(args, "name");
    char*font = lu(args, "font");
    int size = (int)(1024*parsePxOrPercent(font, lu(args, "size")));
    int width = parseTwip(lu(args, "width"));
    int height = parseTwip(lu(args, "height"));
    char*text  = lu(args, "text");
    RGBA color = parseColor(lu(args, "color"));
    int maxlength = parseInt(lu(args, "maxlength"));
    char*variable = lu(args, "variable");
    char*passwordstr = lu(args, "password");
    char*wordwrapstr = lu(args, "wordwrap");
    char*multilinestr = lu(args, "multiline");
    char*htmlstr = lu(args, "html");
    char*noselectstr = lu(args, "noselect");
    char*readonlystr = lu(args, "readonly");
    char*borderstr = lu(args, "border");
    char*autosizestr = lu(args, "autosize");
    char*alignstr = lu(args, "align");
    int align = -1;

    int flags = 0;
    if(!strcmp(passwordstr, "password")) flags |= ET_PASSWORD;
    if(!strcmp(wordwrapstr, "wordwrap")) flags |= ET_WORDWRAP;
    if(!strcmp(multilinestr, "multiline")) flags |= ET_MULTILINE;
    if(!strcmp(readonlystr, "readonly")) flags |= ET_READONLY;
    if(!strcmp(htmlstr, "html")) flags |= ET_HTML;
    if(!strcmp(noselectstr, "noselect")) flags |= ET_NOSELECT;
    if(!strcmp(borderstr, "border")) flags |= ET_BORDER;
    if(!strcmp(autosizestr, "autosize")) flags |= ET_AUTOSIZE;
    if(!strcmp(alignstr, "left") || !*alignstr) align = ET_ALIGN_LEFT;
    else if(!strcmp(alignstr, "right")) align = ET_ALIGN_RIGHT;
    else if(!strcmp(alignstr, "center")) align = ET_ALIGN_CENTER;
    else if(!strcmp(alignstr, "justify")) align = ET_ALIGN_JUSTIFY;
    else syntaxerror("Unknown alignment: %s", alignstr);

    s_edittext(name, font, size, width, height, text, &color, maxlength, variable, flags, align);
    return 0;
}

static int c_morphshape(map_t*args) {return fakechar(args);}
static int c_movie(map_t*args) {return fakechar(args);}

static char* readfile(const char*filename)
{
    FILE*fi = fopen(filename, "rb");
    int l;
    char*text;
    if(!fi) 
	syntaxerror("Couldn't find file %s: %s", filename, strerror(errno));
    fseek(fi, 0, SEEK_END);
    l = ftell(fi);
    fseek(fi, 0, SEEK_SET);
    text = rfx_alloc(l+1);
    fread(text, l, 1, fi);
    text[l]=0;
    fclose(fi);
    return text;
}

static int c_action(map_t*args) 
{
    char* filename  = map_lookup(args, "filename");
    if(!filename ||!*filename) {
	readToken();
	if(type != RAWDATA) {
	    syntaxerror("colon (:) expected");
	}
	s_action(text);
    } else {
	s_action(readfile(filename));
    }
   
    return 0;
}

static int c_initaction(map_t*args) 
{
    char* character = lu(args, "name");
    char* filename  = map_lookup(args, "filename");
    if(!filename ||!*filename) {
	readToken();
	if(type != RAWDATA) {
	    syntaxerror("colon (:) expected");
	}
	s_initaction(character, text);
    } else {
	s_initaction(character, readfile(filename));
    }
   
    return 0;
}

static struct {
    char*command;
    command_func_t* func;
    char*arguments;
} arguments[] =
{{"flash", c_flash, "bbox=autocrop background=black version=6 fps=50 name= filename= @compress=default"},
 {"frame", c_frame, "n=<plus>1 name= @cut=no @anchor=no"},
 // "import" type stuff
 {"swf", c_swf, "name filename"},
 {"shape", c_swf, "name filename"},
 {"jpeg", c_image, "name filename quality=80%"},
 {"png", c_image, "name filename"},
 {"movie", c_movie, "name filename"},
 {"sound", c_sound, "name filename"},
 {"font", c_font, "name filename"},
 {"soundtrack", c_soundtrack, "filename"},
 {"quicktime", c_quicktime, "url"},

    // generators of primitives

 {"point", c_point, "name x=0 y=0"},
 {"gradient", c_gradient, "name @radial=0 rotate=0 scale= scalex= scaley= x= y= width= height= r= shear="}, //extra parameters like .texture
 {"outline", c_outline, "name format=simple"},
 {"textshape", c_textshape, "name font size=100% text"},

    // filters
 {"blur", c_blur, "name blur= blurx= blury= passes=1"},
 {"gradientglow", c_gradientglow, "name gradient blur= blurx= blury= angle=0.0 distance=0.0 strength=1.0 @innershadow=0 @knockout=0 @composite=0 @ontop=0 passes=1"},
 {"dropshadow", c_dropshadow, "name color blur= blurx= blury= angle=0.0 distance=0.0 strength=1.0 @innershadow=0 @knockout=0 @composite=0 passes=1"},
 {"bevel", c_bevel, "name shadow highlight blur= blurx= blury= angle=0.0 distance=0.0 strength=1.0 @innershadow=0 @knockout=0 @composite=0 @ontop=0 passes=1"},

    // character generators
 {"box", c_primitive, "name width height color=white line=1 @fill=none"},
 {"circle", c_primitive, "name r color=white line=1 @fill=none"},
 {"filled", c_primitive, "name outline color=white line=1 @fill=none"},

 {"egon", c_egon, "name vertices color=white line=1 @fill=none"},
 {"text", c_text, "name text font size=100% color=white"},
 {"edittext", c_edittext, "name font= size=100% width height text="" color=white maxlength=0 variable="" @password=0 @wordwrap=0 @multiline=0 @html=0 @noselect=0 @readonly=0 @border=0 @autosize=0 align="},
 {"morphshape", c_morphshape, "name start end"},
 {"button", c_button, "name"},
    {"show", c_show,             "name x=0 y=0 red=+0 green=+0 blue=+0 alpha=+0 luminance= scale= scalex= scaley= blend= filter= pivot= pin= shear= rotate= ratio= above= below= as="},
    {"on_press", c_on_press, "position=inside"},
    {"on_release", c_on_release, "position=anywhere"},
    {"on_move_in", c_on_move_in, "state=not_pressed"},
    {"on_move_out", c_on_move_out, "state=not_pressed"},
    {"on_key", c_on_key, "key=any"},
 
    // control tags
 {"play", c_play, "name loop=0 @nomultiple=0"},
 {"stop", c_stop, "name= "},
 {"nextframe", c_nextframe, "name"},
 {"previousframe", c_previousframe, "name"},

    // object placement tags
 {"put", c_put,             "<i> x=0 y=0 red=+0 green=+0 blue=+0 alpha=+0 luminance= scale= scalex= scaley= blend= filter= pivot= pin= shear= rotate= ratio= above= below="},
 {"startclip", c_startclip, "<i> x=0 y=0 red=+0 green=+0 blue=+0 alpha=+0 luminance= scale= scalex= scaley= blend= filter= pivot= pin= shear= rotate= ratio= above= below="},
 {"change", c_change,   "name x= y= red= green= blue= alpha= luminance= scale= scalex= scaley= blend= filter= pivot= pin= shear= rotate= ratio= above= below="},
 {"arcchange", c_arcchange,   "name pivot= angle= red= green= blue= alpha= luminance= scale= scalex= scaley= blend= filter= pivot= pin= shear= rotate= ratio= above= below="},
 {"qchange", c_qchange, "name x= y= red= green= blue= alpha= luminance= scale= scalex= scaley= blend= filter= pivot= pin= shear= rotate= ratio= above= below="},
 {"jump", c_jump,       "name x= y= red= green= blue= alpha= luminance= scale= scalex= scaley= blend= filter= pivot= pin= shear= rotate= ratio= above= below="},
 {"del", c_del, "name"},
    // virtual object placement
 {"texture", c_texture, "<i> x=0 y=0 width= height= scale= scalex= scaley= r= shear= rotate="},

    // commands which start a block
//startclip (see above)
 {"sprite", c_sprite, "name"},
 {"action", c_action, "filename="},
 {"initaction", c_initaction, "name filename="},

 {"end", c_end, ""}
};


static map_t parseArguments(char*command, char*pattern)
{
    char*x;
    char*d,*e;
   
    string_t name[64];
    string_t value[64];
    int set[64];
    int isboolean[64];
    int pos;
    int len;
    int t;
    string_t t1,t2;
    map_t result;
    map_init(&result);

    string_set(&t1, "commandname");
    string_set(&t2, command);
    map_put(&result, t1, t2);

    if(!pattern || !*pattern)
	return result;

    x = pattern;

    pos = 0;

    if(!strncmp("<i> ", x, 3)) {
	readToken();
	if(type == COMMAND || type == RAWDATA) {
	    pushBack();
	    syntaxerror("character name expected");
	}
	name[pos].str = "instance";
	name[pos].len = 8;
	value[pos].str = text;
	value[pos].len = strlen(text);
	set[pos] = 1;
	pos++;

	if(type == ASSIGNMENT)
	    readToken();

	name[pos].str = "character";
	name[pos].len = 9;
	value[pos].str = text;
	value[pos].len = strlen(text);
	set[pos] = 1;
	pos++;

	x+=4;
    }

    while(*x) {
	isboolean[pos] = (x[0] =='@');
	if(isboolean[pos])
	    x++;

	d = strchr(x, ' ');
	e = strchr(x, '=');
	if(!d)
	    d=&x[strlen(x)];
	set[pos] = 0;

	if(!e || d<e) { 
	    // no default
	    name[pos].str = x;
	    name[pos].len = d-x;
	    value[pos].str = 0;
	    value[pos].len = 0;
	} else {
	    name[pos].str = x;
	    name[pos].len = e-x;
	    value[pos].str = e+1;
	    value[pos].len = d-e-1;
	}
	pos++;
	if(!*d) break;
	x=d+1;
    }
    len = pos;

/*    for(t=0;t<len;t++) {
	printf("(%d) %s=%s %s\n", t, strdup_n(name[t], namelen[t]), strdup_n(value[t], valuelen[t]),
		isboolean[t]?"(boolean)":"");
    }*/

    while(1) {
	readToken();
	if(type == RAWDATA || type == COMMAND) {
	    pushBack();
	    break;
	}

	// first, search for boolean arguments
	for(pos=0;pos<len;pos++)
	{
	    if(!set[pos] && isboolean[pos] && !strncmp(name[pos].str, text, name[pos].len>textlen?name[pos].len:textlen)) {
		set[pos] = 1;
		if(type == ASSIGNMENT)
		    readToken();
		value[pos].str = text;
		value[pos].len = strlen(text);
		/*printf("setting boolean parameter %s (to %s)\n",
			strdup_n(name[pos], namelen[pos]),
			strdup_n(value[pos], valuelen[pos]));*/
		break;
	    }
	}

	// second, search for normal arguments
	if(pos==len)
	for(pos=0;pos<len;pos++)
	{
	    if((type == ASSIGNMENT && !strncmp(name[pos].str, text, name[pos].len>textlen?name[pos].len:textlen)) ||
	       (type != ASSIGNMENT && !set[pos])) {
		if(set[pos]) {
		    syntaxerror("value %s set twice (old value:%s)", text, strdup_n(value[pos].str, value[pos].len));
		}
		if(type == ASSIGNMENT)
		    readToken();
		set[pos] = 1;
		value[pos].str = text;
		value[pos].len = strlen(text);
#if 0//def DEBUG
		printf("setting parameter %s (to %s)\n",
			strdup_n(name[pos].str, name[pos].len),
			strdup_n(value[pos].str, value[pos].len));
#endif
		break;
	    }
	}
	if(pos==len) {
	    syntaxerror("Illegal argument \"%s\" to .%s", text, command);
	}
    }
#if 0//def DEBUG
    for(t=0;t<len;t++) {
	printf("%s=%s\n", strdup_n(name[t].str, name[t].len), strdup_n(value[t].str, value[t].len));
    }
#endif
    for(t=0;t<len;t++) {
	if(value[t].str && value[t].str[0] == '*') {
	    //relative default- take value from some other parameter
	    int s;
	    for(s=0;s<len;s++) {
		if(value[s].len == value[t].len-1 &&
		   !strncmp(&value[t].str[1], value[s].str, value[s].len))
		    value[t].str = value[s].str;
	    }
	}
	if(value[t].str == 0) {
	    pushBack();
	    syntaxerror("value for parameter %s missing (no default)", strdup_n(name[t].str, name[t].len));
	}
    }

    /* ok, now construct the dictionary from the parameters */

    for(t=0;t<len;t++) 
    {
	map_put(&result, name[t], value[t]);
    }
    return result;
}
static void parseArgumentsForCommand(char*command)
{
    int t;
    map_t args;
    int nr = -1;
    msg("<verbose> parse Command: %s (line %d)", command, line);

    for(t=0;t<sizeof(arguments)/sizeof(arguments[0]);t++) {
	if(!strcmp(arguments[t].command, command)) {

	    /* ugly hack- will be removed soon (once documentation and .sc generating
	       utilities have been changed) */
	    if(!strcmp(command, "swf") && !stackpos) {
		warning("Please use .flash instead of .swf- this will be mandatory soon");	
		command = "flash";
		t = 0;
	    }

	    args = parseArguments(command, arguments[t].arguments);
	    nr = t;
	    break;
	}
    }
    if(nr<0)
	syntaxerror("command %s not known", command);
   
    // catch missing .flash directives at the beginning of a file
    if(strcmp(command, "flash") && !stackpos)
    {
	syntaxerror("No movie defined- use .flash first");
    }

#ifdef DEBUG
    printf(".%s\n", command);fflush(stdout);
    map_dump(&args, stdout, "\t");fflush(stdout);
#endif

    (*arguments[nr].func)(&args);

    /*if(!strcmp(command, "button") ||
       !strcmp(command, "action")) {
	while(1) {
	    readToken();
	    if(type == COMMAND) {
		if(!strcmp(text, "end"))
		    break;
		else {
		    pushBack();
		    break;
		}
	    }
	}
    }*/

    map_clear(&args);
    return;
}

int main (int argc,char ** argv)
{ 
    int t;
    processargs(argc, argv);
    initLog(0,-1,0,0,-1,verbose);

    if(!filename) {
	args_callback_usage(argv[0]);
	exit(1);
    }
    
    file = generateTokens(filename);
    if(!file) {
	fprintf(stderr, "parser returned error.\n");
	return 1;
    }
    pos=0;
    t=0;

    while(!noMoreTokens()) {
	readToken();
	if(type != COMMAND)
	    syntaxerror("command expected");
	parseArgumentsForCommand(text);
    }

    s_close();
    freeTokens(file);

    return 0;
}

