/*
 * Tapioca library
 * Copyright (C) 2007 INdT.
 * @author  Abner Jose de Faria Silva <abner.silva@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "tpa-audio-stream.h"
#include "tpa-audio-stream-priv.h"
#include "tpa-channel-priv.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CHANNEL

#include <tapioca/base/tpa-debug.h>

struct _TpaAudioStreamPrivate {
    DBusGProxy *stream_engine_proxy;
    gboolean disposed;
    guint volume;
    gboolean mute_in;
    gboolean mute_out;
};

G_DEFINE_TYPE(TpaAudioStream, tpa_audio_stream, TPA_TYPE_STREAM)

static GObject*
tpa_audio_stream_constructor (GType type,
                              guint n_construct_params,
                              GObjectConstructParam *construct_params)
{
    GObject *object;
    TpaAudioStream *self;
    DBusGConnection *bus;
    GError *error = NULL;

    object = G_OBJECT_CLASS (tpa_audio_stream_parent_class)->constructor
                            (type, n_construct_params, construct_params);
    self = TPA_AUDIO_STREAM (object);

    bus = dbus_g_bus_get (DBUS_BUS_SESSION, &error);

    if (!bus) {
        ERROR ("failed to open connection to dbus");
        g_error_free (error);
        return NULL;
    }

    self->priv->stream_engine_proxy = dbus_g_proxy_new_for_name (bus,
                                        "org.freedesktop.Telepathy.StreamEngine",
                                        "/org/freedesktop/Telepathy/StreamEngine",
                                        "org.freedesktop.Telepathy.StreamEngine");
    tpa_object_add_proxy (TPA_OBJECT (self), self->priv->stream_engine_proxy);

    return object;
}

static void
tpa_audio_stream_dispose (GObject *object)
{
    TpaAudioStream *self = TPA_AUDIO_STREAM (object);

    if (self->priv->disposed)
       /* If dispose did already run, return. */
       return;

    if (self->priv->stream_engine_proxy)
        g_object_unref (self->priv->stream_engine_proxy);

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    G_OBJECT_CLASS (tpa_audio_stream_parent_class)->dispose (object);
}

static void
tpa_audio_stream_class_init (TpaAudioStreamClass *klass)
{
    GObjectClass *gobject_class;

    gobject_class = (GObjectClass *) klass;
    tpa_audio_stream_parent_class = g_type_class_peek_parent (klass);

    g_type_class_add_private (klass, sizeof (TpaAudioStreamPrivate));

    gobject_class->dispose = tpa_audio_stream_dispose;
    gobject_class->constructor = tpa_audio_stream_constructor;
}

static void
tpa_audio_stream_init (TpaAudioStream *self)
{
    self->priv = TPA_AUDIO_STREAM_GET_PRIVATE (self);
    self->priv->stream_engine_proxy = NULL;
    self->priv->volume = 50;
    self->priv->mute_in = FALSE;
    self->priv->mute_out = FALSE;
    self->priv->disposed = FALSE;
}

TpaAudioStream *
tpa_audio_stream_new (TpaChannel *channel,
                      guint id,
                      guint contact,
                      guint type,
                      guint state,
                      guint direction,
                      guint flags)
{
    TpaAudioStream *self = NULL;

    g_return_val_if_fail (channel != NULL, NULL);

    /* Pass givin arguments to constructor */
    self = TPA_AUDIO_STREAM (g_object_new (TPA_TYPE_AUDIO_STREAM,
                                           "channel", channel,
                                           "id", id,
                                           "contact", contact,
                                           "type", type,
                                           "state", state,
                                           "direction", direction,
                                           "flags", flags,
                                           NULL));

    return self;
}

TpaAudioStream *
tpa_audio_stream_new_incoming (TpaChannel *channel,
                               guint id,
                               guint contact,
                               guint type)
{
    TpaAudioStream *self = NULL;

    g_return_val_if_fail (channel != NULL, NULL);

    /* Pass givin arguments to constructor */
    self = TPA_AUDIO_STREAM (g_object_new (TPA_TYPE_AUDIO_STREAM,
                                           "channel", channel,
                                           "id", id,
                                           "contact", contact,
                                           "type", type,
                                           NULL));

    return self;
}

/**
 * tpa_audio_stream_pause:
 * @self: #TpaAudioStream instance.
 */
void
tpa_audio_stream_set_volume (TpaAudioStream *self,
                             guint volume)
{
    const gchar *path;
    GError *error = NULL;

    VERBOSE ("(%p, %d)", self,  volume);
    g_assert (self);

    if (self->priv->volume != volume) {
        path = tpa_channel_get_object_path (TPA_CHANNEL (tpa_stream_get_channel (TPA_STREAM (self))));
        dbus_g_proxy_call (self->priv->stream_engine_proxy, "SetOutputVolume", &error,
                           DBUS_TYPE_G_OBJECT_PATH, path,
                           G_TYPE_UINT, tpa_stream_get_id (TPA_STREAM (self)),
                           G_TYPE_UINT, volume,
                           G_TYPE_INVALID, G_TYPE_INVALID);
        if (error) {
            ERROR ("%s", error->message);
            g_error_free (error);
        }
        else
            self->priv->volume = volume;
    }

    VERBOSE ("return");
}

guint
tpa_audio_stream_get_volume (TpaAudioStream *self)
{
    VERBOSE ("(%p)", self);

    g_assert (self);

    VERBOSE ("return");

    return self->priv->volume;
}

void
tpa_audio_stream_mute_in (TpaAudioStream *self,
                          gboolean mute)
{
    const gchar *path;
    GError *error = NULL;

    VERBOSE ("(%p, %d)", self, mute);
    g_assert (self);

    if (self->priv->mute_in != mute) {
        path = tpa_channel_get_object_path (TPA_CHANNEL (tpa_stream_get_channel (TPA_STREAM (self))));
        dbus_g_proxy_call (self->priv->stream_engine_proxy, "MuteInput", &error,
                           DBUS_TYPE_G_OBJECT_PATH, path,
                           G_TYPE_UINT, tpa_stream_get_id (TPA_STREAM (self)),
                           G_TYPE_BOOLEAN, mute,
                           G_TYPE_INVALID, G_TYPE_INVALID);
        if (error) {
            ERROR ("%s", error->message);
            g_error_free (error);
        }
        else
            self->priv->mute_in = mute;
    }

    VERBOSE ("return");
}

void
tpa_audio_stream_mute_out (TpaAudioStream *self,
                          gboolean mute)
{
    const gchar *path;
    GError *error = NULL;

    VERBOSE ("(%p, %d)", self, mute);
    g_assert (self);

    if (self->priv->mute_out != mute) {
        path = tpa_channel_get_object_path (TPA_CHANNEL (tpa_stream_get_channel (TPA_STREAM (self))));
        dbus_g_proxy_call (self->priv->stream_engine_proxy, "MuteOutput", &error,
                           DBUS_TYPE_G_OBJECT_PATH, path,
                           G_TYPE_UINT, tpa_stream_get_id (TPA_STREAM (self)),
                           G_TYPE_BOOLEAN, mute,
                           G_TYPE_INVALID, G_TYPE_INVALID);
        if (error) {
            ERROR ("%s", error->message);
            g_error_free (error);
        }
        else
            self->priv->mute_out = mute;
    }

    VERBOSE ("return");
}
