/*
 * Tapioca library
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>

#include "tpa-channel-priv.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CHANNEL

#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-channel-bindings.h>
#include <tapioca/base/tpa-signals-marshal.h>

struct _TpaChannelPrivate {
    DBusGProxy *proxy;
    TpaUserContact *user;
    TpaContactGroup *group;
    TpaChannelTarget *target;
    TpaChannelType type;
    gboolean join;
    gboolean disposed;
};

enum
{
    CLOSED,
    LAST_SIGNAL
};

enum
{
    ARG_0,
    ARG_PROXY,
    ARG_USER,
    ARG_TARGET,
    ARG_TYPE
};

static guint tpa_channel_signals[LAST_SIGNAL] = { 0 };

G_DEFINE_TYPE(TpaChannel, tpa_channel, TPA_TYPE_OBJECT)

static void
proxy_closed_channel_signal (DBusGProxy *proxy,
                             gpointer user_data)
{
    TpaChannel *self = TPA_CHANNEL (user_data);

    g_assert (self);

    g_signal_emit (self, tpa_channel_signals[CLOSED], 0);
}

static GObject*
tpa_channel_constructor (GType type,
                         guint n_construct_params,
                         GObjectConstructParam *construct_params)
{
    GObject *object;
    TpaChannel *self;

    object = G_OBJECT_CLASS (tpa_channel_parent_class)->constructor
                            (type, n_construct_params, construct_params);

    self = TPA_CHANNEL (object);

    tpa_object_add_proxy (TPA_OBJECT (self), self->priv->proxy);

    tpa_object_connect_signal (TPA_OBJECT (self),
                               TPA_INTERFACE_CHANNEL,
                               "Closed",
                               G_CALLBACK (proxy_closed_channel_signal),
                               self);

    return object;
}

static void
tpa_channel_set_property (GObject *object,
                          guint prop_id,
                          const GValue *value,
                          GParamSpec *pspec)
{
    TpaChannel *self = TPA_CHANNEL (object);

    switch (prop_id) {
        case ARG_PROXY:
          self->priv->proxy = g_value_get_pointer (value);
          break;
         case ARG_USER:
          self->priv->user = g_value_get_object (value);
          break;
        case ARG_TARGET:
          self->priv->target = g_value_get_object (value);
          break;
        case ARG_TYPE:
          self->priv->type = g_value_get_uint (value);
          break;
        default:
          G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
          break;
    }
}

static void
tpa_channel_get_property (GObject *object,
                          guint prop_id,
                          GValue *value,
                          GParamSpec *pspec)
{
    TpaChannel *self = TPA_CHANNEL (object);

    switch (prop_id) {
        case ARG_PROXY:
          g_value_set_pointer (value, self->priv->proxy);
          break;
        case ARG_USER:
          g_value_set_object (value, self->priv->user);
          break;
        case ARG_TARGET:
          g_value_set_object (value, self->priv->target);
          break;
        case ARG_TYPE:
          g_value_set_uint (value, self->priv->type);
          break;
        default:
          G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
          break;
    }
}

static void
tpa_channel_dispose (GObject *object)
{
    TpaChannel *self = TPA_CHANNEL (object);

    if (self->priv->disposed)
       /* If dispose did already run, return. */
       return;

    /* Free proxy */
    if (self->priv->proxy)
        g_object_unref (self->priv->proxy);

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    VERBOSE ("channel %p disposed", object);

    G_OBJECT_CLASS (tpa_channel_parent_class)->dispose (object);
}

static void
tpa_channel_class_init (TpaChannelClass *klass)
{
    GObjectClass *gobject_class;

    gobject_class = (GObjectClass *) klass;
    tpa_channel_parent_class = g_type_class_peek_parent (klass);

    g_type_class_add_private (klass, sizeof (TpaChannelPrivate));

    gobject_class->dispose = tpa_channel_dispose;
    gobject_class->constructor = tpa_channel_constructor;
    gobject_class->get_property = tpa_channel_get_property;
    gobject_class->set_property = tpa_channel_set_property;

    g_object_class_install_property (gobject_class,
                                     ARG_PROXY,
                                     g_param_spec_pointer ("proxy",
                                     "proxy",
                                     "proxy",
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_WRITABLE));

    g_object_class_install_property (gobject_class,
                                     ARG_USER,
                                     g_param_spec_object ("user",
                                     "user",
                                     "user",
                                     TPA_TYPE_USER_CONTACT,
                                     G_PARAM_CONSTRUCT | G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     ARG_TARGET,
                                     g_param_spec_object ("target",
                                     "target",
                                     "target",
                                     TPA_TYPE_CHANNEL_TARGET,
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_WRITABLE));

    g_object_class_install_property (gobject_class,
                                     ARG_TYPE,
                                     g_param_spec_uint ("type",
                                     "type",
                                     "type",
                                     TPA_CHANNEL_TYPE_TEXT,
                                     TPA_CHANNEL_TYPE_STREAM,
                                     TPA_CHANNEL_TYPE_TEXT,
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_WRITABLE));

    tpa_channel_signals[CLOSED] = g_signal_new ("closed",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL,
        NULL,
        g_cclosure_marshal_VOID__VOID,
        G_TYPE_NONE,
        0);
}

static void
tpa_channel_init (TpaChannel *self)
{
    self->priv = TPA_CHANNEL_GET_PRIVATE (self);
    self->priv->target = NULL;
    self->priv->proxy = NULL;
    self->priv->user = NULL;
    self->priv->disposed = FALSE;
}

/**
 * tpa_channel_new:
 * @service: Service name.
 * @path: Object path.
 * @user: #TpaUserContact instance.
 * @type: #TpaChannelType of the channel.
 * @target: #TpaChannelTarget instance.
 *
 * Creates a new instance of #TpaChannel.
 */
TpaChannel *
tpa_channel_new (const gchar *service,
                 const gchar *path,
                 TpaUserContact *user,
                 TpaChannelType type,
                 TpaChannelTarget *target)
{
    TpaChannel *self;

    g_return_val_if_fail (service != NULL, NULL);
    g_return_val_if_fail (path != NULL, NULL);
    g_return_val_if_fail (user != NULL, NULL);
    g_return_val_if_fail (target != NULL, NULL);

    self = TPA_CHANNEL (g_object_new (TPA_TYPE_CHANNEL,
                        "service", service,
                        "path", path,
                        "user", user,
                        "type", type,
                        "target", target,
                        NULL));

    return self;
}

/**
 * tpa_channel_get_target:
 * @self: #TpaChannel instance.
 * @returns: #TpaChannelTarget target.
 *
 * Get target of the channel.
 */
TpaChannelTarget *
tpa_channel_get_target (TpaChannel *self)
{
    VERBOSE ("(%p)", self);

    g_assert (self);

    VERBOSE ("return %p", self->priv->target);
    return g_object_ref (self->priv->target);
}

/**
 * tpa_channel_get_channel_type:
 * @self: #TpaChannel instance.
 * @returns: #TpaChannelType type.
 *
 * Get type of the channel.
 */
TpaChannelType
tpa_channel_get_channel_type (TpaChannel *self)
{
    VERBOSE ("(%p)", self);

    g_assert (self);

    VERBOSE ("return %d", self->priv->type);
    return self->priv->type;
}

/**
 * tpa_channel_get_owner:
 * @self: #TpaChannel instance.
 */
TpaUserContact *
tpa_channel_get_owner (TpaChannel *self)
{
    VERBOSE ("(%p)", self);

    g_assert (self);

    VERBOSE ("return %p", self->priv->user);
    return g_object_ref (self->priv->user);
}

/**
 * tpa_channel_close:
 * @self: #TpaChannel instance.
 */
void
tpa_channel_close (TpaChannel *self)
{
    GError *error = NULL;

    VERBOSE ("(%p)", self);

    g_assert (self);

    /* Close telepathy channel */
    if (!org_freedesktop_Telepathy_Channel_close (self->priv->proxy, &error) || error)
    {
        if(error)
        {
            ERROR ("%s", error->message);
            g_error_free (error);
        }
    }

    VERBOSE ("return");
}

void
tpa_channel_join (TpaChannel *self)
{
    VERBOSE ("(%p)", self);
    g_assert (self);

    VERBOSE ("return %s", self->priv->join ? "true" : "false");
    self->priv->join = TRUE;
}

gboolean
tpa_channel_has_joined (TpaChannel *self)
{
    VERBOSE ("(%p)", self);
    g_assert (self);

    VERBOSE ("return %s", self->priv->join ? "true" : "false");
    return self->priv->join;
}

gboolean
tpa_channel_has_group_support (TpaChannel *self)
{
    gboolean ret;

    VERBOSE ("(%p)", self);
    g_assert (self);

    ret = (self->priv->group != NULL);
    VERBOSE ("return %s", ret ? "true" : "false");
    return ret;
}

TpaContactGroup *
tpa_channel_get_contact_group (TpaChannel *self)
{
    VERBOSE ("(%p)", self);
    g_assert (self);

    VERBOSE ("return %p", self->priv->group);
    return g_object_ref (self->priv->group);
}

const gchar *
tpa_channel_get_object_path (TpaChannel *self)
{
    const gchar *object_path;
    VERBOSE ("(%p)", self);
    g_assert (self);

    object_path = dbus_g_proxy_get_path (self->priv->proxy);

    VERBOSE ("return %s", object_path);
    return object_path;
}
