/*
 * Tapioca library
 * Copyright (C) 2006 INdT.
 * @author  Abner Jose de Faria Silva <abner.silva@indt.org.br>
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with self library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>

#include "tpa-manager-factory.h"
#include "tpa-manager-priv.h"
#include "tpa-thread.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CONNECTION_MANAGER

#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-connection-manager-bindings.h>
#include <tapioca/base/tpa-signals-marshal.h>

#include <dbus/dbus-glib-bindings.h>

#define _MANAGER_PATH "/org/freedesktop/Telepathy/ConnectionManager"

struct _TpaManagerFactoryPrivate {
    DBusGConnection *bus;
    DBusGProxy *proxy;
    GSList *paths;
    gboolean disposed;
};

static void tpa_manager_factory_load (TpaManagerFactory *self);

G_DEFINE_TYPE(TpaManagerFactory, tpa_manager_factory, G_TYPE_OBJECT)

static gboolean tpa_initialized = FALSE;
static TpaManagerFactory *factory;
static GHashTable *cms = NULL;

static GObject*
tpa_manager_factory_constructor (GType type,
                                 guint n_construct_params,
                                 GObjectConstructParam *construct_params)
{
    GObject *object;
    TpaManagerFactory *self;
    DBusGConnection *bus;
    GError *error = NULL;


    if (!tpa_initialized) {
        tpa_initialized = TRUE;

        /* register our own marshallers */
        dbus_g_object_register_marshaller (tpa_marshal_VOID__STRING_STRING_STRING,
            G_TYPE_NONE, G_TYPE_STRING, DBUS_TYPE_G_OBJECT_PATH, G_TYPE_STRING, G_TYPE_INVALID);
        dbus_g_object_register_marshaller (tpa_marshal_VOID__STRING_STRING_UINT_UINT_BOOLEAN,
            G_TYPE_NONE, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_BOOLEAN, G_TYPE_INVALID);
        dbus_g_object_register_marshaller (tpa_marshal_VOID__UINT_UINT,
            G_TYPE_NONE, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_INVALID);
        dbus_g_object_register_marshaller (tpa_marshal_VOID__UINT_UINT_STRING,
            G_TYPE_NONE, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_STRING, G_TYPE_INVALID);
        dbus_g_object_register_marshaller (tpa_marshal_VOID__UINT_UINT_UINT_UINT_UINT_STRING,
            G_TYPE_NONE, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_STRING, G_TYPE_INVALID);
        dbus_g_object_register_marshaller (tpa_marshal_VOID__UINT_UINT_UINT_STRING,
            G_TYPE_NONE, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_STRING, G_TYPE_INVALID);
        dbus_g_object_register_marshaller (tpa_marshal_VOID__UINT_UINT,
            G_TYPE_NONE, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_INVALID);
        dbus_g_object_register_marshaller (tpa_marshal_VOID__STRING_BOXED_BOXED_BOXED_BOXED_UINT_UINT,
            G_TYPE_NONE, G_TYPE_STRING, G_TYPE_BOXED, G_TYPE_BOXED,
            G_TYPE_BOXED, G_TYPE_BOXED, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_INVALID);
        dbus_g_object_register_marshaller (tpa_marshal_VOID__STRING_STRING_UINT_UINT_BOOLEAN,
            G_TYPE_NONE,
            DBUS_TYPE_G_OBJECT_PATH, G_TYPE_STRING, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_BOOLEAN, G_TYPE_INVALID);
        dbus_g_object_register_marshaller (tpa_marshal_VOID__UINT_STRING,
            G_TYPE_NONE, G_TYPE_UINT, G_TYPE_STRING, G_TYPE_INVALID);
        dbus_g_object_register_marshaller (tpa_marshal_VOID__UINT_UINT_UINT,
            G_TYPE_NONE, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_INVALID);

        if (!cms) {
            cms = g_hash_table_new_full (g_str_hash,
                                         g_str_equal,
                                         g_free,
                                         g_object_unref);
        }

        bus = tpa_thread_get_bus ();
        if (!bus) {
            ERROR ("failed to open connection to dbus");
            return NULL;
        }

        object = G_OBJECT_CLASS (tpa_manager_factory_parent_class)->constructor 
                                (type, n_construct_params, construct_params);

        self = TPA_MANAGER_FACTORY (object);
        self->priv->bus = bus;
        self->priv->proxy = dbus_g_proxy_new_for_name (bus,
            DBUS_SERVICE_DBUS, DBUS_PATH_DBUS, DBUS_INTERFACE_DBUS);

        if (error)
            g_error_free (error);

        tpa_manager_factory_load (self);
        factory = self;
    }
    else
        object = g_object_ref (factory);

    return object;
}

static void
tpa_manager_factory_dispose (GObject *object)
{
    TpaManagerFactory *self = TPA_MANAGER_FACTORY (object);

    if (self->priv->disposed)
       /* If dispose did already run, return. */
       return;

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    if (cms) {
        g_hash_table_destroy (cms);
        cms = NULL;
    }

    /* Free proxy */
    if (self->priv->proxy)
        g_object_unref (self->priv->proxy);

    G_OBJECT_CLASS (tpa_manager_factory_parent_class)->dispose (object);
}

static void
tpa_manager_factory_class_init (TpaManagerFactoryClass *klass)
{
    GObjectClass *gobject_class;

    gobject_class = (GObjectClass *) klass;
    tpa_manager_factory_parent_class = g_type_class_peek_parent (klass);

    g_type_class_add_private (klass, sizeof (TpaManagerFactoryPrivate));

    gobject_class->constructor = tpa_manager_factory_constructor;
    gobject_class->dispose = tpa_manager_factory_dispose;
}

static void
tpa_manager_factory_init (TpaManagerFactory *self)
{
    self->priv = TPA_MANAGER_FACTORY_GET_PRIVATE (self);
    self->priv->disposed = FALSE;
    self->priv->bus = NULL;
    self->priv->proxy = NULL;
    self->priv->paths = NULL;
}

static void
_get_all (gpointer key,
          gpointer value,
          gpointer user_data)
{
    g_ptr_array_add (user_data, value);
}

static void
tpa_manager_factory_load_manager_files (TpaManagerFactory *self,
                                        GSList *paths)
{
    TpaManager *manager;
    GSList *tmp;
    GDir *dir;
    const gchar *filename;

    /* Load installed CMs files */
    for (tmp = paths; tmp; tmp = tmp->next) {
        gchar *path = g_strdup_printf ("%s/managers", (gchar *) tmp->data);

        VERBOSE ("### Reading %s ###", path);
        dir = g_dir_open (path, 0, NULL);
        if (dir == NULL)
            continue;

        while ((filename = g_dir_read_name (dir)) != NULL) {
            if (g_str_has_suffix (filename, ".manager")) {
                gchar *name = g_strndup (filename, strlen (filename) - strlen (".manager"));
                gchar *file = g_build_filename (path, filename, NULL);

                VERBOSE ("Found %s", file);
                if (!g_hash_table_lookup (cms, name)) {
                    manager = tpa_manager_new_from_file (file);
                    if (manager) {
                        g_hash_table_insert (cms, name, manager);
                        VERBOSE ("%s loaded", file);
                    }
                }
                g_free (file);
            }
        }
        g_dir_close (dir);
    }
}

static void
tpa_manager_factory_load_profile_files (TpaManagerFactory *self,
                                        GSList *paths)
{
    TpaManager *manager;
    GSList *tmp;
    GDir *dir;
    const gchar *filename;
    GError *error = NULL;

    /* Load installed CMs files */
    for (tmp = paths; tmp; tmp = tmp->next) {
        gchar *path = g_strdup_printf ("%s/profiles", (gchar *) tmp->data);

        VERBOSE ("### Reading %s ###", path);
        dir = g_dir_open (path, 0, NULL);
        if (dir == NULL)
            continue;

        while ((filename = g_dir_read_name (dir)) != NULL) {
            if (g_str_has_suffix (filename, ".profile")) {
                gchar *file = g_build_filename (path, filename, NULL);
                GKeyFile *key_file = g_key_file_new ();

                VERBOSE ("Found %s", file);
                if (g_key_file_load_from_file (key_file,
                                   file,
                                   G_KEY_FILE_NONE,
                                   &error)) {
                    if (g_key_file_has_group (key_file, "Profile")) {
                        gchar *name = g_key_file_get_value (key_file, "Profile", "Manager", &error);

                        if ((manager = g_hash_table_lookup (cms, name))) {
                            tpa_manager_add_profile (manager, file);
                            VERBOSE ("%s loaded", file);
                        }

                        if (name)
                            g_free (name);
                    }
                    g_key_file_free (key_file);
                    g_free (file);
                }
            }
        }
        g_dir_close (dir);
    }
}

/**
 * tpa_manager_factory_load:
 * @self: #TpaManagerFactory instance
 *
 * Internal function to load connection managers.
 *
 * Priority is:
 *      1. DBus bus.
 *      2. Files under $TELEPATHY_DATA_PATH/managers.
 *      3. Files under $HOME/.telepathy/managers.
 *      4. Files under /usr/local/share/telepathy/managers.
 *      5. Files under /usr/share/telepathy/managers.
 */
static void
tpa_manager_factory_load (TpaManagerFactory *self)
{
    TpaManager *manager = NULL;
    guint i;
    GSList *paths = NULL;
    const gchar *path = g_getenv ("TELEPATHY_DATA_PATH");
    gchar **services;
    GError *error = NULL;

    g_assert (self);

    VERBOSE ("Loading...");

    /* Load active CMs */
    VERBOSE ("### Scanning bus ###");
    if (dbus_g_proxy_call (self->priv->proxy, "ListNames", &error,
        G_TYPE_INVALID, G_TYPE_STRV, &services, G_TYPE_INVALID))
    {
        for (i = 0; services[i]; i++) {
            if (g_str_has_prefix (services[i], TPA_INTERFACE_CM)) {
                VERBOSE ("Found %s", services[i]);
                gchar *name = g_strdup (services[i] + strlen (TPA_INTERFACE_CM) + strlen ("."));
                gchar *object_path = g_strdup_printf ("%s/%s", _MANAGER_PATH, name);

                if (!g_hash_table_lookup (cms, name)) {
                    manager = tpa_manager_new_from_bus (services[i], object_path, (const gchar **)services);
                    if (manager) {
                        g_hash_table_insert (cms, name, manager);
                        VERBOSE ("%s loaded", name);
                    }
                }
            }
        }
        g_strfreev (services);
    }

    /* Includes paths to look for installed CMs */
    if (path) {
        gchar **env_paths = g_strsplit (path, ":", 0);

        for (i = 0; i < g_strv_length (env_paths); ++i)
            paths = g_slist_append (paths, env_paths[i]);
    }
    else {
        paths = g_slist_append (paths, g_strdup_printf ("%s/.telepathy", g_get_home_dir()));
        paths = g_slist_append (paths, g_strdup ("/usr/local/share/telepathy"));
        paths = g_slist_append (paths, g_strdup ("/usr/share/telepathy"));
    }

    tpa_manager_factory_load_manager_files (self, paths);
    tpa_manager_factory_load_profile_files (self, paths);

    VERBOSE ("Done");
    g_slist_free (paths);
}

/**
 * tpa_manager_factory_new:
 * @returns: #TpaManagerFactory instance even if no connection manager
 * are available or NULL if fail.
 *
 * Ctor of #TpaManagerFactory upon its call all available connection
 * managers are instantiated. It will try to instantiate managers
 * that do offer a .manager an those that dont but are already
 * activated in dbus session. For those that are not actived
 * yet they will not be activated until the user perform some
 * operation on #TpaManager instance that represents it.
 *
 */
TpaManagerFactory *
tpa_manager_factory_new ()
{
    TpaManagerFactory *self;

    VERBOSE ("");

    self = TPA_MANAGER_FACTORY (g_object_new (TPA_TYPE_MANAGER_FACTORY, NULL));

    VERBOSE ("return %p", self);
    return self;
}

/**
 * tpa_manager_factory_get_all_managers:
 * @self: #TpaManagerFactory instance
 * @returns: #GPtrArray containing #TpaManager instances or NULL
 * if no connection manager is available.
 *
 * Get all connection manager available.
 *
 */
GPtrArray *
tpa_manager_factory_get_all_managers (TpaManagerFactory *self)
{
    GPtrArray *managers = NULL;

    VERBOSE ("(%p)", self);

    g_assert (self);

    managers = g_ptr_array_new ();

    if (cms)
        g_hash_table_foreach (cms, _get_all, managers);

    return managers;
}

/**
 * tpa_manager_factory_get_all_profiles:
 * @self: #TpaManagerFactory instance
 * @returns: #GPtrArray containing protocol string or NULL
 * if no protocol is available.
 *
 * Get all protocols available.
 *
 */
GPtrArray *
tpa_manager_factory_get_all_profiles (TpaManagerFactory *self)
{
    GPtrArray *managers = NULL;
    GPtrArray *protocols = NULL;
    guint i = 0;

    VERBOSE ("(%p)", self);

    g_assert (self);

    managers = g_ptr_array_new ();
    protocols = g_ptr_array_new ();

    if (cms)
        g_hash_table_foreach (cms, _get_all, managers);

    for (i = 0; i < managers->len; i++) {
        GPtrArray *tmp = NULL;
        guint j = 0;
        TpaManager *manager = g_ptr_array_index (managers, i);

        tmp = tpa_manager_get_supported_profiles (manager);
        for (j = 0; j < tmp->len; j++) {
            guint k = 0;
            gboolean found = FALSE;
            for (k = 0; k < protocols->len; k++) {
                if (g_str_equal (g_ptr_array_index (tmp, j),
                    g_ptr_array_index (protocols, k))) {
                    found = TRUE;
                    break;
                }
            }

            if (!found)
                g_ptr_array_add (protocols, g_ptr_array_index (tmp, j));
            else
                g_free (g_ptr_array_index (tmp, j));
        }

        g_ptr_array_free (tmp, FALSE);
    }

    g_ptr_array_free (managers, FALSE);

    return protocols;
}

/**
 * tpa_manager_factory_reload:
 * @self: #TpaManagerFactory instance.
 *
 * Check whether new managers are available reloading all.
 *
 * Note: Managers that has no refences will be disposed.
 *
 */
void
tpa_manager_factory_reload (TpaManagerFactory *self)
{
    GPtrArray *managers = NULL;
    guint i = 0;

    VERBOSE ("(%p)", self);

    managers = tpa_manager_factory_get_all_managers (self);

    /* Remove all managers previous loaded */
    for (i = 0; i < managers->len; ++i)
        g_hash_table_remove (cms, g_ptr_array_index (managers, i));

    tpa_manager_factory_load (self);
    g_ptr_array_free (managers, TRUE);
}

/**
 * tpa_manager_factory_get_managers:
 * @self: #TpaManagerFactory instance
 * @protocol: The name of protocol
 * @returns: #GPtrArray containing #TpaManager instances or NULL
 * if no connection manager is available.
 *
 * Get all connection manager that offer the given protocol.
 *
 */
GPtrArray *
tpa_manager_factory_get_managers (TpaManagerFactory *self,
                                  const gchar *protocol)
{
    GPtrArray *managers = NULL;
    TpaManager *manager = NULL;
    guint i = 0;

    VERBOSE ("(%p, %s)", self, protocol);

    g_assert (self);
    g_return_val_if_fail (protocol != NULL, NULL);

    managers = tpa_manager_factory_get_all_managers (self);

    for (i = 0; i < managers->len; ++i) {
        manager = g_ptr_array_index (managers, i);
        if (!tpa_manager_supports (manager, protocol)) {
            g_ptr_array_remove_index (managers, i);
            i--;
        }
    }

    return managers;
}

/**
 * tpa_manager_factory_get_manager:
 * @self: #TpaManagerFactory instance
 * @protocol: The name of protocol
 * @returns: #TpaManager instance or NULL if none of the available
 * connection manager offer the given protocol.
 *
 * Get first connection manager reference that offer the given protocol.
 *
 */
TpaManager *
tpa_manager_factory_get_manager (TpaManagerFactory *self,
                                 const gchar *protocol)
{
    GPtrArray *managers = NULL;
    TpaManager *manager = NULL;
    guint i = 0;

    VERBOSE ("(%p, %s)", self, protocol);

    g_assert (self);
    g_return_val_if_fail (protocol != NULL, NULL);

    managers = tpa_manager_factory_get_all_managers (self);

    for (i = 0; i < managers->len; i++) {
        manager = g_ptr_array_index (managers, i);
        if (tpa_manager_supports (manager, protocol))
            break;
    }

    g_ptr_array_free (managers, FALSE);

    /* Return a reference */
    return g_object_ref (manager);
}

/**
 * tpa_manager_factory_get_manager_by_name:
 * @self: #TpaManagerFactory instance.
 * @name: The connection manager name.
 * @returns: #TpaManager reference.
 *
 * Get the connection manager reference that matches with given name.
 *
 */
TpaManager *
tpa_manager_factory_get_manager_by_name (TpaManagerFactory *self,
                                         const gchar *name)
{
    TpaManager *manager = NULL;

    VERBOSE ("(%p, %s)", self, name);

    g_assert (self);
    g_return_val_if_fail (name != NULL, NULL);

    if (cms)
        manager = g_object_ref (g_hash_table_lookup (cms, name));

    /* Return a reference */
    return manager;
}
