/*
 * Tapioca library
 * Copyright (C) 2006 INdT.
 * @author  Abner Jose de Faria Silva <abner.silva@indt.org.br>
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with self library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <stdlib.h>

#include "tpa-manager.h"
#include "tpa-profile-priv.h"
#include "tpa-connection-priv.h"
#include "tpa-thread.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CONNECTION_MANAGER

#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-connection-manager-bindings.h>
#include <tapioca/base/tpa-signals-marshal.h>

#include <dbus/dbus-glib-bindings.h>

/* Signals */
enum {
    NEW_CONNECTION,
    LAST_SIGNAL
};

/* Property */
enum {
    ARG_0,
    ARG_FILE,
    ARG_SERVICES
};

struct _TpaManagerPrivate {
    DBusGProxy *proxy;
    GHashTable *connections;
    GHashTable *profiles;
    gchar *file;
    gchar **services;
    gchar *name;
    gboolean running;
    gboolean disposed;
};

G_DEFINE_TYPE(TpaManager, tpa_manager, TPA_TYPE_OBJECT)

static guint tpa_manager_signals[LAST_SIGNAL] = { 0 };

static void
proxy_status_changed_signal (TpaConnection *connection,
                             guint status,
                             guint reason,
                             gpointer user_data)
{
    TpaManager *self = TPA_MANAGER (user_data);
    const gchar *path;

    if (status == TPA_CONNECTION_STATUS_DISCONNECTED) {
        path = tpa_connection_get_path (connection);
        if (g_hash_table_lookup (self->priv->connections, path)) {
            g_hash_table_remove (self->priv->connections, path);
            INFO ("%s unloaded", path);
        }
    }
}

static void
proxy_new_connection_signal (DBusGProxy *proxy,
                             const gchar *service,
                             const gchar *obj_path,
                             const gchar *protocol,
                             gpointer user_data)
{
    TpaManager *self = TPA_MANAGER (user_data);
    TpaConnection *connection;

    g_assert (self);

    if (!(connection = g_hash_table_lookup (self->priv->connections, obj_path))) {
        connection = tpa_connection_new (service, obj_path, protocol);
        g_hash_table_insert (self->priv->connections, g_strdup (obj_path), connection);
        g_signal_connect (connection, "status-changed", G_CALLBACK (proxy_status_changed_signal), self);
        INFO ("[new-connection] %s", obj_path);
        g_signal_emit (self, tpa_manager_signals[NEW_CONNECTION], 0, connection);
    }
    else
       INFO ("%s already loaded", obj_path);
}

static void
tpa_manager_create_proxy (TpaManager *self,
                          const gchar *service,
                          const gchar *path)
{
    DBusGProxy *proxy;
    DBusGConnection *bus;

    g_return_if_fail (service != NULL);
    g_return_if_fail (path != NULL);

    bus = tpa_thread_get_bus ();
    if (!bus) {
        ERROR ("failed to open connection to dbus");
        return;
    }

    proxy = dbus_g_proxy_new_for_name (bus,
                                       service,
                                       path,
                                       TPA_INTERFACE_CM);

    tpa_object_add_proxy (TPA_OBJECT (self), proxy);

    /* Connect all DBus signals */
    tpa_object_connect_signal (TPA_OBJECT (self),
                               TPA_INTERFACE_CM,
                               "NewConnection",
                               G_CALLBACK (proxy_new_connection_signal),
                               self);

    self->priv->proxy = tpa_object_get_proxy (TPA_OBJECT (self), TPA_INTERFACE_CM);
    g_object_unref (proxy);
}

static void
tpa_manager_parse_protocol_parameters (TpaManager *self,
                                       GKeyFile *key_file,
                                       const gchar *group,
                                       GError **error)
{
    guint i;
    gsize length;
    gchar **keys =
        g_key_file_get_keys (key_file, group, &length, error);
    gchar *protocol = g_strdup ((group) + strlen ("Protocol "));
    TpaProfile *profile = tpa_profile_new (protocol, "", protocol, "");

    VERBOSE ("Protocol %s", protocol);
    for (i = 0; i < length; ++i) {
        const gchar *value;
        TpaParameter *parameter;

        value = g_key_file_get_value (key_file, group, keys[i], error);
        VERBOSE ("%s = %s", keys[i], value);

        if (g_str_has_prefix (keys[i], "param-")) {
            gchar *name = g_strdup (keys[i] + strlen ("param-"));

            parameter = tpa_profile_add_parameter_signature (profile, name, value);

            if (g_strstr_len (value, strlen (value), "required"))
                tpa_parameter_set_required_flag (parameter);
            if (g_strstr_len (value, strlen (value), "register"))
                tpa_parameter_set_register_flag (parameter);
        }
        else if (g_str_has_prefix (keys[i], "default-")) {
            gchar *name = g_strdup (keys[i] + strlen ("default-"));

            parameter = tpa_profile_get_parameter (profile, name);
            tpa_parameter_set_default_flag (parameter);
            tpa_parameter_set_default_value_as_string (parameter, value);
        }
    }

    g_hash_table_insert (self->priv->profiles, protocol, profile);
}

static void
tpa_manager_load_from_file (TpaManager *self)
{
    gchar *service;
    gchar *path;
    GError *error = NULL;
    GKeyFile *key_file = g_key_file_new ();

    VERBOSE ("(%p)", self);
    g_assert (self);
    g_return_if_fail (self->priv->file != NULL);

    if (g_key_file_load_from_file (key_file,
                                   self->priv->file,
                                   G_KEY_FILE_NONE,
                                   &error)) {
        if (g_key_file_has_group (key_file, "ConnectionManager")) {
            if ((service = g_key_file_get_value (key_file, "ConnectionManager", "BusName", &error)) &&
                (path = g_key_file_get_value (key_file, "ConnectionManager", "ObjectPath", &error))) {
                gint i = 0;
                gsize size = 0;
                gchar *name = NULL;
                gchar **groups = g_key_file_get_groups (key_file, &size);
                gchar **filename = g_strsplit (self->priv->file, "/", -1);

                VERBOSE ("Parsing %s...", filename[g_strv_length (filename) - 1]);
                name = g_strndup (filename[g_strv_length (filename) - 1],
                    strlen (filename[g_strv_length (filename) - 1]) - strlen (".manager"));
                /* filename and service name MUST match */
                if (!g_str_equal (name, &(service[strlen (TPA_INTERFACE_CM) +1]))) {
                    ERROR ("Fail to load: %s doesnt match %s.", name, service);
                    goto OUT;
                }

                tpa_manager_create_proxy (self, service, path);

                VERBOSE ("Name %s", name);
                VERBOSE ("BusName %s", service);
                VERBOSE ("ObjectPath %s", path);
                /* Parse parameters for each protocol */
                for (i = 0; i < size; ++i) {
                    if (g_str_has_prefix (groups[i], "Protocol ")) {
                        tpa_manager_parse_protocol_parameters (self, key_file, groups[i], &error);
                    }
                }
                self->priv->name = g_strdup (name);
OUT:
                g_free (name);
                g_strfreev(groups);
                g_strfreev(filename);
            }
        }
    }
    g_key_file_free (key_file);

    if (error) {
        ERROR ("%s", error->message);
        g_error_free (error);
    }
    VERBOSE ("return");
}

static TpaProfile *
tpa_manager_check_parameters (TpaManager *self,
                              const gchar *protocol,
                              GPtrArray *array)
{
    GValueArray *param;
    TpaProfile *profile;
    guint i;
    gboolean ok = TRUE;

    profile = tpa_profile_new (protocol, "", protocol, "");
    for (i = 0; ok && i < array->len; i++) {
        param = g_ptr_array_index (array, i);

        ok = param->n_values == 4;

        if (ok) {
            ok = G_VALUE_HOLDS_STRING (&(param->values[0])) &&
                G_VALUE_HOLDS_UINT (&(param->values[1])) &&
                G_VALUE_HOLDS_STRING (&(param->values[2])) &&
                G_VALUE_HOLDS (&(param->values[3]), G_TYPE_VALUE);
            if (ok) {
                const gchar *name = g_value_get_string (&(param->values[0]));
                TpaParameter *parameter = tpa_profile_add_parameter (profile, name,
                    G_VALUE_TYPE (g_value_get_boxed (&(param->values[3]))));

                tpa_parameter_set_flags (parameter,
                    g_value_get_uint (&(param->values[1])));
                VERBOSE ("param-%s = %s", name,
                    g_value_get_string (&(param->values[2])));
                tpa_parameter_set_default_value (parameter,
                    g_value_get_boxed (&(param->values[3])));
            }
        }
    }

    g_ptr_array_free (array, TRUE);

    return profile;
}

static void
tpa_manager_load_active_connection (TpaManager *self,
                                    const gchar *service)
{
    TpaConnection *connection;
    gchar *path = g_strdup (service);
    gchar **split = g_strsplit (service, ".", -1);

    VERBOSE ("(%p, %s)", self, service);

    path = g_strdelimit (path, ".", '/');
    path = g_strconcat ("/", path, NULL);
    connection = tpa_connection_new (service, path, split[5]);
    if (connection) {
        self->priv->running = TRUE;
        g_hash_table_insert (self->priv->connections, path, connection);
        g_signal_connect (connection,
                          "status-changed",
                          G_CALLBACK (proxy_status_changed_signal),
                          self);
        INFO ("connection %s loaded", path);
    }
    g_strfreev (split);
    VERBOSE ("return");
}

static void
tpa_manager_load_from_bus (TpaManager *self)
{
    GPtrArray *array;
    gchar **protocols;
    gchar *service;
    gchar *path;
    guint i;
    GError *error = NULL;

    VERBOSE ("(%p)", self);
    g_assert (self);
    g_return_if_fail (self->priv->services != NULL);

    self->priv->proxy = tpa_object_get_proxy (TPA_OBJECT (self), TPA_INTERFACE_CM);
    /* Connect all DBus signals */
    tpa_object_connect_signal (TPA_OBJECT (self),
                               TPA_INTERFACE_CM,
                               "NewConnection",
                               G_CALLBACK (proxy_new_connection_signal),
                               self);

    if (self->priv->proxy) {
        array = g_ptr_array_new ();

        /* List protocols available */
        if (!org_freedesktop_Telepathy_ConnectionManager_list_protocols (self->priv->proxy,
            &protocols, &error) || error) {
            ERROR ("%s", error->message);
            g_error_free (error);
            g_object_unref (self);
            return;
        }

        g_object_get (TPA_OBJECT (self),
                      "service", &service,
                      "path", &path,
                      NULL);
        self->priv->name = g_strdup (service + strlen (TPA_INTERFACE_CM) + strlen ("."));

        VERBOSE ("Name %s", self->priv->name);
        VERBOSE ("BusName %s", service);
        VERBOSE ("ObjectPath %s", path);
        for (i = 0; protocols[i]; i++) {
            TpaProfile *profile;
            GPtrArray *array;

            VERBOSE ("Protocol %s", protocols[i]);
            /* List get protocols */
            if (!org_freedesktop_Telepathy_ConnectionManager_get_parameters (self->priv->proxy,
                protocols[i], &array, &error) || error) {
                VERBOSE ("%s", error->message);
                g_error_free (error);
                g_object_unref (self);
                return;
            }
            profile = tpa_manager_check_parameters (self, protocols[i], array);
            g_hash_table_insert (self->priv->profiles, g_strdup (protocols[i]), profile);
        }

        self->priv->running = TRUE;
    }
    else
        g_object_unref (self);

    for (i = 0; self->priv->services[i]; i++) {
        gchar *prefix = g_strconcat (TPA_INTERFACE_CONNECTION, ".", self->priv->name, ".", NULL);
        if (g_str_has_prefix (self->priv->services[i], prefix))
            tpa_manager_load_active_connection (self, self->priv->services[i]);

        g_free (prefix);
    }
    VERBOSE ("return");
}

static GObject*
tpa_manager_constructor (GType type,
                         guint n_construct_params,
                         GObjectConstructParam *construct_params)
{
    GObject *object;
    TpaManager *self;

    object = G_OBJECT_CLASS (tpa_manager_parent_class)->constructor
                            (type, n_construct_params, construct_params);
    self = TPA_MANAGER (object);

    if (self->priv->file)
        tpa_manager_load_from_file (self);
    else if (self->priv->services)
        tpa_manager_load_from_bus (self);

    return object;
}

static void
tpa_manager_set_property (GObject *object,
                         guint prop_id,
                         const GValue *value,
                         GParamSpec *pspec)
{
    TpaManager *self = TPA_MANAGER (object);

    switch (prop_id) {
        case ARG_SERVICES:
            self->priv->services = g_value_get_pointer (value);
            break;
        case ARG_FILE:
            self->priv->file = g_value_dup_string (value);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static void
tpa_manager_get_property (GObject *object,
                         guint prop_id,
                         GValue *value,
                         GParamSpec *pspec)
{
    TpaManager *self = TPA_MANAGER (object);

    switch (prop_id) {
        case ARG_SERVICES:
            g_value_set_pointer (value, self->priv->services);
            break;
        case ARG_FILE:
            g_value_set_string (value, self->priv->file);
            break;
        default:
          G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
          break;
    }
}

static void
tpa_manager_dispose (GObject *object)
{
    TpaManager *self = TPA_MANAGER (object);

    if (self->priv->disposed)
       /* If dispose did already run, return. */
       return;

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;
    self->priv->running = FALSE;

    /* Free data */
    if (self->priv->connections)
        g_hash_table_destroy (self->priv->connections);

    if (self->priv->profiles)
        g_hash_table_destroy (self->priv->profiles);

    if (self->priv->name)
        g_free (self->priv->name);

    if (self->priv->file)
        g_free (self->priv->file);

    if (self->priv->proxy)
        g_object_unref (self->priv->proxy);

    G_OBJECT_CLASS (tpa_manager_parent_class)->dispose (object);
}

static void
tpa_manager_class_init (TpaManagerClass *klass)
{
    GObjectClass *gobject_class;

    gobject_class = (GObjectClass *) klass;
    tpa_manager_parent_class = g_type_class_peek_parent (klass);

    g_type_class_add_private (klass, sizeof (TpaManagerPrivate));

    gobject_class->dispose = tpa_manager_dispose;
    gobject_class->constructor = tpa_manager_constructor;
    gobject_class->get_property = tpa_manager_get_property;
    gobject_class->set_property = tpa_manager_set_property;

    g_object_class_install_property (gobject_class,
                                     ARG_SERVICES,
                                     g_param_spec_pointer ("services",
                                     "services",
                                     "services",
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));

    g_object_class_install_property (gobject_class,
                                     ARG_FILE,
                                     g_param_spec_string ("file",
                                     "file",
                                     "file",
                                     NULL,
                                     G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));

    /* Let's create our signals */
    tpa_manager_signals[NEW_CONNECTION] = g_signal_new ("new-connection",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL,
        NULL,
        g_cclosure_marshal_VOID__POINTER,
        G_TYPE_NONE,
        1,
        G_TYPE_POINTER);
}

static void
tpa_manager_init (TpaManager *self)
{
    self->priv = TPA_MANAGER_GET_PRIVATE (self);
    self->priv->disposed = FALSE;
    self->priv->name = NULL;
    self->priv->file = NULL;
    self->priv->proxy = NULL;
    self->priv->running = FALSE;
    self->priv->connections =
        g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_object_unref);
    self->priv->profiles =
        g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_object_unref);
    self->priv->services = NULL;
}

static void
_get_all_profiles  (gpointer key,
                    gpointer value,
                    gpointer user_data)
{
    g_ptr_array_add (user_data, g_strdup (key));
}

static void
_get_all_connections (gpointer key,
                      gpointer value,
                      gpointer user_data)
{
    g_ptr_array_add (user_data, g_object_ref (value));
}

/**
 * tpa_manager_new_from_file:
 * @file: String with complete path to .manager
 * file.
 * @returns: #TpaManager instance or NULL if fail.
 *
 * Returns a new #TpaManager instance from a .manager
 * file.
 *
 */
TpaManager *
tpa_manager_new_from_file (const gchar *file)
{
    TpaManager *self;

    VERBOSE ("(%s)", file);

    self = TPA_MANAGER (g_object_new (TPA_TYPE_MANAGER,
                        "file", file,
                        NULL));

    VERBOSE ("return %p", self);
    return self;
}

/**
 * tpa_manager_new_from_bus:
 * @service_name: String with dbus service name.
 * @object_path: String with dbus object path.
 * @services: Array of Strings to search for active connections.
 * @returns: #TpaManager instance or NULL if fail.
 *
 * Returns a new #TpaManager instance created from a bus service.
 *
 */
TpaManager *
tpa_manager_new_from_bus (const gchar *service,
                          const gchar *path,
                          const gchar **services)
{
    TpaManager *self;

    VERBOSE ("(%s, %s, %p)", service, path, services);

    self = TPA_MANAGER (g_object_new (TPA_TYPE_MANAGER,
                                      "service", service,
                                      "path", path,
                                      "interface", TPA_INTERFACE_CM,
                                      "services", services,
                                      NULL));

    VERBOSE ("return %p", self);
    return self;
}

/**
 * tpa_manager_get_name:
 * @self: #TpaManager instance.
 * @returns: Connection manager's name.
 *
 * Get name of the connection manager instance.
 *
 */
gchar *
tpa_manager_get_name (TpaManager *self)
{
    g_assert (self);

    return self->priv->name;
}

/**
 * tpa_manager_get_supported_profiles:
 * @self: #TpaManager instance.
 * @returns: #GPtrArray with all profiles supported.
 *
 * Get profiles supported by the connection
 * manager instance.
 *
 */
GPtrArray *
tpa_manager_get_supported_profiles (TpaManager *self)
{
    GPtrArray *profiles = g_ptr_array_new ();

    g_hash_table_foreach (self->priv->profiles,
                          _get_all_profiles,
                          profiles);

    return profiles;
}

/**
 * tpa_manager_get_connections:
 * @self: #TpaManager instance.
 * @returns: #GPtrArray with all connections.
 *
 * <para>
 * Get all current connections.
 * </para>
 */
GPtrArray *
tpa_manager_get_connections (TpaManager *self)
{
    GPtrArray *connections = g_ptr_array_new ();
    g_assert (self);

    g_hash_table_foreach (self->priv->connections,
                          _get_all_connections,
                          connections);

    return connections;
}

/**
 * tpa_manager_get_profile:
 * @self: #TpaManager instance.
 * @name: profile that will be used.
 * @returns: a instance of #TpaProfile.
 *
 * <para>
 * Get a list of the parameters for the given profiles that is
 * necessary for the RequestConnection method.
 * </para>
 */
TpaProfile *
tpa_manager_get_profile (TpaManager *self,
                         const gchar *name)
{
    TpaProfile *profile;

    g_assert (self);
    g_return_val_if_fail (name != NULL, NULL);

    profile = g_hash_table_lookup (self->priv->profiles, name);

    return profile;
}

/**
 * tpa_manager_is_running:
 * @self: #TpaManager instance.
 * @returns: TRUE if it is active otherwise FALSE.
 *
 * Checks if the connection manager is active.
 *
 */
gboolean
tpa_manager_is_running (TpaManager *self)
{
    g_assert (self);

    return self->priv->running;
}

/**
 * tpa_manager_request_connection:
 * @self: #TpaManager instance.
 * @profile:
 * @returns: #TpaConnection instance or NULL if fail.
 *
 * Request a new #TpaConnection instance.
 * The instance state is disconnected to connect it call
 * #tpa_connection_connect.
 *
 */
TpaConnection *
tpa_manager_request_connection (TpaManager *self,
                                TpaProfile *profile)
{
    TpaConnection *connection = NULL;
    gchar *service;
    gchar *object_path;
    const gchar *protocol;
    GError *error = NULL;
    GHashTable *params;

    VERBOSE ("(%p, %p)", self, profile);
    g_assert (self);

    g_return_val_if_fail (profile != NULL, NULL);

    protocol = tpa_profile_get_protocol (profile);
    params = tpa_profile_get_formatted_table (profile);

    if (!(org_freedesktop_Telepathy_ConnectionManager_request_connection (self->priv->proxy,
                                                                          protocol,
                                                                          params,
                                                                          &service,
                                                                          &object_path,
                                                                          &error))) {
        ERROR ("%s", error->message);
        VERBOSE ("return NULL");
        return NULL;
    }
    else {
        connection = tpa_connection_new (service, object_path, protocol);
        self->priv->running = TRUE;
        g_hash_table_insert (self->priv->connections, object_path, connection);
        g_signal_connect (connection, "status-changed", G_CALLBACK (proxy_status_changed_signal), self);
    }

    if (error)
        g_error_free (error);

    VERBOSE ("return %p", connection);
    return connection;
}

/**
 * tpa_manager_supports:
 * @self: #TpaManager instance.
 * @profile: Profile name
 * @returns: TRUE if the instance support the protocol
 * otherwise FALSE.
 *
 * Checks wether #TpaManager instance support the givin
 * protocol.
 *
 */
gboolean
tpa_manager_supports (TpaManager *self,
                      const gchar *profile)
{
    gboolean ret;

    VERBOSE ("(%p, %s)", self, profile);
    g_assert (self);

    ret = (g_hash_table_lookup (self->priv->profiles, profile) != NULL);
    VERBOSE ("return %s", ret ? "true" : "false");
    return ret;
}

TpaProfile *
tpa_manager_add_profile (TpaManager *self,
                         const gchar *file)
{
    TpaProfile *profile = NULL;
    TpaProfile *parent;
    GError *error = NULL;
    GKeyFile *key_file = g_key_file_new ();

    VERBOSE ("(%p)", self);
    g_assert (self);
    g_return_val_if_fail (file != NULL, NULL);

    if (g_key_file_load_from_file (key_file,
                                   file,
                                   G_KEY_FILE_NONE,
                                   &error)) {
        if (g_key_file_has_group (key_file, "Profile")) {
            guint i;
            gsize length;
            gchar *protocol = g_key_file_get_value (key_file, "Profile", "Protocol", &error);
            gchar *name = g_key_file_get_value (key_file, "Profile", "Name", &error);
            gchar *description = g_key_file_get_value (key_file, "Profile", "Description", &error);
            gchar *icon = g_key_file_get_value (key_file, "Profile", "Icon", &error);
            gchar **keys;
            GPtrArray *parameters;

            if (error)
                goto out;

            if (!(parent = tpa_manager_get_profile (self, protocol)))
                goto out;

            profile = tpa_profile_new (name, description, protocol, icon);
            parameters = tpa_profile_get_all_parameters (parent);

            /* Copy parameters from parent/protocol profile */
            for (i = 0; i < parameters->len; i++) {
                    TpaParameter *parm0;
                    TpaParameter *parm1;
                    GValue *default_value;
                    const gchar *parm;

                    parm0 = g_ptr_array_index (parameters, i);
                    parm = tpa_parameter_get_name (parm0);
                    default_value = tpa_parameter_get_default_value (parm0);
                    parm1 = tpa_profile_add_parameter (profile, parm, G_VALUE_TYPE (default_value));

                    if (tpa_parameter_has_default_flag (parm0))
                        tpa_parameter_set_default_value (parm1, default_value);
                    if (tpa_parameter_has_required_flag (parm0))
                        tpa_parameter_set_required_flag (parm1);
                    if (tpa_parameter_has_register_flag (parm0))
                        tpa_parameter_set_register_flag (parm1);
            }

            keys = g_key_file_get_keys (key_file, "Profile", &length, &error);
            for (i = 0; i < length; ++i) {
                const gchar *value;

                value = g_key_file_get_value (key_file, "Profile", keys[i], &error);

                if (g_str_has_prefix (keys[i], "Default-")) {
                    TpaParameter *parameter;
                    gchar *param = g_strdup (keys[i] + strlen ("Default-"));

                    parameter = tpa_profile_get_parameter (profile, param);
                    if (parameter) {
                        tpa_parameter_set_default_flag (parameter);
                        tpa_parameter_set_default_value_as_string (parameter, value);
                    }
                }
            }
            g_hash_table_insert (self->priv->profiles, g_strdup (name), profile);
            g_strfreev (keys);
out:
            g_free (name);
            g_free (description);
            g_free (protocol);
            g_free (icon);
        }
    }
    g_key_file_free (key_file);

    if (error) {
        ERROR ("%s", error->message);
        g_error_free (error);
    }
    VERBOSE ("return");
    return profile;
}
