/*
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "config.h"

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>

#include "tpa-aliasing.h"

#include "tpa-aliasing-private.h"

#define DEBUG_DOMAIN TPA_DOMAIN_ALIASING

#include <tapioca/base/tpa-signals-marshal.h>
#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-errors.h>

/* signal enum */
enum
{
    ALIASES_CHANGED,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0};

struct _TpaIAliasingPrivate {
    GObject *object;
};

/* we need to define the get_type function */
GType
tpa_aliasing_get_type()
{
    static GType object_type = 0;

    if (!object_type) {
        static const GTypeInfo object_info = {
            sizeof(TpaIAliasing),
            NULL,	/* base init */
            NULL,	/* base finalize */
        };
        object_type =
            g_type_register_static(G_TYPE_INTERFACE,
                "TpaIAliasing",
                &object_info, 0);
    }
    return object_type;
}

void
tpa_aliasing_init (TpaIAliasing *iface,
                   gpointer data)
{
    VERBOSE ("(%p, %p)", iface, data);

    iface->get_alias_flags = NULL;
    iface->request_aliases = NULL;
    iface->set_aliases = NULL;

    /* Interface signals */
    signals[ALIASES_CHANGED] =
        g_signal_new ("aliases-changed",
                      G_OBJECT_CLASS_TYPE (iface),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      g_cclosure_marshal_VOID__BOXED,
                      G_TYPE_NONE, 1, (dbus_g_type_get_collection ("GPtrArray", (dbus_g_type_get_struct ("GValueArray", G_TYPE_UINT, G_TYPE_STRING, G_TYPE_INVALID)))));

    iface->priv = g_new0 (TpaIAliasingPrivate, 1);
}

void
tpa_aliasing_finalize (GObject *obj)
{
    TpaIAliasing *iface = TPA_IALIASING (obj);
    VERBOSE ("(%p)", obj);

    if (iface->priv)
        g_free (iface->priv);
}

/**
 * tpa_aliasing_get_alias_flags
 *
 * Implements DBus method GetAliasFlags
 * on interface org.freedesktop.Telepathy.Connection.Interface.Aliasing
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_aliasing_get_alias_flags (GObject *obj,
                              guint *ret,
                              GError **error)
{
    TpaIAliasing *iface = TPA_IALIASING (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p)", obj, ret);

    g_return_error_val_if_fail (iface->get_alias_flags != NULL, error,
        TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->get_alias_flags (obj, ret);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_aliasing_request_aliases
 *
 * Implements DBus method RequestAliases
 * on interface org.freedesktop.Telepathy.Connection.Interface.Aliasing
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_aliasing_request_aliases (GObject *obj,
                              const GArray *contacts,
                              gchar ***ret,
                              GError **error)
{
    TpaIAliasing *iface = TPA_IALIASING (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p, %p)", obj, contacts, ret);

    g_return_error_val_if_fail (iface->request_aliases != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->request_aliases (obj, contacts, ret);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_aliasing_set_aliases
 *
 * Implements DBus method SetAliases
 * on interface org.freedesktop.Telepathy.Connection.Interface.Aliasing
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_aliasing_set_aliases (GObject *obj,
                          GHashTable *aliases,
                          GError **error)
{
    TpaIAliasing *iface = TPA_IALIASING (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p)", obj, aliases);

    g_return_error_val_if_fail (iface->set_aliases != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->set_aliases (obj, aliases);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_aliasing_signal_aliases_changed
 *
 * Implements DBus signal StatusChanged
 * on interface org.freedesktop.Telepathy.Connection.Interface.Aliasing
 */
void
tpa_aliasing_signal_aliases_changed (GObject *obj,
                                     GPtrArray *aliases)
{
    TpaIAliasing *iface = TPA_IALIASING (obj);

    if (!iface)
        return;

    VERBOSE ("(%p, %p)", obj, aliases);

    g_signal_emit (iface, signals[ALIASES_CHANGED], 0, aliases);
}
