/*
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "config.h"

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>

#include "tpa-avatars.h"

#include "tpa-avatars-private.h"

#define DEBUG_DOMAIN TPA_DOMAIN_AVATARS

#include <tapioca/base/tpa-signals-marshal.h>
#include <tapioca/base/tpa-errors.h>


/* signal enum */
enum
{
    AVATAR_UPDATED,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0};

struct _TpaIAvatarsPrivate {
    GObject *object;
};

/* we need to define the get_type function */
GType
tpa_avatars_get_type()
{
    static GType object_type = 0;

    if (!object_type) {
        static const GTypeInfo object_info = {
            sizeof(TpaIAvatars),
            NULL,	/* base init */
            NULL,	/* base finalize */
        };
        object_type =
            g_type_register_static(G_TYPE_INTERFACE,
                "TpaIAvatars",
                &object_info, 0);
    }
    return object_type;
}

void
tpa_avatars_init (TpaIAvatars *iface,
                  gpointer data)
{
    VERBOSE ("(%p, %p)", iface, data);

    iface->get_avatar_requirements = NULL;
    iface->get_avatar_tokens = NULL;
    iface->request_avatar = NULL;
    iface->set_avatar = NULL;

    /* Interface signals */
    signals[AVATAR_UPDATED] =
        g_signal_new ("avatar-updated",
                      G_OBJECT_CLASS_TYPE (iface),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      tpa_marshal_VOID__UINT_STRING,
                      G_TYPE_NONE, 2, G_TYPE_UINT, G_TYPE_STRING);

    iface->priv = g_new0 (TpaIAvatarsPrivate, 1);
}

void
tpa_avatars_finalize (GObject *obj)
{
    TpaIAvatars *iface = TPA_IAVATARS (obj);
    VERBOSE ("(%p)", obj);

    g_free (iface->priv);
}

/**
 * tpa_avatars_get_avatar_requirements
 *
 * Implements DBus method GetAvatarRequirements
 * on interface org.freedesktop.Telepathy.Connection.Interface.Avatars
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_avatars_get_avatar_requirements (GObject *obj,
                                     gchar ***ret,
                                     guint *ret1,
                                     guint *ret2,
                                     guint *ret3,
                                     guint *ret4,
                                     guint *ret5,
                                     GError **error)
{
    TpaIAvatars *iface = TPA_IAVATARS (obj);
    TpaError error_code = TPA_ERROR_NONE;

    if (!iface)
        g_return_error_val_if_fail (iface->priv != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p, %p, %p, %p, %p, %p)", obj, ret, ret1, ret2, ret3, ret4, ret5);

    g_return_error_val_if_fail (iface->get_avatar_requirements != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->get_avatar_requirements (obj, ret, ret1, ret2, ret3, ret4, ret5);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_avatars_get_avatar_requirements
 *
 * Implements DBus method GetAvatarTokens
 * on interface org.freedesktop.Telepathy.Connection.Interface.Avatars
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_avatars_get_avatar_tokens (GObject *obj,
                               const GArray *contacts,
                               DBusGMethodInvocation *context)
{
    TpaIAvatars *iface = TPA_IAVATARS (obj);
    TpaError error_code = TPA_ERROR_NONE;
    GError *error = NULL;

    if (!iface)
        g_return_context_error_val_if_fail (iface != NULL, context, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p)", obj, contacts);

    g_return_context_error_val_if_fail (iface->get_avatar_tokens != NULL, context, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->get_avatar_tokens (obj, contacts, context);

    g_return_context_error_val_if_fail (error_code == TPA_ERROR_NONE, context, error, error_code);

    if (context)
        dbus_g_method_return (context);

    return TRUE;
}

/**
 * tpa_avatars_request_avatar
 *
 * Implements DBus method RequestAvatar
 * on interface org.freedesktop.Telepathy.Connection.Interface.Avatars
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_avatars_request_avatar (GObject *obj,
                            guint contact,
                            DBusGMethodInvocation *context)
{
    TpaIAvatars *iface = TPA_IAVATARS (obj);
    TpaError error_code = TPA_ERROR_NONE;
    GError *error = NULL;

    if (!iface)
        g_return_context_error_val_if_fail (iface != NULL, context, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %d)", obj, contact);

    g_return_context_error_val_if_fail (iface->request_avatar != NULL, context, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->request_avatar (obj, contact, context);

    g_return_context_error_val_if_fail (error_code == TPA_ERROR_NONE, context, error, error_code);

    if (context)
        dbus_g_method_return (context);

    return TRUE;
}

/**
 * tpa_avatars_set_avatar
 *
 * Implements DBus method SetAvatar
 * on interface org.freedesktop.Telepathy.Connection.Interface.Avatars
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_avatars_set_avatar (GObject *obj,
                        const GArray *avatar,
                        const gchar *mime_type,
                        DBusGMethodInvocation *context)
{
    TpaIAvatars *iface = TPA_IAVATARS (obj);
    TpaError error_code = TPA_ERROR_NONE;
    GError *error = NULL;

    if (!iface)
        g_return_context_error_val_if_fail (iface != NULL, context, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p, %s)", obj, avatar, mime_type);

    g_return_context_error_val_if_fail (iface->set_avatar != NULL, context, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->set_avatar (obj, avatar, mime_type, context);

    g_return_context_error_val_if_fail (error_code == TPA_ERROR_NONE, context, error, error_code);

    if (context)
        dbus_g_method_return (context);

    return TRUE;
}

/**
 * tpa_avatars_signal_avatar_updated
 *
 * Implements DBus signal AvatarUpdated
 * on interface org.freedesktop.Telepathy.Connection.Interface.Avatars
 */
void
tpa_avatars_signal_avatar_updated (GObject *obj,
                                   guint handle,
                                   const gchar *new_avatar_token)
{
    TpaIAvatars *iface = TPA_IAVATARS (obj);

    if (!iface)
        return;

    VERBOSE ("(%p, %d, %s)", obj, handle, new_avatar_token);

    g_signal_emit (iface, signals[AVATAR_UPDATED], 0, handle, new_avatar_token);
}
