/*
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <dbus/dbus-glib.h>
#include <dbus/dbus-protocol.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "tpa-connection-manager.h"
#include "tpa-ifaces.h"

#include "tpa-connection-manager-private.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CONNECTION_MANAGER

#include <tapioca/base/tpa-signals-marshal.h>
#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-enums.h>

#define _DBUS_PATH "/org/freedesktop/Telepathy/ConnectionManager"

#define TPA_TYPE_PARAM (dbus_g_type_get_struct ("GValueArray", \
      G_TYPE_STRING, \
      G_TYPE_UINT, \
      G_TYPE_STRING, \
      G_TYPE_VALUE, \
      G_TYPE_INVALID))

/* signal enum */
enum
{
    NEW_CONNECTION,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0};

/* we need to define the get_type function */
GType
tpa_connection_manager_get_type ()
{
    static GType object_type = 0;

    if (!object_type) {
        static const GTypeInfo object_info = {
            sizeof(TpaIConnectionManager),
            NULL,   /* base init */
            NULL,   /* base finalize */
        };
        object_type =
            g_type_register_static(G_TYPE_INTERFACE,
                "TpaIConnectionManager",
                &object_info, 0);
    }
    return object_type;
}

void
tpa_connection_manager_init (TpaIConnectionManager *iface,
                             gpointer data)
{
    VERBOSE ("(%p, %p)", iface, data);

    iface->request_connection = NULL;

    signals[NEW_CONNECTION] =
        g_signal_new ("new-connection",
                      G_OBJECT_CLASS_TYPE (iface),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      tpa_marshal_VOID__STRING_STRING_STRING,
                      G_TYPE_NONE, 3, G_TYPE_STRING, DBUS_TYPE_G_OBJECT_PATH, G_TYPE_STRING);
    iface->parameters = g_ptr_array_new ();
    iface->protocols = NULL;
    iface->name = NULL;
}

void
tpa_connection_manager_finalize (GObject *obj)
{
    VERBOSE ("(%p)", obj);
}

/**
 * tpa_connnection_manager_request_connection
 *
 * Implements DBus method Connect
 * on interface org.freedesktop.Telepathy.ConnectionManager
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_connection_manager_request_connection (GObject *obj,
                                           const gchar *protocol,
                                           GHashTable *parameters,
                                           gchar **service_name,
                                           gchar **object_path,
                                           GError **error)
{
    TpaIConnectionManager *iface = TPA_ICONNECTION_MANAGER (obj);
    TpaError error_code = TPA_ERROR_NONE;
    gchar *name = NULL;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %s, %p, %p, %p)", obj, protocol, parameters, service_name,
        object_path);

    g_return_error_val_if_fail (iface->request_connection != NULL,
        error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->request_connection (obj, protocol, parameters, &name);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    *service_name = g_strdup_printf ("%s.%s.%s.%s", DBUS_CONNECTION_IFACE, iface->name,
        protocol, name);
    *object_path = g_strdup_printf ("%s/%s/%s/%s", DBUS_CONNECTION_PATH, iface->name,
        protocol, name);

    VERBOSE ("returning %s, %s", *service_name, *object_path);

    g_free (name);

    return TRUE;
}

/**
 * tpa_connection_manager_get_parameters
 *
 * Implements DBus method GetMandatoryParameters
 * on interface org.freedesktop.Telepathy.ConnectionManager
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_connection_manager_get_parameters (GObject *obj,
                                       const gchar *protocol,
                                       GPtrArray **ret,
                                       GError **error)
{
    TpaIConnectionManager *iface = TPA_ICONNECTION_MANAGER (obj);
    TpaConnectionManagerParam *parameter = NULL;
    guint i;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %s, %p)", obj, protocol, ret);

    if (iface->parameters && ret) {
        *ret = g_ptr_array_new ();

        for (i = 0; i < iface->parameters->len; i++) {
            parameter =
                (TpaConnectionManagerParam *) g_ptr_array_index (iface->parameters, i);
            GValue param = {0,};
            GValue val = {0,};

            g_value_init (&param, TPA_TYPE_PARAM);
            g_value_set_static_boxed (&param,
                dbus_g_type_specialized_construct (TPA_TYPE_PARAM));

            if (g_str_equal (parameter->type, "s")) {
                g_value_init (&val, G_TYPE_STRING);
                g_value_set_string (&val, parameter->value);
            }
            else if (g_str_equal (parameter->type, "i") ||
                     g_str_equal (parameter->type, "n")) {
                g_value_init (&val, G_TYPE_INT);
                g_value_set_int (&val, *((gint *) parameter->value));
            }
            else if (g_str_equal (parameter->type, "u")  ||
                     g_str_equal (parameter->type, "q")) {
                g_value_init (&val, G_TYPE_UINT);
                g_value_set_uint (&val, *((guint *) parameter->value));
            }
            else if (g_str_equal (parameter->type, "b")) {
                g_value_init (&val, G_TYPE_BOOLEAN);
                g_value_set_boolean (&val, *((gboolean *) parameter->value));
            }

            dbus_g_type_struct_set (&param,
                                    0, parameter->name,
                                    1, parameter->flag,
                                    2, parameter->type,
                                    3, &val,
                                    G_MAXUINT);

            g_value_unset (&val);

            g_ptr_array_add (*ret, g_value_dup_boxed (&param));

            g_value_unset (&param);
        }
    }

    return TRUE;
}

/**
 * tpa_connection_manager_list_protocols
 *
 * Implements DBus method ListProtocols
 * on interface org.freedesktop.Telepathy.ConnectionManager
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_connection_manager_list_protocols (GObject *obj,
                                       gchar ***ret,
                                       GError **error)
{
    TpaIConnectionManager *iface = TPA_ICONNECTION_MANAGER (obj);

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p)", obj, ret);

    if (ret) {
        *ret = g_new0 (gchar *, g_strv_length (iface->protocols));
        *ret = g_strdupv (iface->protocols);
    }

    return TRUE;
}

/**
 * tpa_connection_manager_signal_new_connection
 *
 * Implements DBus signal NewConnnection
 * on interface org.freedesktop.Telepathy.Manager
 */
void
tpa_connection_manager_signal_new_connection (GObject *obj,
                                              const gchar *bus_name,
                                              const gchar *obj_path,
                                              const gchar *protocol)
{
    g_assert (TPA_IS_ICONNECTION_MANAGER (obj));

    VERBOSE ("(%p, %s, %s, %s)", obj, bus_name, obj_path, protocol);

    g_signal_emit (obj, signals[NEW_CONNECTION], 0, bus_name, obj_path, protocol);
}
