/*
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "config.h"

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "tpa-presence.h"
#include "tpa-account.h"

#include "tpa-presence-private.h"

#define DEBUG_DOMAIN TPA_DOMAIN_PRESENCE

#include <tapioca/base/tpa-signals-marshal.h>
#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-errors.h>

/* signal enum */
enum
{
    PRESENCE_UPDATE,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0};

struct _TpaIPresencePrivate {
    GObject *object;
    GHashTable *presence;
};

/* we need to define the get_type function */
GType
tpa_presence_get_type()
{
    static GType object_type = 0;

    if (!object_type) {
        static const GTypeInfo object_info = {
            sizeof(TpaIPresence),
            NULL,   /* base init */
            NULL,   /* base finalize */
        };
        object_type =
            g_type_register_static(G_TYPE_INTERFACE,
                "TpaIPresence",
                &object_info, 0);
    }
    return object_type;
}

void
tpa_presence_init (TpaIPresence *iface,
                   gpointer data)
{
    VERBOSE ("(%p, %p)", iface, data);

    iface->add_status = NULL;
    iface->clear_status = NULL;
    iface->get_statuses = NULL;
    iface->request_presence = NULL;
    iface->remove_status = NULL;
    iface->set_last_activity_time = NULL;
    iface->set_status = NULL;

    /* Interface signals */
    signals[PRESENCE_UPDATE] =
        g_signal_new ("presence-update",
                      G_OBJECT_CLASS_TYPE (iface),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      g_cclosure_marshal_VOID__BOXED,
                      G_TYPE_NONE, 1, (dbus_g_type_get_map ("GHashTable", G_TYPE_UINT, (dbus_g_type_get_struct ("GValueArray", G_TYPE_UINT, (dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, (dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, G_TYPE_VALUE)))), G_TYPE_INVALID)))));
    iface->priv = g_new0 (TpaIPresencePrivate, 1);
}

void
tpa_presence_finalize (GObject *obj)
{
    TpaIPresence *iface = TPA_IPRESENCE (obj);
    VERBOSE ("(%p)", obj);

    if (iface->priv)
        g_free (iface->priv);
}

/**
 * tpa_presence_add_status
 *
 * Implements DBus method AddStatus
 * on interface org.freedesktop.Telepathy.Connection.Interface.Presence
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_presence_add_status (GObject *obj,
                         const gchar *status,
                         GHashTable *parms,
                         GError **error)
{
    TpaIPresence *iface = TPA_IPRESENCE (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %s, %p)", obj, status, parms);

    g_return_error_val_if_fail (iface->add_status != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->add_status (obj, status, parms);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_presence_clear_status
 *
 * Implements DBus method ClearStatus
 * on interface org.freedesktop.Telepathy.Connection.Interface.Presence
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_presence_clear_status (GObject *obj,
                           GError **error)
{
    TpaIPresence *iface = TPA_IPRESENCE (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p)", obj);

    g_return_error_val_if_fail (iface->clear_status != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->clear_status (obj);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_presence_get_statuses
 *
 * Implements DBus method GetStatuses
 * on interface org.freedesktop.Telepathy.Connection.Interface.Presence
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_presence_get_statuses (GObject *obj,
                           GHashTable **ret,
                           GError **error)
{
    TpaIPresence *iface = TPA_IPRESENCE (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p)", obj, ret);

    g_return_error_val_if_fail (iface->get_statuses != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->get_statuses (obj, ret);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_presence_remove_status
 *
 * Implements DBus method RemoveStatus
 * on interface org.freedesktop.Telepathy.Connection.Interface.Presence
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_presence_remove_status (GObject *obj,
                            const gchar *status,
                            GError **error)
{
    TpaIPresence *iface = TPA_IPRESENCE (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p)", obj, status);

    g_return_error_val_if_fail (iface->remove_status != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->remove_status (obj, status);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_presence_request_presence
 *
 * Implements DBus method RequestPresence
 * on interface org.freedesktop.Telepathy.Connection.Interface.Presence
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_presence_request_presence (GObject *obj,
                               const GArray *contacts,
                               GError **error)
{
    TpaIPresence *iface = TPA_IPRESENCE (obj);
    TpaError error_code = TPA_ERROR_NONE;
    gchar *contact = NULL;
    guint handle = 0;
    guint c = 0;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p)", obj, contacts);

    g_return_error_val_if_fail (iface->request_presence != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    for (c = 0; c < contacts->len; c++) {
        handle = g_array_index (contacts, guint, c);
        contact = tpa_connection_get_string (G_OBJECT (obj), handle);
        if (contact)
            error_code = iface->request_presence (obj, contact);
        else
            error_code = TPA_ERROR_INVALID_HANDLE;
        if (error_code != TPA_ERROR_NONE)
            break;
    }

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_presence_set_last_activity_time
 *
 * Implements DBus method SetLastActivityTime
 * on interface org.freedesktop.Telepathy.Connection.Interface.Presence
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_presence_set_last_activity_time (GObject *obj,
                                     guint time,
                                     GError **error)
{
    TpaIPresence *iface = TPA_IPRESENCE (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %d)", obj, time);

    g_return_error_val_if_fail (iface->set_last_activity_time != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->set_last_activity_time (obj, time);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_presence_set_status
 *
 * Implements DBus method SetStatus
 * on interface org.freedesktop.Telepathy.Connection.Interface.Presence
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_presence_set_status (GObject *obj,
                         GHashTable *statuses,
                         GError **error)
{
    TpaIPresence *iface = TPA_IPRESENCE (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p)", obj, statuses);

    g_return_error_val_if_fail (iface->request_presence != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->set_status (obj, statuses);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_presence_update
 *
 * Format presences status to telepathy spec.
 *
 * @return: GHashTable to be emmit with tpa_presence_signal_presence_update.
 */
gboolean
tpa_presence_update (GObject *obj,
                     const gchar *contact,
                     const gchar *status,
                     const gchar *message)
{
    TpaIPresence *iface = TPA_IPRESENCE (obj);
    GValueArray *vals;
    GHashTable *contact_status, *parameters;
    GValue *msg;
    time_t timestamp = 0;
    TpaHandle *handle = NULL;

    g_assert (TPA_IS_ACCOUNT (obj));

    handle = tpa_connection_get_contact_handle (G_OBJECT (obj), contact);

    if (!iface->priv->presence)
        iface->priv->presence = g_hash_table_new_full (g_direct_hash, g_direct_equal, NULL,
                                        (GDestroyNotify) g_value_array_free);

    if (g_hash_table_lookup (iface->priv->presence, contact)) {
        VERBOSE ("Presence %s %s %s already on list", contact, status, message);
        return FALSE;
    }

    msg = g_new0 (GValue, 1);
    g_value_init (msg, G_TYPE_STRING);
    g_value_set_static_string (msg, message);

    parameters =
        g_hash_table_new_full (g_str_hash, g_str_equal,
                           NULL, (GDestroyNotify) g_hash_table_destroy);

    g_hash_table_insert (parameters, "message", msg);

    contact_status =
        g_hash_table_new_full (g_str_hash, g_str_equal,
                           NULL, (GDestroyNotify) g_hash_table_destroy);
    g_hash_table_insert (contact_status, (gpointer) status, parameters);

    vals = g_value_array_new (2);

    g_value_array_append (vals, NULL);
    g_value_init (g_value_array_get_nth (vals, 0), G_TYPE_UINT);
    timestamp = time (NULL);
    g_value_set_uint (g_value_array_get_nth (vals, 0), timestamp);

    g_value_array_append (vals, NULL);
    g_value_init (g_value_array_get_nth (vals, 1),
        dbus_g_type_get_map ("GHashTable", G_TYPE_STRING,
        dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, G_TYPE_VALUE)));
    g_value_take_boxed (g_value_array_get_nth (vals, 1), contact_status);

    g_hash_table_insert (iface->priv->presence, GINT_TO_POINTER (handle->id), vals);

    VERBOSE ("Presence %s %s %s added", contact, status, message);

    return TRUE;
}

/**
 * tpa_presence_signal_presence_update
 *
 * Implements DBus signal PresenceUpdate
 * on interface org.freedesktop.Telepathy.Connection.Interface.Presence
 */
void
tpa_presence_signal_presence_update (GObject *obj)
{
    TpaIPresence *iface = TPA_IPRESENCE (obj);

    g_assert (TPA_IS_IPRESENCE (obj));

    VERBOSE ("(%p)", obj);

    if (iface->priv->presence) {
        g_signal_emit (obj, signals[PRESENCE_UPDATE], 0, iface->priv->presence);
        g_hash_table_destroy (iface->priv->presence);
        iface->priv->presence = NULL;
    }
}
