/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test preferences managers.
 * 
 * ti-preferences.h:
 * Singleton that holds global application preferences.
 * 
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "ti-preferences.h"
#include "ti-constants.h"
#include "ti-signals-marshal.h"

#include <dbus/dbus-glib.h>

struct _TIPreferencesClass {
    GObjectClass parent;

    // ID for "handle-display-mode-changed" signal.
    guint handle_display_mode_changed_id;
};

G_DEFINE_TYPE (TIPreferences, ti_preferences, G_TYPE_OBJECT);

/**
 * Instance private data.
 */
struct _TIPreferencesPrivate {
    guint handle_display_mode;
    gboolean auto_inspect_handles;
};
typedef struct _TIPreferencesPrivate TIPreferencesPrivate;

#define TI_PREFERENCES_GET_PRIVATE(object)  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_PREFERENCES, TIPreferencesPrivate))

/**
 * Drop all references to other objects.
 */
/*static void
ti_preferences_dispose (GObject *object)
{
    TIPreferences* preferences = TI_PREFERENCES (object);
    TIPreferencesPrivate* priv = TI_PREFERENCES_GET_PRIVATE (preferences);

    G_OBJECT_CLASS (ti_preferences_parent_class)->dispose (object);
}*/

/**
 * Finalizes the object, marking the memory as ready for reuse
 */
/*static void
ti_preferences_finalize (GObject *object)
{
    TIPreferences* preferences = TI_PREFERENCES (object);
    TIPreferencesPrivate* priv = TI_PREFERENCES_GET_PRIVATE (preferences);

	G_OBJECT_CLASS (ti_preferences_parent_class)->finalize (object);
}*/

/**
 * Class initialization.
 */
static void
ti_preferences_class_init (TIPreferencesClass *ti_preferences_class)
{
    //GObjectClass *gobject_class = G_OBJECT_CLASS (ti_preferences_class);

    // Override base object methods
    //gobject_class->dispose = ti_preferences_dispose;
    //gobject_class->finalize = ti_preferences_finalize;

    // Add private
    g_type_class_add_private (ti_preferences_class, sizeof (TIPreferencesPrivate));

    /**
     * Signal: handle-display-mode-changed
     * Emitted every time handle_display_mode changes.
     *
     * Param #1 - UINT The new handle_display_mode value.
     */
    ti_preferences_class->handle_display_mode_changed_id =
        g_signal_new ("handle-display-mode-changed",
                      G_OBJECT_CLASS_TYPE (ti_preferences_class),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      g_cclosure_marshal_VOID__UINT,
                      G_TYPE_NONE,
                      1, G_TYPE_UINT);
}

/**
 * Instance initialization.
 */
static void
ti_preferences_init (TIPreferences *ti_preferences)
{
    TIPreferencesPrivate* priv = TI_PREFERENCES_GET_PRIVATE (ti_preferences);

    // Hard-coded default values:
    priv->handle_display_mode = TI_PREFERENCES_HANDLE_DISPLAY_HANDLE;
    priv->auto_inspect_handles = FALSE;
}

/**
 * Creates a new preferences
 */
TIPreferences*
ti_preferences_new ()
{
    static TIPreferences* preferences = NULL;

    // It's a singleton
    if (preferences == NULL)
    {
        preferences = g_object_new (TI_TYPE_PREFERENCES, NULL);
    }
    else
    {
        g_object_ref (preferences);
    }

    return preferences;
}

/**
 * Get Handle Display Mode
 */
guint
ti_preferences_get_handle_display_mode (TIPreferences* preferences)
{
    TIPreferencesPrivate* priv = TI_PREFERENCES_GET_PRIVATE (preferences);

    return priv->handle_display_mode;
}

/**
 * Set Handle Display Mode
 */
void
ti_preferences_set_handle_display_mode (TIPreferences* preferences, guint new_mode)
{
    TIPreferencesPrivate* priv = TI_PREFERENCES_GET_PRIVATE (preferences);
    TIPreferencesClass* klass = TI_PREFERENCES_GET_CLASS (preferences);

    priv->handle_display_mode = new_mode;

    g_signal_emit (preferences, klass->handle_display_mode_changed_id, 0, priv->handle_display_mode);
}

/**
 * Get Auto Inspect Handles
 */
gboolean
ti_preferences_get_auto_inspect_handles (TIPreferences* preferences)
{
    TIPreferencesPrivate* priv = TI_PREFERENCES_GET_PRIVATE (preferences);

    return priv->auto_inspect_handles;
}

/**
 * Set Auto Inspect Handles
 */
void
ti_preferences_set_auto_inspect_handles (TIPreferences* preferences, gboolean auto_inspect)
{
    TIPreferencesPrivate* priv = TI_PREFERENCES_GET_PRIVATE (preferences);

    priv->auto_inspect_handles = auto_inspect;
}
