/*  spawx11 - the Simplest Previewer with Antialiasing in the World for X11  */
/*  Copyright (C) 1997-2000  Hirotsugu Kakugawa. All rights reserved.        */
/*  See "COPYING" for disctribution of this software.                        */
#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#if HAVE_STRING_H
#  include <string.h>
#endif
#if HAVE_STRINGS_H
#  include <strings.h>
#endif
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#include <X11/keysym.h>
#include "libdvi29.h"
#include "defs.h"
#define SPAWX11_VERSION   "1.2.1" 

int              param_dpi             = DEFAULT_DPI;
char            *param_kpathsea_mode   = DEFAULT_KPATHSEA_MODE;
char            *param_vflibcap        = DEFAULT_VFLIBCAP;
int              param_delayed_fopen   = 1;
int              param_draw_eps_figure = 1;
char            *param_paper           = "A4"; 
int              param_orient          = 0;
double           param_offset_x        = DEFAULT_OFFSET_X;
double           param_offset_y        = DEFAULT_OFFSET_Y;
int              param_aa              = DEFAULT_AA;
char            *param_fg              = DEFAULT_FG_COLOR;
char            *param_bg              = DEFAULT_BG_COLOR;
DVI              dvi       = NULL;
DVI_DEVICE       dev       = NULL;
char            *dvi_file  = NULL;
DVI_FRAME_BUFFER framebuff = NULL;
int              f_w, f_h;
DVI_PROPERTY     dvi_props = NULL;
double           mag;
double           shrink;
int              ui_cmd, thePage, aa_nlevels;
Display         *x_disp;
Window           x_win;
GC               x_gc_win;
XEvent           xevent;
Cursor           x_c_reading, x_c_ready, x_c_drawing, x_c_ps; 
int              x_w, x_h, x_ofsx, x_ofsy;
unsigned long    x_pix_fg, x_pix_bg;
unsigned long   *x_pix_table  = NULL;

#define EXIT_MSG1(a1)   {fprintf(stderr,a1); fprintf(stderr,"\n"); exit(1);}
#define EXIT_MSG2(a1,a2){fprintf(stderr,a1,a2); fprintf(stderr,"\n"); exit(1);}
char  *parse_args(int,char**);
void  usage(void);
int   window_size(char*,int,int*,int*,int*,int*);
void  calc_window_params(int);
int   main_loop(DVI_DEVICE,DVI);
DVI_DEVICE  create_dev(void);
void  DEV_put_bitmap(DVI_DEVICE,DVI,DVI_BITMAP,int,long,long,long,long);
void  DEV_put_rectangle(DVI_DEVICE,DVI,long,long,long,long);
int   DEV_before_ps_figure(DVI_DEVICE,DVI,char*,long,long);
void  DEV_after_ps_figure(DVI_DEVICE,DVI);
int   DEV_draw_ps_figures(DVI_DEVICE,DVI);
int   DEV_poll(DVI_DEVICE,DVI,int);
void  x_init(void);
void  x_change_cursor(Cursor);
int   x_parse_event(DVI,XEvent*);
void  x_aa_put(long,long,long,long);

int  main(int argc, char **argv)
{
  char          params[1024];

  if ((dvi_file = parse_args(argc, argv)) == NULL)
    EXIT_MSG1("No DVI file.\n");
  x_init();
  x_change_cursor(x_c_reading);
  sprintf(params, "TeX_DPI=%d, TeX_KPATHSEA_MODE=%s, TeX_KPATHSEA_PROGRAM=%s",
	  param_dpi, param_kpathsea_mode, "spawx11");
  if ((DVI_INIT(param_vflibcap, params) < 0) || (create_dev() == NULL))
    exit(1);
  dvi_props = DVI_PROPERTY_ALLOC_DEFAULT();
  if (param_delayed_fopen == 1)
    DVI_PROPERTY_SET(dvi_props, DVI_PROP_DELAYED_FONT_OPEN);
  DVI_PROPERTY_SET(dvi_props,DVI_PROP_ASYNC_GS_INVOCATION);
  DVI_PROPERTY_SET(dvi_props, DVI_PROP_INCREMENTAL_EPS_DISPLAY);
  if ((dvi = DVI_CREATE(dev, dvi_file, dvi_props)) == NULL)
    EXIT_MSG2("Can't open DVI file: %s\n", dvi_file);
  if (DVI_OPEN_FONT(dvi, dev) < 0)
    EXIT_MSG1("Cannot find all fonts.\n");
  main_loop(dev,dvi);
  return 0;
}

char  *parse_args(int argc, char **argv)
{
  double   d;
  char    *f;

  f = NULL;
  mag = 1.0;

  for (--argc, argv++; argc > 0;  argc--, argv++){
    if ((strcmp(*argv, "-m") == 0) && (argc > 1)){
      argc--; argv++;
      mag = atof(*argv);
      if (mag < MAG_MIN)
	EXIT_MSG1("Maginification is too small.\n");
      if (MAG_MAX < mag)
	EXIT_MSG1("Maginification is too large.\n");
    } else if ((strcmp(*argv, "-v") == 0)  && (argc > 1)){
      argc--;  argv++;
      param_vflibcap = *argv;
    } else if (DVI_parse_device_mode(&argv[0][1], NULL, NULL, NULL) >= 0){
      param_kpathsea_mode = &argv[0][1];
    } else if ((strcmp(*argv, "-mode") == 0) && (argc > 1)){
      argc--;  argv++;
      param_kpathsea_mode = *argv;
    } else if ((strcmp(*argv, "-dpi") == 0) && (argc > 1)){
      argc--;  argv++;
      if ((param_dpi = atoi(*argv)) <= 0)
	EXIT_MSG1("Device resolution must be positive value.\n");
    } else if (strcmp(*argv, "-nd") == 0){
      param_delayed_fopen = 0;
    } else if (strcmp(*argv, "-no-eps") == 0){
      param_draw_eps_figure = 0;
    } else if ((strcmp(*argv, "-fg") == 0) && (argc > 1)){
      argc--;  argv++;
      param_fg = *argv;
    } else if ((strcmp(*argv, "-bg") == 0) && (argc > 1)){
      argc--;  argv++;
      param_bg = *argv;
    } else if ((strcmp(*argv, "-c") == 0) && (argc > 1)){
      argc--;  argv++;
      switch (atoi(*argv)){
      case 1:  param_fg = "DarkBlue";  param_bg = "white";         break;
      case 2:  param_fg = "white";     param_bg = "DarkGreen";     break;
      case 3:  param_fg = "green";     param_bg = "black";         break;
      case 4:  param_fg = "black";     param_bg = "AntiqueWhite";  break;
      case 5:  param_fg = "magenta";   param_bg = "aquamarine";    break;
      case 6:  param_fg = "blue";      param_bg = "yellow";        break;
      case 7:  param_fg = "red";       param_bg = "black";         break;
      default: param_fg = "black";     param_bg = "white";         break;
      }
    } else if ((strcmp(*argv, "-aa") == 0) && (argc > 1)){
      argc--;  argv++;
      param_aa = atoi(*argv);
      if ((param_aa <= 0) || (8 <= param_aa))
	EXIT_MSG1("Anti-alising factor is out of range. (1..8)\n");
    } else if ((strcmp(*argv, "-ox") == 0) && (argc > 1)){
      argc--;  argv++;
      param_offset_x = atof(*argv);
    } else if ((strcmp(*argv, "-oy") == 0) && (argc > 1)){
      argc--;  argv++;
      param_offset_y = atof(*argv);
    } else if (strcmp(*argv, "-l") == 0){
      param_orient = 1;
    } else if (window_size(&argv[0][1], param_orient, 
			   NULL, NULL, NULL, NULL) >= 0){
      param_paper = &argv[0][1];
    } else if (**argv != '-'){
      f = *argv;
    } else if ((strcmp(*argv, "-h") == 0) || (strcmp(*argv, "--help") == 0)){
      usage();
    } else 
      EXIT_MSG2("Unknow option: %s\n", *argv);
  }
  if (DVI_parse_device_mode(param_kpathsea_mode, &d, NULL, NULL) >= 0){
    param_dpi = d;
  } else {
    param_kpathsea_mode = DEFAULT_KPATHSEA_MODE;
    param_dpi = DEFAULT_DPI;
  }
  aa_nlevels = param_aa * param_aa + 1;
  shrink = DEFAULT_SHRINK * param_dpi / (DEFAULT_DPI * mag);
  calc_window_params(0);

  return  f;
}

void  usage(void)
{
  printf("spawx11 %s - %s\n", SPAWX11_VERSION,
	 "the Simplest Previewer with Anti-aliasing in the World for X11");
  printf("Usage: spwx11 [PaperSize] [Options] DVI-FILE\n");
  printf("PaperSize: -A1,..,-A7,-B1,..,-B7,-LETTER,-US,-LEGAL\n");
  printf("Options:  (Each value enclosed by [ ] is the default.)\n");
  printf(" -l         Landscape mode\n");
  printf(" -m MAG     Magnification (M < %d) [%.3f]\n", MAG_MAX, 1.0);
  printf(" -v PATH    A path name for font database file `vflibcap' [%s]\n", 
	 DEFAULT_VFLIBCAP);
  printf(" -cx, -sparcptr, -ljfour, ..., -MODE\n");
  printf(" -mode MODE  Device name for kpathsea [%s]\n", 
	 DEFAULT_KPATHSEA_MODE);
  printf(" -dpi DPI   Device resolution in dpi [%d]\n", DEFAULT_DPI);
  printf(" -fg C, -bg C    Foreground/background color [%s, %s]\n", 
	 DEFAULT_FG_COLOR, DEFAULT_BG_COLOR);
  printf(" -ox N, -oy N    Horizoltal/vertical offset in inch [%.3f, %.3f]\n",
	 DEFAULT_OFFSET_X, DEFAULT_OFFSET_Y);
  printf("Key and Mouse operation:\n");
  printf(" Right, Left Button   Go to the next / previous page\n");
  printf(" SPC, b  Go to the next / previous page\n");
  printf(" [, ]    Shrink / magnify the preview window\n");
  printf(" <, >    Goto the first / last page\n");
  printf(" l       Change paper orientation (portrait/landscape)\n");
  printf(" q       Quit\n");
  exit(0);
}

int  window_size(char *paper_name, int ori, int *w, int *h, int *w0, int *h0)
{
  int     i;
  struct { char *name; double w, h; } p[] ={
    {"A1", 23.4, 33.1},   {"A2", 16.5, 23.4},   {"A3", 11.7, 16.5},
    {"A4", 8.27, 11.7},   {"A5", 5.85, 8.27},   {"A6", 4.13, 5.85},
    {"A7", 2.92, 4.13},   {"B1", 27.8, 39.5},   {"B2", 19.7, 27.8},
    {"B3", 13.9, 19.7},   {"B4", 9.84, 13.9},   {"B5", 6.93, 9.84},
    {"B6", 5.32, 6.93},   {"B7", 3.47, 5.32},   {"LETTER", 8.5, 14.0},
    {"LEGAL", 8.5, 14.0}, {"US", 8.5, 11.0},    {NULL, 0, 0}  };

  for (i = 0; p[i].name != NULL; i++){
    if (strcmp(paper_name, p[i].name) == 0){
      if (w0 != NULL) 
	*w0 = ((ori==1)?(p[i].h):(p[i].w))*param_dpi*mag / shrink;
      if (w != NULL) 
	*w = ((ori==1)?(p[i].h):(p[i].w)) *param_dpi*mag / (shrink*param_aa);
      if (h0 != NULL)
	*h0 = ((ori==1)?(p[i].w):(p[i].h))*param_dpi*mag / shrink;
      if (h != NULL)
	*h = ((ori==1)?(p[i].w):(p[i].h))*param_dpi*mag  /(shrink*param_aa);
      return 0;
    }
  }
  return -1;
}

void calc_window_params(int resize_window)
{
  if (window_size(param_paper, param_orient, &x_w, &x_h, &f_w, &f_h) < 0)
    EXIT_MSG2("Unknown paper size: %s\n", param_paper);
  if (framebuff != NULL)
    DVI_fb_dispose(framebuff);
  if ((framebuff = DVI_fb_create((long)f_w, (long)f_h)) == NULL)
    EXIT_MSG1("No memory for frame buffer.\n");
  DVI_fb_clear(framebuff);
  x_ofsx = param_offset_x * param_dpi / shrink;
  x_ofsy = param_offset_y * param_dpi / shrink;
  if (resize_window == 1)
    XResizeWindow(x_disp, x_win, x_w, x_h);
}

int  main_loop(DVI_DEVICE dev, DVI dvi)
{
  int  last, need_draw;
  double  s;
  XEvent  xev;

  ui_cmd = CMD_NONE;
  for (thePage = last = 1; ; last = thePage){
    while (ui_cmd == CMD_NONE){
      XNextEvent(x_disp, &xevent);
      ui_cmd = x_parse_event(dvi, &xevent);
    }

    need_draw = 0;
    switch (ui_cmd){
    case CMD_EXPOSE:
      while (XCheckWindowEvent(x_disp, x_win, ExposureMask, &xev) == True)
	;
      need_draw = 1;
      break;
    case CMD_NEXT:
      thePage++; 
      need_draw = 1;
      break;
    case CMD_PREV:
      thePage--; 
      need_draw = 1;
      break;
    case CMD_PAGE_FIRST:
      thePage = 1;
      need_draw = 1;
      break;
    case CMD_PAGE_LAST:
      thePage = dvi->pages; 
      need_draw = 1;
      break;
    case CMD_PREV10:
      thePage -= 10; 
      if (thePage < 1)
	thePage = 1;
      need_draw = 1;
      break;
    case CMD_NEXT10:
      thePage += 10; 
      if (thePage > dvi->pages)
	thePage = dvi->pages;
      need_draw = 1;
      break;
    case CMD_SWAP_ORIENT: 
      param_orient = (param_orient == 0) ? 1 : 0; 
      calc_window_params(1);
      break;
    case CMD_ENLARGE: 
      if ((s = mag*10/8) < MAG_MAX)
	mag = s;
      calc_window_params(1);
      break;
    case CMD_SHRINK: 
      if ((s = mag*8/10) > MAG_MIN)
	mag = s;
      calc_window_params(1);
    case CMD_RELOAD:
      DVI_DISPOSE(dvi, dev);
      dvi = NULL;
      if ((dvi = DVI_CREATE(dev, dvi_file, dvi_props)) == NULL)
	EXIT_MSG2("Can't open DVI file: %s\n", dvi_file);
      if (DVI_OPEN_FONT(dvi, dev) < 0)
	EXIT_MSG1("Cannot find all fonts.\n");
      if (dvi->pages < thePage)
	thePage = dvi->pages;
      need_draw = 1;
      break;
    }

    if (ui_cmd != CMD_EXPOSE){
      x_change_cursor(x_c_drawing);
      DVI_fb_clear(framebuff);
      XClearWindow(x_disp, x_win);
    }
    if (need_draw == 1){
      if (DVI_DRAW_PAGE(dvi, dev, thePage, shrink/mag)
	  != DVI_DRAW_INTERRUPTED){
	ui_cmd = CMD_NONE;
	x_change_cursor(x_c_ready);
      }
    } else {
      ui_cmd = CMD_NONE;
    }
  }
}

DVI_DEVICE  create_dev(void)
{
  if ((dev = DVI_DEVICE_ALLOC()) == NULL)
    return NULL;
  dev->h_dpi = dev->v_dpi   = param_dpi;
  dev->device_polling       = DEV_poll;
  dev->put_rectangle        = DEV_put_rectangle;
  dev->put_bitmap           = DEV_put_bitmap;
#if 1
  dev->message_advice       = NULL;
#endif
  dev->draw_ps_figures      = DEV_draw_ps_figures;
  dev->before_ps_figure     = DEV_before_ps_figure;
  dev->after_ps_figure      = DEV_after_ps_figure;
  return dev;
}

void  DEV_put_bitmap(DVI_DEVICE dev, DVI dvi, DVI_BITMAP bm, int font_id,
		     long key2, long code_point, long x, long y)
{
  DVI_fb_put_bitmap(framebuff, bm, x + x_ofsx, y + x_ofsy);
  x_aa_put(x + x_ofsx, y + x_ofsy, bm->width, bm->height);
}

void  DEV_put_rectangle(DVI_DEVICE dev, DVI dvi, 
			long x, long y, long w, long h)
{
  DVI_fb_put_rectangle(framebuff, x + x_ofsx, y + x_ofsy, w, h);
  x_aa_put(x + x_ofsx, y + x_ofsy, w, h);
}

int   DEV_before_ps_figure(DVI_DEVICE dev, DVI dvi, char *cmd, long x, long y)
{
  x_change_cursor(x_c_ps);
  return 1;
}

void  DEV_after_ps_figure(DVI_DEVICE dev, DVI dvi)
{
  x_change_cursor(x_c_drawing);
}

int  DEV_draw_ps_figures(DVI_DEVICE dev, DVI dvi)
{
  if (param_draw_eps_figure == 1)
    return 1;
  return 0;
}

int  DEV_poll(DVI_DEVICE dev, DVI dvi, int poll_type)
{
  static int  t = 0;

  if ((poll_type != DVI_POLL_PAGE) || ((--t) > 0))
    return 0;
  t = DEFAULT_POLLING_INTERVAL;
  if ((XCheckWindowEvent(x_disp, x_win, XEVENTS, &xevent) == False)
      || ((ui_cmd = x_parse_event(dvi, &xevent)) == CMD_NONE))
    return 0;
  return 1;
}

void  x_init(void)
{
  XColor        col, xc_fg, xc_bg, xc;
  int           i;

  if ((x_disp = XOpenDisplay(NULL)) == NULL)
    EXIT_MSG1("Can't open X display.\n");
  XAllocNamedColor(x_disp, DefaultColormap(x_disp,0), param_fg, &xc_fg, &xc); 
  XAllocNamedColor(x_disp, DefaultColormap(x_disp,0), param_bg, &xc_bg, &xc); 
  x_pix_fg = xc_fg.pixel;
  x_pix_bg = xc_bg.pixel;
  x_c_reading = XCreateFontCursor(x_disp, XC_shuttle);
  x_c_drawing = XCreateFontCursor(x_disp, XC_watch);
  x_c_ready   = XCreateFontCursor(x_disp, XC_circle);
  x_c_ps      = XCreateFontCursor(x_disp, XC_coffee_mug);
  x_win = XCreateSimpleWindow(x_disp, RootWindow(x_disp, 0), 0, 0, x_w, x_h, 
			      2, XBlackPixel(x_disp,0), x_pix_bg);
  x_gc_win = XCreateGC(x_disp, x_win, 0, 0);
  x_pix_table = (unsigned long*)malloc(aa_nlevels * sizeof(unsigned long));
  if (x_pix_table == NULL)
    EXIT_MSG1("No memory.\n");
  for (i = 0; i < aa_nlevels; i++){
    col.red   = xc_bg.red   
      + ceil(((double)(xc_fg.red   - xc_bg.red)   * i)/(double)(aa_nlevels-1));
    col.green = xc_bg.green 
      + ceil(((double)(xc_fg.green - xc_bg.green) * i)/(double)(aa_nlevels-1));
    col.blue  = xc_bg.blue
      + ceil(((double)(xc_fg.blue  - xc_bg.blue)  * i)/(double)(aa_nlevels-1));
    col.flags = DoRed | DoGreen | DoBlue;
    if (XAllocColor(x_disp, DefaultColormap(x_disp, 0), &col) == 0)
      EXIT_MSG1("Cannot allocate colors.\n");
    x_pix_table[i] = col.pixel;
  }
  XSetForeground(x_disp, x_gc_win, x_pix_table[0]);
  XSetBackground(x_disp, x_gc_win, x_pix_table[aa_nlevels-1]);
  XStoreName(x_disp, x_win, "spawx11");
  XSelectInput(x_disp, x_win, XEVENTS);
  XMapWindow(x_disp, x_win);
  XClearWindow(x_disp, x_win);
}

int  x_parse_event(DVI dvi, XEvent *xev)
{
  char   keyin[16];
  KeySym ks;

  switch (xev->type){
  case Expose:     return CMD_EXPOSE;
  case ButtonPress:
    switch (xev->xbutton.button){
    case Button1:   return (1 < thePage) ?          CMD_PREV : CMD_NONE;
    case Button3:   return (thePage < dvi->pages) ? CMD_NEXT : CMD_NONE;
    }
    break;
  case KeyPress:
    if (XLookupString(&(xev->xkey), keyin, sizeof(keyin), &ks, NULL) != 1){
      switch (ks){
      case XK_space:    return (thePage < dvi->pages) ? CMD_NEXT : CMD_NONE;
      case XK_Delete:   return (1 < thePage) ? CMD_PREV : CMD_NONE;
      case XK_Begin: case XK_Home:   return CMD_PAGE_FIRST;
      case XK_End:                   return CMD_PAGE_LAST;
      }
    } else {
      switch (keyin[0]){
      case '[': case '{': case '-':  return CMD_SHRINK;
      case ']': case '}': case '+':  return CMD_ENLARGE;
      case ' ':            return (thePage < dvi->pages) ? CMD_NEXT : CMD_NONE;
      case 'b': case 'B':  return (1 < thePage) ? CMD_PREV : CMD_NONE;
      case '<':            return CMD_PAGE_FIRST;
      case '>':            return CMD_PAGE_LAST;
      case ',':            return CMD_PREV10;
      case '.':            return CMD_NEXT10;
      case 'l': case 'L':  return CMD_SWAP_ORIENT;
      case 'r': case 'R':  return CMD_RELOAD;
      case 'q': case 'Q':  exit(0);
      }
    }
  }
  return CMD_NONE;
}

void  x_change_cursor(Cursor new)
{
  XDefineCursor(x_disp, x_win, new);
  XFlush(x_disp); 
}

void  x_aa_put(long pos_x, long pos_y, long w, long h)
{
  XImage         *x_image;
  long            aabw, aabh, x, y, z;
  unsigned short *aa_buff;

  if (pos_x < 0){
    w += pos_x;
    pos_x = 0;
  }
  if (pos_y < 0){
    h += pos_y;
    pos_y = 0;
  }
  if (pos_x + w > framebuff->width)
    w = framebuff->width - pos_x;
  if (pos_y + h > framebuff->height)
    h = framebuff->height - pos_y;
  if ((w <= 0) || (h <= 0))
    return;

  if ((aa_buff = DVI_fb_antialias(framebuff, param_aa, pos_x, pos_y, 
				  w, h, &aabw, &aabh)) == NULL)
    return;
  if ((x_image = XCreateImage(x_disp, 
			      DefaultVisual(x_disp,0), DefaultDepth(x_disp,0),
			      ZPixmap, 0, NULL, aabw, aabh, 8, 0)) == NULL)
    return;
  z = x_image->bytes_per_line * aabh;
  if ((x_image->data = (char*)malloc((z!=0)?z:1)) == NULL)
    return;
  for (y = 0; y < aabh; y++)
    for (x = 0; x < aabw; x++)
      XPutPixel(x_image, x, y, x_pix_table[aa_buff[aabw * y + x]]);
  XPutImage(x_disp, x_win, x_gc_win, x_image, 0, 0, 
	    pos_x/param_aa, pos_y/param_aa, aabw, aabh);
  XDestroyImage(x_image);
  free(aa_buff);
}
