/*
    Theseus - maximum likelihood superpositioning of macromolecular structures

    Copyright (C) 2004-2007 Douglas L. Theobald

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the:

    Free Software Foundation, Inc.,
    59 Temple Place, Suite 330,
    Boston, MA  02111-1307  USA

    -/_|:|_|_\-
*/

#include "pdbStats_local.h"
#include "myassert.h"

static void
Vars2Bfacts(CoordsArray *cdsA);

void
CheckVars(CoordsArray *cdsA)
{
    int     i;

    for(i = 0; i < cdsA->vlen; ++i)
    {
        if (!isfinite(cdsA->var[i]) || cdsA->var[i] < DBL_EPSILON)
        {
            printf("Bad variance: %4d % e\n", i, cdsA->var[i]);
            fflush(NULL);
        }
    }
}


/* Calculate superposition stats */
void
CalcStats(CoordsArray *incdsA)
{
    int             i, round;
    int tmph;
    double          smallestRMSD, n;
    Algorithm      *algo = incdsA->algo;
    const int       cnum = incdsA->cnum;
    const int       vlen = incdsA->vlen;
    double         *evals = malloc(3 * sizeof(double));
    double        **rotmat = MatInit(3, 3);
    double        **lastmat = MatInit(3,3);
    CoordsArray    *cdsA = NULL;

    cdsA = CoordsArrayInit();
    CoordsArrayAlloc(cdsA, cnum, vlen);
    CoordsArraySetup(cdsA);
    CoordsArrayCopy(cdsA, incdsA);

    if (algo->covweight == 1)
        SetupCovWeighting(cdsA);
    else if (algo->pca > 0)
        if (cdsA->CovMat == NULL)
            cdsA->CovMat = MatInit(vlen, vlen);

    if (algo->alignment == 1)
        CalcDf(cdsA);

    if (algo->bfact > 0)
    {
        for (i = 0; i < cnum; ++i)
            Bfacts2PrVars(cdsA, i);
    }

    if (algo->noave == 0 &&
        algo->jackknife == 0)
    {
        AveCoords(cdsA);

        if (algo->mbias == 1)
            UnbiasMean(cdsA);
    }

    round = 0;
    while(1)
    {
        ++round;

        Mat3Cpy(lastmat, (const double **) cdsA->AxCovMat);

        /* weighting by dimensional, axial Xi covariance matrix, here diagonal */
        CalcCovariances(cdsA);
        /* CheckVars(cdsA); */
        CalcWts(cdsA);

        if (MatFrobNorm((const double **) lastmat, (const double **) cdsA->AxCovMat, 3, 3) < algo->precision)
               break;
    }

    /* CheckVars(cdsA); */
    if (algo->leastsquares == 1)
        CalcNormResidualsLS(cdsA);
    else
        CalcNormResiduals(cdsA);
    /* CheckVars(cdsA); */

    if (algo->write_file == 1)
        WriteResiduals(cdsA, mystrcat(algo->rootname, "_residuals.txt"));

    CalcLogL(cdsA);
    /* Mat3Print(cdsA->AxCovMat); */
    CalcAIC(cdsA);
    CalcBIC(cdsA);
    CalcMLRMSD(cdsA);
    CalcPRMSD(cdsA);

    Vars2Bfacts(cdsA);

/*     SkewnessCoords(cdsA); */
/*     KurtosisCoords(cdsA); */
    MomentsCoords(cdsA);

    cdsA->stats->omnibus_chi2 = (vlen * cdsA->stats->hierarch_chi2 + vlen * cnum * 3 * cdsA->stats->chi2) / (vlen * cnum * 3 + vlen);
    cdsA->stats->omnibus_chi2_P = chisqr_sdf(vlen * cdsA->stats->hierarch_chi2 + vlen * cnum * 3 * cdsA->stats->chi2,
                                      vlen * cnum * 3 + vlen, 0);

    n = (double) vlen * cnum * 3;
	cdsA->stats->SES = sqrt((6.0 * n * (n-1)) / ((n-2) * (n+1) * (n+3))); /* exact formulas */
	cdsA->stats->SEK = sqrt((24.0 * n * (n-1) * (n-1)) / ((n-3) * (n-2) * (n+3) * (n+5)));

    if (algo->write_file == 1)
        WriteVariance(cdsA, mystrcat(algo->rootname, "_variances.txt"));

    if (algo->covweight == 1 && (algo->write_file > 0 || algo->info != 0) && algo->pca == 0)
    {
/*         if (algo->alignment == 1) */
/*             CalcCovMatOcc(cdsA); */
/*         else */
/*             CalcCovMat(cdsA); */

        PrintCovMatGnuPlot((const double **) cdsA->CovMat, vlen, mystrcat(algo->rootname, "_cov.mat"));
        CovMat2CorMat(cdsA->CovMat, vlen);
        PrintCovMatGnuPlot((const double **) cdsA->CovMat, vlen, mystrcat(algo->rootname, "_cor.mat"));
    }

    if (algo->fullpca == 1)
    {
        printf("    Calculating anisotropic Principal Components of the superposition ... \n");
        fflush(NULL);

        if (cdsA->FullCovMat == NULL)
            cdsA->FullCovMat = MatInit(3 * vlen, 3 * vlen);

        CalcFullCovMat(cdsA);
        /* CalcLedoitFullCovMat(cdsA); */
        Calc3NPCA(cdsA); /* PCA analysis of covariance matrix */
    }
    else if (algo->pca > 0)
    {
        printf("    Calculating isotropic Principal Components of the superposition ... \n");
        fflush(NULL);

        if (algo->alignment == 1)
            CalcCovMatOcc(cdsA);
        else
            CalcCovMat(cdsA);

        tmph = cdsA->algo->hierarch;

/*         if (cdsA->algo->hierarch >= 1 && cdsA->algo->hierarch <= 8) */ /* DLT -- I don't understand why I was using this; do I need it? */
/*             cdsA->algo->hierarch = 7; */
/*         else */
/*             cdsA->algo->hierarch = 12; */

        HierarchVars(cdsA);
        cdsA->algo->hierarch = tmph;
        //CalcLedoitCovMatPar(cdsA);
        CalcPCA(cdsA); /* PCA analysis of covariance matrix */
    }

    if (algo->modelpca == 1)
    {
        printf("    Calculating Principal Components across models ... \n");
        fflush(NULL);

        CalcStructPCA(cdsA);
    }

    if (algo->stats == 1)
    {
        RadiusGyration(cdsA->avecoords, cdsA->w);

        for (i = 0; i < cnum; ++i)
            RadiusGyration(cdsA->coords[i], cdsA->w);

        Durbin_Watson(cdsA);
    }

    printf("    Calculating likelihood statistics ... \n");
    fflush(NULL);

    smallestRMSD = DBL_MAX;
    for (i = 0; i < cnum; ++i)
    {
        if (smallestRMSD > cdsA->coords[i]->wRMSD_from_mean)
        {
            smallestRMSD = cdsA->coords[i]->wRMSD_from_mean;
            cdsA->stats->median = i;
        }
    }

    CopyStats(incdsA, cdsA);

    free(evals);
    MatDestroy(rotmat);
    MatDestroy(lastmat);
    CoordsArrayDestroy(cdsA);
}


void
CalcPreStats(CoordsArray *cdsA)
{
    CalcStats(cdsA);

    cdsA->stats->starting_stddev = cdsA->stats->stddev;
    cdsA->stats->starting_paRMSD = cdsA->stats->ave_paRMSD;
    cdsA->stats->starting_pawRMSD = cdsA->stats->ave_pawRMSD;
    cdsA->stats->starting_ave_wRMSD_from_mean = cdsA->stats->ave_pawRMSD * sqrt((double)(cdsA->cnum - 1) / (double)(2 * cdsA->cnum));
    cdsA->stats->starting_mlRMSD = cdsA->stats->mlRMSD;
    cdsA->stats->starting_logL = cdsA->stats->logL;
}


void
CopyStats(CoordsArray *cdsA1, CoordsArray *cdsA2)
{
    int             i;
    Coords        **coords1 = cdsA1->coords, **coords2 = cdsA2->coords;
    Statistics     *stats1 = cdsA1->stats, *stats2 = cdsA2->stats;
    const int       cnum = cdsA1->cnum;
    const int       vlen = cdsA1->vlen;

    memcpy(stats1, stats2, sizeof(Statistics));

    memcpy(stats1->skewness, stats2->skewness, 4 * sizeof(double));
    memcpy(stats1->kurtosis, stats2->kurtosis, 4 * sizeof(double));
    memcpy(cdsA1->axesw, cdsA2->axesw, 3 * sizeof(double));
    //memcpy(&cdsA1->AxCovMat[0][0], &cdsA2->AxCovMat[0][0], 9 * sizeof(double)); /* DLT debug FIX */

    cdsA1->avecoords->radgyr = cdsA2->avecoords->radgyr;

    memcpy(cdsA1->var, cdsA2->var, vlen * sizeof(double));
    memcpy(cdsA1->w, cdsA2->w, vlen * sizeof(double));

    for (i = 0; i < cnum; ++i)
    {
        coords1[i]->radgyr = coords2[i]->radgyr;
        MatCpySym(coords1[i]->matrix, (const double **) coords2[i]->matrix, 3);
        MatCpySym(coords1[i]->evecs, (const double **) coords2[i]->evecs, 4);
        memcpy(coords1[i]->evals, coords2[i]->evals, 4 * sizeof(double));
        memcpy(coords1[i]->center, coords2[i]->center, 4 * sizeof(double));
        memcpy(coords1[i]->translation, coords2[i]->translation, 4 * sizeof(double));
        coords1[i]->ref_wRMSD_from_mean = coords2[i]->ref_wRMSD_from_mean;
        coords1[i]->wRMSD_from_mean = coords2[i]->wRMSD_from_mean;
    }

    CoordsCopyAll(cdsA1->avecoords, cdsA2->avecoords);
    cdsA1->pcamat = cdsA2->pcamat;
    cdsA1->pcavals = cdsA2->pcavals;
    cdsA1->modpcamat = cdsA2->modpcamat;
    cdsA1->modpcavals = cdsA2->modpcavals;
}


void
CalcDf(CoordsArray *cdsA)
{
    int         i,j;

	for (i = 0; i < cdsA->vlen; ++i)
	{
		cdsA->df[i] = 0;
		for (j = 0; j < cdsA->cnum; ++j)
			cdsA->df[i] += cdsA->coords[j]->o[i];

		//cdsA->df[i] *= 3;
    }
}


/* Calculates the atomic variances for a family of Coords */
/* returns the standard deviation */
double
VarianceCoordsNoVec(CoordsArray *cdsA)
{
    int             i, j;
    double          sqrx, sqry, sqrz, sqrdist;
    double          tmpx, tmpy, tmpz;
    double          variance;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const double    idf = 1.0 / (3 * cnum);
    double         *var = cdsA->var;
    const Coords  **coords = (const Coords **) cdsA->coords;
    Coords         *coordsj;
    const double   *axesw = cdsA->axesw;
    const double   *avex = (const double *) cdsA->avecoords->x,
                   *avey = (const double *) cdsA->avecoords->y,
                   *avez = (const double *) cdsA->avecoords->z;

    variance = 0.0;
    for (i = 0; i < vlen; ++i)
    {
        sqrx = sqry = sqrz = 0.0;
        for (j = 0; j < cnum; ++j)
        {
            coordsj = (Coords *) coords[j];
            tmpx = coordsj->x[i] - avex[i];
            sqrx += tmpx * tmpx;
            tmpy = coordsj->y[i] - avey[i];
            sqry += tmpy * tmpy;
            tmpz = coordsj->z[i] - avez[i];
            sqrz += tmpz * tmpz;
        }

        sqrdist = (sqrx * axesw[0]) + (sqry * axesw[1]) + (sqrz * axesw[2]);
        var[i] = sqrdist * idf;
        variance += sqrdist;
    }

    variance /= (vlen * cnum);
    cdsA->stats->stddev = sqrt(variance);

    return(cdsA->stats->stddev);
}


double
VarianceCoords(CoordsArray *cdsA)
{
    int             i, j;
    double          tmpx, tmpy, tmpz;
    double          variance;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const double    idf = 1.0 / (3 * cnum);
    double         *var = cdsA->var;
    const Coords  **coords = (const Coords **) cdsA->coords;
    Coords         *coordsj;
    const double   *axesw = cdsA->axesw;
    const double   *avex = (const double *) cdsA->avecoords->x,
                   *avey = (const double *) cdsA->avecoords->y,
                   *avez = (const double *) cdsA->avecoords->z;

    memset(var, 0, vlen * sizeof(double));

    for (j = 0; j < cnum; ++j)
    {
        coordsj = (Coords *) coords[j];

		for (i = 0; i < vlen; ++i)
		{
			tmpx = coordsj->x[i] - avex[i];
			var[i] += tmpx * tmpx * axesw[0];
			tmpy = coordsj->y[i] - avey[i];
			var[i] += tmpy * tmpy * axesw[1];
			tmpz = coordsj->z[i] - avez[i];
			var[i] += tmpz * tmpz * axesw[2];
		}
    }

    for (i = 0; i < vlen; ++i)
        var[i] *= idf;

    variance = 0.0;
    for (i = 0; i < vlen; ++i)
        variance += var[i];

    variance /= vlen;
    cdsA->stats->stddev = sqrt(variance);

    return(cdsA->stats->stddev);
}
            

/* Calculates variances weighted by the inverse of a non-diagonal 3x3 dimensional covariance matrix */
double
VarianceCoordsFullAx(CoordsArray *cdsA)
{
    int             i, j, k, m;
    double          variance, tmpv[3];
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const double    idf = 1.0 / (3 * cnum);
    double         *var = cdsA->var;
    const Coords  **coords = (const Coords **) cdsA->coords;
    Coords         *coordsm;
    const double   *avex = (const double *) cdsA->avecoords->x,
                   *avey = (const double *) cdsA->avecoords->y,
                   *avez = (const double *) cdsA->avecoords->z;
    double        **WtMat = cdsA->AxWtMat;

    InvSymEigenOp(WtMat, (const double **) cdsA->AxCovMat, 3, cdsA->tmpvec3a, cdsA->tmpmat3d, DBL_MIN);

    memset(var, 0, vlen * sizeof(double));

    for (m = 0; m < cnum; ++m)
    {
        /* (i x k)(k x j) = (i x j) */
        for (i = 0; i < vlen; ++i)
        {
            coordsm = (Coords *) coords[m];
            tmpv[0] = coordsm->x[i] - avex[i];
            tmpv[1] = coordsm->y[i] - avey[i];
            tmpv[2] = coordsm->z[i] - avez[i];

            for (j = 0; j < 3; ++j)
                for (k = 0; k < 3; ++k)
                    var[i] += tmpv[k] * WtMat[k][j] * tmpv[j];
        }
    }

    variance = 0.0;
    for (i = 0; i < vlen; ++i)
    {
        variance += var[i];
        var[i] *= idf;
    }

    variance /= (vlen * cnum);
    cdsA->stats->stddev = sqrt(variance);

    return(cdsA->stats->stddev);
}


/* Same as VarianceCoords() but weights by occupancies */
double
VarianceCoordsOcc(CoordsArray *cdsA)
{
    int             i, j;
    double          sqrx, sqry, sqrz, sqrdist;
    double          tmpx, tmpy, tmpz;
    double          variance;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    double         *var = cdsA->var;
    const Coords  **coords = (const Coords **) cdsA->coords;
    Coords         *coordsj;
    const double   *axesw = cdsA->axesw;
    const double   *avex = (const double *) cdsA->avecoords->x,
                   *avey = (const double *) cdsA->avecoords->y,
                   *avez = (const double *) cdsA->avecoords->z;
    double         occ;

    variance = 0.0;
    for (i = 0; i < vlen; ++i)
    {
        sqrx = sqry = sqrz = 0.0;
        for (j = 0; j < cnum; ++j)
        {
            coordsj = (Coords *) coords[j];
            occ = coordsj->o[i];
            tmpx = coordsj->x[i] - avex[i];
            sqrx += occ * tmpx * tmpx;
            tmpy = coordsj->y[i] - avey[i];
            sqry += occ * tmpy * tmpy;
            tmpz = coordsj->z[i] - avez[i];
            sqrz += occ * tmpz * tmpz;

//         printf("\n%4d %4d %e - %e %e %e - %e %e %e",
//                i, j, occ,
//                coordsj->x[i], coordsj->y[i], coordsj->z[i],
//                avex[i], avey[i], avez[i]);
        }

        sqrdist = (sqrx * axesw[0]) + (sqry * axesw[1]) + (sqrz * axesw[2]);

        var[i] = sqrdist / (3.0 * cdsA->df[i]);
        variance += sqrdist;
    }

    //WriteVariance(cdsA, "jacob.log"); exit(1);
    variance /= (double) (vlen * cnum);
    cdsA->stats->stddev = sqrt(variance);

    return(cdsA->stats->stddev);
}


/* void */
/* WriteTransformations(CoordsArray *cdsA, char *outfile_name) */
/* { */
/*     FILE           *transfile = NULL; */
/*     int             i, j; */
/*  */
/*     transfile = myfopen(outfile_name, "w"); */
/*     if (transfile == NULL) */
/*     { */
/*         perror("\n  ERROR"); */
/*         fprintf(stderr, */
/*                 "\n  ERROR99: could not open file '%s' for writing. \n", outfile_name); */
/*         PrintTheseusTag(); */
/*         exit(EXIT_FAILURE); */
/*     } */
/*  */
/*     fprintf(transfile, "# Translation vectors\n"); */
/*  */
/* 	for (i = 0; i < cdsA->cnum; ++i) */
/* 	{ */
/* 		fprintf(transfile, */
/* 		        "MODEL %3d, t: %9.4f %9.4f %9.4f\n", */
/* 		        i+1, */
/* 			    cdsA->coords[i]->translation[0], */
/* 			    cdsA->coords[i]->translation[1], */
/* 			    cdsA->coords[i]->translation[2]); */
/* 	} */
/*  */
/*     fprintf(transfile, "\n# Rotation matrices\n"); */
/*  */
/* 	for (i = 0; i < cdsA->cnum; ++i) */
/* 	{ */
/* 		fprintf(transfile, "MODEL %3d, R: ", i+1); */
/*  */
/* 		for (j = 0; j < 3; ++j) */
/*         { */
/*             fprintf(transfile, */
/*                     "% 10.7f % 10.7f % 10.7f    ", */
/*                     cdsA->coords[i]->matrix[j][0], */
/*                     cdsA->coords[i]->matrix[j][1], */
/*                     cdsA->coords[i]->matrix[j][2]); */
/*         } */
/*  */
/*         fputc('\n', transfile); */
/* 	} */
/*  */
/*     fprintf(transfile, "\n\n"); */
/* 	fflush(NULL); */
/*  */
/*     fclose(transfile); */
/* } */


void
WriteTransformations(CoordsArray *cdsA, char *outfile_name)
{
    FILE           *transfile = NULL;
    int             i, j;
    double          angle, *v = malloc(3*sizeof(double));

    transfile = myfopen(outfile_name, "w");
    if (transfile == NULL)
    {
        perror("\n  ERROR");
        fprintf(stderr,
                "\n  ERROR99: could not open file '%s' for writing. \n", outfile_name);
        PrintTheseusTag();
        exit(EXIT_FAILURE);
    }

    fprintf(transfile, "# Translation vectors\n");

	for (i = 0; i < cdsA->cnum; ++i)
	{
		fprintf(transfile,
		        "MODEL %3d, t: %9.4f %9.4f %9.4f\n",
		        i+1,
			    cdsA->coords[i]->translation[0],
			    cdsA->coords[i]->translation[1],
			    cdsA->coords[i]->translation[2]);
	}

    fprintf(transfile, "\n# Rotation matrices\n");

	for (i = 0; i < cdsA->cnum; ++i)
	{
		fprintf(transfile, "MODEL %3d, R: ", i+1);

		for (j = 0; j < 3; ++j)
        {
            fprintf(transfile,
                    "% 10.7f % 10.7f % 10.7f    ",
                    cdsA->coords[i]->matrix[j][0],
                    cdsA->coords[i]->matrix[j][1],
                    cdsA->coords[i]->matrix[j][2]);
        }

        fputc('\n', transfile);
	}

    fprintf(transfile, "\n# Rotations, angle-axis representation\n");

	for (i = 0; i < cdsA->cnum; ++i)
	{
	    angle = RotMat2AxisAngle(cdsA->coords[i]->matrix, v);

		fprintf(transfile,
		        "MODEL %3d, Angle: % 10.7f  Axis: % 10.7f % 10.7f % 10.7f\n",
		        i+1, angle, v[0], v[1], v[2]);
	}

    fprintf(transfile, "\n\n");
	fflush(NULL);

    free(v);
    fclose(transfile);
}


void
WriteVariance(CoordsArray *cdsA, char *outfile_name)
{
    FILE           *varfile = NULL;
    int             i, cnum = cdsA->cnum;

    varfile = myfopen(outfile_name, "w");
    if (varfile == NULL)
    {
        perror("\n  ERROR");
        fprintf(stderr,
                "\n  ERROR99: could not open file '%s' for writing. \n",
                outfile_name);
        PrintTheseusTag();
        exit(EXIT_FAILURE);
    }

    fprintf(varfile, "#ATOM   resName resSeq     variance      std dev         RMSD pseudo-Bfact\n");

    if (cdsA->algo->varweight == 1 || cdsA->algo->leastsquares == 1)
    {
		for (i = 0; i < cdsA->vlen; ++i)
		{
			fprintf(varfile,
					"%-5d       %3s %6d %12.6f %12.6f %12.6f %12.6f\n",
					i+1,
					cdsA->coords[0]->resName[i],
					cdsA->coords[0]->resSeq[i],
					cdsA->var[i],
					sqrt(cdsA->var[i]),
					sqrt(2.0 * cdsA->var[i] * (cnum - 1) / cnum),
					cdsA->avecoords->b[i]);
		}
    }
    else if (cdsA->algo->covweight == 1)
    {
		for (i = 0; i < cdsA->vlen; ++i)
		{
			fprintf(varfile,
					"%-5d      %3s %6d %12.6f %12.6f %12.6f %12.6f\n",
					i+1,
					cdsA->coords[0]->resName[i],
					cdsA->coords[0]->resSeq[i],
					cdsA->CovMat[i][i],
					sqrt(cdsA->CovMat[i][i]),
					sqrt(2.0 * cdsA->CovMat[i][i] * (cnum - 1) / cnum),
					cdsA->avecoords->b[i]);
		}
    }

    fputc('\n', varfile);
	fflush(NULL);

    /* printf("\nVariance range: %f\n", log(VecBiggest(cdsA->var, cdsA->vlen)/VecSmallest(cdsA->var, cdsA->vlen))/log(10.0)); */

    fclose(varfile);
}


static void
Vars2Bfacts(CoordsArray *cdsA)
{
    int         i;
    double      bfact = 8.0 * MY_PI * MY_PI;

	for (i = 0; i < cdsA->vlen; ++i)
	{
		 cdsA->avecoords->b[i] = cdsA->var[i] * bfact;

		 if (cdsA->avecoords->b[i] > 99.99)
			 cdsA->avecoords->b[i] = 99.99;
    }
}


void
Bfacts2PrVars(CoordsArray *cdsA, int coord)
{
    int         i;
    double      bfact = 1.0 / (24.0 * MY_PI * MY_PI);

	for (i = 0; i < cdsA->vlen; ++i)
		cdsA->coords[coord]->prvar[i] = cdsA->coords[coord]->b[i] * bfact;

    for (i = 0; i < cdsA->vlen; ++i)
    {
        myassert(cdsA->coords[coord]->prvar[i] > 0.0);
        if (cdsA->coords[coord]->prvar[i] == 0.0)
            cdsA->coords[coord]->prvar[i] = 0.3;
    }
}


/* average of all possible unique pairwise RMSDs */
/* Eqn 1 and following paragraph, Kearsley, S.K. (1990) "An algorithm for the 
   simultaneous superposition of a structural series." Journal of Computational 
   Chemistry 11(10):1187-1192. */
double
CalcPRMSD(CoordsArray *cdsA)
{
    int             i, j, k;
    double          sqrdist;
    double          wsqrdist;
    double          paRMSD, pawRMSD;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const Coords  **coords = (const Coords **) cdsA->coords;
    const Coords   *coordsi, *coordsj;
    const double   *w = (const double *) cdsA->w;
    double   axesw[3];

    for (i = 0; i < 3; ++i)
        axesw[i]= 1.0 / cdsA->AxCovMat[i][i];

    sqrdist = wsqrdist = 0.0;
    for (i = 0; i < cnum; ++i)
    {
        coordsi = coords[i];

        for (j = 0; j < i; ++j)
        {
            coordsj = coords[j];

            for (k = 0; k < vlen; ++k)
            {
                sqrdist += SqrCoordsDist(coordsi, k, coordsj, k);
                wsqrdist += (w[k] * SqrCoordsDistMahal(coordsi, k, coordsj, k, axesw));
            }
        }
    }

    paRMSD  = (2.0 *  sqrdist) / (double) (vlen * cnum * (cnum - 1));
    pawRMSD = (2.0 * wsqrdist) / (double) (vlen * cnum * (cnum - 1));
    cdsA->stats->ave_pawRMSD = sqrt(pawRMSD);
    cdsA->stats->ave_paRMSD  = sqrt(paRMSD);

    return (cdsA->stats->ave_paRMSD);
}


/* Calculate a matrix normal maximum likelihood RMSD,
   based on the traces of the inverse covariance matrices --
   this is an RMSD from the mean (a sigma value),
   _not_ an average pairwise RMSD. */
double
CalcMLRMSD(CoordsArray *cdsA)
{
    int             i, vlen = cdsA->vlen;
    double         *variance = cdsA->var;
    double          trAx;
    Algorithm      *algo = cdsA->algo;

    trAx = (1.0 / cdsA->AxCovMat[0][0]) +
           (1.0 / cdsA->AxCovMat[1][1]) +
           (1.0 / cdsA->AxCovMat[2][2]);

    if (algo->covweight == 1)
    {
        cdsA->stats->mlRMSD = sqrt(3.0 * cdsA->stats->wtnorm / trAx);
    }
    else if (algo->varweight == 1)
    {
        cdsA->stats->mlRMSD = 0.0;
        for (i = 0; i < vlen; ++i)
            cdsA->stats->mlRMSD += (1.0 / variance[i]);

        cdsA->stats->mlRMSD = sqrt(3.0 * vlen / (cdsA->stats->mlRMSD * trAx));
    }
    else
    {
        cdsA->stats->mlRMSD = 0.0;
        for (i = 0; i < vlen; ++i)
            cdsA->stats->mlRMSD += variance[i];

        cdsA->stats->mlRMSD = sqrt(3.0 * cdsA->stats->mlRMSD / (vlen * trAx));
    }

    return(cdsA->stats->mlRMSD);
}


double
SqrCoordsDist(const Coords *coords1, const int atom1,
              const Coords *coords2, const int atom2)
{
    double          tmpx, tmpy, tmpz;

    tmpx = coords2->x[atom2] - coords1->x[atom1];
    tmpy = coords2->y[atom2] - coords1->y[atom1];
    tmpz = coords2->z[atom2] - coords1->z[atom1];

    return(tmpx*tmpx + tmpy*tmpy + tmpz*tmpz);
}


/* double */
/* SqrCoordsDistMahal(const Coords *coords1, const int atom1, */
/*                    const Coords *coords2, const int atom2, */
/*                    const double *weights) */
/* { */
/*     double          xdist, ydist, zdist; */
/*  */
/*     xdist = weights[0] * mysquare(coords2->x[atom2] - coords1->x[atom1]); */
/*     ydist = weights[1] * mysquare(coords2->y[atom2] - coords1->y[atom1]); */
/*     zdist = weights[2] * mysquare(coords2->z[atom2] - coords1->z[atom1]); */
/*  */
/*     return(xdist + ydist + zdist); */
/* } */


double
SqrCoordsDistMahal(const Coords *coords1, const int atom1,
                   const Coords *coords2, const int atom2,
                   const double *weights)
{
    return(weights[0] * mysquare(coords2->x[atom2] - coords1->x[atom1]) +
           weights[1] * mysquare(coords2->y[atom2] - coords1->y[atom1]) + 
           weights[2] * mysquare(coords2->z[atom2] - coords1->z[atom1]));
}


double
SqrCoordsDistMahal2(const Coords *coords1, const int atom1,
                    const Coords *coords2, const int atom2,
                    const double weight)
{
    return(weight * (mysquare(coords2->x[atom2] - coords1->x[atom1]) +
                     mysquare(coords2->y[atom2] - coords1->y[atom1]) + 
                     mysquare(coords2->z[atom2] - coords1->z[atom1])));
}


double
SqrPDBCoordsDist(PDBCoords *coords1, int atom1, PDBCoords *coords2, int atom2)
{
    double          xdist, ydist, zdist;

    xdist = coords2->x[atom2] - coords1->x[atom1];
    ydist = coords2->y[atom2] - coords1->y[atom1];
    zdist = coords2->z[atom2] - coords1->z[atom1];

    return(xdist * xdist + ydist * ydist + zdist * zdist);
}


double
CoordsDist(Coords *coords1, int atom1, Coords *coords2, int atom2)
{
    double           dist;
    double           xdist, ydist, zdist;
    double           xx, yy, zz;
    double           sum;

    xdist = coords2->x[atom2] - coords1->x[atom1];
    ydist = coords2->y[atom2] - coords1->y[atom1];
    zdist = coords2->z[atom2] - coords1->z[atom1];

    xx = xdist * xdist;
    yy = ydist * ydist;
    zz = zdist * zdist;

    sum = xx + yy + zz;
    dist = sqrt(sum);

    return(dist);
}


double
VecMag(const double *vec)
{
    double           dist;
    double           xx, yy, zz;
    double           sum;

    xx = vec[0] * vec[0];
    yy = vec[1] * vec[1];
    zz = vec[2] * vec[2];

    sum = xx + yy + zz;
    dist = sqrt(sum);

    return(dist);
}


double
CoordMag(const Coords *coords, const int vec)
{
    double           dist;
    double           xx, yy, zz;
    double           sum;

    xx = coords->x[vec] * coords->x[vec];
    yy = coords->y[vec] * coords->y[vec];
    zz = coords->z[vec] * coords->z[vec];

    sum = xx + yy + zz;
    dist = sqrt(sum);

    return(dist);
}


double
SqrCoordMag(const Coords *coords, const int vec)
{
    double           xx, yy, zz;
    double           sqrmag;

    xx = coords->x[vec] * coords->x[vec];
    yy = coords->y[vec] * coords->y[vec];
    zz = coords->z[vec] * coords->z[vec];

    sqrmag = xx + yy + zz;

    return(sqrmag);
}


double
CoordMult(const Coords *coords1, const Coords *coords2, const int vec)
{
    double           xx, yy, zz;
    double           mag;

    xx = coords1->x[vec] * coords2->x[vec];
    yy = coords1->y[vec] * coords2->y[vec];
    zz = coords1->z[vec] * coords2->z[vec];

    mag = xx + yy + zz;

    return(mag);
}


double
RadiusGyration(Coords *coords, const double *weights)
{
    double          sum;
    int             i;

    sum = 0.0;

    for (i = 0; i < coords->vlen; ++i)
        sum += (weights[i] * SqrCoordMag(coords, i));

    coords->radgyr = sqrt(sum / coords->vlen);
    return(coords->radgyr);
}


double
TraceCoords(const Coords *coords1, const Coords *coords2, const double *weights)
{
    double          sum;
    int             i;

    sum = 0.0;
    for (i = 0; i < coords1->vlen; ++i)
        sum += (weights[i] * CoordMult(coords1, coords2, i));
    sum /= coords1->vlen;

    return(sqrt(sum));
}


/* calculate all weighted residuals */
void
CalcResiduals(CoordsArray *cdsA)
{
    int             i, j;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const double   *w = (const double *) cdsA->w;
    double         *sqrtw = malloc(cdsA->vlen * sizeof(double));
    double          weight;
    const double   *avex = (const double *) cdsA->avecoords->x,
                   *avey = (const double *) cdsA->avecoords->y,
                   *avez = (const double *) cdsA->avecoords->z;
    Coords         *coords;

    for (j = 0; j < vlen; ++j)
        sqrtw[j] = sqrt(w[j]);

    for (i = 0; i < cnum; ++i)
    {
        for (j = 0; j < vlen; ++j)
        {
            weight = /* sqrtw[j] */ 1.0;
            coords = cdsA->coords[i];
            coords->residual_x[j] = weight * (coords->x[j] - avex[j]);
            coords->residual_y[j] = weight * (coords->y[j] - avey[j]);
            coords->residual_z[j] = weight * (coords->z[j] - avez[j]);
        }
    }

    free(sqrtw);
    /* StudentizeResiduals(cdsA); */
}


void
StudentizeResiduals(CoordsArray *cdsA)
{
    int             i, j;
    double          sum, h, tmp;
    const double    ninv = 1.0 / cdsA->vlen;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const Coords  **coords = (const Coords **) cdsA->coords;
    const double   *var = (const double *) cdsA->var;
    const double   *avex = (const double *) cdsA->avecoords->x,
                   *avey = (const double *) cdsA->avecoords->y,
                   *avez = (const double *) cdsA->avecoords->z;

    for (i = 0; i < cnum; ++i)
    {
        for (j = 0; j < vlen; ++j)
        {
            coords[i]->residual_x[j] = (avex[j] - coords[i]->x[j]);
            coords[i]->residual_y[j] = (avey[j] - coords[i]->y[j]);
            coords[i]->residual_z[j] = (avez[j] - coords[i]->z[j]);
        }
    }

    sum = 0.0;
    for (i = 0; i < cnum; ++i)
    {
        for (j = 0; j < vlen; ++j)
        {
            sum += mysquare(coords[i]->residual_x[j]);
            sum += mysquare(coords[i]->residual_y[j]);
            sum += mysquare(coords[i]->residual_z[j]);
        }
    }
    sum /= 3.0;

    for (i = 0; i < cnum; ++i)
    {
        for (j = 0; j < vlen; ++j)
        {
            h = ninv + mysquare(coords[i]->residual_x[j]) / sum;
            tmp = var[j] * (1.0 - h);
            coords[i]->residual_x[j] /= tmp;
            coords[i]->residual_y[j] /= tmp;
            coords[i]->residual_z[j] /= tmp;
        }
    }
}


void
PrintResiduals(CoordsArray *cdsA)
{
    int             i, j;
    const Coords  **coords = (const Coords **) cdsA->coords;

    putchar('\n');
    for (i = 0; i < cdsA->cnum; ++i)
    {
        for (j = 0; j < cdsA->vlen; ++j)
        {   
            fprintf(stderr, "\n%-3d %12.6f %12.6f %12.6f",
                    j+1,
                    coords[i]->residual_x[j],
                    coords[i]->residual_y[j],
                    coords[i]->residual_z[j]);
        }
    }
}


void
WriteResiduals(CoordsArray *cdsA, char *outfile_name)
{
    FILE           *residualfile = NULL;
    int             i;

    residualfile = myfopen(outfile_name, "w");
    if (residualfile == NULL)
    {
        perror("\n  ERROR");
        fprintf(stderr,
                "\n  ERROR99: could not open file '%s'. \n", outfile_name);
        PrintTheseusTag();
        exit(EXIT_FAILURE);
    }

    for (i = 0; i < cdsA->cnum * cdsA->vlen * 3; ++i)
    {
		fprintf(residualfile,
				"%-3d %12.6f\n",
				i+1,
				cdsA->residuals[i]);
    }
    fputc('\n', residualfile);

    fclose(residualfile);
}


double
Durbin_Watson(CoordsArray *cdsA)
{
    int             i, j, jm;
    double          sumn, sumd;
    Coords         *coords;

    sumn = 0.0;
    for (i = 0; i < cdsA->cnum; ++i)
    {
        for (j = 1; j < cdsA->vlen; ++j)
        {
            coords = cdsA->coords[i];
            jm = j-1;
            sumn += mysquare(coords->residual_x[j] - coords->residual_x[jm]);
            sumn += mysquare(coords->residual_y[j] - coords->residual_y[jm]);
            sumn += mysquare(coords->residual_z[j] - coords->residual_z[jm]);
        }
    }

    sumd = 0.0;
    for (i = 0; i < cdsA->cnum; ++i)
    {
        for (j = 0; j < cdsA->vlen; ++j)
        {
            coords = cdsA->coords[i];
            sumd += mysquare(coords->residual_x[j]);
            sumd += mysquare(coords->residual_y[j]);
            sumd += mysquare(coords->residual_z[j]);
        }
    }

    cdsA->stats->dw = sumn / sumd;

    return(cdsA->stats->dw);
}


void
MomentsCoords(CoordsArray *cdsA)
{
    double ave, median, adev, mdev, sdev, var, skew, kurt, hrange, lrange;

    moments((const double *) cdsA->residuals,
            cdsA->cnum * cdsA->vlen * 3, /* data array and length */
            &ave, &median,       /* mean and median */
            &adev, &mdev,        /* ave dev from mean and median */
            &sdev, &var,         /* std deviation, variance */
            &skew, &kurt,        /* skewness and kurtosis */
            &hrange, &lrange);   /* range of data, high and low */

    cdsA->stats->skewness[3] = skew;
    cdsA->stats->kurtosis[3] = kurt;
}


void
SkewnessCoords(CoordsArray *cdsA)
{
    int             i;
    double          skew;

    skew = 0.0;
    for (i = 0; i < cdsA->cnum * cdsA->vlen * 3; ++i)
        skew += mycube(cdsA->residuals[i]);

    cdsA->stats->skewness[3] = skew / (cdsA->cnum * cdsA->vlen * 3);
}


/* find the multivariate normal skewness
   Mardia, K.V. (1970) "Measures of multivariate skewness and kurtosis with applications."
   Biometrika 57, 519530.
*/


/* takes a dataset and finds the kurtosis */
void
KurtosisCoords(CoordsArray *cdsA)
{
    int             i;
    double          kurt;

    kurt = 0.0;
    for (i = 0; i < cdsA->cnum * cdsA->vlen * 3; ++i)
        kurt += mypow4(cdsA->residuals[i]);

    cdsA->stats->kurtosis[3] = kurt / (cdsA->cnum * cdsA->vlen * 3) - 3.0;
}


double
CoordxMatxCoord(const double *v1, const double *v2, const double **sigma)
{
    int             j, k;
    double          vm[3] = {0.0, 0.0, 0.0};
    double          val;

    for (j = 0; j < 3; ++j)
    {
        vm[j] = 0.0;
        for (k = 0; k < 3; ++k)
            vm[j] += (v1[k] * sigma[k][j]);
    }

    val = 0.0;
    for (j = 0; j < 3; ++j)
        val += (vm[j] * v2[j]);

    /* printf("\n %f", val); */

    return(val);
}


double
CalcANOVAF(CoordsArray *cdsA)
{
    int                i, j;
    double            *array1, *array2;
    long unsigned int  signsum;
    int                wilcoxZplus, mean, sigma;
    CoordsArray       *anova;

    anova = CoordsArrayInit();
    CoordsArrayAlloc(anova, (cdsA->cnum * 2), cdsA->vlen);
    anova->algo->method   = cdsA->algo->method;
    anova->algo->writestats = 0;

    for (i = 0; i < cdsA->cnum; ++i)
    {
        for (j = 0; j < cdsA->vlen; ++j)
        {
            anova->coords[i]->x[j] = cdsA->coords[i]->x[j];
            anova->coords[i]->y[j] = cdsA->coords[i]->y[j];
            anova->coords[i]->z[j] = cdsA->coords[i]->z[j];
        }
    }

    for (i = 0; i < cdsA->cnum; ++i)
    {
        for (j = 0; j < cdsA->vlen; ++j)
        {
            anova->coords[i + cdsA->cnum]->x[j] = -(cdsA->coords[i]->x[j]);
            anova->coords[i + cdsA->cnum]->y[j] = cdsA->coords[i]->y[j];
            anova->coords[i + cdsA->cnum]->z[j] = cdsA->coords[i]->z[j];
        }
    }

    /*for (i = 0; i < anova->cnum; ++i)
        PrintCoords(anova->coords[i]);
    fflush(NULL);*/

    AveCoords(anova);
    cdsA->algo->rounds = MultiPose(anova);

    /* Kolmogorov-Smirnov distribution comparison */
    array1 = malloc((cdsA->vlen+1) * sizeof(double));
    array2 = malloc((cdsA->vlen+1) * sizeof(double));

    memcpy(array1,  anova->var, cdsA->vlen * sizeof(double));
    memcpy(array2, cdsA->var, cdsA->vlen * sizeof(double));

/*     for (i = 0; i < cdsA->vlen; ++i) */
/*     { */
/*         array1[i] = anova->var[i]; */
/*         array2[i] = cdsA->var[i]; */
/*     } */

    cdsA->stats->KSp = kstwo(array1, cdsA->vlen, array2, cdsA->vlen) / 2.0;

    /* one-tailed, paired sign-test distribution comparison */
    signsum = 0.0;
    for (i = 0; i < cdsA->vlen; ++i)
    {
        if (anova->var[i] > cdsA->var[i])
            ++signsum;
    }

    cdsA->stats->signp = Binomial_sum((long unsigned int) cdsA->vlen, signsum, 0.5);

    /* one-tailed, Wilcoxon ranked sign test */
    for (i = 0; i < cdsA->vlen; ++i)
    {
        array2[i] = anova->var[i] - cdsA->var[i];
        array1[i] = fabs(array1[i]);
    }

    array1[cdsA->vlen] = array2[cdsA->vlen] = DBL_MAX;
    quicksort2d(array1, array2, cdsA->vlen);

    wilcoxZplus = 0;
    for (i = 0; i < cdsA->vlen; ++i)
    {
        if(array2[i] > 0.0)
            wilcoxZplus += i;
    }

    sigma = sqrt((double)cdsA->vlen * ((double)cdsA->vlen + 1) * (2.0 * (double)cdsA->vlen + 1) / 24.0);
    mean = (double)cdsA->vlen * ((double)cdsA->vlen + 1) / 4.0;
    cdsA->stats->wilcoxonp = normal_pdf((double)wilcoxZplus, mean, mysquare(sigma));

    cdsA->stats->anova_RMSD = anova->stats->wRMSD_from_mean;

    CalcLogL(anova);
    CalcAIC(anova);

    cdsA->stats->anova_logL = anova->stats->logL;
    cdsA->stats->anova_AIC = anova->stats->AIC;

    CoordsArrayDestroy(anova);
    free(array1);
    free(array2);

    return(cdsA->stats->refl_RMSD);
}


void
CalcNormResidualsOld(CoordsArray *cdsA)
{
    int             i, j, k, m;
    double          logL, rootv, dimwtx, dimwty, dimwtz;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const double   *var = (const double *) cdsA->var;
    const double  **AxCovMat = (const double **) cdsA->AxCovMat;
    const Coords  **coords = (const Coords **) cdsA->coords;
    const Coords   *coordsm;
    const double   *avex = (const double *) cdsA->avecoords->x,
                   *avey = (const double *) cdsA->avecoords->y,
                   *avez = (const double *) cdsA->avecoords->z;
    double         *invvar = malloc(vlen * sizeof(double));
    double         *normresid;

    if (cdsA->residuals == NULL)
        cdsA->residuals = malloc(vlen * 3 * cnum * sizeof(double));

    normresid = cdsA->residuals;

    for (i = 0; i < vlen; ++i)
        invvar[i] = 1.0 / var[i];

    dimwtx = sqrt(1.0 / AxCovMat[0][0]);
    dimwty = sqrt(1.0 / AxCovMat[1][1]);
    dimwtz = sqrt(1.0 / AxCovMat[2][2]);

    /* memset(&SumMat[0][0], 0, 9 * sizeof(double)); */
    /* printf("\n%f %f %f", 1.0 / AxCovMat[0][0], 1.0 / AxCovMat[1][1], 1.0 / AxCovMat[2][2]); */
    j = 0;
    
    for (k = 0; k < vlen; ++k)
    {
        rootv = sqrt(invvar[k]);
        for (m = 0; m < cnum; ++m)
        {
            coordsm = (Coords *) coords[m];
            normresid[j] = (coordsm->x[k] - avex[k]) * dimwtx * rootv;
            ++j;
            normresid[j] = (coordsm->y[k] - avey[k]) * dimwty * rootv;
            ++j;
            normresid[j] = (coordsm->z[k] - avez[k]) * dimwtz * rootv;
            ++j;
        }
    }

    cdsA->stats->chi2 = chi_sqr_adapt(normresid, vlen * 3 * cnum, 0, &logL, 0.0, 1.0, normal_pdf, normal_lnpdf, normal_int);

    free(invvar);
}


void
CalcNormResiduals(CoordsArray *cdsA)
{
    int             j, k, m;
    double          logL, rootv, dimwtx, dimwty, dimwtz;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const double   *var = (const double *) cdsA->var;
    const double  **AxCovMat = (const double **) cdsA->AxCovMat;
    const Coords  **coords = (const Coords **) cdsA->coords;
    const Coords   *coordsm;
    const double   *avex = (const double *) cdsA->avecoords->x,
                   *avey = (const double *) cdsA->avecoords->y,
                   *avez = (const double *) cdsA->avecoords->z;
    double         *normresid;

    if (cdsA->residuals == NULL)
        cdsA->residuals = malloc(vlen * 3 * cnum * sizeof(double));

    normresid = cdsA->residuals;

    dimwtx = sqrt(1.0 / AxCovMat[0][0]);
    dimwty = sqrt(1.0 / AxCovMat[1][1]);
    dimwtz = sqrt(1.0 / AxCovMat[2][2]);

    j = 0;
    for (k = 0; k < vlen; ++k)
    {
        rootv = sqrt(1.0 / var[k]);

        myassert(isfinite(rootv));
        myassert(rootv > DBL_EPSILON);

        for (m = 0; m < cnum; ++m)
        {
            coordsm = (Coords *) coords[m];

            if (coordsm->o[k] == 1)
            {
				normresid[j] = (coordsm->x[k] - avex[k]) * dimwtx * rootv;
				++j;
				normresid[j] = (coordsm->y[k] - avey[k]) * dimwty * rootv;
				++j;
				normresid[j] = (coordsm->z[k] - avez[k]) * dimwtz * rootv;
				++j;
            }
        }
    }

    //VecPrint(normresid, j-1); exit(1);

    cdsA->stats->chi2 = chi_sqr_adapt(normresid, j-1, 0, &logL, 0.0, 1.0, normal_pdf, normal_lnpdf, normal_int);
//    printf("\nchi^2: %f\n", cdsA->stats->chi2);
}


void
CalcNormResidualsLS(CoordsArray *cdsA)
{
    int             j, k, m;
    double          logL, dimwtx, dimwty, dimwtz, avevar;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const double   *var = (const double *) cdsA->var;
    const double  **AxCovMat = (const double **) cdsA->AxCovMat;
    const Coords  **coords = (const Coords **) cdsA->coords;
    const Coords   *coordsm;
    const double   *avex = (const double *) cdsA->avecoords->x,
                   *avey = (const double *) cdsA->avecoords->y,
                   *avez = (const double *) cdsA->avecoords->z;
    double         *normresid;

    if (cdsA->residuals == NULL)
        cdsA->residuals = malloc(vlen * 3 * cnum * sizeof(double));

    normresid = cdsA->residuals;

    avevar = 0.0;
    for (m = 0; m < vlen; ++m)
        avevar += var[m];
    avevar /= vlen;
//    printf("\n##### %f %f", avevar, cdsA->stats->stddev*cdsA->stats->stddev); fflush(NULL);
    dimwtx = sqrt(1.0 / (AxCovMat[0][0] * avevar));
    dimwty = sqrt(1.0 / (AxCovMat[1][1] * avevar));
    dimwtz = sqrt(1.0 / (AxCovMat[2][2] * avevar));
    // printf("\nHERE %d %d", vlen, cnum); fflush(NULL);
    j = 0;
    for (k = 0; k < vlen; ++k)
    {
        for (m = 0; m < cnum; ++m)
        {
            coordsm = (Coords *) coords[m];
            // printf("\n%4d %4d %f", k, m, coordsm->o[k]); fflush(NULL);
            if (coordsm->o[k] == 1)
            {
				normresid[j] = (coordsm->x[k] - avex[k]) * dimwtx;
				++j;
				normresid[j] = (coordsm->y[k] - avey[k]) * dimwty;
				++j;
				normresid[j] = (coordsm->z[k] - avez[k]) * dimwtz;
				++j;
				// printf("\n%4d %f", j, normresid[j]); fflush(NULL);
            }
        }
    }

    //VecPrint(normresid, j-1); exit(1);

    cdsA->stats->chi2 = chi_sqr_adapt(normresid, j-1, 0, &logL, 0.0, 1.0, normal_pdf, normal_lnpdf, normal_int);
//    printf("\nchi^2: %f\n", cdsA->stats->chi2);
}


double
FrobTerm(CoordsArray *cdsA)
{
    int             i;
    double          trace, logL;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const int       len = vlen * 3 * cnum;
    double         *residuals = cdsA->residuals;

    trace = 0.0;
    for (i = 0; i < len; ++i)
        trace += residuals[i] * residuals[i];

    cdsA->stats->chi2 = chi_sqr_adapt(residuals, len, 0, &logL, 0.0, 1.0, normal_pdf, normal_lnpdf, normal_int);

    return(-0.5 * trace);
}


double
FrobTerm2(CoordsArray *cdsA)
{
    int             i, j, k, m;
    double          trace, dimwt;
    const int       cnum = cdsA->cnum, vlen = cdsA->vlen;
    const double  **AxCovMat = (const double **) cdsA->AxCovMat;
    const Coords  **coords = (const Coords **) cdsA->coords;
    const Coords   *coordsi;
    const double   *avex = (const double *) cdsA->avecoords->x,
                   *avey = (const double *) cdsA->avecoords->y,
                   *avez = (const double *) cdsA->avecoords->z;
    double        **ErrMat = MatInit(vlen, 3);
    double        **TmpMat = MatInit(3, vlen);
    double        **SumMat = MatInit(3, 3);
    double        **InvCovMat = MatInit(vlen, vlen);

    memset(&SumMat[0][0], 0, 9 * sizeof(double));

    pseudoinv_sym(cdsA->CovMat, InvCovMat, vlen, DBL_MIN);

    for (m = 0; m < cnum; ++m)
    {
        for (i = 0; i < vlen; ++i)
        {
            coordsi = (Coords *) coords[m];
            ErrMat[i][0] = coordsi->x[i] - avex[i];
            ErrMat[i][1] = coordsi->y[i] - avey[i];
            ErrMat[i][2] = coordsi->z[i] - avez[i];
        }

        /* (i x k)(k x j) = (i x j) */
        for (i = 0; i < 3; ++i)
        {
            for (j = 0; j < vlen; ++j)
            {
                TmpMat[i][j] = 0.0;
                for (k = 0; k < vlen; ++k)
                    TmpMat[i][j] += ErrMat[k][i] * InvCovMat[k][j];
            }
        }

        for (i = 0; i < 3; ++i)
        {
            for (j = 0; j < 3; ++j)
            {
                dimwt = 1.0 / AxCovMat[j][j];
                for (k = 0; k < vlen; ++k)
                    SumMat[i][j] += TmpMat[i][k] * ErrMat[k][j] * dimwt ;
            }
        }
    }

    trace = SumMat[0][0] + SumMat[1][1] + SumMat[2][2];

    MatDestroy(ErrMat);
    MatDestroy(SumMat);
    MatDestroy(TmpMat);
    MatDestroy(InvCovMat);

    return(-0.5 * trace);
}


double
CalcHierarchLogL(CoordsArray *cdsA)
{
    Algorithm      *algo = cdsA->algo;
    Statistics     *stats = cdsA->stats;
    const int       vlen = cdsA->vlen, cnum = cdsA->cnum;
    const int       nd = cnum * 3;
    /* double          logL; */

    switch(algo->hierarch)
    {
        case 0:
        case 15:
        case 16:
            return(0.0);
            break;

        case 1: /* ML fit of variances to an inverse gamma distribution - 2 param */
            if (algo->varweight != 0)
            {
                double *newvar = malloc(vlen * sizeof(double));
                double logL;

				memcpy(newvar, cdsA->var, vlen * sizeof(double));
				qsort(newvar, vlen, sizeof(double), dblcmp_rev);
				logL = invgamma_logL(newvar, vlen - 3, stats->hierarch_p1, stats->hierarch_p2);
				free(newvar);

				return(logL);
				/* return(dist_logL(invgamma_lnpdf, stats->hierarch_p1, stats->hierarch_p2, cdsA->var, cdsA->vlen - 3)); */
				/* invgamma_fit(newvar, cdsA->vlen - 3, &stats->hierarch_p1, &stats->hierarch_p12, &logL); */
				/* return(cdsA->vlen * invgamma_logL(stats->hierarch_p1, stats->hierarch_p2)); */
            }
            else if (cdsA->algo->covweight != 0)
            {
				double **evecs = cdsA->tmpmatKK2;
				int newlen;
				double logL;

				if (vlen - 3 < nd - 6)
					newlen = vlen - 3;
				else
					newlen = nd - 6;

				eigenvalsym2((const double **) cdsA->CovMat, cdsA->var, evecs, vlen, &cdsA->tmpmatKK1[0][0]);
				logL = invgamma_logL(cdsA->var + vlen - newlen, newlen, stats->hierarch_p1, stats->hierarch_p2);

				return(logL);
            }
            break;

        case 2:
        case 3:
        case 4:
        case 5:
        case 6:
        case 7:
        case 8:
        case 9:
        case 11:
        case 12:
        case 13:
        case 14:
        case 17:
        case 18:
            return(invgamma_logL(cdsA->var, cdsA->vlen, stats->hierarch_p1, stats->hierarch_p2));
            break;

//        case 8: /* Reciprocal Inverse Gaussian */ /* DLT debug */
//            /* return(dist_logL(recinvgauss_lnpdf, stats->hierarch_p1, stats->hierarch_p2, cdsA->var, cdsA->vlen)); */
//            break;

 //       case 9: /* Lognormal */
 //           return(cdsA->vlen * lognormal_logL(stats->hierarch_p1, stats->hierarch_p2));
 //           break;

//        case 10:
//            return(dist_logL(invgauss_lnpdf, stats->hierarch_p1, stats->hierarch_p2, cdsA->var, cdsA->vlen));
//            break;

        default:
            printf("\n  ERROR:  Bad -g option \"%d\" \n", algo->hierarch);
            Usage(0);
            exit(EXIT_FAILURE);
            break;
    }

    return(0.0);
}


/* Calculates the likelihood for a specified Gaussian model, given a
   structural superposition.

     NOTA BENE: This function assumes that the variances, covariance matrices,
     hierarchical model parameters, average coordinates, rotations, and 
     translations have all been pre-calculated. Even when not calculating the
     optimal ML rotations and translation transformations, the other parameters
     in general must be estimated iteratively, as described below.

   This is not nearly as trivial as it may first appear. For the dimensionally
   weighted case, this involves an iterative ML estimate of the covariance
   matrices, even when the atomic row-wise matrix is assumed to be diagonal or
   proportional to the identity matrix. The way I do it, the superposition as a
   whole is rotated to bring it into alignment with the principal axes of the
   dimensional covariance matrix. Furthermore, the first term of the likelihood
   equation (the Mahalonobius Frobenius matrix norm term) is normally equal to
   NKD/2 at the maximum. However, when using shrinkage or hierarchical estimates
   of the covariance matrices, this convenient simplification no longer holds,
   and the double matrix-weighted Frobenius norm must be calcualted explicitly.
*/
double
CalcLogL(CoordsArray *cdsA)
{
    const double    vlen = cdsA->vlen;
    const double    cnum = cdsA->cnum;
    const double    nk = cnum * vlen;
    const double    nd = cnum * 3.0;
    const double    ndk = nk * 3.0;
    const double    ndk2 = 0.5 * ndk;
    const double   *var = (const double *) cdsA->var;
    double          lndetdim, lndetrow , frobterm, igL;
    Algorithm      *algo = cdsA->algo;
    Statistics     *stats = cdsA->stats;
    int             i;

    lndetdim = lndetrow = frobterm = igL = 0.0;

    if (algo->dimweight == 1)
    {
        lndetdim = 0.0;
        for (i = 0; i < 3; ++i)
            lndetdim += log(cdsA->AxCovMat[i][i]);
    }

    if (algo->leastsquares == 1)
    {
        frobterm = FrobTerm(cdsA);
        lndetrow = 2.0 * vlen * log(cdsA->stats->stddev);
    }
    else if (algo->varweight == 1)
    {
        lndetrow = 0.0;
        for (i = 0; i < vlen; ++i)
            lndetrow += log(var[i]);
/* 		for (i = 0; i < vlen; ++i) */
/* 			printf("\nevals: % 12.6e", var[i]); */
        if (algo->hierarch != 0)
        {
            frobterm = FrobTerm(cdsA);
            igL = CalcHierarchLogL(cdsA);
        }
        else
            frobterm = -ndk2 /* FrobTerm(cdsA) */;
    }
    else if (algo->covweight == 1)
    {
        lndetrow = MatSymLnDet((const double **) cdsA->CovMat, vlen);

        if (algo->hierarch != 0)
        {
            frobterm = FrobTerm2(cdsA);
            igL = CalcHierarchLogL(cdsA);
        }
        else
            frobterm = -ndk2;
    }

/* printf("\n  * Axial variances:                [ %5.3f  %5.3f  %5.3f ] * ", */
/*     cdsA->AxCovMat[0][0], cdsA->AxCovMat[1][1], cdsA->AxCovMat[2][2]); */

    if (algo->verbose == 1)
    {
        printf("!     frobterm        -ndk2          igL     lndetrow     lndetdim         covs\n");
        printf("! % 12.4f % 12.4f % 12.4f % 12.4f % 12.4f % 12.4f\n",
                frobterm, -ndk2, igL, - 0.5 * nd * lndetrow, - 0.5 * nk * lndetdim,
                - 0.5 * nd * lndetrow - 0.5 * nk * lndetdim);
    }

/* printf("\n _>_>_>_>_>_>_>_>_> Frobterm: %f, -NDK/2: %f", FrobTerm(cdsA), -ndk2); */

    stats->logL = frobterm - ndk2 * log(2.0*MY_PI) - 0.5 * nd * lndetrow - 0.5 * nk * lndetdim + igL;

    return(stats->logL);
}


/* Calculate number of parameters fit for the specified algorithm and model */
double
CalcParamNum(CoordsArray *cdsA)
{
    Algorithm      *algo = cdsA->algo;
    const double    vlen = cdsA->vlen;
    const double    cnum = cdsA->cnum;
    double          params;

    params = 0.0;

    /* for the atomic covariances/variances */
    if (algo->leastsquares == 1)
        params += 1.0;

    if (algo->varweight != 0)
        params += vlen;

    if (algo->covweight != 0)
        params += vlen * (vlen + 1.0) / 2.0;

    /* for the dimensional covmat (eigenvalues only) */
    if (algo->dimweight != 0)
        params += 3.0;

    /* for the hierarchical parameters */
    switch(algo->hierarch)
    {
        case 0:
            break;

        case 1: /* ML fit of variances to an inverse gamma distribution - 2 param */
        case 2:
        case 5:
        case 6:
        case 7:
        case 8:
        case 9:
        case 10:
        case 11:
        case 12:
        case 13:
            params += 2.0;
            break;

        case 3: /* ML fit of variances to an inverse gamma distribution - 1 param */
        case 4:
            params += 1.0;
            break;
    }

    /* for the mean */
    if (algo->noave == 0)
        params += 3.0 * vlen;

    /* translations */
    if (algo->notrans == 0)
        params += 3.0 * cnum;

    /* rotations */
    if (algo->norot == 0)
        params += 3.0 * cnum;

    return(params);
}


void
CalcAIC(CoordsArray *cdsA)
{
    double         n, p;

    cdsA->stats->nparams = p = CalcParamNum(cdsA);
    cdsA->stats->ndata = n = 3.0 * cdsA->cnum * cdsA->vlen;

    cdsA->stats->AIC = cdsA->stats->logL - p * n / (n - p - 1);
}


void
CalcBIC(CoordsArray *cdsA)
{
    double         n, p;

    p = CalcParamNum(cdsA);
    n = 3.0 * cdsA->cnum * cdsA->vlen;

    cdsA->stats->BIC = cdsA->stats->logL - (log(n) * p / 2.0);
}


/* Calculate the trace of the inner product of some coordinates.
   This is the same as the squared radius of gyration without normalization
   for the number of atoms. */
double
TrCoordsInnerProd(Coords *coords, const int len)
{
    int             i;
    double          sum, tmpx, tmpy, tmpz;

    sum = 0.0;
    for (i = 0; i < len; ++i)
    {
        tmpx = coords->x[i];
        tmpy = coords->y[i];
        tmpz = coords->z[i];
        sum += (tmpx*tmpx + tmpy*tmpy + tmpz*tmpz);
    }

    return(sum);
}


void
UnbiasMean(CoordsArray *scratchA)
{
    const int       vlen = scratchA->vlen;
    int             i;
    double          term, trsiginv;
    Coords         *coords = scratchA->avecoords;

    trsiginv = 0.0;
    for (i = 0; i < vlen; ++i)
    {
        trsiginv += (coords->x[i] * coords->x[i] +
                     coords->y[i] * coords->y[i] +
                     coords->z[i] * coords->z[i]) / scratchA->var[i];
    }

    term = 1.0 - 6.0 / trsiginv;

    if (term > 0.0)
        term = sqrt(term);
    else
        term = 0.0;

    printf(" bias constant of the mean % f % f\n", trsiginv, term);
    fflush(NULL);

    for (i = 0; i < vlen; ++i)
    {
        coords->x[i] *= term;
        coords->y[i] *= term;
        coords->z[i] *= term;
    }
}
